# -*- coding: utf-8 -*-
#####################################################################
#  Rafael Proença <cypherbios@ubuntu.com>
#  Laudeci Oliveira <laudeci@gmail.com> 
#
#  Copyright 2006 APTonCD DevTeam.
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published
#  by the Free Software Foundation; version 2 only.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#####################################################################
import gtk
import time
import os
import gobject

from APTonCD.core import *
from APTonCD.core import gui 
from APTonCD.core import constants
from APTonCD.core import utils
from APTonCD.widgets import *
from APTonCD.widgets.progresswindow import ProgressDialog
from APTonCD.restore import restoreFromDialog
from APTonCD.core import dbus_helper
from APTonCD.core import mediainfo

WINDOW_NAME = 'restore'

class RestoreWindow(controller.IController):
    """
        Restore window
    """
    def __init__(self, controller = None, temp_path = '' , packages_path = '', filename = ''):
        """
            Constructor
        """
        self.controller = controller
        self.glade = gui.get_glade(constants.RESTORE_GUI, WINDOW_NAME)
        
        self.temp_path = temp_path
        self.cd_path = ''
        self.root_shell = None
        if self.temp_path == '' or self.temp_path == None:
            self.temp_path = constants.TEMP_FOLDER
        else:
            self.temp_path = temp_path

        self.packages_path = packages_path
        self.isoFileName = filename
        self.restoretype = constants.RESTORE_TYPE.RESTORE_FROM_NONE
        
        self.window = gui.get_widget(self.glade, WINDOW_NAME)
        if self.controller:
            self.window.set_decorated(False)
            self.content = gui.set_parent_widget_from_glade(self.glade, WINDOW_NAME, self.controller.restoreContainer)
            self.window.destroy()
            gui.setCursorToBusy(self.controller.get_main_window(), True)
        else:
            #Window Widgets
            self.window.show()
            gui.setCursorToBusy(self.window, True)
        
        self.ListContainer = gui.get_widget(self.glade,'ListContainer')
       
        self.lblPkgDesc = gui.get_widget(self.glade, 'lblPkgDesc')
        self.pkgImg = gui.get_widget(self.glade, 'pkgImg')

        self.btnRestorePackages = gui.get_widget(self.glade, 'restorePackages')
        self.btnLoadFrom = gui.get_widget(self.glade, 'btnLoadFrom')
        self.btnRestorePackages.set_sensitive(False)


        self.packageList = PackageList.PackageView(self)
        self.ListContainer.add(self.packageList)

        self.ListContainer.show_all()
        gui.processEvents()
        
        if self.controller:
            self.controller.show_status_message(constants.MESSAGE_0001)
            
        #self.packageList.load_package_folder(self.packages_path)
        
        self.bind_signals()
        
        if self.isoFileName != '':
            self.restoreFromIso(self.isoFileName, self.temp_path)

    def bind_signals(self):
        gui.connect(self.btnLoadFrom, 'clicked', self.on_btnLoadFrom)
        gui.connect(self.btnRestorePackages, 'clicked', self.on_btnRestorePackages)
    
    def rebuild_list(self):
        self.packageList.clear()
        self.packageList.load_package_folder(self.packages_path)
        
    def get_main_window(self):
        """
            implements IController interface method.
            for access the class windows object.
        """
        if self.controller:
            return self.controller.get_main_window()
        else:
            return self.window
        
    def get_parent_widget(self):
        """
            implements IController interface method.
            for container widget
        """
        if self.controller:
            return self.controller.get_parent_widget()
        else:
            return self.ListContainer
    
    def get_parent(self):
        return self.controller
        
    def show_selected(self, text, img):
        self.lblPkgDesc.set_markup(text)
        self.pkgImg.set_from_pixbuf(img)
        
    def show_selected_count(self, value):
        if self.controller:
            self.controller.show_status_message(constants.MESSAGE_0007 % value)
        
    def show(self):
        self.window.show()
    
    def hide(self):
        self.window.hide()


    def copy_files(self):
        gui.setCursorToBusy(self.get_parent_widget(),True)
        self.controller.get_main_window().set_sensitive(False)
        
        progress = self.get_parent().progressbar
        progress.show()
        
        self.controller.show_status_message('preparing to copy files')
        script_file =os.path.join(constants.TEMP_FOLDER,'aoc.sc')
        ifile = open(script_file, 'w')
        indx = 1
        self.file_count = 1
        f_len = len(self.packageList.store)
        for nItens in self.packageList.store:
            if nItens[0]:
                pkg = nItens[PackageList.COL_PACKAGE]
                source = pkg.deb_full_filename
                destination =  utils.join_path(constants.LOCAL_APT_FOLDER, pkg.deb_filename)
                if not utils.fileExist(destination):
                    ifile.write('%s|%s\n' % (source , destination)) 
                    self.file_count += 1
            percent = (float(indx) / f_len)
            progress.set_fraction(percent)
            progress.set_text(str(int(percent * 100)) + '%' )
            gui.processEvents()
            indx +=1
        ifile.close()
        
        self.controller.show_status_message('Wait while your system is being restored.')
        progress.hide()
        gui.processEvents()
        
        parentwd = self.controller.get_main_window()
        winid = parentwd.window.xid
        
        file_exec = constants.COPY_SCRIPT
        command = "gksu --desktop " + constants.DESKTOP_FILE + " 'python %s %s %s'" % (file_exec, winid, script_file)
        os.system(command)

        gui.processEvents()
        self.packageList.update_parent_count()
        self.controller.get_main_window().set_sensitive(True)
        gui.setCursorToBusy(self.get_parent_widget())
  
    #Event handler
    def on_btnRestorePackages(self, widget, *args):
        restore_from = ''
        
        if self.restoretype == constants.RESTORE_TYPE.RESTORE_FROM_ISO:
            restore_from = self.temp_path
            result, msg = self.__mount_iso(self.isoFileName)
            if result:
                self.copy_files()
                self.__umount_iso(self.isoFileName)
        elif self.restoretype == constants.RESTORE_TYPE.RESTORE_FROM_MEDIA:
            result, msg = self.is_aptoncd_media(self.cd_path)
            if result:
                self.copy_files()
            else:
                gui.erro_message(constants.MESSAGE_0067,self.get_main_window())
        else:
            gui.warn_message(constants.MESSAGE_0069,self.get_main_window())
    
    def  on_btnLoadFrom(self, widget, *args):
        #load choose dialog
        self.btnRestorePackages.set_sensitive(False)
        bus = dbus_helper.DeviceList()
        
        #get mounted device list
        drives = bus.get_devices()

        dlg = restoreFromDialog.RestoreFrom(self, drives)
        result, self.restoretype, FileName = dlg.run()
        dlg.destroy()
        
        if result:
            if self.restoretype == constants.RESTORE_TYPE.RESTORE_FROM_ISO:
                #default place to mount the iso file
                self.isoFileName = FileName
                self.restoreFromIso(self.isoFileName, self.temp_path)
                
            elif self.restoretype == constants.RESTORE_TYPE.RESTORE_FROM_MEDIA:
                self.cd_path = FileName
                self.restoreFromMedia(self.cd_path)
            else:
                gui.warn_message(constants.MESSAGE_0069,self.get_main_window())

    def is_aptoncd_media(self, from_Path):
        try:
            #creates a cdinfo object to check media
            cdinfo = mediainfo.MediaInfo(utils.join_path(from_Path ,'aptoncd.info'))
            result, msgError = cdinfo.infoFromFile()
            if result:
                isValid, strMsg = cdinfo.compare_version()
                if isValid:
                    return True , strMsg
                else:
                    if gui.question_message(strMsg + '\n\n' + constants.MESSAGE_0071):    
                        return True, strMsg
                    else:    
                        return False , strMsg
            else:
                return False, msgError
        except Exception, eMsg:
            return False, eMsg
    
    def __mount_iso(self, filename):
        
        fromPath = os.path.join(self.temp_path, "aptoncd-mnt-image/")
        print 'mount point = ', fromPath
        print 'iso image = ', filename
        utils.mkdir(fromPath, True)
        
        command = "gksu --desktop " + constants.DESKTOP_FILE + " 'mount -o loop %s %s'" % (filename.replace(' ','\ '), fromPath.replace(' ','\ '))
        ret = utils.run_command(command)
        
        #wait a few seconds so mount would complete
        time.sleep(0.1)
        
        result, msgError = self.is_aptoncd_media(fromPath)
        if not result:
            self.__umount_iso(fromPath)
            return False, msgError
        else:
            return True, ''
    
    def __umount_iso(self, fromPath):
        command = "gksu --desktop " + constants.DESKTOP_FILE + " 'umount %s'" % fromPath.replace(' ','\ ')
        ret = utils.run_command(command)
        utils.removePath(os.path.join(self.temp_path, "aptoncd-mnt-image/"))
    
    def restoreFromMedia(self, path):
        result, msg = self.is_aptoncd_media(path)
        if result:
            self.packageList.load_package_folder(path, recursive = True)
            self.btnRestorePackages.set_sensitive(True)
        else:
            gui.erro_message(constants.MESSAGE_0067,self.get_main_window())
            
    def restoreFromIso(self, filename = '', temp_dir= ''):
        """
            This will restore packages from isofiles.
            Actually it will get all packages from a given iso file location.
        """
        result, msgError = self.__mount_iso(filename)
        
        print result, msgError
        if result:
            self.packageList.clear()
            self.packageList.load_package_folder(os.path.join(self.temp_path, "aptoncd-mnt-image/"), False,  True)
            self.btnRestorePackages.set_sensitive(True)
            self.__umount_iso(os.path.join(self.temp_path, "aptoncd-mnt-image/"))
        else:
        	print msgError   
