/*
    Copyright (C) 2000-2003 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: editor_scrub.cc,v 1.10 2003/12/13 20:53:09 pbd Exp $
*/

#include <cstdlib>
#include <cmath>

#include <string>

#include <pbd/error.h>
#include <pbd/basename.h>
#include <gtkmmext/utils.h>
#include <ardour/audioengine.h>
#include <ardour/session.h>
#include <ardour/playlist.h>
#include <ardour/region.h>
#include <ardour/diskstream.h>
#include <ardour/filesource.h>
#include <ardour/sndfilesource.h>
#include <ardour/utils.h>
#include <ardour/location.h>

#include "ardour_ui.h"
#include "editor.h"
#include "time_axis_view.h"
#include "canvas-simplerect.h"
#include "streamview.h"
#include "regionview.h"
#include "rgb_macros.h"
#include "extra_bind.h"

#include "gtk-custom-hruler.h"

using namespace ARDOUR;
using namespace SigC;
using namespace Gtk;

static struct timeval tv;

void
Editor::start_scrubbing_at (jack_nframes_t frame)
{
	if (session == 0) {
		return;
	}

	scrub_frame = frame;
	last_scrub_frame = frame;
	last_scrub_speed = 0.0;

	gdk_window_set_cursor (track_canvas_scroller.get_window(), wait_cursor);
	gdk_flush ();

	scrub_connection = session->ScrubReady.connect (slot (*this, &Editor::scrub_ready_proxy));
	session->start_scrub (last_scrub_frame);

	double x1, x2, y1, y2;
	jack_nframes_t half = session->scrub_buffer_size() / 2;
	x1 = frame_to_pixel (frame - half);
	x2 = frame_to_pixel (frame + half);
	y1 = 0;
	y2 = canvas_height;

	if (scrub_rect == 0) {
		scrub_rect = gtk_canvas_item_new (gtk_canvas_root (GTK_CANVAS(track_gtk_canvas)),
						  gtk_canvas_simplerect_get_type(),
						  "x1", x1,
						  "x2", x2,
						  "y1", y1,
						  "y2", y2,
						  "outline_color_rgba", RGBA_TO_UINT(0,0,0,255),
						  "fill_color_rgba", RGBA_TO_UINT(24,35,64,98),
						  NULL);
	} else {
		gtk_object_set (GTK_OBJECT(scrub_rect),
				"x1", x1,
				"x2", x2,
				"y1", y1,
				"y2", y2, 
				NULL);
	}
}

void 
Editor::scrub_ready_proxy ()
{
	ARDOUR_UI::instance()->call_slot (slot (*this, &Editor::scrub_ready));
}

void
Editor::scrub_ready ()
{
	cerr << "SCRUB READY, use null cursor\n" << endl;
	scrub_connection.disconnect ();

	_scrubbing = true;
	gdk_window_set_cursor (track_canvas_scroller.get_window(), null_cursor);
	gtk_canvas_item_show (scrub_rect);
	gtk_canvas_item_raise_to_top (scrub_rect);

	gettimeofday (&tv, 0);
	scrub_timeout_tag = gtk_timeout_add (250, &Editor::_check_scrub, this);
}
	
void
Editor::stop_scrubbing_at ()
{
	if (scrub_rect) {
		gtk_canvas_item_hide (scrub_rect);
	}

	if (session == 0 || !_scrubbing) {
		return;
	}

	_scrubbing = false;
	session->stop_scrub ();
	gdk_window_set_cursor (track_canvas_scroller.get_window(), current_canvas_cursor);
	gtk_timeout_remove (scrub_timeout_tag);
}

gint
Editor::_check_scrub (gpointer arg)
{
	return reinterpret_cast<Editor*> (arg)->check_scrub();
}

gint
Editor::check_scrub ()
{
	float new_speed;

	if (!_scrubbing) {
		return FALSE;
	}

	struct timeval now;
	gettimeofday (&now, 0);

	if (scrub_frame == last_scrub_frame) {

		new_speed = 0;
		
	} else {
		float elapsed = ((now.tv_sec - tv.tv_sec) * 1000000 + (now.tv_usec - tv.tv_usec)) / 1000000.0f;
		new_speed = (((signed long) (scrub_frame - last_scrub_frame)) / elapsed) / session->frame_rate();
	}
	
	tv = now;

	new_speed = min (4.0f, new_speed);
	new_speed = max (-4.0f, new_speed);

	if (new_speed != last_scrub_speed) {
		session->set_scrub_speed (new_speed);
		last_scrub_speed = new_speed;
	}

	last_scrub_frame = scrub_frame;

	return TRUE;
}

void
Editor::scrub_motion (jack_nframes_t frame)
{
	scrub_frame = frame;
}

