/*
    Copyright (C) 2001 Paul Davis
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: main.cc,v 1.64 2004/02/22 23:04:00 taybin Exp $
*/

#include <sys/mman.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <cerrno>
#include <cstdlib>
#include <signal.h>
#include <unistd.h>
#include <mcheck.h>

#include <pbd/error.h>
#include <pbd/textreceiver.h>
#include <pbd/platform.h>
#include <pbd/platform_factory.h>
#include <pbd/failed_constructor.h>
#include <pbd/pthread_utils.h>

#include <jack/jack.h>

#include <ardour/version.h>
#include <ardour/ardour.h>
#include <ardour/audioengine.h>

#include <gtk--/main.h>
#include <gtkmmext/popup.h>

#include "version.h"
#include "ardour_ui.h"
#include "opts.h"

#include "i18n.h"

using namespace GTK_ARDOUR;
using namespace ARDOUR;

Transmitter  error (Transmitter::Error);
Transmitter  info (Transmitter::Info);
Transmitter  fatal (Transmitter::Fatal);
Transmitter  warning (Transmitter::Warning);
TextReceiver text_receiver ("ardour");

static jack_client_t *jack_client = 0;

static ARDOUR_UI  *ui = 0;

static void
shutdown (int status)
{
	char *msg;
	
	if (status) {

		msg = _("ardour is killing itself for a clean exit\n");
		write (1, msg, strlen (msg));
		/* drastic, but perhaps necessary */
		kill (-getpgrp(), SIGKILL);	
		/*NOTREACHED*/

	} else {

		if (ui) {
			msg = _("stopping user interface\n");
			write (1, msg, strlen (msg));
			ui->kill();
		}
		
		pthread_cancel_all ();
	}

	msg = _("Ardour is done. Come again. Have a nice day. Etc.\n");
	write (1, msg, strlen (msg));
	exit (status);
}

static void 
handler (int sig)
{
	char buf[64];
	int n;

	/* XXX its doubtful that snprintf() is async-safe */
	n = snprintf (buf, sizeof(buf), _("%d: received signal %d\n"), getpid(), sig);
	write (1, buf, n);

	shutdown (1);
}

static void *
signal_thread (void *arg)
{
	int sig;
	sigset_t blocked;

	pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, 0);
	
	/* find out what's blocked right now */

	sigprocmask (SIG_SETMASK, 0, &blocked);

	/* wait for any signal */

	sigwait (&blocked, &sig);

	cerr << "Signal " << sig << " received\n";

	if (sig != SIGSEGV) {

		/* unblock signals so we can see them during shutdown.
		   this will help prod developers not to lose sight
		   of bugs that cause segfaults etc. during shutdown.
		*/

		sigprocmask (SIG_UNBLOCK, &blocked, 0);
	}

	shutdown (1);
	/*NOTREACHED*/
	return 0;
}

int
catch_signals (void)
{
	struct sigaction action;
	pthread_t signal_thread_id;
	sigset_t signals;

	setsid ();

	sigemptyset (&signals);
	sigaddset(&signals, SIGHUP);
	sigaddset(&signals, SIGINT);
	sigaddset(&signals, SIGQUIT);
	sigaddset(&signals, SIGPIPE);
	sigaddset(&signals, SIGTERM);
	sigaddset(&signals, SIGUSR1);
	sigaddset(&signals, SIGUSR2);


	/* install a handler because otherwise
	   pthreads behaviour is undefined when we enter
	   sigwait.
	*/
	
	action.sa_handler = handler;
	action.sa_mask = signals;
	action.sa_flags = SA_RESTART|SA_RESETHAND;

	for (int i = 1; i < 32; i++) {
		if (sigismember (&signals, i)) {
			if (sigaction (i, &action, 0)) {
				cerr << compose (_("cannot setup signal handling for %1"), i) << endl;
				return -1;
			}
		}
	} 

	/* this sets the signal mask for this and all 
	   subsequent threads that do not reset it.
	*/
	
	if (pthread_sigmask (SIG_SETMASK, &signals, 0)) {
		cerr << compose (_("cannot set default signal mask (%1)"), strerror (errno)) << endl;
		return -1;
	}

	/* start a thread to wait for signals */

	if (pthread_create_and_store ("signal", &signal_thread_id, 0, signal_thread, 0)) {
		cerr << "cannot create signal catching thread" << endl;
		return -1;
	}

	pthread_detach (signal_thread_id);
	return 0;
}

string
which_ui_rcfile ()
{
	string rcfile;

	rcfile = find_config_file ("ardour_ui.rc");
	
	if (rcfile.length() == 0) {
		warning << _("Without a UI style file, ardour will look strange.\n"
			     "Please set ARDOUR_UI_RC to point to a valid UI style file")
			<< endmsg;
	}
	
	return rcfile;
}

gint
show_ui_callback (void *arg)
{
 	ARDOUR_UI * ui = (ARDOUR_UI *) arg;

	ui->hide_splash();
	ui->show ();
	
	return FALSE;
}

gint
jack_fooey (GdkEventAny* ignored)
{
	Gtk::Main::quit ();
	return FALSE;
}

gint
jack_foobar (GdkEventButton *ignored)
{
	Gtk::Main::quit ();
	return FALSE;
}

void
gui_jack_error ()
{
	Gtk::Window win (GTK_WINDOW_DIALOG);
	Gtk::Label label (_("Ardour could not connect to JACK.\n\
There are several possible reasons:\n\
\n\
1) JACK is not running.\n\
2) JACK is running as another user, perhaps root.\n\
3) There is already another client called \"ardour\".\n\
\n\
Please consider the possibilities, and perhaps (re)start JACK."));

	win.set_title (_("ardour: unplugged"));
	win.set_border_width (7);
	win.add (label);
	win.show_all ();
	win.delete_event.connect (SigC::slot (jack_fooey));
	win.add_events (GDK_BUTTON_RELEASE_MASK|GDK_BUTTON_PRESS_MASK);
	win.set_position (GTK_WIN_POS_CENTER);
	win.button_release_event.connect (SigC::slot (jack_foobar));

	Gtk::Main::run ();
}

int
main (int argc, char *argv[])
{
	ARDOUR::AudioEngine *engine;
	
	gtk_set_locale ();
	
	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);

	pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, 0);

	catch_signals ();

	text_receiver.listen_to (error);
	text_receiver.listen_to (info);
	text_receiver.listen_to (fatal);
	text_receiver.listen_to (warning);
	
	if (parse_opts (argc, argv)) {
		exit (1);
	}

	if (getenv("GTK_RC_FILES"))
		unsetenv("GTK_RC_FILES");

	cout << _("Ardour/GTK ") 
	     << gtk_ardour_version
	     << _(" running with libardour ")
	     << ardour_major_version << '.'
	     << ardour_minor_version << '.' 
	     << ardour_micro_version << endl;
	
	if (just_version) {
		exit (0);
	}

	if (no_splash) {
		cerr << _("Copyright 1999-2004 Paul Davis")
		     << endl
		     << _("Ardour comes with ABSOLUTELY NO WARRANTY") << '\n'
		     << _("This is free software, and you are welcome to redistribute it ") << '\n'
		     << _("under certain conditions; see the file COPYING for details")
		     << endl;
	}

	try { 
		ui = new ARDOUR_UI (&argc, &argv, which_ui_rcfile());
	} 

	catch (failed_constructor& err) {
		error << _("could not create ARDOUR GUI") << endmsg;
		exit (1);
	}

	if ((jack_client = jack_client_new (jack_client_name.c_str())) == 0) {
		gui_jack_error ();
		error << compose (_("Could not connect to JACK server as  \"%1\""), jack_client_name) <<  endmsg;
		kill (-getpgrp(), SIGKILL);
		exit (1);
	}

	if (!no_splash) {
		ui->show_splash ();
		if (session_name.length()) {  
			gtk_timeout_add (4000, show_ui_callback, ui);
		}
	}
	
	try { 
		engine = new ARDOUR::AudioEngine (jack_client);
		ARDOUR::init (*engine);
		ui->set_engine (*engine);
	} 
	
	catch (failed_constructor& err) {
		error << _("could not initialize Ardour.") << endmsg;
		exit (1);
	}

	/* load session, if given */
	string name, path;

	if (session_name.length()){
		bool isnew;

		if(Session::find_session (session_name, name, path, isnew)) {
			error << compose(_("could not load command line session \"%1\""), session_name) << endmsg;
		} else {
			
			if (isnew) {

				/* default for a new session:
				   stereo master outs, auto-connect inputs
				   to physical channels, autoconnect
				   outputs to masters. no control outs.
				*/

				ui->build_session (name, path,
						   0, 2, 
						   Session::AutoConnectPhysical,
						   Session::AutoConnectMaster);
			} else {
				ui->load_session (name, path);
			}
		}

		if (no_splash) {
			ui->show();
		}

	} else {
		ui->hide_splash ();
		ui->show ();
		ui->new_session ();
	}

	ui->run (text_receiver);
	
	delete ui;
	ui = 0;

	ARDOUR::cleanup ();

	shutdown (0);
}

