/*
    Copyright (C) 1999-2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: new_session_dialog.cc,v 1.5 2004/01/26 03:51:46 trutkin Exp $
*/

#include <list>
#include <string>
#include <ardour/session.h>
#include <ardour/audioengine.h>
#include <gtkmmext/utils.h>

#include "prompter.h"
#include "new_session_dialog.h"

using namespace Gtkmmext;
using namespace Gtk;
using namespace ARDOUR;

#include "i18n.h"

static const gchar *new_session_title[] = {
	N_("Available Mix Templates"),
	0
};

extern void channel_choice_printer (char buf[32], Adjustment& adj, void *ignored);

NewSessionDialog::NewSessionDialog (ARDOUR::AudioEngine& engine)
	: file_selector (_("Session name:"), _("Create")),
	  control_out_config_adjustment (2, 1, 99, 1, 4),
	  master_out_config_adjustment (2, 1, 99, 1, 4),
	  use_control_button (_("use control outs")),
	  use_master_button (_("use master outs")),
	  connect_to_master_button (_("autoconnect track outputs to master outs")),
	  connect_to_physical_inputs_button (_("autoconnect track inputs to physical ports")),
	  connect_to_physical_outputs_button (_("autoconnect track outputs to physical ports")),
	  manual_connect_inputs_button (_("manually connect track inputs")),
	  manual_connect_outputs_button (_("manually connect track outputs")),
	  use_manual_button (_("new setup")),
	  use_template_button (_("use existing template")),
	  template_selector (_mix_template_refiller, this, internationalize (new_session_title), false, true),
	  output_label (_("Output Options")),
	  input_label (_("Input Options")),
	  control_out_configuration_clickbox (&control_out_config_adjustment, "NewSessionChannelChoice"),
	  master_out_configuration_clickbox (&master_out_config_adjustment, "NewSessionChannelChoice"),
	  expansion_button (_("Advanced...")),
	  out_table (2, 2)
	
{
	set_name ("NewSessionDialog");
	set_title (_("new session setup"));
	set_wmclass (_("ardour_new_session"), "Ardour");
	set_position (GTK_WIN_POS_MOUSE);
	set_keyboard_input (true);
	set_policy (false, true, false);
	set_modal (true);

	use_manual_button.set_group (use_template_button.group());
	use_manual_button.set_active (true);

	/* sample rate */


	sr_label1.set_text (compose 
			   (_("This session will playback and record at %1 Hz"),
			    engine.frame_rate()));
	sr_label2.set_text (_("This rate is set by JACK and cannot be changed.\n"
			      "If you want to use a different sample rate\n"
			      "please exit and restart JACK"));
	sr_box.set_spacing (12);
	sr_box.set_border_width (12);
	sr_box.pack_start (sr_label1, false, false);
	sr_box.pack_start (sr_label2, false, false);
	sr_frame.add (sr_box);
	
	/* input */

	manual_connect_inputs_button.set_group (connect_to_physical_inputs_button.group());
	connect_to_physical_inputs_button.set_active (true);

	input_label.set_alignment (0.1, 0.5);
	input_vbox.pack_start (input_label, true, true, 7);
	input_vbox.pack_start (connect_to_physical_inputs_button);
	input_vbox.pack_start (manual_connect_inputs_button);

	/* output */

	use_master_button.set_active (true);
	
	connect_to_physical_outputs_button.set_group (connect_to_master_button.group());
	manual_connect_outputs_button.set_group (connect_to_master_button.group());
	connect_to_master_button.set_active (true);

	master_out_configuration_clickbox.set_print_func (channel_choice_printer, 0);
	set_usize_to_display_given_text (master_out_configuration_clickbox, _("99 channels"), 2, 2);	

	control_out_configuration_clickbox.set_print_func (channel_choice_printer, 0);
	set_usize_to_display_given_text (control_out_configuration_clickbox, _("99 channels"), 2, 2);	

	master_out_config_frame.add (master_out_configuration_clickbox);
	master_out_config_frame.set_shadow_type (GTK_SHADOW_IN);

	control_out_config_frame.add (control_out_configuration_clickbox);
	control_out_config_frame.set_shadow_type (GTK_SHADOW_IN);

	out_table.set_col_spacings (7);
	out_table.set_row_spacings (7);
	out_table.attach (use_control_button, 0, 1, 0, 1, 0, 0);
	out_table.attach (control_out_config_frame, 1, 2, 0, 1, 0, 0);
	out_table.attach (use_master_button, 0, 1, 1, 2, 0, 0);
	out_table.attach (master_out_config_frame, 1, 2, 1, 2, 0, 0);
	
	output_label.set_alignment (0.1, 0.5);
	output_vbox.pack_start (output_label, true, true, 7);
	output_vbox.pack_start (out_table);
	output_vbox.pack_start (connect_to_master_button);
	output_vbox.pack_start (connect_to_physical_outputs_button);
	output_vbox.pack_start (manual_connect_outputs_button);
	
	output_hbox.pack_start (output_vbox, false, false);

	/* set it up */

	template_selector.clist().column_titles_hide ();
	template_selector.clist().set_selection_mode (GTK_SELECTION_SINGLE);
	template_selector.set_usize(-1, 75);

	template_box.set_border_width (5);
	template_box.set_spacing (5);
	template_box.pack_start (use_template_button, false, false);
	template_box.pack_start (template_selector);
	template_frame.add (template_box);
	template_selector.rescan ();
	template_selector.clist().row(0).select();

	manexp_box.set_spacing (7);
	manexp_box.pack_start (use_manual_button, false, false);
	manexp_box.pack_end (expansion_button, false, false, 7);

	io_box.set_spacing (7);
	io_box.pack_start (input_vbox);
	io_box.pack_start (output_hbox);

	manual_vbox.set_border_width (5);
	manual_vbox.set_spacing (7);
	manual_vbox.pack_start (manexp_box, false, false);
	manual_frame.add (manual_vbox);

	option_hbox.set_spacing (7);
	option_hbox.pack_start (manual_frame);
	option_hbox.pack_start (template_frame);

	vbox.set_border_width (12);
	vbox.set_spacing (7);
	vbox.pack_start (file_selector.table, false, false);
	vbox.pack_start (*(manage (new HSeparator)), false, false);
	vbox.pack_start (option_hbox, false, false);
	vbox.pack_start (sr_frame, false, false);
	vbox.pack_start (file_selector.button_box, false, false);

	add (vbox);

	template_selector.shift_made.connect (
		slot (*this, &NewSessionDialog::mix_template_shift));
	template_selector.control_made.connect (
		slot (*this, &NewSessionDialog::mix_template_control));

	file_selector.cancel_button.clicked.connect (bind (slot (*this, &ArdourDialog::stop), -1));
	file_selector.op_button.clicked.connect (bind (slot (*this, &ArdourDialog::stop), 0));
	file_selector.Expanded.connect (slot (*this, &NewSessionDialog::file_selector_expansion));

	expansion_button.clicked.connect (slot (*this, &NewSessionDialog::toggle_expansion));

	delete_event.connect (slot (*this, &ArdourDialog::wm_close_event));
	show.connect (slot (*this, &NewSessionDialog::fixup_at_show));

	file_selector.entry_label.set_name ("NewSessionMainLabel");
	file_selector.where_label.set_name ("NewSessionMainLabel");
	input_label.set_name ("NewSessionIOLabel");
	output_label.set_name ("NewSessionIOLabel");
	sr_label1.set_name ("NewSessionSR1Label");
	sr_label2.set_name ("NewSessionSR2Label");
	use_manual_button.set_name ("NewSessionMainButton");
	use_template_button.set_name ("NewSessionMainButton");

	Session::FavoriteDirs favs;
	Session::read_favorite_dirs (favs);
	file_selector.set_favorites (favs);


	set_default_size(531, 358);
}

void
NewSessionDialog::file_selector_expansion (bool expanded)
{
	if (expanded) {
		vbox.pack_start (file_selector.expansion_vbox);
		vbox.reorder_child (file_selector.expansion_vbox, 2);
	} else {
		vbox.remove (file_selector.expansion_vbox);
	}
}

void
NewSessionDialog::toggle_expansion ()
{
	if (io_box.is_visible()) {
		manual_vbox.remove (io_box);
		io_box.hide ();
		static_cast<Gtk::Label*>(expansion_button.get_child())->set_text (_("Advanced ..."));
	} else {
		static_cast<Gtk::Label*>(expansion_button.get_child())->set_text (_("Hide Advanced"));
		manual_vbox.pack_start (io_box, false, false);
		io_box.show_all ();
	}
}

void
NewSessionDialog::fixup_at_show ()
{

	if (template_selector.clist().rows().size() == 0) {
		use_template_button.set_sensitive (false);
	}

	Session::FavoriteDirs favs;
	Session::read_favorite_dirs (favs);
	file_selector.set_favorites (favs);

	file_selector.entry.grab_focus ();
}

void
NewSessionDialog::_mix_template_refiller (CList &clist, void *arg)

{
	((NewSessionDialog*) arg)->mix_template_refiller (clist);
}

void
NewSessionDialog::mix_template_refiller (CList &clist)
{
	const gchar *rowdata[2];
	list<string> templates;
	list<string>::iterator i;
	
	Session::get_template_list(templates);
	
	rowdata[0] = _("blank");
	clist.insert_row (0, rowdata);

	guint row;
	for (row=1, i=templates.begin(); i != templates.end(); ++row, ++i) {
		rowdata[0] = (*i).c_str();
		clist.insert_row (row, rowdata);
	}
}

void
NewSessionDialog::mix_template_shift (Gtkmmext::Selector* selector, Gtkmmext::SelectionResult* res)
{
	if (res && res->text){
		Session::delete_template(*res->text);
		template_selector.rescan();
	}
}

void
NewSessionDialog::mix_template_control (Gtkmmext::Selector* selector, Gtkmmext::SelectionResult* res)
{
	if (res && res->text) {
		ArdourPrompter prompter (true);
		prompter.set_prompt(_("Name for mix template:"));

		string old_name = *(res->text);
		prompter.set_initial_text (old_name);
		prompter.done.connect (Gtk::Main::quit.slot());
		prompter.show_all();
		
		Gtk::Main::run();
		
		if (prompter.status == Gtkmmext::Prompter::entered) {
			string name;

			prompter.get_result (name);

			if (name.length() && name != old_name) {
				Session::rename_template(old_name, name);
				template_selector.rescan();
			}
		}
	}
}
