/*
    Copyright (C) 2001 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: option_editor.cc,v 1.68 2004/02/12 17:21:24 pauld Exp $
*/

#include "option_editor.h"

#include <midi++/manager.h>

#include <ardour/session.h>
#include <ardour/audioengine.h>
#include <ardour/configuration.h>
#include <ardour/auditioner.h>
#include <gtkmmext/stop_signal.h>
#include <gtkmmext/utils.h>

#include "public_editor.h"
#include "mixer_ui.h"
#include "keyboard.h"
#include "ardour_ui.h"
#include "io_selector.h"
#include "gain_pan_meter.h"
#include "library_ui.h"
#include "utils.h"
#include "editing.h"
#include "i18n.h"

using namespace ARDOUR;
using namespace Gtk;
using namespace Editing;

OptionEditor::OptionEditor (ARDOUR_UI& uip, PublicEditor& ed, Mixer_UI& mixui)
	: ui (uip),
	  editor (ed),
	  mixer (mixui),

	  path_table (9, 2),

	  /* Fades */

	  auto_xfade_button (_("Automatic crossfade on overlap")),

	  /* solo */
	  solo_latched_button (_("Latched solo")),
	  solo_via_bus_button (_("Solo via bus")),

	  /* display */

	  show_waveforms_button (_("Show waveforms")),
	  mixer_strip_width_button (_("Narrow mixer strips")),
	  show_measures_button (_("Show measure lines")),
	  follow_playhead_button (_("Follow playhead")),

	  /* Sync */

	  send_mtc_button (_("Send MTC")),
	  send_mmc_button (_("Send MMC")),
	  mmc_control_button (_("MMC Control")),
	  jack_time_master_button (_("JACK time master")),
	  jack_transport_master_button (_("JACK transport master")),

	  /* MIDI */

	  trace_midi_in_button (_("Trace Input")),
	  trace_midi_out_button (_("Trace Output")),
	  
	  /* Click */

	  click_table (2, 3),
	  click_browse_button (_("Browse")),
	  click_emphasis_browse_button (_("Browse")),

	  /* kbd/mouse */

	  keyboard_mouse_table (3, 4),
	  delete_button_adjustment (3, 1, 5),
	  delete_button_spin (delete_button_adjustment),
	  edit_button_adjustment (3, 1, 5),
	  edit_button_spin (edit_button_adjustment),

	  /* Misc */

	  hw_monitor_button(_("Use Hardware Monitoring")),
	  sw_monitor_button(_("Use Software Monitoring")),
	  auto_connect_ports_button (_("Auto-connect new tracks")),
	  plugins_stop_button (_("Stop plugins with transport")),
	  plugins_on_rec_button (_("Run plugins while recording")),
	  stop_rec_on_xrun_button (_("Stop recording on xrun")),
	  stop_at_end_button (_("Stop transport at end of session")),
	  debug_keyboard_button (_("Debug keyboard events"))
	  
{
	using namespace Notebook_Helpers;

	click_io_selector = 0;
	auditioner_io_selector = 0;

	set_default_size (300, 300);
	set_title (_("ardour: options editor"));
	set_wmclass (_("ardour_option_editor"), "Ardour");

	set_name ("OptionsWindow");
	add_events (GDK_KEY_PRESS_MASK|GDK_KEY_RELEASE_MASK);
	
	VBox *vbox = manage(new VBox);
	add (*vbox);
	set_border_width (3);

	vbox->set_spacing (4);
	vbox->pack_start(notebook);

	delete_event.connect (slot (*this, &OptionEditor::wm_close));

	notebook.set_show_tabs (true);
	notebook.set_show_border (true);
	notebook.set_name ("OptionsNotebook");

	setup_sync_options();
	setup_path_options();
	setup_fade_options ();
	setup_solo_options ();
	setup_display_options ();
	setup_misc_options ();
	setup_keyboard_options ();

	notebook.pages().push_back (TabElem (sync_packer, _("Sync")));
	notebook.pages().push_back (TabElem (path_table, _("Paths/Files")));
	notebook.pages().push_back (TabElem (display_packer, _("Display")));
	notebook.pages().push_back (TabElem (keyboard_mouse_table, _("Kbd/Mouse")));
	notebook.pages().push_back (TabElem (click_packer, _("Click")));
	notebook.pages().push_back (TabElem (audition_packer, _("Audition")));
	notebook.pages().push_back (TabElem (fade_packer, _("Fades")));
	notebook.pages().push_back (TabElem (solo_packer, _("Solo")));
	notebook.pages().push_back (TabElem (misc_packer, _("Misc")));

	if (!MIDI::Manager::instance()->get_midi_ports().empty()) {
		setup_midi_options ();
		notebook.pages().push_back (TabElem (midi_packer, _("MIDI")));
	}

	set_session (0);
}

void
OptionEditor::set_session (Session *s)
{
	clear_click_editor ();
	clear_auditioner_editor ();

	click_path_entry.set_text ("");
	click_emphasis_path_entry.set_text ("");
	session_raid_entry.set_text ("");

	send_mtc_button.set_sensitive (false);
	send_mmc_button.set_sensitive (false);
	mmc_control_button.set_sensitive (false);
	trace_midi_in_button.set_sensitive (false);
	trace_midi_out_button.set_sensitive (false);
	click_path_entry.set_sensitive (false);
	click_emphasis_path_entry.set_sensitive (false);
	session_raid_entry.set_sensitive (false);
	plugins_on_rec_button.set_sensitive (false);
	slave_type_combo.set_sensitive (false);
	align_style_combo.set_sensitive (false);
	solo_latched_button.set_sensitive (false);
	solo_via_bus_button.set_sensitive (false);

	if ((session = s) == 0) {
		return;
	}

	send_mtc_button.set_sensitive (true);
	send_mmc_button.set_sensitive (true);
	mmc_control_button.set_sensitive (true);
	trace_midi_in_button.set_sensitive (true);
	trace_midi_out_button.set_sensitive (true);
	click_path_entry.set_sensitive (true);
	click_emphasis_path_entry.set_sensitive (true);
	session_raid_entry.set_sensitive (true);
	plugins_on_rec_button.set_sensitive (true);
	slave_type_combo.set_sensitive (true);
	align_style_combo.set_sensitive (true);
	solo_latched_button.set_sensitive (true);
	solo_via_bus_button.set_sensitive (true);

	send_mtc_button.set_active (session->get_send_mtc());
	send_mmc_button.set_active (session->get_send_mmc());
	mmc_control_button.set_active (session->get_mmc_control());

	setup_click_editor ();
	setup_auditioner_editor ();

	plugins_on_rec_button.set_active (session->get_recording_plugins ());
	send_mtc_button.set_active (session->get_send_mtc());
	send_mmc_button.set_active (session->get_send_mmc());
	mmc_control_button.set_active (session->get_mmc_control());
	auto_xfade_button.set_active (session->get_auto_crossfade());
	solo_latched_button.set_active (session->solo_latched());
	solo_via_bus_button.set_active (session->solo_model() == Session::SoloBus);

	add_session_paths ();

	session_control_changed (Session::SlaveType);
	session_control_changed (Session::AlignChoice);
	session->ControlChanged.connect (slot (*this, &OptionEditor::queue_session_control_changed));
}

OptionEditor::~OptionEditor ()
{
}

static const gchar *native_format_strings[] = {
	N_("Broadcast WAVE/floating point"),
	N_("WAVE/floating point"),
	0
};

void
OptionEditor::setup_path_options()
{
	Gtk::Label* label;

	path_table.set_homogeneous (true);
	path_table.set_border_width (12);
	path_table.set_row_spacings (5);

	session_raid_entry.set_name ("OptionsEntry");

	session_raid_entry.activate.connect (slot (*this, &OptionEditor::raid_path_changed));

	session_raid_entry.focus_out_event.connect (bind (slot (*this, &OptionEditor::focus_out_event_handler), &OptionEditor::raid_path_changed));

	label = manage(new Label(_("session RAID path")));
	path_table.attach (*label, 0, 1, 0, 1, 0, 0);
	path_table.attach (session_raid_entry, 1, 3, 0, 1, GTK_FILL|GTK_EXPAND, 0);

	label = manage(new Label(_("Native Format")));
	path_table.attach (*label, 0, 1, 1, 2, 0, 0);
	path_table.attach (native_format_combo, 1, 3, 1, 2, GTK_FILL|GTK_EXPAND, 0);

	native_format_combo.set_popdown_strings (internationalize(native_format_strings)),
	native_format_combo.get_entry()->set_editable (false);
	native_format_combo.get_entry()->set_name ("OptionsEntry");
	native_format_combo.set_use_arrows_always (true);
	native_format_combo.get_popwin()->unmap_event.connect (slot (*this, &OptionEditor::native_format_chosen));

	/* find the widest string */

	int n;
	int maxindex;
	size_t maxlen = 0;
	string maxstring;

	for (n = 0; native_format_strings[n]; ++n) {
		size_t l;
		if ((l = strlen (native_format_strings[n])) > maxlen) {
			maxlen = l;
			maxindex = n;
			maxstring = native_format_strings[n];
		}
	}

	/* be sure to include ascenders and descenders */

	maxstring[0] = 'g';
	maxstring[1] = 'l';

	const guint32 FUDGE = 10; // Combo's are stupid - they steal space from the entry for the button

	Gtkmmext::set_usize_to_display_given_text (*native_format_combo.get_entry(), maxstring.c_str(), 10 + FUDGE, 10);

	/* sigh. combo's have the worst API ... */

	if (Config->get_native_format_is_bwf()) {
		native_format_combo.get_entry()->set_text (native_format_strings[0]);
	} else {
		native_format_combo.get_entry()->set_text (native_format_strings[1]);
	}
	
	path_table.show_all();
}

void
OptionEditor::add_session_paths ()
{
	click_path_entry.set_sensitive (true);
	click_emphasis_path_entry.set_sensitive (true);
	session_raid_entry.set_sensitive (true);

	if (session->click_sound.length() == 0) {
		click_path_entry.set_text (_("internal"));
	} else {
		click_path_entry.set_text (session->click_sound);
	}

	if (session->click_emphasis_sound.length() == 0) {
		click_emphasis_path_entry.set_text (_("internal"));
	} else {
		click_emphasis_path_entry.set_text (session->click_emphasis_sound);
	}

	session_raid_entry.set_text(session->raid_path());
}

void
OptionEditor::setup_fade_options ()
{
	Gtk::HBox* hbox;

	auto_xfade_button.set_name ("OptionEditorToggleButton");

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (auto_xfade_button, false, false);
	fade_packer.pack_start (*hbox, false, false);

	auto_xfade_button.clicked.connect (slot (*this, &OptionEditor::auto_xfade_clicked));

	fade_packer.show_all ();
}

void
OptionEditor::setup_solo_options ()
{
	Gtk::HBox* hbox;

	solo_via_bus_button.set_name ("OptionEditorToggleButton");
	solo_latched_button.set_name ("OptionEditorToggleButton");

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (solo_via_bus_button, false, false);
	solo_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (solo_latched_button, false, false);
	solo_packer.pack_start (*hbox, false, false);

	solo_via_bus_button.clicked.connect 
		(slot (*this, &OptionEditor::solo_via_bus_clicked));
	solo_latched_button.clicked.connect 
		(slot (*this, &OptionEditor::solo_latched_clicked));

	solo_packer.show_all ();
}

void
OptionEditor::solo_via_bus_clicked ()
{
	if (!session) {
		return;
	}

	if (solo_via_bus_button.get_active()) {
		session->set_solo_model (Session::SoloBus);
	} else {
		session->set_solo_model (Session::InverseMute);
	}
}

void
OptionEditor::solo_latched_clicked ()
{
	if (!session) {
		return;
	}

	bool x = solo_latched_button.get_active();

	if (x != session->solo_latched()) {
		session->set_solo_latched (x);
	}
}

void
OptionEditor::setup_display_options ()
{
	HBox* hbox;

	display_packer.set_border_width (12);
	display_packer.set_spacing (5);

	show_waveforms_button.set_name ("OptionEditorToggleButton");
	show_measures_button.set_name ("OptionEditorToggleButton");
	follow_playhead_button.set_name ("OptionEditorToggleButton");

	show_waveforms_button.set_active (editor.show_waveforms());
	show_measures_button.set_active (editor.show_measures());
	mixer_strip_width_button.set_active (mixer.get_strip_width() == Narrow);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (show_waveforms_button, false, false);
	display_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (show_measures_button, false, false);
	display_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (mixer_strip_width_button, false, false);
	display_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (follow_playhead_button, false, false);
	display_packer.pack_start (*hbox, false, false);

	show_waveforms_button.clicked.connect (slot (*this, &OptionEditor::show_waveforms_clicked));
	show_measures_button.clicked.connect (slot (*this, &OptionEditor::show_measures_clicked));
	mixer_strip_width_button.clicked.connect (slot (*this, &OptionEditor::strip_width_clicked));
	follow_playhead_button.clicked.connect (slot (*this, &OptionEditor::follow_playhead_clicked));

	editor.DisplayControlChanged.connect (slot (*this, &OptionEditor::display_control_changed));

}

void
OptionEditor::display_control_changed (Editing::DisplayControl dc)
{
	ToggleButton* button = 0;
	bool val = true;

	switch (dc) {
	case ShowMeasures:
		val = editor.show_measures ();
		button = &show_measures_button;
		break;
	case ShowWaveforms:
		val = editor.show_waveforms ();
		button = &show_waveforms_button;
		break;
	case FollowPlayhead:
		val = editor.follow_playhead ();
		button = &follow_playhead_button;
		break;
	}

	if (button->get_active() != val) {
		button->set_active (val);
	}
}

void
OptionEditor::setup_sync_options ()
{
	Label *slave_type_label = manage (new Label (_("Sync Source")));
	vector<string> dumb;
	HBox* hbox;

	dumb.push_back (_("Master"));
	dumb.push_back (_("MTC (SMPTE)"));
	dumb.push_back (_("ADAT"));
	dumb.push_back (_("JACK"));
	
	slave_type_combo.set_popdown_strings (dumb);
	slave_type_combo.set_use_arrows_always (true);
	slave_type_combo.set_value_in_list (true, false);
	slave_type_combo.get_entry()->set_editable (false);
	slave_type_combo.get_entry()->set_name ("OptionsEntry");
	slave_type_combo.get_popwin()->unmap_event.connect (slot (*this, &OptionEditor::slave_type_chosen));


	dumb.clear ();
	dumb.push_back (_("Existing material"));
	dumb.push_back (_("Capture time"));

	Label *align_style_label = manage (new Label (_("Align captured material with")));

	align_style_combo.set_popdown_strings (dumb);
	align_style_combo.set_use_arrows_always (true);
	align_style_combo.set_value_in_list (true, false);
	align_style_combo.get_entry()->set_editable (false);
	align_style_combo.get_entry()->set_name ("OptionsEntry");
	align_style_combo.get_popwin()->unmap_event.connect (slot (*this, &OptionEditor::align_style_chosen));

	send_mtc_button.set_name ("OptionEditorToggleButton");
	send_mmc_button.set_name ("OptionEditorToggleButton");
	mmc_control_button.set_name ("OptionEditorToggleButton");
	jack_time_master_button.set_name ("OptionEditorToggleButton");
	jack_transport_master_button.set_name ("OptionEditorToggleButton");

	send_mtc_button.unset_flags (GTK_CAN_FOCUS);
	send_mmc_button.unset_flags (GTK_CAN_FOCUS);
	mmc_control_button.unset_flags (GTK_CAN_FOCUS);
	jack_time_master_button.unset_flags (GTK_CAN_FOCUS);
	jack_transport_master_button.unset_flags (GTK_CAN_FOCUS);

	hbox = manage (new HBox);
	hbox->set_border_width (5);
	hbox->set_spacing (10);
	hbox->pack_start (*slave_type_label, false, false);
	hbox->pack_start (slave_type_combo, false, false);

	sync_packer.pack_start (*hbox, false, false);
	
	hbox = manage (new HBox);
	hbox->set_border_width (5);
	hbox->pack_start (send_mtc_button, false, false);
	sync_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (5);
	hbox->pack_start (send_mmc_button, false, false);
	sync_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (5);
	hbox->pack_start (mmc_control_button, false, false);
	sync_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (5);
	hbox->pack_start (jack_time_master_button, false, false);
	sync_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (5);
	hbox->pack_start (jack_transport_master_button, false, false);
	sync_packer.pack_start (*hbox, false, false);


	hbox = manage (new HBox);
	hbox->set_border_width (5);
	hbox->set_spacing (10);
	hbox->pack_start (*align_style_label, false, false);
	hbox->pack_start (align_style_combo, false, false);

	sync_packer.pack_start (*hbox, false, false);

	jack_time_master_button.set_active (Config->get_jack_time_master());
	jack_transport_master_button.set_active (Config->get_jack_transport_master());

	send_mtc_button.button_press_event.connect (bind (slot (*this, &OptionEditor::send_mtc_toggled), &send_mtc_button));
	send_mmc_button.button_press_event.connect (bind (slot (*this, &OptionEditor::send_mmc_toggled), &send_mmc_button));
	mmc_control_button.button_press_event.connect (bind (slot (*this, &OptionEditor::mmc_control_toggled), &mmc_control_button));
	jack_time_master_button.clicked.connect (slot (*this, &OptionEditor::jack_time_master_clicked));
	jack_transport_master_button.clicked.connect (slot (*this, &OptionEditor::jack_transport_master_clicked));
}

void
OptionEditor::setup_midi_options ()
{
	vector<string> dumb;
	Label *mtc_port_label = manage (new Label (_("MMC Port")));
	Label *mmc_port_label = manage (new Label (_("MTC Port")));
	HBox* hbox;

	MIDI::Manager::PortMap::const_iterator i;
	const MIDI::Manager::PortMap& ports = MIDI::Manager::instance()->get_midi_ports();

	dumb.clear ();
	for (i = ports.begin(); i != ports.end(); ++i) {
		dumb.push_back (i->first);
	}
	
	mmc_port_combo.set_popdown_strings (dumb);
	mmc_port_combo.set_use_arrows_always (true);
	mmc_port_combo.set_value_in_list (true, false);
	mmc_port_combo.get_entry()->set_editable (false);
	mmc_port_combo.get_entry()->set_name ("OptionsEntry");
	mmc_port_combo.get_popwin()->unmap_event.connect (slot (*this, &OptionEditor::mmc_port_chosen));
	
	mtc_port_combo.set_popdown_strings (dumb);
	mtc_port_combo.set_use_arrows_always (true);
	mtc_port_combo.set_value_in_list (true, false);
	mtc_port_combo.get_entry()->set_editable (false);
	mtc_port_combo.get_entry()->set_name ("OptionsEntry");
	mtc_port_combo.get_popwin()->unmap_event.connect (slot (*this, &OptionEditor::mtc_port_chosen));
	
	trace_midi_in_button.set_name ("OptionEditorToggleButton");
	trace_midi_out_button.set_name ("OptionEditorToggleButton");

	trace_midi_in_button.unset_flags (GTK_CAN_FOCUS);
	trace_midi_out_button.unset_flags (GTK_CAN_FOCUS);

	midi_packer.set_border_width (12);
	midi_packer.set_spacing (5);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->set_spacing (10);
	hbox->pack_start (*mtc_port_label, false, false);
	hbox->pack_start (mtc_port_combo, false, false);

	midi_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->set_spacing (10);
	hbox->pack_start (*mmc_port_label, false, false);
	hbox->pack_start (mmc_port_combo, false, false);

	midi_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (trace_midi_in_button, false, false);
	midi_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (trace_midi_out_button, false, false);
	midi_packer.pack_start (*hbox, false, false);

	trace_midi_in_button.clicked.connect (slot (*this, &OptionEditor::midi_trace_in_clicked));
	trace_midi_out_button.clicked.connect (slot (*this, &OptionEditor::midi_trace_out_clicked));
}

gint
OptionEditor::send_mtc_toggled (GdkEventButton *ev, CheckButton *button)
{
	if (session) {
		session->set_send_mtc (!button->get_active());
	}
	return stop_signal (*button, "button_press_event");
}

gint
OptionEditor::send_mmc_toggled (GdkEventButton *ev, CheckButton *button)
{
	if (session) {
		session->set_send_mmc (!button->get_active());
	}

	return stop_signal (*button, "button_press_event");
}

gint
OptionEditor::mmc_control_toggled (GdkEventButton *ev, CheckButton *button)
{
	if (session) {
		session->set_mmc_control (!button->get_active());
	}
	return stop_signal (*button, "button_press_event");
}

void
OptionEditor::save ()
{
	ui.save_ardour_state ();

	if (session) {
		session->save_state (session->snap_name());
	}
}

gint
OptionEditor::wm_close (GdkEventAny *ev)
{
	save ();
	just_close_win();
	return TRUE;
}

void
OptionEditor::midi_trace_in_clicked ()
{
	if (session) {
		session->set_trace_midi_input (trace_midi_in_button.get_active());
	}
}

void
OptionEditor::midi_trace_out_clicked ()
{
	if (session) {
		session->set_trace_midi_output (trace_midi_out_button.get_active());
	}
}

void
OptionEditor::jack_time_master_clicked ()
{
	bool yn = jack_time_master_button.get_active();

	Config->set_jack_time_master (yn);

	if (session) {
		session->engine().reset_timebase ();
	}
}

void
OptionEditor::jack_transport_master_clicked ()
{
	Config->set_jack_transport_master (jack_transport_master_button.get_active());
}

void
OptionEditor::raid_path_changed ()
{
	if (session) {
		session->set_raid_path (session_raid_entry.get_text());
	}
}

void
OptionEditor::click_browse_clicked ()
{
	SoundFileSelector& sfdb (ARDOUR_UI::instance()->get_sfdb_window());

	sfdb.run (_("Use as click"), false);
	
	if (sfdb.run_status ()) {
		return;
	}

	string path;
	bool multi;
	
	sfdb.get_result (path, multi);

	click_path_entry.set_text (path);
	click_sound_changed ();
}

void
OptionEditor::click_emphasis_browse_clicked ()
{
	SoundFileSelector& sfdb (ARDOUR_UI::instance()->get_sfdb_window());

	sfdb.run (_("Use as click emphasis"), false);
	
	if (sfdb.run_status ()) {
		return;
	}

	string path;
	bool multi;
	
	sfdb.get_result (path, multi);

	click_emphasis_path_entry.set_text (path);
	click_emphasis_sound_changed ();
}

void
OptionEditor::click_sound_changed ()
{
	if (session) {
		string path = click_path_entry.get_text();

		if (path == session->click_sound) {
			return;
		}

		if (path.length() == 0) {

			session->set_click_sound ("");

		} else {

			strip_whitespace_edges (path);
			
			if (path == _("internal")) {
				session->set_click_sound ("");
			} else {
				session->set_click_sound (path);
			}
		}
	}
}

void
OptionEditor::click_emphasis_sound_changed ()
{
	if (session) {
		string path = click_emphasis_path_entry.get_text();

		if (path == session->click_emphasis_sound) {
			return;
		}

		if (path.length() == 0) {

			session->set_click_emphasis_sound ("");

		} else {

			strip_whitespace_edges (path);

			if (path == _("internal")) {
				session->set_click_emphasis_sound ("");
			} else {
				session->set_click_emphasis_sound (path);
			}
		}
	}
}

void
OptionEditor::show_waveforms_clicked ()
{
	editor.set_show_waveforms (show_waveforms_button.get_active());
}

void
OptionEditor::show_measures_clicked ()
{
	editor.set_show_measures (show_measures_button.get_active());
}

void
OptionEditor::follow_playhead_clicked ()
{
	editor.set_follow_playhead (follow_playhead_button.get_active());
}

void
OptionEditor::strip_width_clicked ()
{
	mixer.set_strip_width (mixer_strip_width_button.get_active() ? Narrow : Wide);
}

void
OptionEditor::just_close_win()
{
	hide_all();
	ARDOUR_UI::instance()->allow_focus(false);
}

void
OptionEditor::queue_session_control_changed (Session::ControlType t)
{
	ui.call_slot (bind (slot (*this, &OptionEditor::session_control_changed), t));
}

void
OptionEditor::session_control_changed (Session::ControlType t)
{
	switch (t) {
	case Session::SlaveType:
		switch (session->slave_source()) {
		case Session::None:
			slave_type_combo.get_entry()->set_text (_("Master"));
			break;
		case Session::MTC:
			slave_type_combo.get_entry()->set_text (_("MTC (SMPTE)"));
			break;
		case Session::ADAT:
			slave_type_combo.get_entry()->set_text (_("ADAT"));
			break;
		case Session::JACK:
			slave_type_combo.get_entry()->set_text (_("JACK"));
			break;
		case Session::Scrub:
			slave_type_combo.get_entry()->set_text (_("Scrub"));
			break;
		default:
			slave_type_combo.get_entry()->set_text (_("--unknown--"));
			break;
		}
		
		break;

	case Session::SendMTC:
		map_some_session_state (send_mtc_button, &Session::get_send_mtc);
		break;

	case Session::SendMMC:
		map_some_session_state (send_mmc_button, &Session::get_send_mmc);
		break;

	case Session::MMCControl:       
		map_some_session_state (mmc_control_button, &Session::get_mmc_control);
		break;

	default:
		break;
	}
}

void
OptionEditor::map_some_session_state (CheckButton& button, bool (Session::*get)() const)
{
	if (session) {
		button.set_active ((session->*get)());
	}
}

gint
OptionEditor::mtc_port_chosen (GdkEventAny *ignored)
{
	if (session) {
		string which = mtc_port_combo.get_entry()->get_text();
		session->set_mtc_port (which);
	}
	return FALSE;
}

gint
OptionEditor::mmc_port_chosen (GdkEventAny *ignored)
{
	if (session) {
		string which = mmc_port_combo.get_entry()->get_text();
		session->set_mmc_port (which);
	}
	return FALSE;
}

gint
OptionEditor::native_format_chosen (GdkEventAny *ignored)
{
	string which;

	if (session == 0) {
		return FALSE;
	}

	bool use_bwf = (native_format_combo.get_entry()->get_text() == native_format_strings[0]);

	if (use_bwf != Config->get_native_format_is_bwf()) {
		Config->set_native_format_is_bwf (use_bwf);
		session->reset_native_file_format ();
	}

	return TRUE;
}

gint
OptionEditor::slave_type_chosen (GdkEventAny *ignored)
{
	string which;

	if (session == 0) {
		return FALSE;
	}

	which = slave_type_combo.get_entry()->get_text();

	if (which == _("Master")) {
		session->request_slave_source (Session::None);
	} else if (which == _("MTC (SMPTE)")) {
		session->request_slave_source (Session::MTC);
	} else if (which == _("ADAT")) {
		session->request_slave_source (Session::ADAT);
	} else if (which == _("JACK")) {
		session->request_slave_source (Session::JACK);
	} 
	return FALSE;
}

gint
OptionEditor::align_style_chosen (GdkEventAny *ignored)
{
	string which;

	if (session == 0) {
		return FALSE;
	}

	which = align_style_combo.get_entry()->get_text();

	if (which == _("Existing material")) {
		session->set_align_style (Session::ExistingMaterial);
	} else if (which == _("Capture time")) {
		session->set_align_style (Session::CaptureTime);
	} 

	return FALSE;
}

void
OptionEditor::clear_click_editor ()
{
	if (click_io_selector) {
		click_packer.remove (*click_io_selector);
		click_packer.remove (*click_gpm);
		delete click_io_selector;
		delete click_gpm;
		click_io_selector = 0;
		click_gpm = 0;
	}
}

void
OptionEditor::setup_click_editor ()
{
	Label* label;
	HBox* hpacker = manage (new HBox);

	click_path_entry.set_sensitive (true);
	click_emphasis_path_entry.set_sensitive (true);

	click_path_entry.set_name ("OptionsEntry");
	click_emphasis_path_entry.set_name ("OptionsEntry");
	
	click_path_entry.activate.connect (slot (*this, &OptionEditor::click_sound_changed));
	click_emphasis_path_entry.activate.connect (slot (*this, &OptionEditor::click_emphasis_sound_changed));

	click_path_entry.focus_out_event.connect (bind (slot (*this, &OptionEditor::focus_out_event_handler), &OptionEditor::click_sound_changed));
	click_emphasis_path_entry.focus_out_event.connect (bind (slot (*this, &OptionEditor::focus_out_event_handler), &OptionEditor::click_emphasis_sound_changed));

	click_browse_button.clicked.connect (slot (*this, &OptionEditor::click_browse_clicked));
	click_emphasis_browse_button.clicked.connect (slot (*this, &OptionEditor::click_emphasis_browse_clicked));

	click_packer.set_border_width (12);
	click_packer.set_spacing (5);

	click_io_selector = new IOSelector (*session, session->click_io(), false);
	click_gpm = new GainPanMeter (session->click_io(), *session);

	click_table.set_col_spacings (10);
	
	label = manage(new Label(_("Click audio file")));
	click_table.attach (*label, 0, 1, 0, 1, 0, 0);
	click_table.attach (click_path_entry, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, 0);
	click_table.attach (click_browse_button, 2, 3, 0, 1, 0, 0);
	
	label = manage(new Label(_("Click emphasis audiofile")));
	click_table.attach (*label, 0, 1, 1, 2, 0, 0);
	click_table.attach (click_emphasis_path_entry, 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, 0);
	click_table.attach (click_emphasis_browse_button, 2, 3, 1, 2, 0, 0);

	hpacker->set_spacing (10);
	hpacker->pack_start (*click_io_selector, false, false);
	hpacker->pack_start (*click_gpm, false, false);

	click_packer.pack_start (click_table, false, false);
	click_packer.pack_start (*hpacker, false, false);

	click_packer.show_all ();
}

void
OptionEditor::clear_auditioner_editor ()
{
	if (auditioner_io_selector) {
		audition_packer.remove (*auditioner_io_selector);
		audition_packer.remove (*auditioner_gpm);
		delete auditioner_io_selector;
		delete auditioner_gpm;
		auditioner_io_selector = 0;
		auditioner_gpm = 0;
	}
}

void
OptionEditor::setup_auditioner_editor ()
{
	HBox* hpacker = manage (new HBox);
	Label* label = manage (new Label);

	audition_packer.set_border_width (12);
	audition_packer.set_spacing (5);

	label->set_name ("OptionEditorAuditionerLabel");
	label->set_text (_("The auditioner is a dedicated mixer strip used\n"
			   "for listening to specific regions outside the context\n"
			   "of the overall mix. It can be connected just like any\n"
			   "other mixer strip."));
	

	auditioner_io_selector = new IOSelector (*session, session->the_auditioner(), false);
	auditioner_gpm = new GainPanMeter (session->the_auditioner(), *session);

	hpacker->set_spacing (10);
	hpacker->pack_start (*auditioner_io_selector, false, false);
	hpacker->pack_start (*auditioner_gpm, false, false);

	audition_packer.pack_start (*label, false, false, 10);
	audition_packer.pack_start (*hpacker, false, false);
}

gint
OptionEditor::enter_notify_event_impl (GdkEventCrossing *ev)
{
	if (ev->detail != GDK_NOTIFY_INFERIOR) {
		ARDOUR_UI::instance()->the_keyboard().allow_focus (true);
	}
	return ArdourDialog::enter_notify_event_impl (ev);
}

gint
OptionEditor::leave_notify_event_impl (GdkEventCrossing *ev)
{
	if (ev->detail != GDK_NOTIFY_INFERIOR) {
		ARDOUR_UI::instance()->the_keyboard().allow_focus (false);
	}
	return ArdourDialog::leave_notify_event_impl (ev);
}

gint
OptionEditor::focus_out_event_handler (GdkEventFocus* ev, void (OptionEditor::*pmf)()) 
{
	(this->*pmf)();
	return FALSE;
}

void
OptionEditor::setup_misc_options()
{
	HBox* hbox;

	misc_packer.set_border_width (12);
	misc_packer.set_spacing (6);
	
	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (hw_monitor_button, false, false);
	misc_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (sw_monitor_button, false, false);
	misc_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (plugins_stop_button, false, false);
	misc_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (plugins_on_rec_button, false, false);
	misc_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (auto_connect_ports_button, false, false);
	misc_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (stop_rec_on_xrun_button, false, false);
	misc_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (stop_at_end_button, false, false);
	misc_packer.pack_start (*hbox, false, false);

	hbox = manage (new HBox);
	hbox->set_border_width (12);
	hbox->pack_start (debug_keyboard_button, false, false);
	misc_packer.pack_start (*hbox, false, false);

	hw_monitor_button.set_name ("OptionEditorToggleButton");
	sw_monitor_button.set_name ("OptionEditorToggleButton");
	plugins_stop_button.set_name ("OptionEditorToggleButton");
	plugins_on_rec_button.set_name ("OptionEditorToggleButton");
	auto_connect_ports_button.set_name ("OptionEditorToggleButton");
	stop_rec_on_xrun_button.set_name ("OptionEditorToggleButton");
	stop_at_end_button.set_name ("OptionEditorToggleButton");
	debug_keyboard_button.set_name ("OptionEditorToggleButton");

	hw_monitor_button.set_active (Config->get_use_hardware_monitoring());
	sw_monitor_button.set_active (!Config->get_no_sw_monitoring());
	plugins_stop_button.set_active (Config->get_plugins_stop_with_transport());
	auto_connect_ports_button.set_active (Config->get_auto_connect_ports());
	stop_rec_on_xrun_button.set_active (Config->get_stop_recording_on_xrun());
	stop_at_end_button.set_active (Config->get_stop_at_session_end());
	debug_keyboard_button.set_active (false);

	hw_monitor_button.clicked.connect (slot (*this, &OptionEditor::hw_monitor_clicked));
	sw_monitor_button.clicked.connect (slot (*this, &OptionEditor::sw_monitor_clicked));
	plugins_stop_button.clicked.connect (slot (*this, &OptionEditor::plugins_stop_with_transport_clicked));
	plugins_on_rec_button.clicked.connect (slot (*this, &OptionEditor::plugins_on_while_recording_clicked));
	auto_connect_ports_button.clicked.connect (slot (*this, &OptionEditor::auto_connect_ports_clicked));
	stop_rec_on_xrun_button.clicked.connect (slot (*this, &OptionEditor::stop_rec_on_xrun_clicked));
	stop_at_end_button.clicked.connect (slot (*this, &OptionEditor::stop_at_end_clicked));
	debug_keyboard_button.clicked.connect (slot (*this, &OptionEditor::debug_keyboard_clicked));
}

void
OptionEditor::auto_xfade_clicked ()
{
	if (session) {
		session->set_auto_crossfade (auto_xfade_button.get_active());
	}
}

void
OptionEditor::debug_keyboard_clicked ()
{
	extern bool debug_keyboard;
	debug_keyboard = debug_keyboard_button.get_active ();
}

void
OptionEditor::auto_connect_ports_clicked ()
{
	Config->set_auto_connect_ports (auto_connect_ports_button.get_active());
}

void
OptionEditor::hw_monitor_clicked ()
{
	Config->set_use_hardware_monitoring (hw_monitor_button.get_active());
}

void
OptionEditor::sw_monitor_clicked ()
{
	Config->set_no_sw_monitoring (!sw_monitor_button.get_active());
}

void
OptionEditor::plugins_stop_with_transport_clicked ()
{
	Config->set_plugins_stop_with_transport (plugins_stop_button.get_active());
}

void
OptionEditor::plugins_on_while_recording_clicked ()
{
	if (session) {
		session->set_recording_plugins (plugins_on_rec_button.get_active());
	}
}

void
OptionEditor::stop_rec_on_xrun_clicked ()
{
	Config->set_stop_recording_on_xrun (stop_rec_on_xrun_button.get_active());
}

void
OptionEditor::stop_at_end_clicked ()
{
	Config->set_stop_at_session_end (stop_at_end_button.get_active());
}
						  

static const struct {
    const char *name;
    guint   modifier;
} modifiers[] = {
	{ "Shift", GDK_SHIFT_MASK },
	{ "Control", GDK_CONTROL_MASK },
	{ "Alt (Mod1)", GDK_MOD1_MASK },
	{ "Control-Shift", GDK_CONTROL_MASK|GDK_SHIFT_MASK },
	{ "Control-Alt", GDK_CONTROL_MASK|GDK_MOD1_MASK },
	{ "Shift-Alt", GDK_SHIFT_MASK|GDK_MOD1_MASK },
	{ "Control-Shift-Alt", GDK_CONTROL_MASK|GDK_SHIFT_MASK|GDK_MOD1_MASK },
	{ "Mod2", GDK_MOD2_MASK },
	{ "Mod3", GDK_MOD3_MASK },
	{ "Mod4", GDK_MOD4_MASK },
	{ "Mod5", GDK_MOD5_MASK },
	{ 0, 0 }
};

void
OptionEditor::setup_keyboard_options ()
{
	vector<string> dumb;
	Label* label;

	keyboard_mouse_table.set_border_width (12);
	keyboard_mouse_table.set_row_spacings (5);
	keyboard_mouse_table.set_col_spacings (5);

	/* internationalize and prepare for use with combos */

	for (int i = 0; modifiers[i].name; ++i) {
		dumb.push_back (_(modifiers[i].name));
	}

	edit_modifier_combo.set_popdown_strings (dumb);
	edit_modifier_combo.get_entry()->set_editable (false);
	edit_modifier_combo.get_entry()->set_name ("OptionsEntry");
	edit_modifier_combo.set_use_arrows_always (true);
	edit_modifier_combo.get_popwin()->unmap_event.connect (slot (*this, &OptionEditor::edit_modifier_chosen));

	for (int x = 0; modifiers[x].name; ++x) {
		if (modifiers[x].modifier == Keyboard::edit_modifier ()) {
			edit_modifier_combo.get_entry()->set_text (_(modifiers[x].name));
			break;
		}
	}

	label = manage (new Label (_("Edit using")));
	label->set_alignment (1.0, 0.5);
		
	keyboard_mouse_table.attach (*label, 0, 1, 0, 1, GTK_FILL|GTK_EXPAND, 0);
	keyboard_mouse_table.attach (edit_modifier_combo, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, 0);

	label = manage (new Label (_("+ button")));

	keyboard_mouse_table.attach (*label, 3, 4, 0, 1, GTK_FILL|GTK_EXPAND, 0);
	keyboard_mouse_table.attach (edit_button_spin, 4, 5, 0, 1, GTK_FILL|GTK_EXPAND, 0);

	edit_button_spin.set_name ("OptionsEntry");
	edit_button_adjustment.set_value (Keyboard::edit_button());
	edit_button_adjustment.value_changed.connect (slot (*this, &OptionEditor::edit_button_changed));

	delete_modifier_combo.set_popdown_strings (dumb);
	delete_modifier_combo.get_entry()->set_editable (false);
	delete_modifier_combo.get_entry()->set_name ("OptionsEntry");
	delete_modifier_combo.set_use_arrows_always (true);
	delete_modifier_combo.get_popwin()->unmap_event.connect (slot (*this, &OptionEditor::delete_modifier_chosen));

	for (int x = 0; modifiers[x].name; ++x) {
		if (modifiers[x].modifier == Keyboard::delete_modifier ()) {
			delete_modifier_combo.get_entry()->set_text (_(modifiers[x].name));
			break;
		}
	}

	label = manage (new Label (_("Delete using")));
	label->set_alignment (1.0, 0.5);
		
	keyboard_mouse_table.attach (*label, 0, 1, 1, 2, GTK_FILL|GTK_EXPAND, 0);
	keyboard_mouse_table.attach (delete_modifier_combo, 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, 0);

	label = manage (new Label (_("+ button")));

	keyboard_mouse_table.attach (*label, 3, 4, 1, 2, GTK_FILL|GTK_EXPAND, 0);
	keyboard_mouse_table.attach (delete_button_spin, 4, 5, 1, 2, GTK_FILL|GTK_EXPAND, 0);

	delete_button_spin.set_name ("OptionsEntry");
	delete_button_adjustment.set_value (Keyboard::delete_button());
	delete_button_adjustment.value_changed.connect (slot (*this, &OptionEditor::delete_button_changed));

	snap_modifier_combo.set_popdown_strings (dumb);
	snap_modifier_combo.get_entry()->set_editable (false);
	snap_modifier_combo.get_entry()->set_name ("OptionsEntry");
	snap_modifier_combo.set_use_arrows_always (true);
	snap_modifier_combo.get_popwin()->unmap_event.connect (slot (*this, &OptionEditor::snap_modifier_chosen));
	
	for (int x = 0; modifiers[x].name; ++x) {
		if (modifiers[x].modifier == (guint) Keyboard::snap_modifier ()) {
			snap_modifier_combo.get_entry()->set_text (_(modifiers[x].name));
			break;
		}
	}

	label = manage (new Label (_("Ignore snap using")));
	label->set_alignment (1.0, 0.5);
	
	keyboard_mouse_table.attach (*label, 0, 1, 2, 3, GTK_FILL|GTK_EXPAND, 0);
	keyboard_mouse_table.attach (snap_modifier_combo, 1, 2, 2, 3, GTK_FILL|GTK_EXPAND, 0);
}

gint
OptionEditor::edit_modifier_chosen (GdkEventAny *ev)
{
	string txt;
	
	txt = edit_modifier_combo.get_entry()->get_text();

	for (int i = 0; modifiers[i].name; ++i) {
		if (txt == _(modifiers[i].name)) {
			Keyboard::set_edit_modifier (modifiers[i].modifier);
			break;
		}
	}
	return TRUE;
}

gint
OptionEditor::delete_modifier_chosen (GdkEventAny *ev)
{
	string txt;
	
	txt = delete_modifier_combo.get_entry()->get_text();

	for (int i = 0; modifiers[i].name; ++i) {
		if (txt == _(modifiers[i].name)) {
			Keyboard::set_delete_modifier (modifiers[i].modifier);
			break;
		}
	}
	return TRUE;
}

gint
OptionEditor::snap_modifier_chosen (GdkEventAny *ev)
{
	string txt;
	
	txt = snap_modifier_combo.get_entry()->get_text();

	for (int i = 0; modifiers[i].name; ++i) {
		if (txt == _(modifiers[i].name)) {
			Keyboard::set_snap_modifier (modifiers[i].modifier);
			break;
		}
	}
	return TRUE;
}

void
OptionEditor::delete_button_changed ()
{
	Keyboard::set_delete_button ((guint) delete_button_adjustment.get_value());
}

void
OptionEditor::edit_button_changed ()
{
	Keyboard::set_edit_button ((guint) edit_button_adjustment.get_value());
}

