/*
    Copyright (C) 2001 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: redirect.h,v 1.29 2004/02/16 04:26:19 pauld Exp $
*/

#ifndef __ardour_redirect_h__
#define __ardour_redirect_h__

#include <string>
#include <vector>
#include <set>
#include <map>
#include <glib.h>
#include <sigc++/signal_system.h>

#include <pbd/lockmonitor.h>
#include <pbd/undo.h>

#include <ardour/ardour.h>
#include <ardour/io.h>
#include <ardour/automation_event.h>

class XMLNode;

namespace ARDOUR {

class Session;

struct RedirectState : public StateManager::State {
    RedirectState (string why) 
	    : StateManager::State (why) {}
    ~RedirectState () {}

    bool active;
};

class Redirect : public IO
{
  public:
	static const string state_node_name;

	enum Placement {
		PreFader,
		PostFader
	};

	Redirect (Session&, const string& name, Placement,
		  int input_min = -1, int input_max = -1, int output_min = -1, int output_max = -1);
	Redirect (const Redirect&);
	virtual ~Redirect ();

	static Redirect *clone (const Redirect&);

	bool active () const { return _active; }
	void set_active (bool yn, void *src);

	unsigned long sort_key() const { return _sort_key; }
	void set_sort_key (unsigned long sk,void *src) { _sort_key = sk; sort_key_changed (this, src); }

	Placement placement() const { return _placement; }
	void set_placement (Placement, void *src);

	static string placement_as_string (Placement);

	virtual void run (vector<Sample *>& ibufs, jack_nframes_t nframes, jack_nframes_t offset) = 0;
	virtual void activate () = 0;
	virtual void deactivate () = 0;
	virtual jack_nframes_t latency() { return 0; }

	SigC::Signal2<void,Redirect*,void*> active_changed;
	SigC::Signal2<void,Redirect*,void*> sort_key_changed;
	SigC::Signal2<void,Redirect*,void*> placement_changed;
	SigC::Signal2<void,Redirect*,bool>  AutomationPlaybackChanged;
	SigC::Signal2<void,Redirect*,unsigned long> AutomationChanged;
	SigC::Signal1<void,Redirect*> GoingAway;

	static SigC::Signal1<void,Redirect*> RedirectCreated;

	XMLNode& get_state (void);
	int set_state (const XMLNode&);

	StateManager::State* state_factory (std::string why) const;
	Change restore_state (StateManager::State&);

	void *get_gui () const { return _gui; }
	void  set_gui (void *p) { _gui = p; }

	virtual string describe_parameter (unsigned long which);
	virtual float default_parameter_value (unsigned long which) {
		return 1.0f;
	}

	int load_automation (string path);
	int save_automation (string path);

	void what_has_automation (set<unsigned long>&) const;
	const set<unsigned long>& what_can_be_automated () const { return can_automate_list; }

	AutomationList& automation_list (unsigned long);
	bool find_next_event (jack_nframes_t, jack_nframes_t, ControlEvent&) const;
	
  protected:
	void set_placement (const string&, void *src);

	/* children may use this stuff as they see fit */

	std::map<unsigned long,AutomationList*> parameter_automation;
	mutable PBD::NonBlockingLock _automation_lock;

	void can_automate (unsigned long);
	set<unsigned long> can_automate_list;

	void store_state (RedirectState&) const;

  private:
	bool _active;
	Placement _placement;
	unsigned long _sort_key;
	void* _gui;  /* generic, we don't know or care what this is */
};

}; /* namespace ARDOUR */

#endif /* __ardour_redirect_h__ */
