/*
    Copyright (C) 2000-2001 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: region.h,v 1.36 2004/02/20 03:14:33 pauld Exp $
*/

#ifndef __ardour_region_h__
#define __ardour_region_h__

#include <pbd/undo.h>

#include <ardour/ardour.h>
#include <ardour/logcurve.h>
#include <ardour/state_manager.h>

class XMLNode;

namespace ARDOUR {

class Playlist;
class Source;

struct RegionState : public StateManager::State {

    RegionState (std::string why) : StateManager::State (why) {}
    
    jack_nframes_t      _start;
    jack_nframes_t      _length;
    jack_nframes_t      _position;
    unsigned int        _flags;
    jack_nframes_t      _sync_offset;
    layer_t        	_layer;
    string              _name;        
    mutable bool        _first_edit;
};

class Region : public Stateful, public StateManager
{
  public:
	enum Flag {
		Muted = 0x1,
		Opaque = 0x2,
		EnvelopeActive = 0x4,
		DefaultFadeIn = 0x8,
		DefaultFadeOut = 0x10,
		Locked = 0x20,
		Automatic = 0x40,
		WholeFile = 0x80,
		FadeIn = 0x100,
		FadeOut = 0x200,
		Copied = 0x400,
		Import = 0x800,
		External = 0x1000,
		//
		range_guarantoor = USHRT_MAX
	};

 	static const Flag DefaultFlags = Flag (Opaque|DefaultFadeIn|DefaultFadeOut|FadeIn|FadeOut);

	static Change FadeChanged;
	static Change SyncOffsetChanged;
	static Change MuteChanged;
	static Change OpacityChanged;
	static Change LockChanged;
	static Change LayerChanged;

	Region (jack_nframes_t start, jack_nframes_t length, 
		const string& name, layer_t = 0, Flag flags = DefaultFlags);
	Region (const Region&, jack_nframes_t start, jack_nframes_t length, const string& name, layer_t = 0, Flag flags = DefaultFlags);
	Region (const Region&, const string& name);
	Region (const Region&);
	Region (const XMLNode&);
	~Region();

	ARDOUR::id_t id() const { return _id; }

	/* Note: changing the name of a Region does not constitute an edit */

	string name() const { return _name; }
	void set_name (string str);

	jack_nframes_t position () const { return _position; }
	jack_nframes_t start () const { return _start; }
	jack_nframes_t length() const { return _length; }
	jack_nframes_t sync_offset() const { return _sync_offset; }
	layer_t layer () const { return _layer; }
	
	/* first_frame() is an alias; last_frame() just hides some math */

	jack_nframes_t first_frame() const { return _position; }
	jack_nframes_t last_frame() const { return _position + _length - 1; }

	bool muted() const { return _flags & Muted; }
	bool opaque () const { return _flags & Opaque; }
	bool envelope_active () const { return _flags & EnvelopeActive; }
	bool locked() const { return _flags & Locked; }
	bool automatic() const { return  _flags & Automatic; }
	bool whole_file() const { return _flags & WholeFile ; }
	Flag flags() const { return _flags; }

	void freeze ();
	void thaw (const string& why);

	bool covers (jack_nframes_t frame) const {
		return _position <= frame && frame < _position + _length;
	}

	OverlapType coverage (jack_nframes_t start, jack_nframes_t end) {
		jack_nframes_t rend = _position + _length - 1;

		if ((start >= _position) && (end <= rend)) {
			return OverlapInternal;
		}
		if ((end >= _position) && (end <= rend)) {
			return OverlapStart;
		}
		if ((start >= _position) && (start <= rend)) {
			return OverlapEnd;
		}
		if ((_position >= start) && (_position <= end) && (rend <= end)) {
			return OverlapExternal;
		}
		return OverlapNone;
	}

	virtual jack_nframes_t read_at (Sample *buf, Sample *mixdown_buffer, 
					float *gain_buffer, jack_nframes_t position, jack_nframes_t cnt, 
					unsigned int chan_n = 0,
					jack_nframes_t read_frames = 0,
					jack_nframes_t skip_frames = 0) const = 0;

	/* EDITING OPERATIONS */

	void set_length (jack_nframes_t, void *src);
	void set_start (jack_nframes_t, void *src);
	void set_position (jack_nframes_t, void *src);
	void set_position_on_top (jack_nframes_t, void *src);

	void nudge_position (long, void *src);

	void trim_start (jack_nframes_t new_position, void *src);
	void trim_front (jack_nframes_t new_position, void *src);
	void trim_end (jack_nframes_t new_position, void *src);
	void trim_to (jack_nframes_t position, jack_nframes_t length, void *src);
	
	void set_layer (layer_t l); /* ONLY Playlist can call this */
	void raise ();
	void lower ();
	void raise_to_top ();
	void lower_to_bottom ();

	void set_sync_offset (jack_nframes_t n);
	void set_muted (bool yn);
	void set_opaque (bool yn);
	void set_envelope_active (bool yn);
	void set_locked (bool yn);

	virtual unsigned long read_data_count() const { return _read_data_count; }

	ARDOUR::Playlist* playlist() const { return _playlist; }

	virtual UndoAction get_memento() const = 0;

	void set_playlist (ARDOUR::Playlist*);

	virtual void lock_sources () {}
	virtual void unlock_sources () {}

	/* serialization */
	
	XMLNode& get_state ();
	int      set_state (const XMLNode&);

	SigC::Signal1<void,Region*> GoingAway;

	/* This is emitted only when a new id is assigned. Therefore,
	   in a pure Region copy, it will not be emitted.

	   It must be emitted by derived classes, not Region
	   itself, to permit dynamic_cast<> to be used to 
	   infer the type of Region.
	*/

	static SigC::Signal1<void,Region*> RegionCreated;

  protected:

	jack_nframes_t        _start;
	jack_nframes_t        _length;
	jack_nframes_t        _position;
	Flag 	              _flags;
	jack_nframes_t        _sync_offset;
	layer_t        	      _layer;
	string                _name;        
	mutable bool          _first_edit;
	int                   _frozen;
	PBD::Lock              lock;
	ARDOUR::id_t          _id;
	ARDOUR::Playlist*     _playlist;
	mutable unsigned long _read_data_count; // modified in read()
	Change                 pending_changed;

	XMLNode& get_short_state (); /* used only by Session */

	/* state management */

	void send_change (Change);
	void send_state_changed ();

	/* derived classes need these during their own state management calls */

	void   store_state (RegionState&) const;
	Change restore_and_return_flags (RegionState&);
	
	XMLNode& _get_state (bool is_short);
	
	void trim_to_internal (jack_nframes_t position, jack_nframes_t length, void *src);

	bool copied() const { return _flags & Copied; }
	void maybe_uncopy ();
	void rename_after_first_edit ();
	
	virtual bool verify_start (jack_nframes_t) = 0;
	virtual bool verify_start_and_length (jack_nframes_t, jack_nframes_t) = 0;
	virtual bool verify_start_mutable (jack_nframes_t&_start) = 0;
	virtual bool verify_length (jack_nframes_t) = 0;
	virtual void recompute_at_start () = 0;
	virtual void recompute_at_end () = 0;
};

} /* namespace ARDOUR */

#endif /* __ardour_region_h__ */
