/*
    Copyright (C) 2000-2003 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: diskstream.cc,v 1.101 2004/02/29 23:33:56 pauld Exp $
*/

#include <fstream>

#include <cstdio>
#include <unistd.h>
#include <cmath>
#include <cerrno>
#include <string>
#include <climits>
#include <fcntl.h>
#include <stdlib.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include <pbd/error.h>
#include <pbd/basename.h>
#include <pbd/lockmonitor.h>
#include <pbd/xml++.h>

#include <ardour/ardour.h>
#include <ardour/audioengine.h>
#include <ardour/diskstream.h>
#include <ardour/utils.h>
#include <ardour/configuration.h>
#include <ardour/filesource.h>
#include <ardour/send.h>
#include <ardour/audioplaylist.h>
#include <ardour/cycle_timer.h>
#include <ardour/audioregion.h>

#include "i18n.h"
#include <locale.h>

using namespace ARDOUR;

jack_nframes_t DiskStream::disk_io_chunk_frames;
SigC::Signal1<void,DiskStream&> DiskStream::DiskStreamCreated;

DiskStream::DiskStream (Session &sess, const string &name, Flag flag)
	: IO (sess, name, 1, -1, 0, 0)
{
	/* prevent any write sources from being created */

	in_set_state = true;
	init (flag);
	in_set_state = false;

	use_new_playlist ();

	 DiskStreamCreated (*this); /* EMIT SIGNAL */
}
	
DiskStream::DiskStream (Session& sess, const XMLNode& node)
	: IO (sess, "to be renamed", 1, -1, 0, 0)
{
	in_set_state = true;
	init (Recordable);
	in_set_state = false;

	ports_created_c = PortsCreated.connect (slot (*this, &DiskStream::ports_created));
	
	if (set_state (node)) {
		throw failed_constructor();
	}

	 DiskStreamCreated (*this); /* EMIT SIGNAL */
}

void
DiskStream::init_channel (ChannelInfo &chan)
{
	chan.wrap_buffer = 0;
	chan.speed_buffer = 0;
	chan.peak_power = 0.0;
	chan.fades_source = 0;
	chan.write_source = 0;
	chan.source = 0;
	chan.current_capture_buffer = 0;
	chan.current_playback_buffer = 0;
	chan.scrub_buffer = 0;
	chan.scrub_forward_buffer = 0;
	chan.scrub_reverse_buffer = 0;
	
	chan.playback_buf = new RingBuffer<Sample> (_session.diskstream_buffer_size());
	chan.capture_buf = new RingBuffer<Sample> (_session.diskstream_buffer_size());

	/* touch the ringbuffer buffers, which will cause
	   them to be mapped into locked physical RAM if
	   we're running with mlockall(). this doesn't do
	   much if we're not.  
	*/
	memset (chan.playback_buf->buffer(), 0, sizeof (Sample) * chan.playback_buf->bufsize());
	memset (chan.capture_buf->buffer(), 0, sizeof (Sample) * chan.capture_buf->bufsize());
}


void
DiskStream::init (Flag f)
{
	_refcnt = 0;
	_flags = f;
	rec_monitoring_off_for_roll = false;
	_playlist = 0;
	i_am_the_modifier = 0;
	atomic_set (&_record_enabled, 0);
	was_recording = false;
	capture_start_frame = 0;
	capture_captured = 0;
	_visible_speed = 1.0f;
	_actual_speed = 1.0f;
	_buffer_reallocation_required = false;
	_seek_required = false;
	legal_recording_frame = 0;
	_capture_offset = 0;
	_processed = false;
	_slaved = false;
	_scrubbing = false;
	scrub_offset = 0;
	scrub_buffer_size = 0;
	adjust_capture_position = false;
	loop_location = 0;
	wrap_buffer_size = 0;
	speed_buffer_size = 0;
	last_phase = 0;
	phi = (guint64) (0x1000000);
	file_frame = 0;
	playback_sample = 0;
	playback_distance = 0;
	_read_data_count = 0;
	_write_data_count = 0;

	/* there are no channels at this point, so these
	   two calls just get speed_buffer_size and wrap_buffer
	   size setup without duplicating their code.
	*/

	set_block_size (_session.get_block_size());
	allocate_temporary_buffers ();

	pending_overwrite = false;
	overwrite_frame = 0;
	overwrite_queued = false;
	input_change_pending = false;

	add_channel ();
	_n_channels = 1;
	
	input_changed.connect (slot (*this, &DiskStream::handle_input_change));
	input_configuration_changed.connect (slot (*this, &DiskStream::handle_input_configuration_change));
}

int
DiskStream::create_input_port ()
{
	return add_input_port ("", this);
}

int
DiskStream::connect_input_port ()
{
	unsigned long which_input = _session.get_next_diskstream_id();

	Port* port = _session.engine().get_port_by_name (_session.engine().get_nth_physical_input (which_input));

	if (port) {
		return set_input (port, this);
	} else {
		// warning << _("No physical inputs: diskstreams not connected by default") << endmsg;
		return 0;
	}
}

void
DiskStream::destroy_channel (ChannelInfo &chan)
{
	if (chan.write_source) {
		chan.write_source->release ();
		chan.write_source = 0;
	}
		
	if (chan.scrub_forward_buffer) {
		delete [] chan.scrub_forward_buffer;
		delete [] chan.scrub_reverse_buffer;

		chan.scrub_forward_buffer = 0;
		chan.scrub_reverse_buffer = 0;
	}

	if (chan.speed_buffer) {
		delete [] chan.speed_buffer;
	}

	if (chan.wrap_buffer) {
		delete [] chan.wrap_buffer;
	}
	
	delete chan.playback_buf;
	delete chan.capture_buf;

	chan.playback_buf = 0;
	chan.capture_buf = 0;
}

DiskStream::~DiskStream ()
{
	LockMonitor lm (state_lock, __LINE__, __FILE__);

	if (_playlist) {
		_playlist->unref ();
	}

	for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
		destroy_channel((*chan));
	}
	
	channels.clear();
}

void
DiskStream::handle_input_configuration_change (void *src)
{
	input_change_pending = true;
	_session.request_input_change_handling ();
}

void
DiskStream::non_realtime_input_configuration_change ()
{
	if (!input_change_pending) {
		return;
	}

	input_change_pending = false;

	if (n_inputs() > _n_channels) {

		// we need to add new channel infos

		LockMonitor lm (state_lock, __LINE__, __FILE__);

		int diff = n_inputs() - channels.size();

		for (int i = 0; i < diff; ++i) {
			add_channel ();
		}

	} else if (n_inputs() < _n_channels) {

		// we need to get rid of channels

		LockMonitor lm (state_lock, __LINE__, __FILE__);
		
		int diff = channels.size() - n_inputs();

		for (int i = 0; i < diff; ++i) {
			remove_channel ();
		}
	}

	/* reset sources */

	handle_input_change (0);

	/* now refill channel buffers */

	if (speed() != 1.0f || speed() != -1.0f) {
		seek ((jack_nframes_t) (_session.transport_frame() * (double) speed()));
	}
	else {
		seek (_session.transport_frame());
	}
}

void 
DiskStream::handle_input_change (void *src)
{
	for (unsigned int n=0; n < n_inputs(); ++n) {

		const char **connections = input(n)->get_connections ();
		ChannelInfo& chan = channels[n];
		
		if (connections == 0 || connections[0] == 0) {
			
			if (chan.source) {
				// _source->disable_metering ();
			}

			chan.source = 0;

		} else {
			chan.source = _session.engine().get_port_by_name (connections[0]);
		}
		
		if (connections) {
			free (connections);
		}
	}

	set_capture_offset ();
}

int
DiskStream::find_and_use_playlist (const string& name)
{
	Playlist* pl;
	AudioPlaylist* playlist;
		
	if ((pl = _session.get_playlist (name)) == 0) {
		error << compose(_("DiskStream: Session doesn't know about a Playlist called \"%1\""), name) << endmsg;
		return -1;
	}

	if ((playlist = dynamic_cast<AudioPlaylist*> (pl)) == 0) {
		error << compose(_("DiskStream: Playlist \"%1\" isn't an audio playlist"), name) << endmsg;
		return -1;
	}

	return use_playlist (playlist);
}

int
DiskStream::use_playlist (AudioPlaylist* playlist)
{
	{
		LockMonitor lm (state_lock, __LINE__, __FILE__);

		plstate_connection.disconnect();
		plmod_connection.disconnect ();
		
		if (_playlist) {
			_playlist->unref();
		}
			
		_playlist = playlist;
		playlist->ref();
			
		if (!in_set_state && recordable()) {
			for (unsigned int n=0; n < channels.size(); ++n) {
				use_new_write_source (n);
			}
		}
			
		plstate_connection = _playlist->StateChanged.connect (slot (this, &DiskStream::playlist_changed));
		plmod_connection = _playlist->Modified.connect (slot (this, &DiskStream::playlist_modified));
	}

	if (!overwrite_queued) {
		_session.request_overwrite_buffer (this);
		overwrite_queued = true;
	}

	 PlaylistChanged (); /* EMIT SIGNAL */
	
	return 0;
}

int
DiskStream::use_new_playlist ()
{
	string newname;
	AudioPlaylist* playlist;

	if (_playlist) {
		newname = Playlist::bump_name (_playlist->name());
	} else {
		newname = Playlist::bump_name (_name);
	}

	if ((playlist = new AudioPlaylist (_session, newname, hidden())) != 0) {
		playlist->set_orig_disktsream_id (id());
		return use_playlist (playlist);
	} else { 
		return -1;
	}
}

int
DiskStream::use_copy_playlist ()
{
	if (_playlist == 0) {
		error << compose(_("DiskStream %1: there is no existing playlist to make a copy of!"), _name) << endmsg;
		return -1;
	}

	string newname;
	AudioPlaylist* playlist;

	newname = Playlist::bump_name (_playlist->name());
	
	if ((playlist  = new AudioPlaylist (*_playlist, newname)) != 0) {
		playlist->set_orig_disktsream_id (id());
		return use_playlist (playlist);
	} else { 
		return -1;
	}
}

void
DiskStream::set_name (string str, void *src)
{
	_playlist->set_name (str);
	IO::set_name (str, src);

	if (!in_set_state && recordable()) {
		/* open new capture files so that they have the correct name */
		for (unsigned int n = 0; n < _n_channels; ++n) {
			use_new_write_source (n);
		}
	}
}

void
DiskStream::set_speed (float sp)
{
	_session.request_diskstream_speed (*this, sp);

	/* to force a rebuffering at the right place */
	playlist_modified();
}

bool
DiskStream::realtime_set_speed (float sp)
{
	bool changed = false;

	{
		float new_speed = sp * _session.transport_speed();

		if (_visible_speed != sp) {
			_visible_speed = sp;
			changed = true;
		}
		
		if (new_speed != _actual_speed) {

			jack_nframes_t required_wrap_size = (jack_nframes_t) floor (_session.get_block_size() * 
										    fabs (new_speed)) + 1;
			
			if (required_wrap_size > wrap_buffer_size) {
				_buffer_reallocation_required = true;
			}

			_actual_speed = new_speed;
			phi = (guint64) (0x1000000 * fabs(_actual_speed));
		}
	}

	if (changed) {
		_seek_required = true;
		 speed_changed (); /* EMIT SIGNAL */
	}

	return true;
}

void
DiskStream::non_realtime_set_speed ()
{
	if (_buffer_reallocation_required)
	{
		LockMonitor lm (state_lock, __LINE__, __FILE__);
		allocate_temporary_buffers ();

		_buffer_reallocation_required = false;
	}

	if (_seek_required) {
		if (speed() != 1.0f || speed() != -1.0f) {
			seek ((jack_nframes_t) (_session.transport_frame() * (double) speed()), true);
		}
		else {
			seek (_session.transport_frame(), true);
		}

		_seek_required = false;
	}
}

void
DiskStream::prepare ()
{
	_processed = false;
	playback_distance = 0;
}

int
DiskStream::process (jack_nframes_t nframes, jack_nframes_t offset, bool can_record, bool rec_monitors_input)
{
	unsigned int n;
	ChannelList::iterator chan;
	int ret = -1;

	/* if we've already processed the frames corresponding to this call,
	   just return. this allows multiple routes that are taking input
	   from this diskstream to call our ::process() method, but have
	   this stuff only happen once.
	*/

	if (_processed) {
		return 0;
	}
	
	if (nframes == 0) {
		_processed = true;
		return 0;
	}

	bool actually_recording;
	bool nominally_recording;

	/* This lock is held until the end of DiskStream::commit, so these two functions
	   must always be called as a pair. The only exception is if this function
	   returns a non-zero value, in which case, ::commit should not be called.
	*/

	if (pthread_mutex_trylock (state_lock.mutex())) {
		return 1;
	}

	adjust_capture_position = false;

	for (chan = channels.begin(); chan != channels.end(); ++chan) {
		(*chan).current_capture_buffer = 0;
		(*chan).current_playback_buffer  = 0;
	}
	
	nominally_recording = (can_record && record_enabled() && channels[0].source);
	actually_recording = (nominally_recording && _session.transport_frame() >= legal_recording_frame);
	
	if (!actually_recording) {

		if (was_recording) {
			finish_capture (rec_monitors_input);
		}

	} else {

		if (Config->get_use_hardware_monitoring() && record_enabled() && rec_monitoring_off_for_roll && rec_monitors_input) {
			for (chan = channels.begin(); chan != channels.end(); ++chan) {
				(*chan).source->ensure_monitor_input (true);
			}
			rec_monitoring_off_for_roll = false;
		}

		if (!was_recording) {
			capture_start_frame = _session.transport_frame() - _capture_offset;
			capture_captured = 0;
			was_recording = true;
		}
	}

	if (nominally_recording) {
		for (n = 0, chan = channels.begin(); chan != channels.end(); ++chan, ++n) {
		
			(*chan).capture_buf->get_write_vector (&(*chan).capture_vector);
			
			if (nframes > (*chan).capture_vector.len[0]) {
				cerr << "DiskStream " << _name << ":" << n
				     << ": capture overrun (called with "
				     << nframes
				     << " samples but only "
				     << (*chan).capture_vector.len[0]
				     << " available; other half: "
				     << (*chan).capture_vector.len[1]
				     << ')'
				     << endl;
				goto out;
			}
			
			(*chan).current_capture_buffer = (*chan).capture_vector.buf[0];
			memcpy ((*chan).current_capture_buffer, input(n)->get_buffer (nframes) + offset, sizeof (Sample) * nframes);
		}
	}

	if (actually_recording) {
		
		/* data will be written to disk */

		for (chan = channels.begin(); chan != channels.end(); ++chan) {
			(*chan).current_playback_buffer = (*chan).current_capture_buffer;
		}

		playback_distance = nframes;
		adjust_capture_position = true;

	} else if (nominally_recording) {

		/* can't do actual capture yet - waiting for latency effects to finish */

		for (chan = channels.begin(); chan != channels.end(); ++chan) {
			(*chan).current_playback_buffer = (*chan).current_capture_buffer;
		}

		playback_distance = nframes;

	} else {

		/* we're doing playback */

		jack_nframes_t necessary_samples;

		if (_actual_speed != 1.0f) {
			necessary_samples = (jack_nframes_t) floor ((nframes * fabs (_actual_speed))) + 1;
		} else {
			necessary_samples = nframes;
		}

		if (_scrubbing) {

			if (_actual_speed < 0.0f) {
				if (scrub_start > _session.transport_frame()) {
					scrub_offset = scrub_start - _session.transport_frame();
				} else {
					for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
						(*chan).current_playback_buffer = 0;
					}
					
					ret = 0;
					goto out;
				}
			} else {
				scrub_offset = _session.transport_frame() - scrub_start;
			}

			
			if (scrub_offset + necessary_samples > scrub_buffer_size) {
				for (chan = channels.begin(); chan != channels.end(); ++chan) {
					(*chan).current_playback_buffer = 0;
				}
				ret = 0;
				goto out;
			}

			for (chan = channels.begin(); chan != channels.end(); ++chan) {
				
				(*chan).playback_vector.buf[0] = &(*chan).scrub_buffer[scrub_offset];
				(*chan).playback_vector.len[0] = scrub_buffer_size - scrub_offset;
				(*chan).playback_vector.buf[1] = 0;
				(*chan).playback_vector.len[1] = 0;
			}

		} else {
			for (chan = channels.begin(); chan != channels.end(); ++chan) {
				(*chan).playback_buf->get_read_vector (&(*chan).playback_vector);
			}
		}

		n = 0;			

		for (chan = channels.begin(); chan != channels.end(); ++chan, ++n) {
		
			if (necessary_samples <= (*chan).playback_vector.len[0]) {
				(*chan).current_playback_buffer = (*chan).playback_vector.buf[0];
				
			} else {
				jack_nframes_t total = (*chan).playback_vector.len[0] + (*chan).playback_vector.len[1];
				
				if (necessary_samples > total) {
					cerr << "DiskStream " << _name << ":" << n
					     << ": playback underrun (called with " 
					     << necessary_samples
					     << " samples but only "
					     << (*chan).playback_vector.len[0]
					     << " available; other half: "
					     << (*chan).playback_vector.len[1]
					     << ") [speed="
					     << _actual_speed
					     << ']'
					     << endl;
					goto out;
					
				} else {
					
					memcpy ((char *) (*chan).wrap_buffer, (*chan).playback_vector.buf[0],
						(*chan).playback_vector.len[0] * sizeof (Sample));
					memcpy ((*chan).wrap_buffer + (*chan).playback_vector.len[0], (*chan).playback_vector.buf[1], 
						(necessary_samples - (*chan).playback_vector.len[0]) * sizeof (Sample));
					
					(*chan).current_playback_buffer = (*chan).wrap_buffer;
				}
			}
		} 

		if (_actual_speed != 1.0f && _actual_speed != -1.0f) {
			
			guint64 phase = last_phase;
			jack_nframes_t i = 0;

			// Linearly interpolate into the alt buffer
			// using 40.24 fixp maths (swh)

			for (chan = channels.begin(); chan != channels.end(); ++chan) {

				float fr;
				
				i = 0;
				phase = last_phase;

				for (jack_nframes_t outsample = 0; outsample < nframes; ++outsample) {
					i = phase >> 24;
					fr = (phase & 0xFFFFFF) / 16777216.0f;
					(*chan).speed_buffer[outsample] = 
						(*chan).current_playback_buffer[i] * (1.0f - fr) +
						(*chan).current_playback_buffer[i+1] * fr;
					phase += phi;
				}
				
				(*chan).current_playback_buffer = (*chan).speed_buffer;
			}

			playback_distance = i + 1;
			last_phase = (phase & 0xFFFFFF);

		} else {
			playback_distance = nframes;
		}

	}

#ifdef DEBUG_BUFFER_CONTENTS
	cerr << _session.transport_frame() << " @ " << _session.transport_speed() << ": " << _name << ": current playback buffer = "
	     << channels.front().current_playback_buffer - channels.front().playback_buf->buffer()
	     << " (" << channels.front().current_playback_buffer 
	     << ") = " << playback_sample 
	     << endl;
#endif

	ret = 0;

  out:
	_processed = true;

	if (ret) {

		/* we're exiting with failure, so ::commit will not
		   be called. unlock the state lock.
		*/
		
		pthread_mutex_unlock (state_lock.mutex());
	} 

	return ret;
}

void
DiskStream::recover ()
{
	pthread_mutex_unlock (state_lock.mutex());
	_processed = false;
}

bool
DiskStream::commit (jack_nframes_t nframes)
{
	bool need_butler = false;

	if (!_scrubbing) {

		if (_actual_speed < 0.0) {
			playback_sample -= playback_distance;
		} else {
			playback_sample += playback_distance;
		}

		for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
			
			(*chan).playback_buf->increment_read_ptr (playback_distance);
			
			if (adjust_capture_position) {
				(*chan).capture_buf->increment_write_ptr (nframes);
			}
		}

		if (adjust_capture_position) {
			capture_captured += nframes;
		}

		if (_slaved) {
			need_butler = channels[0].playback_buf->write_space() >= channels[0].playback_buf->bufsize() / 2;
		} else {
			need_butler = channels[0].playback_buf->write_space() >= disk_io_chunk_frames
				|| channels[0].capture_buf->read_space() >= disk_io_chunk_frames;
		}
		
	}

	pthread_mutex_unlock (state_lock.mutex());

	_processed = false;

	return need_butler;
}

void
DiskStream::start_scrub (jack_nframes_t start)
{
	unsigned int n;
	ChannelList::iterator c;

	scrub_buffer_size = _session.scrub_buffer_size();

	if (start > _session.frame_rate()) {
		scrub_start = start - _session.frame_rate();
	} else {
		scrub_start = 0;
	}

	for (n = 0, c = channels.begin(); c != channels.end(); ++c, ++n) {

		ChannelInfo& channel (*c);
	
		if (channel.scrub_forward_buffer) {
			delete [] channel.scrub_forward_buffer;
			delete [] channel.scrub_reverse_buffer;
		}

		channel.scrub_forward_buffer = new Sample[scrub_buffer_size];
		channel.scrub_reverse_buffer = new Sample[scrub_buffer_size];
		scrub_offset = 0;
		_scrubbing = true;

		Sample* mixdown_buffer = new Sample[scrub_buffer_size];
		gain_t* gain_buffer = new gain_t[scrub_buffer_size];
	
		_playlist->read (channel.scrub_forward_buffer, mixdown_buffer, gain_buffer, scrub_start, scrub_buffer_size, n);

		delete [] mixdown_buffer;
		delete [] gain_buffer;

		for (jack_nframes_t a = 0, b = scrub_buffer_size-1; a < scrub_buffer_size; ++a, --b) {
			channel.scrub_reverse_buffer[a] = channel.scrub_forward_buffer[b];
		}

		channel.scrub_buffer = channel.scrub_forward_buffer;
	}
}

void
DiskStream::end_scrub ()
{
	for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {	
		if ((*chan).scrub_forward_buffer) {
			delete [] (*chan).scrub_forward_buffer;
			delete [] (*chan).scrub_reverse_buffer;
			(*chan).scrub_forward_buffer = 0;
			(*chan).scrub_reverse_buffer = 0;
			scrub_buffer_size = 0;
		}
	}
	
	_scrubbing = false;
	scrub_offset = 0;
}

void
DiskStream::set_pending_overwrite (bool yn)
{
	/* called from audio thread, so we can use the read ptr and playback sample as we wish */
	
	pending_overwrite = yn;

	overwrite_frame = playback_sample;
	overwrite_offset = channels.front().playback_buf->get_read_ptr();
}

int
DiskStream::overwrite_existing_buffers ()
{
 	Sample* mixdown_buffer;
 	float* gain_buffer;
 	int ret = -1;
	bool reversed = (_visible_speed * _session.transport_speed()) < 0.0f;

	overwrite_queued = false;
 
	/* assume all are the same size */
	jack_nframes_t size = channels[0].playback_buf->bufsize();
	
 	mixdown_buffer = new Sample[size];
 	gain_buffer = new float[size];
 
	/* reduce size so that we can fill the buffer correctly. */
	size--;
	
	unsigned int n=0;
	jack_nframes_t start;

	for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan, ++n) {

		start = overwrite_frame;
		jack_nframes_t cnt = size;
		
		/* to fill the buffer without resetting the playback sample, we need to
		   do it one or two chunks (normally two).

		   |----------------------------------------------------------------------|

		                       ^
				       overwrite_offset
		    |<- second chunk->||<----------------- first chunk ------------------>|
		   
		*/
		
		jack_nframes_t to_read = size - overwrite_offset;

		if (read ((*chan).playback_buf->buffer() + overwrite_offset, mixdown_buffer, gain_buffer, 
			  start, to_read, *chan, n, reversed)) {
			error << compose(_("DiskStream %1: when refilling, cannot read %2 from playlist at frame %3"),
					 _id, size, playback_sample) << endmsg;
			goto out;
		}
			
		if (cnt > to_read) {

			cnt -= to_read;
		
			if (read ((*chan).playback_buf->buffer(), mixdown_buffer, gain_buffer, 
				  start, cnt, *chan, n, reversed)) {
				error << compose(_("DiskStream %1: when refilling, cannot read %2 from playlist at frame %3"),
						 _id, size, playback_sample) << endmsg;
				goto out;
			}
		}
	}

	ret = 0;
 
  out:
	pending_overwrite = false;
 	delete [] gain_buffer;
 	delete [] mixdown_buffer;
 	return ret;
}

int
DiskStream::seek (jack_nframes_t frame, bool complete_refill)
{
	LockMonitor lm (state_lock, __LINE__, __FILE__);
	unsigned int n;
	int ret;
	ChannelList::iterator chan;

	legal_recording_frame = _session.transport_frame() + _capture_offset;	

	for (n = 0, chan = channels.begin(); chan != channels.end(); ++chan, ++n) {
		(*chan).playback_buf->reset ();
		(*chan).capture_buf->reset ();
	}

	playback_sample = frame;
	file_frame = frame;

	if (complete_refill) {
		while ((ret = do_refill (0, 0)) > 0);
	} else {
		ret = do_refill (0, 0);
	}

	return ret;
}

int
DiskStream::read (Sample* buf, Sample* mixdown_buffer, float* gain_buffer, jack_nframes_t& start, jack_nframes_t cnt, 
		  ChannelInfo& channel_info, int channel, bool reversed)
{
	jack_nframes_t this_read = 0;
	bool reloop = false;
	jack_nframes_t loop_end = 0;
	jack_nframes_t loop_start = 0;
	jack_nframes_t loop_length = 0;
	jack_nframes_t offset = 0;
	Location *loc = 0;

	if (!reversed) {
		/* Make the use of a Location atomic for this read operation.
		   
		   Note: Locations don't get deleted, so all we care about
		   when I say "atomic" is that we are always pointing to
		   the same one and using a start/length values obtained
		   just once.
		*/
		
		if ((loc = loop_location) != 0) {
			loop_start = loc->start();
			loop_end = loc->end();
			loop_length = loop_end - loop_start + 1;
		}
		
		/* if we are looping, ensure that the first frame we read is at the correct
		   position within the loop.
		*/
		
		if (loc && start > loop_end) {
			start = loop_start + (start % loop_length);
		}
	}

	while (cnt) {

		/* take any loop into account. we can't read past the end of the loop. */

		if (loc && (loop_end - start < cnt)) {
			this_read = loop_end - start;
			reloop = true;
		} else {
			reloop = false;
			this_read = cnt;
		}

		if (this_read == 0) {
			break;
		}

		this_read = min(cnt,this_read);

#ifdef DEBUG_BUFFER_CONTENTS
		cerr << _name 
		     << ": fill buffer["
		     << (buf+offset) - channel_info.playback_buf->buffer()
		     << "] from " << start << ".." << this_read+start
		     << endl;
#endif

		if (_playlist->read (buf+offset, mixdown_buffer, gain_buffer, start, this_read, channel) != this_read) {
			error << compose(_("DiskStream %1: cannot read %2 from playlist at frame %3"), _id, this_read, 
					 start) << endmsg;
			return -1;
		}

		_read_data_count = _playlist->read_data_count();
		
		if (reversed) {

			/* don't adjust start, since caller has already done that
			 */

			swap_by_ptr (buf, buf + this_read - 1);
			
		} else {
			
			/* if we read to the end of the loop, go back to the beginning */
			
			if (reloop) {
				start = loop_start;
			} else {
				start += this_read;
			}
		} 

		cnt -= this_read;
		offset += this_read;
	}

	return 0;
}

int
DiskStream::do_refill (Sample* mixdown_buffer, float* gain_buffer)
{
	gint32 ret = 0;
	jack_nframes_t to_read;
	RingBuffer<Sample>::rw_vector vector;
	bool free_mixdown;
	bool free_gain;
	bool reversed = (_visible_speed * _session.transport_speed()) < 0.0f;
	jack_nframes_t total_space;
	jack_nframes_t zero_fill;
	unsigned int chan_n;
	ChannelList::iterator i;
	jack_nframes_t ts;

	channels.front().playback_buf->get_write_vector (&vector);
	
	if ((total_space = vector.len[0] + vector.len[1]) == 0) {
		return 0;
	}
	
	/* if there are 2+ chunks of disk i/o possible for
	   this track, let the caller know so that it can arrange
	   for us to be called again, ASAP.
	*/
	
	if (total_space >= (_slaved?3:2) * disk_io_chunk_frames) {
		ret = 1;
	}
	
	/* if we're running close to normal speed and there isn't enough 
	   space to do disk_io_chunk_frames of I/O, then don't bother.  
	   
	   at higher speeds, just do it because the sync between butler
	   and audio thread may not be good enough.
	*/
	
	if ((total_space < disk_io_chunk_frames) && fabs (_actual_speed) < 2.0f) {
		return 0;
	}
	
	/* when slaved, don't try to get too close to the read pointer. this
	   leaves space for the buffer reversal to have something useful to
	   work with.
	*/
	
	if (_slaved && total_space < (channels.front().playback_buf->bufsize() / 2)) {
		return 0;
	}

	total_space = min (disk_io_chunk_frames, total_space);

	if (reversed) {

		if (file_frame == 0) {

			/* at start: nothing to do but fill with silence */

			for (chan_n = 0, i = channels.begin(); i != channels.end(); ++i, ++chan_n) {
					
				ChannelInfo& chan (*i);
				chan.playback_buf->get_write_vector (&vector);
				memset (vector.buf[0], 0, sizeof(Sample) * vector.len[0]);
				if (vector.len[1]) {
					memset (vector.buf[1], 0, sizeof(Sample) * vector.len[1]);
				}
				chan.playback_buf->increment_write_ptr (vector.len[0] + vector.len[1]);
			}
			return 0;
		}

		if (file_frame < total_space) {

			/* too close to the start: read what we can, 
			   and then zero fill the rest 
			*/

			zero_fill = total_space - file_frame;
			total_space = file_frame;
			file_frame = 0;

		} else {
			
			/* move read position backwards because we are going
			   to reverse the data.
			*/
			
			file_frame -= total_space;
			zero_fill = 0;
		}

	} else {

		if (file_frame == max_frames) {

			/* at end: nothing to do but fill with silence */
			
			for (chan_n = 0, i = channels.begin(); i != channels.end(); ++i, ++chan_n) {
					
				ChannelInfo& chan (*i);
				chan.playback_buf->get_write_vector (&vector);
				memset (vector.buf[0], 0, sizeof(Sample) * vector.len[0]);
				if (vector.len[1]) {
					memset (vector.buf[1], 0, sizeof(Sample) * vector.len[1]);
				}
				chan.playback_buf->increment_write_ptr (vector.len[0] + vector.len[1]);
			}
			return 0;
		}
		
		if (file_frame > max_frames - total_space) {

			/* to close to the end: read what we can, and zero fill the rest */

			zero_fill = total_space - (max_frames - file_frame);
			total_space = max_frames - file_frame;

		} else {
			zero_fill = 0;
		}
	}

	/* Please note: the code to allocate buffers isn't run
	   during normal butler thread operation. Its there
	   for other times when we need to call do_refill()
	   from somewhere other than the butler thread.
	*/

	if (mixdown_buffer == 0) {
		mixdown_buffer = new Sample[disk_io_chunk_frames];
		free_mixdown = true;
	} else {
		free_mixdown = false;
	}

	if (gain_buffer == 0) {
		gain_buffer = new float[disk_io_chunk_frames];
		free_gain = true;
	} else {
		free_gain = false;
	}

	jack_nframes_t file_frame_tmp = 0;

	for (chan_n = 0, i = channels.begin(); i != channels.end(); ++i, ++chan_n) {

		ChannelInfo& chan (*i);
		Sample* buf1;
		Sample* buf2;
		jack_nframes_t len1, len2;

		chan.playback_buf->get_write_vector (&vector);

		ts = total_space;
		file_frame_tmp = file_frame;

		if (reversed) {
			buf1 = vector.buf[1];
			len1 = vector.len[1];
			buf2 = vector.buf[0];
			len2 = vector.len[0];
		} else {
			buf1 = vector.buf[0];
			len1 = vector.len[0];
			buf2 = vector.buf[1];
			len2 = vector.len[1];
		}


		to_read = min (ts, len1);
		to_read = min (to_read, disk_io_chunk_frames);

		if (to_read) {

			if (read (buf1, mixdown_buffer, gain_buffer, file_frame_tmp, to_read, chan, chan_n, reversed)) {
				ret = -1;
				goto out;
			}
			
			chan.playback_buf->increment_write_ptr (to_read);
			ts -= to_read;
		}

		to_read = min (ts, len2);

		if (to_read) {

			
			/* we read all of vector.len[0], but it wasn't an entire disk_io_chunk_frames of data,
			   so read some or all of vector.len[1] as well.
			*/

			if (read (buf2, mixdown_buffer, gain_buffer, file_frame_tmp, to_read, chan, chan_n, reversed)) {
				ret = -1;
				goto out;
			}
		
			chan.playback_buf->increment_write_ptr (to_read);
		}

		if (zero_fill) {
			/* do something */
		}

	}
	
	file_frame = file_frame_tmp;

  out:
	if (free_mixdown) {
		delete [] mixdown_buffer;
	}
	if (free_gain) {
		delete [] gain_buffer;
	}

	return ret;
}	

int
DiskStream::do_flush (bool force_flush)
{
	unsigned long to_write;
	gint32 ret = 0;
	RingBuffer<Sample>::rw_vector vector;
	jack_nframes_t total;
	
	/* important note: this function will write *AT MOST* 
	   disk_io_chunk_frames of data to disk. it will never 
	   write more than that. if its writes that much and there 
	   is more than that waiting to be written, it will return 1,
	   otherwise 0 on success or -1 on failure.

	   if there is less than disk_io_chunk_frames to be written, 
	   no data will be written at all unless `force_flush' is true.  
	*/

	_write_data_count = 0;

	for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {

	
		(*chan).capture_buf->get_read_vector (&vector);

		total = vector.len[0] + vector.len[1];

		if (total == 0 || (total < disk_io_chunk_frames && !force_flush)) {
			goto out;
		}

		/* if there are 2+ chunks of disk i/o possible for
		   this track, let the caller know so that it can arrange
		   for us to be called again, ASAP.
		   
		   if we are forcing a flush, then if there is* any* extra
		   work, let the caller know.
		*/

		if (total >= 2 * disk_io_chunk_frames || (force_flush && total > disk_io_chunk_frames)) {
			ret = 1;
		} 

		to_write = min (disk_io_chunk_frames, (jack_nframes_t) vector.len[0]);
	
		if ((!(*chan).write_source) || (*chan).write_source->write (vector.buf[0], to_write) != to_write) {
			error << compose(_("DiskStream %1: cannot write to disk"), _id) << endmsg;
			return -1;
		}
	
		(*chan).capture_buf->increment_read_ptr (to_write);

		if ((to_write == vector.len[0]) && (total > to_write) && (to_write < disk_io_chunk_frames)) {
		
			/* we wrote all of vector.len[0] but it wasn't an entire
			   disk_io_chunk_frames of data, so arrange for some part 
			   of vector.len[1] to be flushed to disk as well.
			*/
		
			to_write = min ((jack_nframes_t)(disk_io_chunk_frames - to_write), (jack_nframes_t) vector.len[1]);
		
			if ((*chan).write_source->write (vector.buf[1], to_write) != to_write) {
				error << compose(_("DiskStream %1: cannot write to disk"), _id) << endmsg;
				return -1;
			}

			_write_data_count += (*chan).write_source->write_data_count();
	
			(*chan).capture_buf->increment_read_ptr (to_write);
		}
	}

  out:
	return ret;
}

void
DiskStream::playlist_changed (Change ignored)
{
	playlist_modified ();
}

void
DiskStream::playlist_modified ()
{
	if (!i_am_the_modifier && !overwrite_queued) {
		_session.request_overwrite_buffer (this);
		overwrite_queued = true;
	} 
}

void
DiskStream::transport_stopped (struct tm& when, time_t twhen, bool abort_capture)
{
	unsigned long buffer_position;
	bool more_work = true;
	int err = 0;
	AudioRegion* region = 0;
	jack_nframes_t total_capture;
	AudioRegion::SourceList srcs;
	AudioRegion::SourceList::iterator src;
	ChannelList::iterator chan;
	list<CaptureInfo*>::iterator ci;
	unsigned int n = 0; 

	legal_recording_frame = _session.transport_frame() + _capture_offset;

	finish_capture (true);

	/* butler is already stopped, but there may be work to do 
	   to flush remaining data to disk.
	*/

	while (more_work && !err) {
		switch (do_flush (true)) {
		case 0:
			more_work = false;
			break;
		case 1:
			break;
		case -1:
			error << compose(_("DiskStream \"%1\": cannot flush captured data to disk!"), _name) << endmsg;
			err++;
		}
	}

	/* XXX is there anything we can do if err != 0 ? */

	if (capture_info.empty()) {
		goto out;
	}

	/* only do this if we captured anything */

	_last_capture_regions.clear ();

	if (abort_capture) {

		for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {

			if ((*chan).write_source) {

				(*chan).write_source->mark_for_remove ();
				(*chan).write_source->release ();
				
				delete (*chan).write_source;
				(*chan).write_source = 0;
			}

			use_new_write_source (n);
		}
		
		for (ci = capture_info.begin(); ci != capture_info.end(); ++ci) {
			delete *ci;
		}
		capture_info.clear ();
		goto out;
	} 

	/* Register a new region with the Session that
	   describes the entire source. Do this first
	   so that any sub-regions will obviously be
	   children of this one (later!)
	*/

	for (total_capture = 0, ci = capture_info.begin(); ci != capture_info.end(); ++ci) {
		total_capture += (*ci)->frames;
	}

	/* figure out the name for this take */

	for (n = 0, chan = channels.begin(); chan != channels.end(); ++chan, ++n) {

		Source* s = (*chan).write_source;
		
		if (s) {

			FileSource* fsrc;

			srcs.push_back (s);

			if ((fsrc = dynamic_cast<FileSource *>(s)) != 0) {
				fsrc->update_header (capture_info.front()->start, when, twhen);
			}

			s->set_captured_for (_name);
			
		}
	}

	/* create a new region describing the entire source file */
	
	try {
		region = new AudioRegion (srcs, 0, total_capture, 
				     Session::region_name_from_path (channels[0].write_source->name()), 
				     0, AudioRegion::Flag (AudioRegion::DefaultFlags|AudioRegion::Automatic|AudioRegion::WholeFile));
	}
	
	catch (failed_constructor& err) {
		error << compose(_("%1: could not create region for complete audio file"), _name) << endmsg;
		/* XXX what now? */
	}

	_last_capture_regions.push_back (region);

	for (buffer_position = 0, ci = capture_info.begin(); ci != capture_info.end(); ++ci) {

		string region_name;
		_session.region_name (region_name, _name, false);
		
		try {
			region = new AudioRegion (srcs, buffer_position, (*ci)->frames, region_name);
		}
		
		catch (failed_constructor& err) {
			error << _("DiskStream: could not create region for captured audio!") << endmsg;
			delete *ci;
			continue; /* XXX is this OK? */
		}

		_last_capture_regions.push_back (region);
		
		buffer_position += (*ci)->frames;
		
		i_am_the_modifier++;
		_session.add_undo (_playlist->get_memento());
		_playlist->add_region (*region, (*ci)->start);
		_session.add_redo_no_execute (_playlist->get_memento());
		i_am_the_modifier--;
		
		delete *ci;
	}

	capture_info.clear ();
	
	for (n = 0, src = srcs.begin(); src != srcs.end(); ++src, ++n) {
		(*src)->mark_streaming_write_completed ();
		use_new_write_source (n);
	}

  out:
	capture_start_frame = 0;
	
}

void
DiskStream::finish_capture (bool rec_monitors_input)
{
	if (Config->get_use_hardware_monitoring() && record_enabled()) {
		if (rec_monitors_input) {
			if (rec_monitoring_off_for_roll) {
				for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
					(*chan).source->ensure_monitor_input (true);
				}
				rec_monitoring_off_for_roll = false;
			}
		} else {
			if (!rec_monitoring_off_for_roll) {
				for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
					(*chan).source->ensure_monitor_input (false);
				}
				rec_monitoring_off_for_roll = true;
			}
		}
	}
	
	was_recording = false;
	
	if (capture_captured == 0) {
		return;
	}
		
	CaptureInfo* ci = new CaptureInfo;
	
	ci->start =  capture_start_frame;
	ci->frames = capture_captured;
	
	/* XXX theoretical race condition here. Need atomic exchange ? 
	   However, the circumstances when this is called right 
	   now (either on record-disable or transport_stopped)
	   mean that no actual race exists. I think ...
	*/
	
	capture_info.push_back (ci);
	capture_captured = 0;
}

void
DiskStream::set_record_enabled (bool yn, void* src)
{
	if (!recordable() || !_session.record_enabling_legal() || channels[0].source == 0) {
		return;
	}

	/* yes, i know that this not proof against race conditions, but its
	   good enough. i think.
	*/

	if (record_enabled() != yn) {
		if (yn) {
			atomic_set (&_record_enabled, 1);
			if (Config->get_use_hardware_monitoring()) {
				for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
					if ((*chan).source) {
						(*chan).source->request_monitor_input (true);
					}
				}
			}
		} else {
			atomic_set (&_record_enabled, 0);
			if (Config->get_use_hardware_monitoring()) {
				for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
					if ((*chan).source) {
						(*chan).source->request_monitor_input (false);
					}
				}
			}
		}

		 record_enable_changed (src); /* EMIT SIGNAL */
	}
}

XMLNode&
DiskStream::get_state ()
{
	XMLNode* node = new XMLNode ("DiskStream");
	char buf[64];

	setlocale (LC_NUMERIC, "POSIX");
	
	snprintf (buf, sizeof(buf), "%d", channels.size());
	node->add_property ("channels", buf);

	node->add_property ("playlist", _playlist->name());
	
	snprintf (buf, sizeof(buf), "%f", _visible_speed);
	node->add_property ("speed", buf);

	node->add_child_nocopy (IO::get_state());

	if (_extra_xml) {
		node->add_child_copy (*_extra_xml);
	}

	setlocale (LC_NUMERIC, "");

	return* node;
}

int
DiskStream::set_state (const XMLNode& node)
{
	const XMLProperty* prop;
	XMLNodeList nlist = node.children();
	XMLNodeIterator niter;
	unsigned int nchans = 1;

	/* prevent write sources from being created */

	
	in_set_state = true;
	
	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		if ((*niter)->name() == IO::state_node_name) {
			if (IO::set_state (**niter)) {
				return -1;
			}
			break;
		}
	}

	if (niter == nlist.end()) {
		error << _("XML node describing a diskstream is missing an IO node") << endmsg;
		return -1;
	}

	if ((prop = node.property ("channels")) != 0) {
		nchans = atoi (prop->value().c_str());
	}
	
	// create necessary extra channels
	// we are always constructed with one
	// and we always need one

	if (nchans > _n_channels) {

		// we need to add new channel infos
		//LockMonitor lm (state_lock, __LINE__, __FILE__);

		int diff = nchans - channels.size();

		for (int i=0; i < diff; ++i) {
			add_channel ();
		}

	} else if (nchans < _n_channels) {

		// we need to get rid of channels
		//LockMonitor lm (state_lock, __LINE__, __FILE__);

		int diff = channels.size() - nchans;
		
		for (int i = 0; i < diff; ++i) {
			remove_channel ();
		}
	}

	if ((prop = node.property ("playlist")) == 0) {
		return -1;
	}
	
	if (find_and_use_playlist (prop->value())) {
		return -1;
	}

	setlocale (LC_NUMERIC, "POSIX");
	
	if ((prop = node.property ("speed")) != 0) {
		float sp = atof (prop->value().c_str());

		if (realtime_set_speed (sp)) {
			non_realtime_set_speed ();
		}
	}
	setlocale (LC_NUMERIC, "");

	_n_channels = channels.size();

	in_set_state = false;

	/* now that we're all done with playlist+channel set up,
	   go ahead and create write sources.
	*/

	if (recordable()) {
		for (nchans = 0; nchans < _n_channels; ++nchans) {
			use_new_write_source (nchans);
		}
	}
		
	return 0;
}

int
DiskStream::use_new_write_source (unsigned int n)
{
	if (n >= channels.size()) {
		error << _("DiskStream: channel out of range") << endmsg;
		return -1;
	}

	ChannelInfo &chan = channels[n];
	
	if (chan.write_source) {
		chan.write_source->release();
		chan.write_source = 0;
	}

	try {
		if ((chan.write_source = _session.create_file_source (*this)) == 0) {
			throw failed_constructor();
		}
	} 

	catch (failed_constructor &err) {
		error << compose (_("%1:%2 new capture file not initialized correctly"), _name, n) << endmsg;
		chan.write_source = 0;
		return -1;
	}

	chan.write_source->use ();
	
	return 0;
}

void
DiskStream::set_block_size (jack_nframes_t nframes)
{
	if (_session.get_block_size() > speed_buffer_size) {
		for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
			if ((*chan).speed_buffer) delete [] (*chan).speed_buffer;
			(*chan).speed_buffer = new Sample[_session.get_block_size()];
		}
		speed_buffer_size = _session.get_block_size();
	}
	allocate_temporary_buffers ();
}

void
DiskStream::allocate_temporary_buffers ()
{
	/* make sure the wrap buffer is at least large enough to deal
	   with the "normal" condition.
	*/

	float sp = max (fabsf (_actual_speed), 1.0f);
	jack_nframes_t required_wrap_size = (jack_nframes_t) floor (_session.get_block_size() * sp) + 1;

	if (required_wrap_size > wrap_buffer_size) {

		for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
			if ((*chan).wrap_buffer) delete [] (*chan).wrap_buffer;
			(*chan).wrap_buffer = new Sample[required_wrap_size];	
		}

		wrap_buffer_size = required_wrap_size;
	}
}

void
DiskStream::monitor_input (bool yn)
{
	for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
		
		if ((*chan).source) {
			(*chan).source->request_monitor_input (yn);
		}
	}
}

void
DiskStream::set_capture_offset ()
{
	jack_nframes_t worst = 0;

	if (_session.get_align_style() == Session::ExistingMaterial) {

		_capture_offset = _session.worst_output_latency ();

	} else {
		
		for (unsigned int n=0; n < n_inputs(); ++n) {
			worst = max (worst, _session.engine().get_port_total_latency (*(input (n))));
		}
		
		_capture_offset = worst;
	}

	legal_recording_frame = _session.transport_frame() + _capture_offset;
}

void
DiskStream::reverse_scrub_buffer (bool forw)
{
	for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {
		if (!forw) {
			(*chan).scrub_buffer = (*chan).scrub_reverse_buffer;
			scrub_start = scrub_start + scrub_buffer_size;
		} else {
			(*chan).scrub_buffer = (*chan).scrub_forward_buffer;
			scrub_start = scrub_start - scrub_buffer_size;
		}
	}

#if 0	
	size_t rptr;
	size_t wptr;
	Sample* buf;
	size_t cnt;
	size_t bufsize;

	for (ChannelList::iterator chan = channels.begin(); chan != channels.end(); ++chan) {

	
		buf =  (*chan).playback_buf->buffer();	
		bufsize = (*chan).playback_buf->bufsize();
		rptr = (*chan).playback_buf->get_read_ptr ();
		wptr = (*chan).playback_buf->get_write_ptr();

		/* reverse the read/write pointers */

		(*chan).playback_buf->set (wptr, rptr);
	
		if (rptr > wptr) {
		
			cnt = rptr - wptr;

			/*
			  |-------------------------------------------|
			  ^                           ^
			  w                           r
		       
			  <--- reverse this area -----> 
                     
			*/

			swap_by_ptr (&buf[wptr], &buf[rptr]);

		} else {

			/*
		  
			   |--------------------------------------------|
			          ^                       ^
			    ----->r                       w<------------

			    swap the indicated area, taking care of
			    buffer wrap-around. thanks to steve harris
			    for this code.

			*/
		       
			Sample tmp;
			int    mask = bufsize - 1;
			int    r, w;

			cnt = (bufsize - (wptr - rptr)) / 2;

			while (cnt--) {
				w = (wptr + cnt) & mask;
				r = (rptr - cnt) & mask;
				tmp = buf[w];
				buf[w] = buf[r];
				buf[r] = tmp;
			}
		
			cnt = bufsize - (wptr - rptr);
		}
	}
	/* we've just reversed the area that was waiting to be overwritten.
	   the next read from the playlist should fetch data from the end
	   of this area.
	*/
	
	if (forw) {

		file_frame = playback_sample + cnt + 1;

	} else {
		
		if (cnt > playback_sample - 1) {
			
			/* the next file frame to read is zero, sort of,
			   because we've gone beyond back to the beginning.
			   code in ::do_refill() ensures that we don't
			   actually do this, however.
			*/
			
			file_frame = 0;

		} else {
			file_frame = playback_sample - cnt - 1;
		}
	}
#endif
}

int
DiskStream::add_channel ()
{
	/* XXX need to take lock??? */

	ChannelInfo chan;

	init_channel (chan);

	chan.speed_buffer = new Sample[speed_buffer_size];
	chan.wrap_buffer = new Sample[wrap_buffer_size];

	channels.push_back (chan);

	_n_channels = channels.size();

	if (!in_set_state && recordable()) {
		use_new_write_source (_n_channels - 1);
	}
	
	return 0;
}

int
DiskStream::remove_channel ()
{
	if (channels.size() > 1) {
		/* XXX need to take lock??? */
		ChannelInfo & chan = channels.back();
		destroy_channel (chan);
		channels.pop_back();

		_n_channels = channels.size();
		return 0;
	}

	return -1;
}


int
DiskStream::ports_created()
{
	// now we can add the stuff we need

	ports_created_c.disconnect();

	// create necessary extra channels
	// we are always constructed with one

	if (n_inputs() > channels.size()) {
		// we need to add new channel infos
		int diff = n_inputs() - channels.size();

		for (int i=0; i < diff; ++i) {
			add_channel ();
		}
	}

	_n_channels = channels.size();
	
	return 0;
}

float
DiskStream::playback_buffer_load () const
{
	return (float) ((double) channels.front().playback_buf->read_space()/
			(double) channels.front().playback_buf->bufsize());
}

float
DiskStream::capture_buffer_load () const
{
	return (float) ((double) channels.front().capture_buf->write_space()/
			(double) channels.front().capture_buf->bufsize());
}

int
DiskStream::set_loop (Location *location)
{
	if (location) {
		if (location->start() >= location->end()) {
			error << compose(_("Location \"%1\" not valid for track loop (start >= end)"), location->name()) << endl;
			return -1;
		}
	}

	loop_location = location;

	 LoopSet (location); /* EMIT SIGNAL */
	return 0;
}

