/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: insert.cc,v 1.36 2004/02/29 23:33:56 pauld Exp $
*/

#include <string.h>

#include <pbd/failed_constructor.h>
#include <pbd/xml++.h>

#include <ardour/insert.h>
#include <ardour/ladspa_plugin.h>
#include <ardour/audioengine.h>
#include <ardour/session.h>

#include "i18n.h"

using namespace ARDOUR;

Insert::Insert(Session& s, Redirect::Placement p)
	: Redirect (s, s.next_insert_name(), p)
{
}

Insert::Insert(Session& s, Redirect::Placement p, int imin, int imax, int omin, int omax)
	: Redirect (s, s.next_insert_name(), p, imin, imax, omin, omax)
{
}

Insert::Insert(Session& s, string name, Redirect::Placement p)
	: Redirect (s, name, p)
{
}

/***************************************************************
 Plugin inserts: send data through a LADSPA plugin
 ***************************************************************/

const string PluginInsert::port_automation_node_name = "PortAutomation";

PluginInsert::PluginInsert (Session& s, LADSPA::Plugin& plug, Redirect::Placement p, unsigned int count)
	: Insert (s, plug.name(), p),
	  _port_automation_playback (0),
	  _port_automation_record (0)
{
	/* the first is the master */
	_plugins.push_back(&plug);

	/* make as many copies as requested */
	for (unsigned int n=1; n < count; ++n) {
		_plugins.push_back(new LADSPA::Plugin(plug));
	}

	_plugins[0]->ControlPortChanged.connect (slot (*this, &PluginInsert::control_port_changed));
	
	init ();

	save_state (_("initial state"));

	 RedirectCreated (this); /* EMIT SIGNAL */
}

PluginInsert::PluginInsert (Session& s, const XMLNode& node)
	: Insert (s, "will change", Redirect::PreFader),
	  _port_automation_playback (0),
	  _port_automation_record (0)

{
	if (set_state (node)) {
		throw failed_constructor();
	}

	set_automatable ();

	save_state (_("initial state"));

	_plugins[0]->ControlPortChanged.connect (slot (*this, &PluginInsert::control_port_changed));
	
}

PluginInsert::PluginInsert (const PluginInsert& other)
	: Insert (other._session, other.plugin().name(), other.placement())
{
	LADSPA::Plugin *plug;
	
	if ((plug = LADSPA::find_plugin (other._session, other.plugin().name())) == 0) {
		error << compose (_("PluginInsert: cannot find the plugin called \"%1\""), other.plugin().name()) << endmsg;
		throw failed_constructor();
	}

	/* the first is the master */
	_plugins.push_back(plug);
	
	unsigned int count = other._plugins.size();

	/* make as many copies as requested */
	for (unsigned int n=1; n < count; ++n) {
		_plugins.push_back(new LADSPA::Plugin(*plug));
	}

	_plugins[0]->ControlPortChanged.connect (slot (*this, &PluginInsert::control_port_changed));

	init ();

	save_state (_("initial state"));

	 RedirectCreated (this); /* EMIT SIGNAL */
}

void
PluginInsert::set_count (unsigned int num)
{
	/* this is a bad idea.... we shouldn't do this while active.
	   only a route holding their redirect_lock should be calling this */
	
	if (num < 1) return;
	else if (num > _plugins.size()) {
		unsigned int diff = num - _plugins.size();
		for (unsigned int n=0; n < diff; ++n) {
			_plugins.push_back(new LADSPA::Plugin(*_plugins[0]));
		}
	}
	else if (num < _plugins.size()) {
		unsigned int diff = _plugins.size() - num;
		for (unsigned int n=0; n < diff; ++n) {
			LADSPA::Plugin * plug = _plugins.back();
			_plugins.pop_back();
			delete plug;
		}
	}
}

void
PluginInsert::init ()
{
	_port_automation_record = new AutomationList::Mode[_plugins[0]->port_count()];
	_port_automation_playback = new bool[_plugins[0]->port_count()];

	for (unsigned long n = 0; n < _plugins[0]->port_count(); ++n) {
		_port_automation_playback[n] = false;
		_port_automation_record[n] = AutomationList::Off;
	}

	set_automatable ();
}

PluginInsert::~PluginInsert ()
{
	 GoingAway (this); /* EMIT SIGNAL */
	delete [] _port_automation_playback;
	delete [] _port_automation_record;

	for (unsigned int n=0; n < _plugins.size(); ++n) {
		delete _plugins[n];
	}
}

void
PluginInsert::set_automatable ()
{
	set<unsigned long> a;
	
	a = _plugins.front()->automatable ();

	for (set<unsigned long>::iterator i = a.begin(); i != a.end(); ++i) {
		can_automate (*i);
	}
}

void
PluginInsert::control_port_changed (unsigned long port, float val)
{
	/* the master has changed, set the clones */

	for (unsigned int n=1; n < _plugins.size(); ++n) {
		_plugins[n]->set_control_port (port, val);
	}
}

void
PluginInsert::activate ()
{
	for (unsigned int n=0; n < _plugins.size(); ++n) {
		_plugins[n]->activate ();
	}
}

void
PluginInsert::deactivate ()
{
	for (unsigned int n=0; n < _plugins.size(); ++n) {
		_plugins[n]->deactivate ();
	}
}

void
PluginInsert::connect_and_run (vector<Sample*>& bufs, jack_nframes_t nframes, jack_nframes_t offset, bool with_auto, jack_nframes_t now)
{
	unsigned long port_index = 0;
	unsigned long in_index = 0;
	unsigned long out_index = 0;
	unsigned long nbufs = bufs.size();

	/* Note that we've already required that plugins
	   be able to handle in-place processing.
	*/

	if (with_auto) {

		map<unsigned long,AutomationList*>::iterator li;

		for (li = parameter_automation.begin(); li != parameter_automation.end(); ++li) {
			
			if (_port_automation_playback[(*li).first]) {
				
				AutomationList& alist (*((*li).second));
				bool locked;
				
				float val = alist.rt_safe_eval (now, locked);
				
				if (!locked) {
					continue;
				}
				
				/* set the first plugin, the others will be set via signals */
				
				_plugins[0]->set_control_port ((*li).first, val);
			}
		}
	}

	/* correct nbufs so that it functions as the highest possible buffer index */
	
	nbufs--;

	for (unsigned int plugn=0; plugn < _plugins.size(); ++plugn)
	{
		port_index = 0;
		
		while (port_index < _plugins[plugn]->port_count()) {
			if (LADSPA_IS_PORT_AUDIO (_plugins[plugn]->port_descriptors()[port_index])) {
				if (LADSPA_IS_PORT_INPUT (_plugins[plugn]->port_descriptors()[port_index])) {
					_plugins[plugn]->connect_port (port_index, &bufs[min(in_index,nbufs)][offset]);
					in_index++;
				} else if (LADSPA_IS_PORT_OUTPUT (_plugins[plugn]->port_descriptors()[port_index])) {
					_plugins[plugn]->connect_port (port_index, &bufs[min(out_index,nbufs)][offset]);
					out_index++;
				}
			}
			port_index++;
		}
	
		_plugins[plugn]->run (nframes);
	}

	/* leave remaining channel buffers alone */
}

void
PluginInsert::run (vector<Sample *>& bufs, jack_nframes_t nframes, jack_nframes_t offset)
{
	if (active()) {
		if (_session.transport_rolling()) {
			automation_run (bufs, nframes, offset);
		} else {
			connect_and_run (bufs, nframes, offset, false);
		}
	}
}

void
PluginInsert::set_control_port (unsigned long port, float val)
{
	/* the others will be set from the event triggered by this */

	_plugins[0]->set_control_port (port, val);
	
	if (_port_automation_record[port]) {
		automation_list (port).add (_session.audible_frame(), val);
	}
}


void
PluginInsert::automation_run (vector<Sample *>& bufs, jack_nframes_t nframes, jack_nframes_t offset)
{
	ControlEvent next_event (0, 0.0f);
	jack_nframes_t now = _session.transport_frame ();
	jack_nframes_t end = now + nframes;

	TentativeLockMonitor lm (_automation_lock, __LINE__, __FILE__);

	if (!lm.locked()) {
		connect_and_run (bufs, nframes, offset, false);
		return;
	}
	
	if (!find_next_event (now, end, next_event)) {
		
 		/* no events have a time within the relevant range */
		
 		connect_and_run (bufs, nframes, offset, true, now);
 		return;
 	}
	
 	while (nframes) {

 		jack_nframes_t cnt = min (((jack_nframes_t) floor (next_event.when) - now), nframes);
  
 		connect_and_run (bufs, cnt, offset, true, now);
 		
 		nframes -= cnt;
 		offset += cnt;
		now += cnt;

		if (!find_next_event (now, end, next_event)) {
			break;
		}
  	}
  
 	/* cleanup anything that is left to do */
  
 	if (nframes) {
 		connect_and_run (bufs, nframes, offset, true, now);
  	}
}	

float
PluginInsert::default_parameter_value (unsigned long port)
{
	if (_plugins.empty()) {
		fatal << _("programming error: ") << X_("PluginInsert::default_parameter_value() called with no plugin")
		      << endmsg;
		/*NOTREACHED*/
	}

	return _plugins[0]->default_value (port);
}
	
void
PluginInsert::set_port_automation_playback (unsigned long port, bool m)
{
	if (port < _plugins[0]->port_count()) {
		
		if (m != _port_automation_playback[port]) {
			_port_automation_playback[port] = m;
			 AutomationPlaybackChanged(); /* EMIT SIGNAL */
		}
	}
}

bool
PluginInsert::get_port_automation_playback (unsigned long port) const
{
	if (port < _plugins[0]->port_count()) {
		return _port_automation_playback[port];
	} else {
		return false;
	}
}

void
PluginInsert::set_port_automation_record (unsigned long port, AutomationList::Mode m)
{
	if (port < _plugins[0]->port_count()) {

		if (_port_automation_record[port] != m) {

			_port_automation_record[port] = m;
			
			/* force creation of an automation list now, rather than later */

			AutomationList& al = automation_list (port);
			al.set_mode (m);

			 AutomationRecordChanged (); /* EMIT SIGNAL */
		}
	}
}

AutomationList::Mode
PluginInsert::get_port_automation_record (unsigned long port) const
{
	if (port < _plugins[0]->port_count()) {
		return _port_automation_record[port];
	} else {
		return AutomationList::Off;
	}
}

XMLNode&
PluginInsert::get_state(void)
{
	gchar buf[256];
	XMLNode *node = new XMLNode("Insert");

	node->add_child_nocopy (Redirect::get_state());

	node->add_property("type", "ladspa");
	snprintf(buf, sizeof(buf), "%s", _plugins[0]->name());
	node->add_property("id", string(buf));
	node->add_property("count", compose("%1", _plugins.size()));
	node->add_child_nocopy (_plugins[0]->get_state());

	/* add port automation state */
	XMLNode *autonode = new XMLNode(port_automation_node_name);
	XMLNode * child;
	
	for (unsigned int i = 0; i < _plugins[0]->port_count(); ++i){
		if (LADSPA_IS_PORT_INPUT(_plugins[0]->port_descriptors()[i]) && 
		    LADSPA_IS_PORT_CONTROL(_plugins[0]->port_descriptors()[i])){
			child = new XMLNode("port");
			snprintf(buf, sizeof(buf), "%u", i);
			child->add_property("number", string(buf));
			
			snprintf(buf, sizeof(buf), "0x%x", _port_automation_playback[i]);
			child->add_property("aplay", string(buf));
			
			snprintf(buf, sizeof(buf), "0x%x", _port_automation_record[i]);
			child->add_property("arec", string(buf));
			
			autonode->add_child_nocopy (*child);
		}
	}

	node->add_child_nocopy (*autonode);
	
	return *node;
}

int
PluginInsert::set_state(const XMLNode& node)
{
	XMLNodeList nlist = node.children();
	XMLNodeIterator niter;
	XMLPropertyList plist;
	const XMLProperty *prop;

	if ((prop = node.property ("type")) == 0) {
		error << _("XML node describing insert is missing the `type' field") << endmsg;
		return -1;
	}
	
	if (prop->value() != "ladspa") {
		error << _("non-LADSPA plugin insert XML used for LADSPA plugin insert") << endmsg;
		return -1;
	}

	if ((prop = node.property ("id")) == 0) {
		error << _("XML node describing insert is missing the `id' field") << endmsg;
 		return -1;
	}

	LADSPA::Plugin * plugin;
	
	if ((plugin = LADSPA::find_plugin (_session, prop->value())) == 0) {
		error << compose(_("Found a reference to a LADSPA plugin (\"%1\") that is unknown.\n"
				   "Perhaps it was removed or moved since it was last used."), prop->value()) 
		      << endmsg;
		return -1;
	}

	_plugins.push_back (plugin);
	
	unsigned int count = 1;
	if ((prop = node.property ("count")) != 0) {
		sscanf (prop->value().c_str(), "%u", &count);
	}
	
	for (unsigned int n=1; n < count; ++n) {
		_plugins.push_back (new LADSPA::Plugin(*plugin));
	}
	
	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		if ((*niter)->name() == LADSPA::Plugin::state_node_name) {
			for (unsigned int n=0; n < _plugins.size(); ++n) {
				_plugins[n]->set_state (**niter);
			}
			break;
		}
	} 

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		if ((*niter)->name() == Redirect::state_node_name) {
			Redirect::set_state (**niter);
			break;
		}
	}

	if (niter == nlist.end()) {
		error << _("XML node describing insert is missing a Redirect node") << endmsg;
		return -1;
	}

	if (niter == nlist.end()) {
		error << compose(_("XML node describing a LADSPA insert is missing the `%1' information"), LADSPA::Plugin::state_node_name) << endmsg;
		return -1;
	}
	
	set_name (_plugins[0]->name(), this);

	/* only allocate if not already done. */

	if (!_port_automation_record && !_port_automation_playback) {
		_port_automation_record = new AutomationList::Mode[_plugins[0]->port_count()];
		_port_automation_playback = new bool[_plugins[0]->port_count()];
		
		for (unsigned long n = 0; n < _plugins[0]->port_count(); ++n) {
			_port_automation_playback[n] = AutomationList::Off;
			_port_automation_record[n] = AutomationList::Off;
		}
	}

	/* look for port automation node */
	
	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		if ((*niter)->name() == port_automation_node_name) {
			XMLNodeList cnodes;
			XMLProperty *cprop;
			XMLNodeConstIterator iter;
			XMLNode *child;
			const char *port;
			unsigned long port_id;

			cnodes = (*niter)->children ("port");
	
			for(iter = cnodes.begin(); iter != cnodes.end(); ++iter){
				
				child = *iter;
				
				if ((cprop = child->property("number")) != 0) {
					port = cprop->value().c_str();
				} else {
					warning << _("PluginInsert: Auto: no ladspa port number") << endmsg;
					continue;
				}

				sscanf (port, "%lu", &port_id);

				if (port_id >= _plugins[0]->port_count()) {
					warning << _("PluginInsert: Auto: port id out of range") << endmsg;
					continue;
				}
				
				if ((cprop = child->property("arec")) != 0) {
					
					/* handle old style */

					if (cprop->value () == "yes") {
						_port_automation_record[port_id] = AutomationList::Write;
					} else if (cprop->value() == "no") {
						_port_automation_record[port_id] = AutomationList::Off;
					} else {
						int x;
						sscanf (cprop->value().c_str(), "0x%x", &x);
						_port_automation_record[port_id] = AutomationList::Mode (x);
					}
				} else {
					warning << _("PluginInsert: Auto: no ladspa arec flag") << endmsg;
				}

				if ((cprop = child->property("aplay")) != 0) {
					int x;
					sscanf (cprop->value().c_str(), "0x%x", &x);
					_port_automation_playback[port_id] = AutomationList::Mode (x);
				} else {
					warning << _("PluginInsert: Auto: no ladspa aplay flag") << endmsg;
				}
			}
			
			break;
		}
	} 

	if (niter == nlist.end()) {
		warning << compose(_("XML node describing a port automation is missing the `%1' information"), port_automation_node_name) << endmsg;
	}
	
	
	return 0;
}

string
PluginInsert::describe_parameter (unsigned long what)
{
	return _plugins[0]->describe_parameter (what);
}

void
PluginInsert::reset_midi_control (MIDI::Port* port, bool on)
{
	_plugins[0]->reset_midi_control (port, on);
}

jack_nframes_t 
PluginInsert::latency() 
{
	return _plugins[0]->latency ();
}
	
void
PluginInsert::store_state (PluginInsertState& state) const
{
	Redirect::store_state (state);
	_plugins[0]->store_state (state.plugin_state);
}

Change
PluginInsert::restore_state (StateManager::State& state)
{
	PluginInsertState* pistate = dynamic_cast<PluginInsertState*> (&state);

	Redirect::restore_state (state);

	_plugins[0]->restore_state (pistate->plugin_state);

	return Change (0);
}

StateManager::State*
PluginInsert::state_factory (std::string why) const
{
	PluginInsertState* state = new PluginInsertState (why);

	store_state (*state);

	return state;
}

/***************************************************************
 Port inserts: send output to a port, pick up input at a port
 ***************************************************************/

PortInsert::PortInsert (Session& s, Redirect::Placement p)
	: Insert (s, p, 1, -1, 1, -1)
{
	init ();
	save_state (_("initial state"));
	 RedirectCreated (this); /* EMIT SIGNAL */
}

PortInsert::PortInsert (const PortInsert& other)
	: Insert (other._session, other.placement(), 1, -1, 1, -1)
{
	init ();
	save_state (_("initial state"));
	 RedirectCreated (this); /* EMIT SIGNAL */
}

void
PortInsert::init ()
{
	if (add_input_port ("", this)) {
		error << _("PortInsert: cannot add input port") << endmsg;
		throw failed_constructor();
	}
	
	if (add_output_port ("", this)) {
		error << _("PortInsert: cannot add output port") << endmsg;
		throw failed_constructor();
	}
}

PortInsert::PortInsert (Session& s, const XMLNode& node)
	: Insert (s, "will change", Redirect::PreFader)
{
	if (set_state (node)) {
		throw failed_constructor();
	}
	 RedirectCreated (this); /* EMIT SIGNAL */
}

PortInsert::~PortInsert ()
{
	GoingAway (this);
}

void
PortInsert::run (vector<Sample *>& bufs, jack_nframes_t nframes, jack_nframes_t offset)
{
	if (active()) {
		// manage_gain (bufs, nframes);
		deliver_output (bufs, nframes, offset);
		collect_input (bufs, nframes, offset);
	} else {
		silence (nframes, offset);
	}
} 

XMLNode&
PortInsert::get_state(void)
{
	XMLNode *node = new XMLNode("Insert");

	node->add_child_nocopy (Redirect::get_state());	node->add_property("type", "port");

	return *node;
}

int
PortInsert::set_state(const XMLNode& node)
{
	XMLNodeList nlist = node.children();
	XMLNodeIterator niter;
	XMLPropertyList plist;
	const XMLProperty *prop;

	if ((prop = node.property ("type")) == 0) {
		error << _("XML node describing insert is missing the `type' field") << endmsg;
		return -1;
	}
	
	if (prop->value() != "port") {
		error << _("non-port insert XML used for port plugin insert") << endmsg;
		return -1;
	}

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		if ((*niter)->name() == Redirect::state_node_name) {
			Redirect::set_state (**niter);
			break;
		}
	}

	if (niter == nlist.end()) {
		error << _("XML node describing insert is missing a Redirect node") << endmsg;
		return -1;
	}

	return 0;
}

jack_nframes_t 
PortInsert::latency() 
{
	/* because we deliver and collect within the same cycle,
	   all I/O is necessarily delayed by at least frames_per_cycle().

	   if the return port for insert has its own latency, we
	   need to take that into account too.
	*/

	return _session.engine().frames_per_cycle() + input_latency();
}
