/*
    Copyright (C) 2001 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: redirect.cc,v 1.31 2004/02/29 23:33:56 pauld Exp $
*/

#include <fstream>
#include <algorithm>
#include <string>
#include <cerrno>
#include <unistd.h>

#include <sigc++/bind.h>

#include <pbd/xml++.h>

#include <ardour/redirect.h>
#include <ardour/session.h>
#include <ardour/utils.h>
#include <ardour/send.h>
#include <ardour/insert.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;

const string Redirect::state_node_name = "Redirect";
Signal1<void,Redirect*> Redirect::RedirectCreated;

Redirect::Redirect (Session& s, const string& name, Redirect::Placement p,
		    int input_min, int input_max, int output_min, int output_max)
	: IO (s, name, input_min, input_max, output_min, output_max)
{
	_placement = p;
	_active = false;
	_sort_key = 0;
	_gui = 0;
	_extra_xml = 0;
}

Redirect::~Redirect ()

{
}

Redirect*
Redirect::clone (const Redirect& other)
{
	const Send *send;
	const PortInsert *port_insert;
	const PluginInsert *plugin_insert;

	if ((send = dynamic_cast<const Send*>(&other)) != 0) {
		return new Send (*send);
	} else if ((port_insert = dynamic_cast<const PortInsert*>(&other)) != 0) {
		return new PortInsert (*port_insert);
	} else if ((plugin_insert = dynamic_cast<const PluginInsert*>(&other)) != 0) {
		return new PluginInsert (*plugin_insert);
	} else {
		fatal << _("programming error: unknown Redirect type in Redirect::Clone!\n")
		      << endmsg;
		/*NOTREACHED*/
	}
	return 0;
}

void
Redirect::set_placement (Placement p, void *src)
{
	if (_placement != p) {
		_placement = p;
		 placement_changed (this, src); /* EMIT SIGNAL */
	}
}

string
Redirect::placement_as_string (Placement p)
{
	switch (p) {
	case PreFader:
		return _("pre");
	default: /* to get g++ to realize we have all the cases covered */
	case PostFader:
		return _("post");
	}
}

void
Redirect::set_placement (const string& str, void *src)
{
	if (str == _("pre")) {
		set_placement (Redirect::PreFader, this);
	} else if (str == _("post")) {
		set_placement (Redirect::PostFader, this);
	} else {
		error << compose(_("Redirect: unknown placement string \"%1\" (ignored)"), str) << endmsg;
	}
}

int
Redirect::load_automation (string path)
{
	ifstream in (path.c_str());

	if (!in) {
		warning << compose(_("%1: cannot open %2 to load automation data (%3)"), _name, path, strerror (errno)) << endmsg;
		return 1;
	}

	LockMonitor lm (_automation_lock, __LINE__, __FILE__);
	parameter_automation.clear ();

	while (in) {
		double when;
		double value;
		unsigned long port;

		in >> port;     if (!in) break;
		in >> when;  if (!in) goto bad;
		in >> value; if (!in) goto bad;
		
		AutomationList& al = automation_list (port);
		al.add (when, value);
	}
	
	return 0;

  bad:
	error << compose(_("%1: cannot load automation data from %2"), _name, path) << endmsg;
	parameter_automation.clear ();
	return -1;
}

int
Redirect::save_automation (string path)
{
	LockMonitor lm (_automation_lock, __LINE__, __FILE__);

	if (parameter_automation.empty()) {
		return 1;
	}

	ofstream out (path.c_str());

	if (!out) {
		error << compose(_("%1: cannot open %2 to store automation data (%3)"), _name, path, strerror (errno)) << endmsg;
		return -1;
	}

	AutomationList::const_iterator i;
	map<unsigned long,AutomationList*>::iterator li;
	
	for (li = parameter_automation.begin(); li != parameter_automation.end(); ++li) {
		for (i = (*li).second->begin(); i != (*li).second->end(); ++i) {
			
			out << (*li).first << ' ' << (*i)->when << ' ' << (*i)->value << endl;
			
			if (!out) {
				break;
			}
		}
		
		if (i != (*li).second->end()) {
			unlink (path.c_str());
			error << compose(_("%1: could not save automation state to %2"), _name, path) << endmsg;
			return -1;
		}
	}

	if (li != parameter_automation.end()) {
		unlink (path.c_str());
		error << compose(_("%1: could not save automation state to %2"), _name, path) << endmsg;
		return -1;
	}

	return 0;
}

XMLNode&
Redirect::get_state (void)
{
	char buf[64];
	XMLNode *node = new XMLNode (state_node_name);
	node->add_property("active", active() ? "yes" : "no");	
	node->add_property("placement", placement_as_string (placement()));
	node->add_child_nocopy (IO::get_state());

	if (_extra_xml){
		node->add_child_copy (*_extra_xml);
	}
	
	string path;
	string legal_name;

	path = _session.path();
	path += "/automation/";
	snprintf (buf, sizeof(buf), "%Lu", id());
	path += buf;
	path += ".auto";

	/* XXX we didn't ask for a state save, we asked for the current state.
	   FIX ME!
	*/

	switch (save_automation (path)) {
	case -1:
		error << compose(_("Could not get state from Redirect (%1).  Problem with save_automation"), _name) << endmsg;
		break;

	case 0:
		XMLNode *aevents = node->add_child("Automation");
		aevents->add_property ("path", path);
		break;
	}

	return *node;
}

void
Redirect::what_has_automation (set<unsigned long>& s) const
{
	LockMonitor (_automation_lock, __LINE__, __FILE__);
	map<unsigned long,AutomationList*>::const_iterator li;
	
	for (li = parameter_automation.begin(); li != parameter_automation.end(); ++li) {
		s.insert  ((*li).first);
	}
}

int
Redirect::set_state (const XMLNode& node)
{
	const XMLProperty *prop;

	if (node.name() != state_node_name) {
		error << compose(_("incorrect XML node \"%1\" passed to Redirect object"), node.name()) << endmsg;
		return -1;
	}

	XMLNodeList nlist = node.children();
	XMLNodeIterator niter;
	bool have_io = false;

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		if ((*niter)->name() == IO::state_node_name) {
			IO::set_state (**niter);
			have_io = true;
		} else if ((*niter)->name() == "Automation") {
			XMLProperty *prop;
			
			if ((prop = (*niter)->property ("path")) != 0) {
				load_automation (prop->value());
			} else {
				warning << compose(_("%1: Automation node has no path property"), _name) << endmsg;
			}
		}
		else if ((*niter)->name() == "extra") {
			_extra_xml = new XMLNode (*(*niter));
		}
	}

	if (!have_io) {
		error << _("XML node describing an IO is missing an IO node") << endmsg;
		return -1;
	}

	if ((prop = node.property ("active")) == 0) {
		error << _("XML node describing a redirect is missing the `flags' field") << endmsg;
		return -1;
	}

	_active = (prop->value() == "yes");
	
	if ((prop = node.property ("placement")) == 0) {
		error << _("XML node describing a redirect is missing the `placement' field") << endmsg;
		return -1;
	}

	set_placement (prop->value(), this);

	return 0;
}

AutomationList&
Redirect::automation_list (unsigned long port)
{
	AutomationList* al = parameter_automation[port];

	if (al == 0) {
		al = parameter_automation[port] = new AutomationList (default_parameter_value (port));
	}

	return *al;
}

string
Redirect::describe_parameter (unsigned long which)
{
	return "";
}

void
Redirect::can_automate (unsigned long what)
{
	can_automate_list.insert (what);
}

bool
Redirect::find_next_event (jack_nframes_t now, jack_nframes_t end, ControlEvent& next_event) const
{
	map<unsigned long,AutomationList*>::const_iterator li;	
	AutomationList::TimeComparator cmp;

	next_event.when = max_frames;
	
  	for (li = parameter_automation.begin(); li != parameter_automation.end(); ++li) {
		
		AutomationList::const_iterator i;
 		const AutomationList& alist (*((*li).second));
		ControlEvent cp (now, 0.0f);
		
 		for (i = lower_bound (alist.const_begin(), alist.const_end(), &cp, cmp); i != alist.const_end() && (*i)->when < end; ++i) {
 			if ((*i)->when > now) {
 				break; 
 			}
 		}
 		
 		if (i != alist.const_end() && (*i)->when < end) {
			
 			if ((*i)->when < next_event.when) {
 				next_event.when = (*i)->when;
 			}
 		}
 	}

 	return next_event.when != max_frames;
}

void
Redirect::store_state (RedirectState& state) const
{
	state.active = _active;
}

Change
Redirect::restore_state (StateManager::State& state)
{
	RedirectState* rstate = dynamic_cast<RedirectState*> (&state);
	set_active (rstate->active, this);
	return Change (0);
}

StateManager::State*
Redirect::state_factory (std::string why) const
{
	RedirectState* state = new RedirectState (why);

	store_state (*state);

	return state;
}

void
Redirect::set_active (bool yn, void* src)
{
	_active = yn; 
	save_state (_("active_changed"));
	active_changed (this, src); 
}
