/*
    Copyright (C) 1999-2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
    $Id: session_state.cc,v 1.69 2004/02/29 23:33:56 pauld Exp $
*/

#include <algorithm>
#include <fstream>
#include <string>

#include <sigc++/bind.h>

#include <cstdio> /* sprintf(3) ... grrr */
#include <cmath>
#include <cerrno>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <climits>
#include <fcntl.h>
#include <poll.h>
#include <signal.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <dirent.h>
#include <libgen.h>

#include <midi++/mmc.h>
#include <midi++/port.h>
#include <pbd/error.h>
#include <pbd/lockmonitor.h>
#include <pbd/pathscanner.h>
#include <pbd/pthread_utils.h>

#include <ardour/audioengine.h>
#include <ardour/configuration.h>
#include <ardour/session.h>
#include <ardour/diskstream.h>
#include <ardour/utils.h>
#include <ardour/audioplaylist.h>
#include <ardour/source.h>
#include <ardour/filesource.h>
#include <ardour/sndfilesource.h>
#include <ardour/sndfile_helpers.h>
#include <ardour/auditioner.h>
#include <ardour/export.h>
#include <ardour/redirect.h>
#include <ardour/send.h>
#include <ardour/insert.h>
#include <ardour/connection.h>
#include <ardour/slave.h>
#include <ardour/tempo.h>
#include <ardour/audio_track.h>
#include <ardour/cycle_timer.h>
#include <ardour/utils.h>
#include <ardour/named_selection.h>
#include <ardour/version.h>
#include <ardour/location.h>
#include <ardour/audioregion.h>

#include "i18n.h"
#include <locale.h>

using namespace std;
using namespace ARDOUR;

void
Session::first_stage_init (string fullpath, string snapshot_name)
{
	if (fullpath.length() == 0) {
		throw failed_constructor();
	}

	_path = fullpath;
	
	if (_path[0] != '/') {

		char buf[PATH_MAX+1];
		getcwd (buf, sizeof (buf));

		/* canonicalize path */

		if (_path[0] == '.' && _path[1] == '/') {
			_path = string(buf) + _path.substr (1);
		} else if (_path[0] == '.' && _path[1] == '.' && _path[2] == '/') {
			_path = string(dirname(buf)) + _path.substr (2);
		} else {
			_path = string(buf) + '/' + _path;
		}
	}

	if (_path[_path.length()-1] != '/') {
		_path += '/';
	}

	/* these two are just provisional settings. set_state()
	   will likely override them.
	*/

	_name = _current_snapshot_name = snapshot_name;
	setup_raid_path (_path);

	_current_frame_rate = _engine.frame_rate ();
	_tempo_map = new TempoMap (_current_frame_rate);
	_tempo_map->StateChanged.connect (slot (*this, &Session::tempo_map_changed));

	atomic_set (&processing_prohibited, 0);
	send_cnt = 0;
	insert_cnt = 0;
	_transport_speed = 0;
	_last_transport_speed = 0;
	transport_sub_state = 0;
	_transport_frame = 0;
	last_stop_frame = 0;
	current_end = 0;
	atomic_set (&_record_status, Disabled);
	auto_play = false;
	punch_in = false;
	punch_out = false;
	auto_loop = false;
	auto_looped = false;
	auto_input = false;
	auto_crossfade = false;
	all_safe = false;
	auto_return = false;
	_last_roll_location = 0;
	_last_record_location = 0;
	pending_locate_frame = 0;
	pending_locate_roll = false;
	pending_locate_flush = false;
	dstream_buffer_size = 0;
	state_tree = 0;
	set_next_event ();
	last_mtc_smpte_frame = 0;
	current_block_size = 0;
	_solo_latched = true;
	_solo_model = InverseMute;
	solo_update_disabled = false;
	currently_soloing = false;
	_worst_output_latency = 0;
	_worst_track_latency = 0;
	_state_of_the_state = StateOfTheState(CannotSave|InitialConnecting);
	_slave = 0;
	_slave_type = None;
	butler_mixdown_buffer = 0;
	butler_gain_buffer = 0;
	auditioner = 0;
	mmc_control = false;
	mmc = 0;
	_mtc_port = 0;
	post_transport_work = PostTransportWork (0);
	atomic_set (&butler_should_do_transport_work, 0);
	pending_audition_region = 0;
	_edit_mode = Slide;
	_play_range = false;
	align_style = ExistingMaterial;
	_control_out = 0;
	_master_out = 0;
	input_auto_connect = AutoConnectOption (0);
	output_auto_connect = AutoConnectOption (0);
	_have_captured = false;

	preroll.type = AnyTime::Frames;
	preroll.frames = 0;
	postroll.type = AnyTime::Frames;
	postroll.frames = 0;

	/* click sounds are unset by default, which causes us to internal
	   waveforms for clicks.
	*/
	
	_click_io = 0;
	_clicking = false;
	click_requested = true;
	click_data = 0;
	click_emphasis_data = 0;
	click_length = 0;
	click_emphasis_length = 0;

	process_function = &Session::process_with_events;

	last_smpte_when = 0;
	_smpte_offset = 0;
	last_smpte_valid = false;

	last_rr_session_dir = session_dirs.begin();
	refresh_disk_space ();

	set_default_fade (0.2, 5.0); /* steepness, millisecs */

	/* default configuration */

	recording_plugins = false;
	over_length_short = 2;
	over_length_long = 10;
	send_midi_timecode = false;
	send_midi_machine_control = false;
	shuttle_speed_factor = 1.0;
	shuttle_speed_threshold = 5;
	rf_speed = 2.0;
	meter_hold = 10; // XXX unknown units: number of calls to meter::set()
	max_level = 0;
	min_level = 0;

	/* default SMPTE type is 30 FPS, non-drop */

	set_smpte_type (30.0, false);

	_engine.GraphReordered.connect (slot (*this, &Session::graph_reordered));

	/* These are all static "per-class" signals */

	Region::RegionCreated.connect (slot (*this, &Session::add_region));
	Source::SourceCreated.connect (slot (*this, &Session::add_source));
	Playlist::PlaylistCreated.connect (slot (*this, &Session::add_playlist));
	Redirect::RedirectCreated.connect (slot (*this, &Session::add_redirect));
	DiskStream::DiskStreamCreated.connect (slot (*this, &Session::add_diskstream));
	NamedSelection::NamedSelectionCreated.connect (slot (*this, &Session::add_named_selection));

	IO::MoreOutputs.connect (slot (*this, &Session::ensure_passthru_buffers));
}

int
Session::second_stage_init (bool new_session)
{
	SndFileSource::set_peak_dir (sound_dir());

	if (!new_session) {
		if (load_state (_current_snapshot_name)) {
			return -1;
		}
		remove_empty_sounds ();
	}

	if (start_butler_thread()) {
		return -1;
	}

	if (start_midi_thread ()) {
		return -1;
	}

	if (state_tree) {
		if (set_state (*state_tree->root())) {
			return -1;
		}
	}

	/* we can't save till after ::when_engine_running() is called,
	   because otherwise we save state with no connections made.
	   therefore, we reset _state_of_the_state because ::set_state()
	   will have cleared it.
	*/

	_state_of_the_state = StateOfTheState (_state_of_the_state|CannotSave);

	find_current_end ();
	set_auto_input (true);
	_locations.changed.connect (slot (this, &Session::locations_changed));
	setup_click_sounds (0);
	setup_midi_control ();

	/* Pay attention ... */

	_engine.Halted.connect (slot (*this, &Session::engine_halted));
	_engine.Xrun.connect (slot (*this, &Session::xrun_recovery));

	if (_engine.running()) {
		when_engine_running();
	} else {
		_engine.Running.connect (slot (*this, &Session::when_engine_running));
	}

	send_full_time_code ();

	_engine.transport_locate (0);
	deliver_mmc (MIDI::MachineControl::cmdMmcReset, 0);
	deliver_mmc (MIDI::MachineControl::cmdLocate, 0);

	return 0;
}

string
Session::raid_path () const
{
	string path;

	for (vector<space_and_path>::const_iterator i = session_dirs.begin(); i != session_dirs.end(); ++i) {
		path += (*i).path;
		path += ':';
	}
	
	return path.substr (0, path.length() - 1); // drop final colon
}

void
Session::set_raid_path (string path)
{
	/* public-access to setup_raid_path() */

	setup_raid_path (path);
}

void
Session::setup_raid_path (string path)
{
	string::size_type colon;
	string remaining;
	space_and_path sp;
	string fspath;
	string::size_type len = path.length();
	int colons;

	colons = 0;

	for (string::size_type n = 0; n < len; ++n) {
		if (path[n] == ':') {
			colons++;
		}
	}

	if (colons == 0) {

		/* no multiple search path, just one directory (common case) */

		sp.path = path;
		sp.blocks = 0;
		session_dirs.push_back (sp);
		
		FileSource::set_search_path (path + '/' + sound_dir_name);

		return;
	}

	remaining = path;

	while ((colon = remaining.find_first_of (':')) != string::npos) {
		
		sp.blocks = 0;
		sp.path = remaining.substr (0, colon);

		fspath += sp.path;
		fspath += '/';
		fspath += sound_dir_name;
		fspath += ':';

		session_dirs.push_back (sp);

		remaining = remaining.substr (colon+1);
	}

	if (remaining.length()) {

		sp.blocks = 0;
		sp.path = remaining;

		fspath += sp.path;
		fspath += '/';
		fspath += sound_dir_name;

		session_dirs.push_back (sp);
	}

	/* set the FileSource search path */

	FileSource::set_search_path (fspath);
}

int
Session::create (bool& new_session, string* mix_template)
{
	string dir;
	
	if (mkdir (_path.c_str(), 0755) < 0) {
		if (errno == EEXIST) {
			new_session = false;
		} else {
			error << compose(_("Session: cannot create session dir \"%1\" (%2)"), _path, strerror (errno)) << endmsg;
			return -1;
		}
	} else {
		new_session = true;
	}

	dir = sound_dir ();

	if (mkdir (dir.c_str(), 0755) < 0) {
		if (errno != EEXIST) {
			error << compose(_("Session: cannot create session sounds dir \"%1\" (%2)"), dir, strerror (errno)) << endmsg;
			return -1;
		}
	}

	dir = dead_sound_dir ();

	if (mkdir (dir.c_str(), 0755) < 0) {
		if (errno != EEXIST) {
			error << compose(_("Session: cannot create session dead sounds dir \"%1\" (%2)"), dir, strerror (errno)) << endmsg;
			return -1;
		}
	}

	dir = automation_dir ();

	if (mkdir (dir.c_str(), 0755) < 0) {
		if (errno != EEXIST) {
			error << compose(_("Session: cannot create session automation dir \"%1\" (%2)"), dir, strerror (errno)) << endmsg;
			return -1;
		}
	}

	
	/* check new_session so we don't overwrite an existing one */
	
	if (mix_template) {
		if (new_session){
			
			string in_path = template_dir();
			in_path += *mix_template;
			in_path += _template_suffix;
			
			ifstream in(in_path.c_str());
			
			if (in){
				string out_path = _path;
				out_path += _name;
				out_path += _statefile_suffix;
				
				ofstream out(out_path.c_str());
				
				if (out){
					out << in.rdbuf();
					
					// okay, session is set up.  Treat like normal saved
					// session from now on.
					
					new_session = false;
					return 0;
					
				} else {
					error << compose (_("Could not open %1 for writing mix template"), out_path) 
					      << endmsg;
					return -1;
				}
				
			} else {
				error << compose (_("Could not open mix template %1 for reading"), in_path) 
				      << endmsg;
				return -1;
			}
			
			
		} else {
			warning << _("Session already exists.  Not overwriting") << endmsg;
			return -1;
		}
	}

	if (new_session) {

		_state_of_the_state = Clean;

		if (save_state (_current_snapshot_name)) {
			return -1;
		}
	}

	return 0;
}

int
Session::load_diskstreams (const XMLNode& node)
{
	XMLNodeList          clist;
	XMLNodeConstIterator citer;
	
	clist = node.children();

	for (citer = clist.begin(); citer != clist.end(); ++citer) {
		
		DiskStream* dstream;

		try {
			dstream = new DiskStream (*this, **citer);
			/* added automatically by DiskStreamCreated handler */
		} 
		
		catch (failed_constructor& err) {
			error << _("Session: could not load diskstream via XML state")			      << endmsg;
			return -1;
		}
	}

	return 0;
}

int
Session::save_state (string snapshot_name)
{
	XMLTree tree;
	string xml_path, bak_path;

	if (_state_of_the_state & CannotSave) {
		return 1;
	}

	tree.set_root (&get_state());

	xml_path = _path;
	xml_path += snapshot_name;
	xml_path += _statefile_suffix;
	bak_path = xml_path;
	bak_path += ".bak";

	// Make backup of main session file
	if ((snapshot_name == _name) 
		&& (access (xml_path.c_str(), F_OK) == 0) 
		&& (rename(xml_path.c_str(), bak_path.c_str()))) {
				error << _("could not backup old state file, current state not saved.")	<< endmsg;
			return -1;
	}
	
	if (!tree.write (xml_path)) {
		error << compose (_("state could not be saved to %1"), xml_path) << endmsg;
		return -1;
	}

	// if it's a snapshot, make it unwritable
	if (snapshot_name != _name) {
		chmod(xml_path.c_str(), S_IRUSR | S_IRGRP | S_IROTH);
	}

	_state_of_the_state = StateOfTheState (_state_of_the_state & ~Dirty);

	 StateSaved (snapshot_name); /* EMIT SIGNAL */

	return 0;
}

int
Session::restore_state (string snapshot_name)
{
	if (load_state (snapshot_name) == 0) {
		set_state (*state_tree->root());
	}
	
	return 0;
}

int
Session::load_state (string snapshot_name)
{
	if (state_tree) {
		delete state_tree;
		state_tree = 0;
	}

	string xmlpath;

	xmlpath = _path;
	xmlpath += snapshot_name;
	xmlpath += _statefile_suffix;

	if (access (xmlpath.c_str(), F_OK)) {
		error << compose(_("%1: session state information file \"%2\" doesn't exist!"), _name, xmlpath) << endmsg;
		return 1;
	}

	state_tree = new XMLTree;

	set_dirty();

	if (state_tree->read (xmlpath)) {
		return 0;
	} else {
		error << compose(_("Could not understand ardour file %1"), xmlpath) << endmsg;
	}
	
	delete state_tree;
	state_tree = 0;
	return -1;
}

int
Session::load_options (const XMLNode& node)
{
	XMLNode* child;
	XMLProperty* prop;
	bool have_fade_msecs = false;
	bool have_fade_steepness = false;
	float fade_msecs = 0;
	float fade_steepness = 0;
	SlaveSource slave_src = None;
	int x;

	setlocale (LC_NUMERIC, "POSIX");
	
	if ((child = find_named_node (node, "input-auto-connect")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			sscanf (prop->value().c_str(), "%x", &x);
			input_auto_connect = AutoConnectOption (x);
		}
	}

	if ((child = find_named_node (node, "output-auto-connect")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			sscanf (prop->value().c_str(), "%x", &x);
			output_auto_connect = AutoConnectOption (x);
		}
	}
				
	if ((child = find_named_node (node, "slave")) != 0) {
		if ((prop = child->property ("type")) != 0) {
			if (prop->value() == "none") {
				slave_src = None;
			} else if (prop->value() == "mtc") {
				slave_src = MTC;
			} else if (prop->value() == "adat") {
				slave_src = ADAT;
			} else if (prop->value() == "jack") {
				slave_src = JACK;
			} else if (prop->value() == "scrub") {
				slave_src = Scrub;
			}
			set_slave_source (slave_src, 0);
		}
	}

	if ((child = find_named_node (node, "edit-mode")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			if (prop->value() == "slide") {
				set_edit_mode (Slide);
			} else if (prop->value() == "splice") {
				set_edit_mode (Splice);
			} 
		}
	}
				
	if ((child = find_named_node (node, "send-midi-timecode")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_send_mtc (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "send-midi-machine-control")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_send_mmc (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "max-level")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			max_level = atoi (prop->value().c_str());
		}
	}
	if ((child = find_named_node (node, "min-level")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			min_level = atoi (prop->value().c_str());
		}
	}
	if ((child = find_named_node (node, "meter-hold")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			meter_hold = atoi (prop->value().c_str());
		}
	}
	if ((child = find_named_node (node, "long-over-length")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			over_length_long = atoi (prop->value().c_str());
		}
	}
	if ((child = find_named_node (node, "short-over-length")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			over_length_short = atoi (prop->value().c_str());
		}
	}
	if ((child = find_named_node (node, "shuttle-speed-factor")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			shuttle_speed_factor = atof (prop->value().c_str());
		}
	}
	if ((child = find_named_node (node, "shuttle-speed-threshold")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			shuttle_speed_threshold = atof (prop->value().c_str());
		}
	}
	if ((child = find_named_node (node, "rf-speed")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			rf_speed = atof (prop->value().c_str());
		}
	}
	if ((child = find_named_node (node, "smpte-frames-per-second")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			smpte_frames_per_second= atof (prop->value().c_str());
		}
	}
	if ((child = find_named_node (node, "click-sound")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			click_sound = prop->value();
		}
	}
	if ((child = find_named_node (node, "click-emphasis-sound")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			click_emphasis_sound = prop->value();
		}
	}

	/* BOOLEAN OPTIONS */

	if ((child = find_named_node (node, "recording-plugins")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			recording_plugins = (prop->value() == "yes");
		}
	}

	if ((child = find_named_node (node, "auto-play")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_auto_play (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "auto-input")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_auto_input (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "auto-loop")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_auto_loop (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "punch-in")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_punch_in (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "punch-out")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_punch_out (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "auto-return")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_auto_return (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "send-mtc")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_send_mtc (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "mmc-control")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_mmc_control (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "recording-plugins")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_recording_plugins (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "auto-crossfade")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_auto_crossfade (prop->value() == "yes");
		}
	}
	if ((child = find_named_node (node, "audible-click")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			set_clicking (prop->value() == "yes");
		}
	}

	if ((child = find_named_node (node, "align-style")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			if (prop->value() == "capture") {
				set_align_style (CaptureTime);
			} else {
				set_align_style (ExistingMaterial);
			}
		}
	}

	/* TIED OPTIONS */

	if ((child = find_named_node (node, "default-fade-steepness")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			fade_steepness = atof (prop->value().c_str());
			have_fade_steepness = true;
		}
	}
	if ((child = find_named_node (node, "default-fade-msec")) != 0) {
		if ((prop = child->property ("val")) != 0) {
			fade_msecs = atof (prop->value().c_str());
			have_fade_msecs = true;
		}
	}

	if (have_fade_steepness || have_fade_msecs) {
		set_default_fade (fade_steepness, fade_msecs);
	}

	setlocale (LC_NUMERIC, "");
	
	return 0;
}

XMLNode&
Session::get_options () const
{
	XMLNode* opthead;
	XMLNode* child;
	char buf[32];

	opthead = new XMLNode ("Options");

	child = opthead->add_child ("recording-plugins");
	child->add_property ("val", recording_plugins?"yes":"no");

	SlaveSource src = slave_source ();
	string src_string;
	switch (src) {
	case None:
		src_string = "none";
		break;
	case MTC:
		src_string = "mtc";
		break;
	case ADAT:
		src_string = "adat";
		break;
	case JACK:
		src_string = "jack";
		break;
	case Scrub:
		src_string = "scrub";
		break;
	}
	child = opthead->add_child ("slave");
	child->add_property ("type", src_string);
	
	child = opthead->add_child ("send-midi-timecode");
	child->add_property ("val", send_midi_timecode?"yes":"no");

	child = opthead->add_child ("send-midi-machine-control");
	child->add_property ("val", send_midi_machine_control?"yes":"no");

	g_snprintf (buf, sizeof(buf)-1, "%x", (int) input_auto_connect);
	child = opthead->add_child ("input-auto-connect");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%x", (int) output_auto_connect);
	child = opthead->add_child ("output-auto-connect");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%d", max_level);
	child = opthead->add_child ("max-level");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%d", min_level);
	child = opthead->add_child ("min-level");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%f", meter_hold);
	child = opthead->add_child ("meter-hold");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%u", over_length_long);
	child = opthead->add_child ("long-over-length");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%u", over_length_short);
	child = opthead->add_child ("short-over-length");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%f", shuttle_speed_factor);
	child = opthead->add_child ("shuttle-speed-factor");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%f", shuttle_speed_threshold);
	child = opthead->add_child ("shuttle-speed-threshold");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%f", rf_speed);
	child = opthead->add_child ("rf-speed");
	child->add_property ("val", buf);

	g_snprintf (buf, sizeof(buf)-1, "%f", smpte_frames_per_second);
	child = opthead->add_child ("smpte-frames-per-second");
	child->add_property ("val", buf);
	
	child = opthead->add_child ("edit-mode");
	switch (_edit_mode) {
	case Splice:
		child->add_property ("val", "splice");
		break;

	case Slide:
		child->add_property ("val", "slide");
		break;
	}

	child = opthead->add_child ("auto-play");
	child->add_property ("val", get_auto_play () ? "yes" : "no");
	child = opthead->add_child ("auto-input");
	child->add_property ("val", get_auto_input () ? "yes" : "no");
	child = opthead->add_child ("auto-loop");
	child->add_property ("val", get_auto_loop () ? "yes" : "no");
	child = opthead->add_child ("punch-in");
	child->add_property ("val", get_punch_in () ? "yes" : "no");
	child = opthead->add_child ("punch-out");
	child->add_property ("val", get_punch_out () ? "yes" : "no");
	child = opthead->add_child ("all-safe");
	child->add_property ("val", get_all_safe () ? "yes" : "no");
	child = opthead->add_child ("auto-return");
	child->add_property ("val", get_auto_return () ? "yes" : "no");
	child = opthead->add_child ("mmc-control");
	child->add_property ("val", get_mmc_control () ? "yes" : "no");
	child = opthead->add_child ("recording-plugins");
	child->add_property ("val", get_recording_plugins () ? "yes" : "no");
	child = opthead->add_child ("auto-crossfade");
	child->add_property ("val", get_auto_crossfade () ? "yes" : "no");
	child = opthead->add_child ("audible-click");
	child->add_property ("val", get_clicking () ? "yes" : "no");

	if (click_sound.length()) {
		child = opthead->add_child ("click-sound");
		child->add_property ("val", click_sound);
	}

	if (click_emphasis_sound.length()) {
		child = opthead->add_child ("click-emphasis-sound");
		child->add_property ("val", click_emphasis_sound);
	}

	child = opthead->add_child ("align-style");
	child->add_property ("val", (align_style == ExistingMaterial ? "existing" : "capture"));

	return *opthead;
}

XMLNode&
Session::get_state()
{
	return state(true);
}

XMLNode&
Session::get_template()
{
	return state(false);
}

XMLNode&
Session::state(bool full_state)
{
	XMLNode* node = new XMLNode("Session");
	XMLNode* child;

	// store libardour version, just in case
	gchar buf[16];
	snprintf(buf, sizeof(buf)-1, "%d.%d.%d", ardour_major_version,
			ardour_minor_version, ardour_micro_version);
	node->add_property("version", string(buf));
		
	/* store configuration settings */

	if (full_state) {
	
		/* store the name */
		node->add_property ("name", _name);

		if (session_dirs.size() > 1) {

			string p;

			vector<space_and_path>::iterator i = session_dirs.begin();
			vector<space_and_path>::iterator next;

			++i; /* skip the first one */
			next = i;
			++next;

			while (i != session_dirs.end()) {

				p += (*i).path;

				if (next != session_dirs.end()) {
					p += ':';
				} else {
					break;
				}

				++next;
				++i;
			}
			
			child = node->add_child ("Path");
			child->add_content (p);
		}
	}

	node->add_child_nocopy (get_options());

	child = node->add_child ("Sources");
	if (full_state) {
		{
			LockMonitor sl (source_lock, __LINE__, __FILE__);
			for (SourceList::iterator siter = sources.begin(); siter != sources.end(); ++siter) {

				/* Don't save information about FileSources that are empty */
			
				FileSource* fs;

				if ((fs = dynamic_cast<FileSource*> ((*siter).second)) != 0) {
					if (fs->length() == 0) {
						continue;
					}
				}

				child->add_child_nocopy ((*siter).second->get_state());
			}
		}
	}

	child = node->add_child ("Regions");
	if (full_state) { 
		{
			LockMonitor rl (region_lock, __LINE__, __FILE__);
			for (AudioRegionList::const_iterator i = audio_regions.begin(); i != audio_regions.end(); ++i) {
				child->add_child_nocopy (i->second->get_short_state());
			}
		}

	}

	child = node->add_child ("DiskStreams");
	{ 
		LockMonitor dl (diskstream_lock, __LINE__, __FILE__);
		for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
			if (!(*i)->hidden()) {
				child->add_child_nocopy ((*i)->get_state());
			}
		}
	}

	node->add_child_nocopy (_locations.get_state());
	
	child = node->add_child ("Connections");
	{
		LockMonitor lm (connection_lock, __LINE__, __FILE__);
		for (ConnectionList::iterator i = _connections.begin(); i != _connections.end(); ++i) {
			if (!(*i)->system_dependent()) {
				child->add_child_nocopy ((*i)->get_state());
			}
		}
	}

	child = node->add_child ("Routes");
	{
		LockMonitor lm (route_lock, __LINE__, __FILE__);
		
		RoutePublicOrderSorter cmp;
		RouteList public_order(routes);
		public_order.sort (cmp);
		
		for (RouteList::iterator i = public_order.begin(); i != public_order.end(); ++i) {
			if (!(*i)->hidden()) {
				if (full_state) {
					child->add_child_nocopy ((*i)->get_state());
				} else {
					child->add_child_nocopy ((*i)->get_template());
				}
			}
		}
	}

	
	child = node->add_child ("EditGroups");
	for (list<RouteGroup *>::iterator i = edit_groups.begin(); i != edit_groups.end(); ++i) {
		child->add_child_nocopy ((*i)->get_state());
	}

	child = node->add_child ("MixGroups");
	for (list<RouteGroup *>::iterator i = mix_groups.begin(); i != mix_groups.end(); ++i) {
		child->add_child_nocopy ((*i)->get_state());
	}

	child = node->add_child ("Playlists");
	for (PlaylistList::iterator i = playlists.begin(); i != playlists.end(); ++i) {
		if (!(*i)->hidden()) {
			if (full_state) {
				child->add_child_nocopy ((*i)->get_state());
			} else {
				child->add_child_nocopy ((*i)->get_template());
			}
		}
	}

	if (_click_io) {
		child = node->add_child ("Click");
		child->add_child_nocopy (_click_io->get_state ());
	}

	if (full_state) {
		child = node->add_child ("NamedSelections");
		for (NamedSelectionList::iterator i = named_selections.begin(); i != named_selections.end(); ++i) {
			if (full_state) {
				child->add_child_nocopy ((*i)->get_state());
			} 
		}
	}

	node->add_child_nocopy (_tempo_map->get_state());

	if (_extra_xml) {
		node->add_child_copy (*_extra_xml);
	}

	return *node;
}

int
Session::set_state (const XMLNode& node)
{
	XMLNodeList nlist;
	XMLNode* child;
	const XMLProperty* prop;

	_state_of_the_state = StateOfTheState (_state_of_the_state|CannotSave);
	
	if (node.name() != "Session"){
		fatal << _("programming error: Session: incorrect XML node sent to set_state()") << endmsg;
		return -1;
	}

	if ((prop = node.property ("name")) != 0) {
		_name = prop->value ();
	}
	
	IO::disable_ports ();
	IO::disable_connecting ();

	/* Object loading order:
	   
	   Path
	   extra
	   Options
	   Sources
	   AudioRegions
	   DiskStreams
	   Connections
	   Locations
	   Routes
	   EditGroups
	   MixGroups
	   Click
	   MIDI
	*/

	if ((child = find_named_node (node, "Path")) != 0) {
		/* XXX this XML content stuff horrible API design */
		string raid_path = _path + ':' + child->children().front()->content();
		setup_raid_path (raid_path);
	} else {
		/* the path is already set */
	}

	if ((child = find_named_node (node, "extra")) != 0) {
		_extra_xml = new XMLNode (*child);
	}

	if ((child = find_named_node (node, "Options")) == 0) {
		error << _("Session: XML state has no options section") << endmsg;
		return -1;
	} else if (load_options (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "Sources")) == 0) {
		error << _("Session: XML state has no sources section") << endmsg;
		return -1;
	} else if (load_sources (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "Regions")) == 0) {
		error << _("Session: XML state has no Regions section") << endmsg;
		return -1;
	} else if (load_regions (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "Playlists")) == 0) {
		error << _("Session: XML state has no playlists section") << endmsg;
		return -1;
	} else if (load_playlists (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "NamedSelections")) != 0) {
		if (load_named_selections (*child)) {
			return -1;
		}
	}

	if ((child = find_named_node (node, "DiskStreams")) == 0) {
		error << _("Session: XML state has no diskstreams section") << endmsg;
		return -1;
	} else if (load_diskstreams (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "Connections")) == 0) {
		error << _("Session: XML state has no connections section") << endmsg;
		return -1;
	} else if (load_connections (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "Locations")) == 0) {
		error << _("Session: XML state has no locations section") << endmsg;
		return -1;
	} else if (_locations.set_state (*child)) {
		return -1;
	}

	Location* location;

	if ((location = _locations.auto_loop_location()) != 0) {
		set_auto_loop_location (location);
	}

	if ((location = _locations.auto_punch_location()) != 0) {
		set_auto_punch_location (location);
	}

	if ((child = find_named_node (node, "EditGroups")) == 0) {
		error << _("Session: XML state has no edit groups section") << endmsg;
		return -1;
	} else if (load_edit_groups (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "MixGroups")) == 0) {
		error << _("Session: XML state has no mix groups section") << endmsg;
		return -1;
	} else if (load_mix_groups (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "TempoMap")) == 0) {
		error << _("Session: XML state has no Tempo Map section") << endmsg;
		return -1;
	} else if (_tempo_map->set_state (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "Routes")) == 0) {
		error << _("Session: XML state has no routes section") << endmsg;
		return -1;
	} else if (load_routes (*child)) {
		return -1;
	}

	if ((child = find_named_node (node, "Click")) == 0) {
		warning << _("Session: XML state has no click section") << endmsg;
	} else if (_click_io) {
		_click_io->set_state (*child);
	}
	
	string port_name;

	port_name = Config->get_mmc_port_name();
	if (port_name.length()) {
		set_mmc_port (port_name);
	}

	port_name = Config->get_mtc_port_name();
	if (port_name.length()) {
		set_mtc_port (port_name);
	}

	_state_of_the_state = Clean;

	return 0;
}

int
Session::load_routes (const XMLNode& node)
{
	XMLNodeList nlist;
	XMLNodeConstIterator niter;
	Route *route;

	nlist = node.children();

	set_dirty();

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {

		if ((route = XMLRouteFactory (**niter)) == 0) {
			error << _("Session: cannot create Route from XML description.")			      << endmsg;
			return -1;
		}

		add_route (*route);
	}

	return 0;
}

Route *
Session::XMLRouteFactory (const XMLNode& node)
{
	if (node.name() != "Route") {
		return 0;
	}

	if (node.property ("diskstream") != 0 || node.property ("diskstream-id") != 0) {
		return new AudioTrack (*this, node);
	} else {
		return new Route (*this, node);
	}
}

int
Session::load_regions (const XMLNode& node)
{
	XMLNodeList nlist;
	XMLNodeConstIterator niter;
	AudioRegion* region;

	nlist = node.children();

	set_dirty();

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {

		if ((region = XMLRegionFactory (**niter, false)) == 0) {
			error << _("Session: cannot create Region from XML description.") << endmsg;
			return -1;
		}
	}

	return 0;
}

AudioRegion *
Session::XMLRegionFactory (const XMLNode& node, bool full)
{
	const XMLProperty* prop;
	id_t s_id;
	id_t r_id;
	AudioRegion* r;
	Source* source;
	AudioRegion::SourceList sources;
	unsigned int nchans = 1;
	char buf[128];
	
	if (node.name() != "Region") {
		return 0;
	}

	if ((prop = node.property ("id")) == 0) {
		error << _("Session: XMLNode describing a AudioRegion is incomplete (no ID)") << endmsg;
		return 0;
	}

	sscanf (prop->value().c_str(), "%Lu", &r_id);

	if ((r = get_region (r_id)) != 0) {

		r = new AudioRegion (*r);

		if (full) {
			r->set_state (node);
		}

		return r;
	} 

	if ((prop = node.property ("channels")) != 0) {
		nchans = atoi (prop->value().c_str());
	}

	
	if ((prop = node.property ("source-0")) == 0) {
		if ((prop = node.property ("source")) == 0) {
			error << _("Session: XMLNode describing a AudioRegion is incomplete (no source)") << endmsg;
			return 0;
		}
	}

	sscanf (prop->value().c_str(), "%Lu", &s_id);

	if ((source = get_source (s_id)) == 0) {
		error << compose(_("Session: XMLNode describing a AudioRegion references an unknown source id =%1"), s_id) << endmsg;
		return 0;
	}

	sources.push_back(source);

	/* pickup other channels */
	for (unsigned int n=1; n < nchans; ++n) {
		snprintf (buf, sizeof(buf), "source-%d", n);
		if ((prop = node.property (buf)) != 0) {
			sscanf (prop->value().c_str(), "%Lu", &s_id);
			
			if ((source = get_source (s_id)) == 0) {
				error << compose(_("Session: XMLNode describing a AudioRegion references an unknown source id =%1"), s_id) << endmsg;
				return 0;
			}
			sources.push_back(source);
		}
	}
	
	
	try {
		return new AudioRegion (sources, node);
	}

	catch (failed_constructor& err) {
		return 0;
	}
}

XMLNode&
Session::get_sources_as_xml ()

{
	XMLNode* node = new XMLNode ("Sources");
	LockMonitor lm (source_lock, __LINE__, __FILE__);

	for (SourceList::iterator i = sources.begin(); i != sources.end(); ++i) {
		node->add_child_nocopy ((*i).second->get_state());
	}

	return *node;
}

string
Session::region_name_from_path (string path)
{
	string::size_type pos;
	
 	/* remove filename suffixes etc. */
	
	if ((pos = path.find_last_of ('.')) != string::npos) {
		return path.substr (0, pos);
	}

	return path;
}	

string
Session::path_from_region_name (string name, string identifier)
{
	char buf[PATH_MAX+1];
	unsigned long n;
	string dir = discover_best_sound_dir ();

	for (n = 0; n < 999999; ++n) {
		if (identifier.length()) {
			snprintf (buf, sizeof(buf), "%s/%s%s%lu.wav", dir.c_str(), name.c_str(), 
				  identifier.c_str(), n);
		} else {
			snprintf (buf, sizeof(buf), "%s/%s-%lu.wav", dir.c_str(), name.c_str(), n);
		}
		if (access (buf, F_OK) != 0) {
			return buf;
		}
	}

	return "";
}
	

int
Session::load_sources (const XMLNode& node)
{
	XMLNodeList nlist;
	XMLNodeConstIterator niter;
	Source* source;

	nlist = node.children();

	set_dirty();

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {

		if ((source = XMLSourceFactory (**niter)) == 0) {
			error << _("Session: cannot create Source from XML description.") << endmsg;
			return -1;
		}
	}

	return 0;
}

Source *
Session::XMLSourceFactory (const XMLNode& node)
{
	Source *src;

	if (node.name() != "Source") {
		return 0;
	}

	try {
		src = new FileSource (node, frame_rate());
	}
	
	catch (failed_constructor& err) {

		try {
			src = new SndFileSource (node);
		}

		catch (failed_constructor& err) {
			error << _("Found a sound file that cannot be used by Ardour. See the progammers.") << endmsg;
			return 0;
		} 
	}

	return src;
}

int
Session::save_template (string template_name)
{
	XMLTree tree;
	string xml_path, bak_path, template_path;

	if (_state_of_the_state & CannotSave) {
		return -1;
	}

	DIR* dp;
	string dir = template_dir();

	if ((dp = opendir (dir.c_str()))) {
		closedir (dp);
	} else {
		if (mkdir (dir.c_str(), S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH)<0) {
			error << compose(_("Could not create mix templates directory \"%1\" (%2)"), dir, strerror (errno)) << endmsg;
			return -1;
		}
	}

	tree.set_root (&get_template());

	xml_path = dir;
	xml_path += template_name;
	xml_path += _template_suffix;

	ifstream in(xml_path.c_str());
	
	if (in) {
		warning << compose(_("Template \"%1\" already exists - new version not created"), template_name) << endmsg;
		return -1;
	} else {
		in.close();
	}

	if (!tree.write (xml_path)) {
		error << _("mix template not saved") << endmsg;
		return -1;
	}

	return 0;
}

int
Session::rename_template (string old_name, string new_name) 
{
	string old_path = template_dir() + old_name + _template_suffix;
	string new_path = template_dir() + new_name + _template_suffix;

	return rename (old_path.c_str(), new_path.c_str());
}

int
Session::delete_template (string name) 
{
	string template_path = template_dir();
	template_path += name;
	template_path += _template_suffix;

	return remove (template_path.c_str());
}

void
Session::refresh_disk_space ()
{
	struct statfs statfsbuf;
	vector<space_and_path>::iterator i;
	LockMonitor lm (space_lock, __LINE__, __FILE__);
	double scale;

	/* get freespace on every FS that is part of the session path */

	_total_free_4k_blocks = 0;
	
	for (i = session_dirs.begin(); i != session_dirs.end(); ++i) {
		statfs ((*i).path.c_str(), &statfsbuf);

		scale = statfsbuf.f_bsize/4096.0;

		(*i).blocks = (unsigned long) floor (statfsbuf.f_bavail * scale);
		_total_free_4k_blocks += (*i).blocks;
	}
}

int
Session::ensure_sound_dir (string path, string& result)
{
	string dead;

	result = path;
	dead = path;

	/* Ensure that the parent directory exists */
	
	if (mkdir (result.c_str(), 0775)) {
		if (errno != EEXIST) {
			error << compose(_("cannot create session directory \"%1\"; ignored"), result) << endmsg;
			return -1;
		}
	}
	
	/* Ensure that the sounds directory exists */
	
	result += '/';
	result += sound_dir_name;
	
	if (mkdir (result.c_str(), 0775)) {
		if (errno != EEXIST) {
			error << compose(_("cannot create sounds directory \"%1\"; ignored"), result) << endmsg;
			return -1;
		}
	}

	dead += '/';
	dead += dead_sound_dir_name;
	
	if (mkdir (result.c_str(), 0775)) {
		if (errno != EEXIST) {
			error << compose(_("cannot create dead sounds directory \"%1\"; ignored"), result) << endmsg;
			return -1;
		}
	}
	
	/* callers expect this to be terminated ... */
			
	result += '/';
	return 0;
}	

string
Session::discover_best_sound_dir ()
{
	vector<space_and_path>::iterator i;
	string result;

	/* handle common case without system calls */

	if (session_dirs.size() == 1) {
		return sound_dir();
	}

	/* OK, here's the algorithm we're following here:

	   We want to select which directory to use for 
	   the next file source to be created. Ideally,
	   we'd like to use a round-robin process so as to
	   get maximum performance benefits from splitting
	   the files across multiple disks.

	   However, in situations without much diskspace, an
	   RR approach may end up filling up a filesystem
	   with new files while others still have space.
	   Its therefore important to pay some attention to
	   the freespace in the filesystem holding each
	   directory as well. However, if we did that by
	   itself, we'd keep creating new files in the file
	   system with the most space until it was as full
	   as all others, thus negating any performance
	   benefits of this RAID-1 like approach.

	   So, we use a user-configurable space threshold. If
	   there are at least 2 filesystems with more than this
	   much space available, we use RR selection between them. 
	   If not, then we pick the filesystem with the most space.

	   This gets a good balance between the two
	   approaches.  
	*/
	
	refresh_disk_space ();
	
	int free_enough = 0;

	for (i = session_dirs.begin(); i != session_dirs.end(); ++i) {
		if ((*i).blocks >= Config->get_disk_choice_space_threshold()) {
			free_enough++;
		}
	}

	if (free_enough >= 2) {

		bool found_it = false;

		/* use RR selection process, ensuring that the one
		   picked works OK.
		*/

		i = last_rr_session_dir;

		do {
			if (++i == session_dirs.end()) {
				i = session_dirs.begin();
			}

			if ((*i).blocks >= Config->get_disk_choice_space_threshold()) {
				if (ensure_sound_dir ((*i).path, result) == 0) {
					last_rr_session_dir = i;
					found_it = true;
					break;
				}
			}

		} while (i != last_rr_session_dir);

		if (!found_it) {
			result = sound_dir();
		}

	} else {

		/* pick FS with the most freespace (and that
		   seems to actually work ...)
		 */
		
		vector<space_and_path> sorted;
		space_and_path_ascending_cmp cmp;

		sorted = session_dirs;
		sort (sorted.begin(), sorted.end(), cmp);
		
		for (i = sorted.begin(); i != sorted.end(); ++i) {
			if (ensure_sound_dir ((*i).path, result) == 0) {
				last_rr_session_dir = i;
				break;
			} 
		}
		
		/* if the above fails, fall back to the most simplistic solution */
		
		if (i == sorted.end()) {
			return sound_dir();
		} 
	}

	return result;
}

int
Session::load_playlists (const XMLNode& node)
{
	XMLNodeList nlist;
	XMLNodeConstIterator niter;
	Playlist *playlist;

	nlist = node.children();

	set_dirty();

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		
		if ((playlist = XMLPlaylistFactory (**niter)) == 0) {
			error << _("Session: cannot create Playlist from XML description.") << endmsg;
			return -1;
		}
	}

	return 0;
}

Playlist *
Session::XMLPlaylistFactory (const XMLNode& node)
{
	try {
		return new AudioPlaylist (*this, node);
	}

	catch (failed_constructor& err) {
		return 0;
	}
}

int
Session::load_named_selections (const XMLNode& node)
{
	XMLNodeList nlist;
	XMLNodeConstIterator niter;
	NamedSelection *ns;

	nlist = node.children();

	set_dirty();

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		
		if ((ns = XMLNamedSelectionFactory (**niter)) == 0) {
			error << _("Session: cannot create Named Selection from XML description.") << endmsg;
			return -1;
		}
	}

	return 0;
}

NamedSelection *
Session::XMLNamedSelectionFactory (const XMLNode& node)
{
	try {
		return new NamedSelection (*this, node);
	}

	catch (failed_constructor& err) {
		return 0;
	}
}

string
Session::dead_sound_dir () const
{
	string res = _path;
	res += '/';
	res += dead_sound_dir_name;
	res += '/';
	return res;
}

string
Session::sound_dir () const
{
	string res = _path;
	res += '/';
	res += sound_dir_name;
	res += '/';
	return res;
}
	
string
Session::automation_dir () const

{
	string res = _path;
	res += '/';
	res += "automation/";
	return res;
}

string
Session::template_dir ()
{
	string path = Config->get_user_ardour_path();
	path += "/templates/";

	return path;
}

int
Session::load_connections (const XMLNode& node)
{
	XMLNodeList nlist = node.children();
	XMLNodeConstIterator niter;

	set_dirty();

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		if ((*niter)->name() == "InputConnection") {
			add_connection (new ARDOUR::InputConnection (**niter));
		} else if ((*niter)->name() == "OutputConnection") {
			add_connection (new ARDOUR::OutputConnection (**niter));
		} else {
			error << compose(_("Unknown node \"%1\" found in Connections list from state file"), (*niter)->name()) << endmsg;
			return -1;
		}
	}

	return 0;
}				

int
Session::load_edit_groups (const XMLNode& node)
{
	return load_route_groups (node, true);
}

int
Session::load_mix_groups (const XMLNode& node)
{
	return load_route_groups (node, false);
}

int
Session::load_route_groups (const XMLNode& node, bool edit)
{
	XMLNodeList nlist = node.children();
	XMLNodeConstIterator niter;
	RouteGroup* route;

	set_dirty();

	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		if ((*niter)->name() == "RouteGroup") {
			if (edit) {
				route = add_edit_group ("");
				route->set_state (**niter);
			} else {
				route = add_mix_group ("");
				route->set_state (**niter);
			}
		}
	}
	
	return 0;
}				

void
Session::swap_configuration(Configuration** new_config)
{
	LockMonitor lm (route_lock, __LINE__, __FILE__);
	Configuration* tmp = *new_config;
	*new_config = Config;
	Config = tmp;
	set_dirty();
}

void
Session::copy_configuration(Configuration* new_config)
{
	LockMonitor lm (route_lock, __LINE__, __FILE__);
	new_config = new Configuration(*Config);
}

static bool
state_file_filter (const string &str, void *arg)
{
	return (str.length() > strlen(Session::statefile_suffix()) &&
		str.find (Session::statefile_suffix()) == (str.length() - strlen (Session::statefile_suffix())));
}

struct string_cmp {
    bool operator()(const string* a, const string* b) {
	    return *a < *b;
    }
};

static string*
remove_end(string* state)
{
	string statename(*state);
	
	string::size_type start,end;
	if ((start = statename.find_last_of ('/')) != string::npos) {
			statename = statename.substr (start+1);
	}
		
	if ((end = statename.rfind(".ardour")) < 0) {
		end = statename.length();
	}

	return new string(statename.substr (0, end));
}

vector<string *> *
Session::possible_states (string path) 
{
	PathScanner scanner;
	vector<string*>* states = scanner (path, state_file_filter, 0, false, false);
	
	transform(states->begin(), states->end(), states->begin(), remove_end);
	
	string_cmp cmp;
	sort (states->begin(), states->end(), cmp);
	
	return states;
}

vector<string *> *
Session::possible_states () const
{
	return possible_states(_path);
}

void
Session::auto_save()
{
	save_state (_current_snapshot_name);
}

RouteGroup *
Session::add_edit_group (string name)
{
	RouteGroup* rg = new RouteGroup (name);
	edit_groups.push_back (rg);
	 edit_group_added (rg); /* EMIT SIGNAL */
	set_dirty();
	return rg;
}

RouteGroup *
Session::add_mix_group (string name)
{
	RouteGroup* rg = new RouteGroup (name, RouteGroup::Relative);
	mix_groups.push_back (rg);
	 mix_group_added (rg); /* EMIT SIGNAL */
	set_dirty();
	return rg;
}

RouteGroup *
Session::mix_group_by_name (string name)
{
	list<RouteGroup *>::iterator i;

	for (i = mix_groups.begin(); i != mix_groups.end(); ++i) {
		if ((*i)->name() == name) {
			return* i;
		}
	}
	return 0;
}

RouteGroup *
Session::edit_group_by_name (string name)
{
	list<RouteGroup *>::iterator i;

	for (i = edit_groups.begin(); i != edit_groups.end(); ++i) {
		if ((*i)->name() == name) {
			return* i;
		}
	}
	return 0;
}

int
Session::save_as (string new_name)
{
	return -1;
}

void
Session::begin_reversible_command (string name, UndoAction* private_undo)
{
	current_cmd.clear ();
	current_cmd.set_name (name);

	if (private_undo) {
		current_cmd.add_undo (*private_undo);
	}
}

void
Session::commit_reversible_command (UndoAction* private_redo)
{
	struct timeval now;

	if (private_redo) {
		current_cmd.add_redo_no_execute (*private_redo);
	}

	gettimeofday (&now, 0);
	current_cmd.set_timestamp (now);

	history.add (current_cmd);
}

Session::GlobalRouteBooleanState 
Session::get_global_route_boolean (bool (Route::*method)(void) const)
{
	GlobalRouteBooleanState s;
	LockMonitor lm (route_lock, __LINE__, __FILE__);

	for (RouteList::iterator i = routes.begin(); i != routes.end(); ++i) {
		if (!(*i)->hidden()) {
			RouteBooleanState v;
			
			v.first =* i;
			v.second = ((*i)->*method)();
			
			s.push_back (v);
		}
	}

	return s;
}

void
Session::set_global_route_boolean (GlobalRouteBooleanState s, void (Route::*method)(bool, void*), void* arg)
{
	LockMonitor lm (route_lock, __LINE__, __FILE__);

	for (GlobalRouteBooleanState::iterator i = s.begin(); i != s.end(); ++i) {
		(i->first->*method) (i->second, arg);
	}
}

void
Session::set_global_mute (GlobalRouteBooleanState s, void* src)
{
	set_global_route_boolean (s, &Route::set_mute, src);
}

void
Session::set_global_solo (GlobalRouteBooleanState s, void* src)
{
	set_global_route_boolean (s, &Route::set_solo, src);
}

void
Session::set_global_record_enable (GlobalRouteBooleanState s, void* src)
{
	set_global_route_boolean (s, &Route::set_record_enable, src);
}

UndoAction
Session::global_mute_memento (void* src)
{
	return bind (slot (*this, &Session::set_global_mute), get_global_route_boolean (&Route::muted), src);
}

UndoAction
Session::global_solo_memento (void* src)
{
	return bind (slot (*this, &Session::set_global_solo), get_global_route_boolean (&Route::soloed), src);
}

UndoAction
Session::global_record_enable_memento (void* src)
{
	return bind (slot (*this, &Session::set_global_record_enable), get_global_route_boolean (&Route::record_enabled), src);
}

static bool
template_filter (const string &str, void *arg)
{
	return (str.length() > strlen(Session::template_suffix()) &&
		str.find (Session::template_suffix()) == (str.length() - strlen (Session::template_suffix())));
}

void
Session::get_template_list (list<string> &template_names)
{
	vector<string *> *templates;
	PathScanner scanner;
	string path;
	char *envvar;

	if ((envvar = getenv ("HOME")) != 0) {
		path = envvar;
		path += "/.ardour/templates:";
	}

	path += DATA_DIR;
	path += "/ardour/templates";
	
	templates = scanner (path, template_filter, 0, false, true);
	
	vector<string*>::iterator i;
	for (i = templates->begin(); i != templates->end(); ++i) {
		string fullpath = *(*i);
		int start, end;

		start = fullpath.find_last_of ('/') + 1;
		if ((end = fullpath.find_last_of ('.')) <0) {
			end = fullpath.length();
		}
		
		template_names.push_back(fullpath.substr(start, (end-start)));
	}
}

int
Session::read_recent_sessions (RecentSessions& rs)
{
	string path = Config->get_user_ardour_path();
	path += "/recent";

	ifstream recent (path.c_str());
	
	if (!recent) {
		if (errno != ENOENT) {
			error << compose (_("cannot open recent session file %1 (%2)"), path, strerror (errno)) << endmsg;
			return -1;
		} else {
			return 1;
		}
	}

	while (true) {

		pair<string,string> newpair;

		getline(recent, newpair.first);

		if (!recent.good()) {
			break;
		}
		
		getline(recent, newpair.second);

		if (!recent.good()) {
			break;
		}

		rs.insert (newpair);
	}

	return 0;
}

int
Session::write_recent_sessions (RecentSessions& rs)
{
	string path = Config->get_user_ardour_path();
	path += "/recent";

	ofstream recent (path.c_str());

	if (!recent) {
		return -1;
	}

	for (RecentSessions::iterator i = rs.begin(); i != rs.end(); ++i) {
		recent << (*i).first << '\n' << (*i).second << endl;
	}
	
	return 0;
}
	
int
Session::store_in_recent_sessions ()
{
	RecentSessions rs;

	if (read_recent_sessions (rs) < 0) {
		return -1;
	}

	if (rs.find (_name) == rs.end()) {
		
		pair<string,string> newpair;

		newpair.first = _name;
		newpair.second = _path;
		
		rs.insert (newpair);

		return write_recent_sessions (rs);
	} 

	return 0;
}

int
Session::read_favorite_dirs (FavoriteDirs & favs)
{
	string path = Config->get_user_ardour_path();
	path += "/favorite_dirs";

	ifstream fav (path.c_str());

	favs.clear();
	
	if (!fav) {
		if (errno != ENOENT) {
			//error << compose (_("cannot open favorite file %1 (%2)"), path, strerror (errno)) << endmsg;
			return -1;
		} else {
			return 1;
		}
	}

	while (true) {

	        string newfav;

		getline(fav, newfav);

		if (!fav.good()) {
			break;
		}

		favs.push_back (newfav);
	}

	return 0;
}

int
Session::write_favorite_dirs (FavoriteDirs & favs)
{
	string path = Config->get_user_ardour_path();
	path += "/favorite_dirs";

	ofstream fav (path.c_str());

	if (!fav) {
		return -1;
	}

	for (FavoriteDirs::iterator i = favs.begin(); i != favs.end(); ++i) {
		fav << (*i) << endl;
	}
	
	return 0;
}


int
Session::cleanup_sources (Session::cleanup_report& rep)
{
	vector<Source*> dead_sources;
	SourceList::iterator tmp;
	AudioRegionList::iterator ir;
	DiskStreamList::iterator ids;

	rep.paths.clear ();
	rep.space = 0;

	for (SourceList::iterator i = sources.begin(); i != sources.end(); ) {

		tmp = i;
		++tmp;

		if ((*i).second->use_cnt() == 0) {
			dead_sources.push_back (i->second);
			sources.erase (i);
		}

		i = tmp;
	}

	/* now move everything not used to the trash directory.
	 */
		
	for (vector<Source*>::iterator i = dead_sources.begin(); i != dead_sources.end();++i) {
		FileSource* fs;

		if ((fs = dynamic_cast<FileSource*> (*i)) != 0) {

			if (remove_file_source (*fs)) {
				return -1;
			}

			rep.paths.push_back (fs->name());
			rep.space += fs->length() * sizeof (jack_nframes_t);

			delete fs;
		}
	}

	return 0;
}

int
Session::cleanup_trash_sources (Session::cleanup_report& rep)
{
	vector<space_and_path>::iterator i;
	string dead_sound_dir;
	struct dirent* dentry;
	struct stat statbuf;
	DIR* dead;

	rep.paths.clear ();
	rep.space = 0;

	for (i = session_dirs.begin(); i != session_dirs.end(); ++i) {
		
		dead_sound_dir = (*i).path;
		dead_sound_dir += '/';
		dead_sound_dir += dead_sound_dir_name;

		cerr << "checking for dead directory " << dead_sound_dir << endl;

		if ((dead = opendir (dead_sound_dir.c_str())) == 0) {
			continue;
		}

		cerr << "checking for dead files in " << dead_sound_dir << endl;

		while ((dentry = readdir (dead)) != 0) {

			/* avoid '.' and '..' */
			
			if ((dentry->d_name[0] == '.' && dentry->d_name[1] == '\0') || 
			    (dentry->d_name[2] == '\0' && dentry->d_name[0] == '.' && dentry->d_name[1] == '.')) {
				continue;
			}

			string fullpath;

			fullpath = dead_sound_dir;
			fullpath += '/';
			fullpath += dentry->d_name;

			if (stat (fullpath.c_str(), &statbuf)) {
				continue;
			}

			if (!S_ISREG (statbuf.st_mode)) {
				continue;
			}

			cerr << "about to unlink " << fullpath << endl;

#if 0
			if (unlink (fullpath.c_str())) {
				error << compose (_("cannot remove dead sound file %1 (%2)"),
						  fullpath, strerror (errno))
				      << endmsg;
			}
#endif

			rep.paths.push_back (dentry->d_name);
			rep.space += statbuf.st_size;
		}

		closedir (dead);
		
	}

	return 0;
}
