/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: source.cc,v 1.34 2004/02/29 23:33:56 pauld Exp $
*/

#include <sys/stat.h>
#include <unistd.h>
#include <cerrno>
#include <ctime>
#include <cmath>
#include <iomanip>

#include <pbd/lockmonitor.h>
#include <pbd/xml++.h>
#include <pbd/pthread_utils.h>

#include <ardour/source.h>
#include <ardour/filesource.h>
#include <ardour/sndfilesource.h>

#include "i18n.h"

using std::min;
using std::max;

using namespace ARDOUR;
using namespace PBD;

SigC::Signal1<void,Source *> Source::SourceCreated;
pthread_t                    Source::peak_thread;
bool                         Source::have_peak_thread = false;
vector<Source*>              Source::pending_peak_sources;
PBD::Lock                    Source::pending_peak_sources_lock;
pthread_cond_t               Source::pending_peak_request;

bool Source::_build_missing_peakfiles = false;
bool Source::_build_peakfiles = false;

Source::Source (bool announce)
{
	_id = ARDOUR::new_id();
	_use_cnt = 0;
	_peaks_built = false;
	next_peak_clear_should_notify = true;
	peakfile = 0;
	_timestamp = 0;
	_read_data_count = 0;
	_write_data_count = 0;
}

Source::Source (const XMLNode& node) 
{
	_use_cnt = 0;
	_peaks_built = false;
	next_peak_clear_should_notify = true;
	peakfile = 0;
	_timestamp = 0;
	_read_data_count = 0;
	_write_data_count = 0;

	if (set_state (node)) {
		throw failed_constructor();
	}
}

Source::~Source ()
{
	if (peakfile) {
		fclose (peakfile);
	}
}

XMLNode&
Source::get_state ()
{
	XMLNode *node = new XMLNode ("Source");
	char buf[64];

	node->add_property ("name", _name);
	snprintf (buf, sizeof(buf)-1, "%Lu", _id);
	node->add_property ("id", buf);

	if (_timestamp != 0) {
		snprintf (buf, sizeof (buf), "%ld", _timestamp);
		node->add_property ("timestamp", buf);
	}

	if (_captured_for.length()) {
		node->add_property ("captured-for", _captured_for);
	}

	return *node;
}

int
Source::set_state (const XMLNode& node)
{
	const XMLProperty* prop;

	if ((prop = node.property ("name")) != 0) {
		_name = prop->value();
	} else {
		return -1;
	}
	
	if ((prop = node.property ("id")) != 0) {
		sscanf (prop->value().c_str(), "%Lu", &_id);
	} else {
		return -1;
	}

	if ((prop = node.property ("timestamp")) != 0) {
		sscanf (prop->value().c_str(), "%ld", &_timestamp);
	}

	if ((prop = node.property ("captured-for")) != 0) {
		_captured_for = prop->value();
	}

	return 0;
}

/***********************************************************************
  PEAK FILE STUFF
 ***********************************************************************/

void*
Source::peak_thread_work (void* arg)
{
	LockMonitor lm (pending_peak_sources_lock, __LINE__, __FILE__);

	while (true) {
		pthread_cond_wait (&pending_peak_request, pending_peak_sources_lock.mutex());

		while (!pending_peak_sources.empty()) {

			Source* s = pending_peak_sources.front();
			pending_peak_sources.erase (pending_peak_sources.begin());

			pthread_mutex_unlock (pending_peak_sources_lock.mutex());
			s->build_peaks();
			pthread_mutex_lock (pending_peak_sources_lock.mutex());
			
		}
	}
	/*NOTREACHED*/
	return 0;
}

int
Source::start_peak_thread ()
{
	if (!_build_peakfiles) {
		return 0;
	}

	if (pthread_create_and_store ("peak file builder", &peak_thread, 0, peak_thread_work, 0)) {
		error << _("Source: could not create peak thread") << endmsg;
		return -1;
	}

	pthread_detach (peak_thread);
	have_peak_thread = true;
	return 0;
}

void
Source::stop_peak_thread ()
{
	if (have_peak_thread) {
		pthread_cancel_one (peak_thread);
	}
}

void 
Source::queue_for_peaks (Source& source)
{
	if (have_peak_thread) {
		LockMonitor lm (pending_peak_sources_lock, __LINE__, __FILE__);
		source.next_peak_clear_should_notify = true;
		pending_peak_sources.push_back (&source);
		pthread_cond_signal (&pending_peak_request);
	}
}

void Source::clear_queue_for_peaks ()
{
	/* this is done to cancel a group of running peak builds */
	if (have_peak_thread) {
		LockMonitor lm (pending_peak_sources_lock, __LINE__, __FILE__);

		while (pending_peak_sources.size() > 0) {
			Source* s = pending_peak_sources.front();
			pending_peak_sources.erase (pending_peak_sources.begin());
		}		
	}
}


bool
Source::peaks_ready (SigC::Slot0<void> the_slot) const
{
	bool ret;
	LockMonitor lm (_lock, __LINE__, __FILE__);

	/* check to see if the peak data is ready. if not
	   connect the slot while still holding the lock.
	*/

	if (!(ret = _peaks_built)) {
		PeaksReady.connect (the_slot);
	}

	return ret;
}

int
Source::initialize_peakfile (bool newfile, string audio_path)
{
	struct stat statbuf;

	peakpath = peak_path (audio_path);

	if (newfile) {

		if (!_build_peakfiles) {
			return 0;
		}

		_peaks_built = false;

	} else {

		if (stat (peakpath.c_str(), &statbuf)) {
			if (errno == ENOENT) {
				_peaks_built = false;
			} else {
				error << compose(_("Source: cannot stat peakfile \"%1\""), peakpath) << endmsg;
				return -1;
			}
		} else {
			if (statbuf.st_size == 0) {
				_peaks_built = false;
			} else {
				// Check if the audio file has changed since the peakfile was built.
				struct stat stat_file;
				int err = stat (audio_path.c_str(), &stat_file);
	
				if (!err && stat_file.st_mtime > statbuf.st_mtime){
					_peaks_built = false;
				} else {
					_peaks_built = true;
				}
			}
		}
	} 

	if ((peakfile = fopen (peakpath.c_str(), "a+")) == 0) {
		error << compose(_("Source: cannot open peakpath \"%1\" (%2)"), peakpath, strerror (errno)) << endmsg;
		return -1;
	}

	if (!newfile && !_peaks_built && _build_missing_peakfiles && _build_peakfiles) {
		info << compose(_("Source: peak file for %1 changed or missing - building."), _name) << endmsg;
		build_peaks_from_scratch ();
	} 

	return 0;
}

int 
Source::read_peaks (PeakData *peaks, jack_nframes_t npeaks, jack_nframes_t start, jack_nframes_t cnt) const
{
	LockMonitor lm (_lock, __LINE__, __FILE__);
	off_t first_peak_byte;
	double scale;
	double expected_peaks;
	double fperpeak;
	PeakData::PeakDatum xmax;
	PeakData::PeakDatum xmin;
	size_t to_read;
	size_t nread;
	jack_nframes_t zero_fill = 0;

	//cerr << "\n\nRP: npeaks = " << npeaks << " start = " << start << " cnt = " << cnt << " len = " << _length << endl;

	expected_peaks = (cnt / (double) frames_per_peak);
	scale = npeaks/expected_peaks;
	fperpeak = cnt / (double) npeaks;
	
	first_peak_byte = (start / frames_per_peak) * sizeof (PeakData);

	if (fseek (peakfile, first_peak_byte, SEEK_SET) < 0) {
		error << compose(_("Source: cannot seek to frame %1 in peakfile!"), start) << endmsg;
		return -1;
	}
	

	/* fix for near-end-of-file conditions */

	if (start + cnt > _length) {
		jack_nframes_t old = npeaks;
		npeaks = min ((_length - start) / frames_per_peak, npeaks);
		zero_fill = old - npeaks;
	}

	//cerr << "actual/npeaks = " << npeaks << " zf = " << zero_fill << endl;

	if (npeaks == cnt) {

		/* no scaling at all, just get the sample data and duplicate it for
		   both max and min peak values.
		*/
		
		Sample staging[cnt];
		
		if (read_unlocked (staging, start, cnt) != cnt) {
			error << _("cannot read sample data for unscaled peak computation") << endmsg;
			return -1;
		}

		for (jack_nframes_t i = 0; i < npeaks; ++i) {
			peaks[i].max = staging[i];
			peaks[i].min = staging[i];
		}
		
		return 0;
		
	}

	if (scale == 1.0) {
		if ((nread = fread (peaks, sizeof (PeakData), npeaks, peakfile)) != npeaks) {
			cerr << "Source["
			      << _name
			      << "]: cannot read peaks from peakfile! (read only " 
			      << nread
			      << " not " 
			      << npeaks
			      << "at sample " 
			      << start
			      << " = byte "
			      << first_peak_byte
			      << ')'
			      << endl;
			return -1;
		}

		if (zero_fill) {
			memset (&peaks[npeaks], 0, sizeof (PeakData) * zero_fill);
		}

		return 0;
	}

	size_t step;
	double istep;
	jack_nframes_t n;
	size_t i, j, nextj, skipj, jbegin, currpeak;
	size_t start_byte;
	jack_nframes_t current_frame = start;

	jack_nframes_t start_peak = (jack_nframes_t) ceil(start * scale / frames_per_peak);
	
	if (scale < 1.0) {


		/* the caller wants:

		    - more frames-per-peak (lower resolution) than the peakfile, or to put it another way,
                    - less peaks than the peakfile holds for the same range

		    So, read a block into a staging area, and then downsample from there.
		*/

		size_t chunksize = (size_t) min (expected_peaks, 4096.0);
		PeakData staging[chunksize];
		
		//ifract = modf (expected_peaks/npeaks, &istep);
		//step = (size_t) floor (istep);

		istep = expected_peaks/npeaks;
		step = (size_t) (istep + 0.5);

		// start at beginning 
		current_frame = 0;
		jack_nframes_t igncnt = 0;

		n = 0;
		skipj = 0;
		currpeak = 0;
		jbegin = 0;
		j = 0;
		
		// calculate starting position
		// and initial values for counters
		// all of this nonsense to prevent jiggling peaks when trimming at
		// the front of a region
		nextj = (size_t) ((currpeak+1) * istep + 0.5) - jbegin;
		
		while (currpeak < start_peak)
		{
			igncnt = nextj - j;
				
			current_frame += igncnt * frames_per_peak;
			++currpeak;
			j = nextj;
			nextj = (size_t) ((currpeak+1) * istep + 0.5) - jbegin;
		}

		jbegin += j;
		
		
		while (n < npeaks) {

			jack_nframes_t tnp;

			start_byte = jbegin * sizeof (PeakData);
			
			if (fseek (peakfile, start_byte, SEEK_SET) < 0) {
				error << compose(_("Source[%1]: cannot seek to frame %2 in peakfile!"), _name, current_frame) << endmsg;
				return -1;
			}

			/* don't try to read more peaks than could possibly exist
			   given the current length.
			*/
			tnp = min ((_length - current_frame)/frames_per_peak, (jack_nframes_t) expected_peaks);
			to_read = min (chunksize, (size_t) tnp);

			if ((nread = fread (staging, sizeof (PeakData), to_read, peakfile)) != to_read) {
				cerr << "Source["
				     << _name
				     << "]: cannot read peak data from peakfile ("
				     << nread 
				     << " peaks instead of "
				     << to_read
				     << ") ("
				     << strerror (errno)
				     << ')'
				     << " at start_byte = " << start_byte 
				     << " and current frame = " << current_frame 
				     << " _length = " << _length
				     << " expected maxpeaks = " << (_length - current_frame)/frames_per_peak
				     << " npeaks was " << npeaks
				     << endl;
				return -1;
			}

			j = 0;

			if (nread <= step) {
				
				/* we didn't read enough data to interpolate,
				   so just copy the data we did read into
				   the actual buffer, and then get out of here.
				*/

				while (j < nread && n < npeaks) {
					peaks[n].max = staging[j].max;
					peaks[n].min = staging[j].min;
					j++;
					n++;
				}

				break;
			}

			j = 0;
			nextj = (size_t) ((currpeak+1) * istep + 0.5) - jbegin;

			if (nread < nextj) {
				nextj = nread;
			}
			
			while ((nextj <= nread) && (n < npeaks))
			{
				igncnt = nextj - j;
				
				xmax = -1.0;
				xmin = 1.0;
				
				for (; j < nextj; ++j) {
					xmax = max (xmax, staging[j].max);
					xmin = min (xmin, staging[j].min);
				}
				
				peaks[n].max = xmax;
				peaks[n].min = xmin;
				
				n++;
				
				current_frame += igncnt * frames_per_peak;

				++currpeak;
				j = nextj;
				nextj = (size_t) ((currpeak+1) * istep + 0.5) - jbegin;
			}

			jbegin += j;

		}

		if (zero_fill) {
			//cerr << "at zero, npeaks = " << npeaks << endl;
			memset (&peaks[npeaks], 0, sizeof (PeakData) * zero_fill);
		}

		return 0;

	} else {
		
//		cerr << "scale > 1.0\n";

		/* the caller wants 

		     - less frames-per-peak (more resolution)
		     - more peaks than stored in the Peakfile

		   So, fetch data from the raw source, and generate peak
		   data on the fly.
		*/

		Sample buf[4096];
		size_t offset = start;

		istep = (double)cnt/(double)npeaks;
		step = (size_t) (istep + 0.5);
		n = 0;
		nextj = j = 0;
		i = 0;
		currpeak = 0;
		
		start_peak = (jack_nframes_t) (offset / istep);
		currpeak = 0;
		jbegin = 0;

		// set up initial positions
		// all of this nonsense to prevent jiggling peaks when trimming at
		// the front of a region
		nextj = (size_t) ((currpeak+1) * istep + 0.5) - jbegin;
		
		while (currpeak < start_peak)
		{
			++currpeak;
			j = nextj;
			nextj = (size_t) ((currpeak+1) * istep + 0.5) - jbegin;
		}

		jbegin += j;

		
		while (n < npeaks) {

			to_read = min ((size_t) (_length - jbegin), (size_t) 4096);

			if ((nread = read_unlocked (buf, jbegin, to_read)) < 0) {
				error << compose(_("Source[%1]: peak read - cannot read %2 samples at offset %3"), _name, to_read, jbegin) << endmsg;
				return -1;
			}

			nread /= sizeof (Sample);

			if (nread == 0) {
				memset (&peaks[n], 0, (npeaks - n) * sizeof (PeakData));
				break;
			}

			j = 0;

			nextj = (size_t) ((currpeak+1) * istep + 0.5) - jbegin;
			//printf ("j is %d  n=%d  npeaks=%d nextj=%d nread=%d  currpeak: %d   offset: %d\n", j, n, npeaks, nextj, nread, currpeak, offset);

			if (nread < nextj) {
				nextj = nread;
			}
			
			while (nextj <= nread && n < npeaks)
			{
				xmax = -1.0;
				xmin = 1.0;

				for (; j < nextj && j < nread; ++j) {
					xmax = max (xmax, buf[j]);
					xmin = min (xmin, buf[j]);
				} 

				peaks[n].min = xmin;
				peaks[n].max = xmax;
				n++;
				++currpeak;

				nextj = (size_t) ((currpeak+1) * istep + 0.5) - jbegin;
				//printf ("j is %d  n=%d  nextj=%d nread=%d\n", j, n, nextj, nread);
			}

			jbegin += j;
			
			//printf ("j is %d  n=%d  i=%d  nread=%d nextj=%d\n", j, n, i, nread, nextj);
		}
		
		if (zero_fill) {
			memset (&peaks[npeaks], 0, sizeof (PeakData) * zero_fill);
		}

		return 0;
	}

	return -1;
}

#undef DEBUG_PEAK_BUILD

int
Source::build_peaks ()
{
	vector<PeakBuildRecord*> built;
	int status = -1;

	{
		LockMonitor lm (_lock, __LINE__, __FILE__);
		
		for (list<PeakBuildRecord *>::iterator i = pending_peak_builds.begin(); i != pending_peak_builds.end(); ++i) {
			
			if ((status = do_build_peak ((*i)->frame, (*i)->cnt)) != 0) { 
				unlink (peakpath.c_str());
				break;
			}

			built.push_back (new PeakBuildRecord (*(*i)));
			delete *i;
		}
		
		pending_peak_builds.clear ();
	}

	if (status == 0) {
		
		for (vector<PeakBuildRecord *>::iterator i = built.begin(); i != built.end(); ++i) {
			 PeakRangeReady ((*i)->frame, (*i)->cnt); /* EMIT SIGNAL */
		}

		if (next_peak_clear_should_notify) {
			_peaks_built = true;
			next_peak_clear_should_notify = false;
			 PeaksReady (); /* EMIT SIGNAL */
		}
	}

	for (vector<PeakBuildRecord *>::iterator i = built.begin(); i != built.end(); ++i) {
		delete *i;
	}

	return status;
}

int
Source::do_build_peak (jack_nframes_t first_frame, jack_nframes_t cnt)
{
	jack_nframes_t current_frame;
	Sample buf[frames_per_peak];
	Sample xmin, xmax;
	PeakData peak;
	jack_nframes_t frames_read;
	jack_nframes_t frames_to_read;
	off_t first_peak_byte;

#ifdef DEBUG_PEAK_BUILD
	cerr << _name << ": building peaks for " << first_frame << " to " << first_frame + cnt - 1 << endl;
#endif

	first_peak_byte = (first_frame / frames_per_peak) * sizeof (PeakData);

#ifdef DEBUG_PEAK_BUILD
	cerr << "seeing to " << first_peak_byte << " before writing new peak data\n";
#endif

	if (fseek (peakfile, first_peak_byte, SEEK_SET) < 0) {
		error << compose(_("%1: could not seek to byte %2 in peakfile (%3"), _name, first_peak_byte, strerror (errno)) << endmsg;
		return -1;
	}
	
	current_frame = first_frame;

	while (cnt) {

		frames_to_read = min (frames_per_peak, cnt);

		if ((frames_read = read_unlocked (buf, current_frame, frames_to_read)) != frames_to_read) {
			error << compose(_("%1: could not write read raw data for peak computation (%2)"), _name, strerror (errno)) << endmsg;
			return -1;
		}

		xmin = 1.0;
		xmax = -1.0;

		for (jack_nframes_t n = 0; n < frames_read; ++n) {
			xmax = max (xmax, buf[n]);
			xmin = min (xmin, buf[n]);
		}

		peak.max = xmax;
		peak.min = xmin;

#ifdef DEBUG_PEAK_BUILD
		off_t where = ftell (peakfile);
		cerr << "writing to " << where << " for first frame = " << first_frame << endl;
#endif

		if (fwrite (&peak, sizeof (peak), 1, peakfile) != 1) {
			error << compose(_("%1: could not write peak file data (%2)"), _name, strerror (errno)) << endmsg;
			return -1;
		}

#ifdef DEBUG_PEAK_BUILD
		off_t where2 = ftell (peakfile);
		cerr << "after write, peakfile position = " << where2 << endl;
#endif

		current_frame += frames_per_peak;
		cnt -= frames_to_read;

#ifdef DEBUG_PEAK_BUILD
		cerr << _name << ": built peaks to " << current_frame << endl;
#endif
	}

	fflush (peakfile);
	return 0;
}

void
Source::build_peaks_from_scratch ()
{
	LockMonitor lp (_lock, __LINE__, __FILE__); 

	next_peak_clear_should_notify = true;
	pending_peak_builds.push_back (new PeakBuildRecord (0, _length));
	queue_for_peaks (*this);
}

bool
Source::file_changed (string path)
{
	struct stat stat_file;
	struct stat stat_peak;

	int e1 = stat (path.c_str(), &stat_file);
	int e2 = stat (peak_path(path).c_str(), &stat_peak);
	
	if (!e1 && !e2 && stat_file.st_mtime > stat_peak.st_mtime){
		return true;
	} else {
		return false;
	}
}

void
Source::use ()
{
	_use_cnt++;
}

void
Source::release ()
{
	if (_use_cnt) --_use_cnt;
}
