/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: editor_timefx.cc,v 1.9 2004/07/05 18:25:17 pauld Exp $
*/

#include <cstdlib>
#include <cmath>

#include <string>

#include <pbd/error.h>
#include <pbd/pthread_utils.h>

#include "editor.h"
#include "audio_time_axis.h"
#include "regionview.h"

#include <ardour/session.h>
#include <ardour/region.h>
#include <ardour/audioplaylist.h>
#include <ardour/audio_track.h>
#include <ardour/audioregion.h>
#include <ardour/diskstream.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;
using namespace Gtk;

void*
Editor::timestretch_thread (void *arg)
{
	PBD::ThreadCreated (pthread_self());

	TimeStretchDialog* args = reinterpret_cast<TimeStretchDialog*>(arg);

	pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, 0);

	args->editor._do_timestretch (*args);

	return 0;
}

Editor::TimeStretchDialog::TimeStretchDialog (Editor& e, TimeAxisView& tv, AudioRegionView& rv)
	: ArdourDialog ("time stretch dialog"),
	  editor (e),
	  quick_button (_("Quick but Ugly")),
	  antialias_button (_("Skip Anti-aliasing")),
	  cancel_button (_("Cancel")),
	  action_button (_("Stretch/Shrink it"))
{
	AudioTimeAxisView* atv;

	if ((atv = dynamic_cast<AudioTimeAxisView*>(&tv)) == 0) {
		fatal << _("attempt to timestretch a non-audio track!") << endmsg;
		/*NOTREACHED*/
		throw failed_constructor();		
	}

	tsr = new Session::TimeStretchRequest (&atv->route(), rv.region);

	set_modal (true);
	set_position (GTK_WIN_POS_MOUSE);
	set_title (_("ardour: timestretch"));
	set_name (N_("TimeStretchDialog"));

	add (packer);

	packer.set_spacing (5);
	packer.set_border_width (5);
	packer.pack_start (upper_button_box);
	packer.pack_start (progress_bar);
	packer.pack_start (lower_button_box);
	
	upper_button_box.set_homogeneous (true);
	upper_button_box.set_spacing (5);
	upper_button_box.set_border_width (5);
	upper_button_box.pack_start (quick_button, true, true);
	upper_button_box.pack_start (antialias_button, true, true);

	lower_button_box.set_homogeneous (true);
	lower_button_box.set_spacing (5);
	lower_button_box.set_border_width (5);
	lower_button_box.pack_start (action_button, true, true);
	lower_button_box.pack_start (cancel_button, true, true);

	action_button.set_name (N_("TimeStretchButton"));
	cancel_button.set_name (N_("TimeStretchButton"));
	quick_button.set_name (N_("TimeStretchButton"));
	antialias_button.set_name (N_("TimeStretchButton"));
	progress_bar.set_name (N_("TimeStretchProgress"));

	action_button.clicked.connect (bind (slot (editor, &Editor::finish_sub_event_loop), 1));
	first_cancel = cancel_button.clicked.connect (bind (slot (editor, &Editor::finish_sub_event_loop), -1));
	first_delete = delete_event.connect (bind (slot (editor, &Editor::finish_sub_event_loop_on_delete), -1));
}

gint
Editor::TimeStretchDialog::update_progress ()
{
	progress_bar.set_percentage (tsr->progress);
	return tsr->running;
}

void
Editor::TimeStretchDialog::cancel_timestretch_in_progress ()
{
	tsr->running = false;
}

gint
Editor::TimeStretchDialog::delete_timestretch_in_progress (GdkEventAny* ev)
{
	tsr->running = false;
	return TRUE;
}

void
Editor::start_timestretch (TimeAxisView& tv, AudioRegionView& rv, jack_nframes_t start, jack_nframes_t end)
{
	TimeStretchDialog *tsd = new TimeStretchDialog (*this, tv, rv);

	tsd->show_all ();

	run_sub_event_loop ();

	if (sub_event_loop_status == 1) {

		tsd->tsr->start = start;
		tsd->tsr->new_length = end - start + 1;
		tsd->tsr->quick_seek = tsd->quick_button.get_active();
		tsd->tsr->antialias = !tsd->antialias_button.get_active();
		tsd->tsr->progress = 0.0f;
		tsd->tsr->running = true;

		do_timestretch (*tsd);
	}

	delete tsd;
	return;
}

void
Editor::do_timestretch (TimeStretchDialog& tsd)
{
	pthread_t thread;

	/* re-connect the cancel button and delete events */

	tsd.first_cancel.disconnect();
	tsd.first_delete.disconnect();

	tsd.cancel_button.clicked.connect (slot (tsd, &TimeStretchDialog::cancel_timestretch_in_progress));
	tsd.delete_event.connect (slot (tsd, &TimeStretchDialog::delete_timestretch_in_progress));

	if (pthread_create_and_store ("timestretch", &thread, 0, timestretch_thread, &tsd)) {
		error << _("cannot create timestretch thread - operation not carried out") << endmsg;
		return;
	}

	pthread_detach (thread);

	Main::timeout.connect (slot (tsd, &TimeStretchDialog::update_progress), 100);

	while (tsd.tsr->running) {
		gtk_main_iteration ();
	}
}

void
Editor::_do_timestretch (TimeStretchDialog& tsd)
{
	AudioTrack* at;
	Playlist* playlist;
	AudioRegion* new_region;

	if ((at = dynamic_cast<AudioTrack*> (tsd.tsr->route)) == 0) {
		tsd.tsr->running = false;
		return;
	}
	
	if ((playlist = at->disk_stream().playlist()) == 0) {
		tsd.tsr->running = false;
		return;
	}

	if ((new_region = session->tempoize_region (*tsd.tsr)) == 0) {
		tsd.tsr->running = false;
		return;
	}

	if (tsd.tsr->running) {
		jack_nframes_t pos;
		
		if (tsd.tsr->start < tsd.tsr->region.position()) {
			pos = tsd.tsr->start;
		} else {
			pos = tsd.tsr->region.position();
		}
		
		session->add_undo (playlist->get_memento());
		playlist->replace_region (tsd.tsr->region, *new_region, pos);
		session->add_redo_no_execute (playlist->get_memento());
		
		tsd.tsr->running = false;
	}
}
