/*
 * Copyright (c) 1995 - 2004 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL").
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define __NO_VERSION__

#include <nnpfs/nnpfs_locl.h>
#include <nnpfs/nnpfs_message.h>
#include <nnpfs/nnpfs_msg_locl.h>
#include <nnpfs/nnpfs_deb.h>
#include <nnpfs/nnpfs_fs.h>
#include <nnpfs/nnpfs_dev.h>

#ifdef RCSID
RCSID("$Id: nnpfs_message.c,v 1.136 2004/06/01 22:10:21 tol Exp $");
#endif

static void
clear_all_children (struct inode *inode, int parent);

static void
nnpfs_d_remove(struct dentry *dentry)
{
    NNPFSDEB(XDEBMSG, ("nnpfs_d_remove %p\n", dentry));
    spin_lock(&dcache_lock);
    dget_locked(dentry);
    spin_unlock(&dcache_lock);
    d_drop(dentry);
    dput(dentry);
}

int
nnpfs_message_installroot(struct nnpfs *nnpfsp,
			  struct nnpfs_message_installroot *message,
			  u_int size)
{
    int error = 0;
    
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installroot\n"));
    
    if (nnpfsp->root != 0) {
	printk(KERN_EMERG "NNPFS Panic: nnpfs_message_installroot again\n");
	error = -EBUSY;
    } else {
	/* VN_HOLD's */
	error = new_nnpfs_node(nnpfsp, &message->node,
			       NNPFS_TO_VFS(nnpfsp)->s_root->d_inode,
			       &nnpfsp->root);
    }

    return error;
}

int
nnpfs_message_installnode(struct nnpfs *nnpfsp,
			  struct nnpfs_message_installnode *message,
			  u_int size)
{
    int error = 0;
    struct nnpfs_node *n, *dp;
    struct dentry *dentry = NULL;
    struct inode *inode;
    struct dentry *parent = NULL;
    struct qstr sqstr;
    struct list_head *alias;
    
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode\n"));
    
    error = nnpfs_node_find(nnpfsp, &message->parent_handle, &dp);
    if (error) {
    	printk(KERN_EMERG "NNPFS Panic: nnpfs_message_install "
	       "could not find parent (%d)\n", error);
	return error;
    }
    inode = XNODE_TO_VNODE(dp);
	
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode: dp: %p aliases:", inode));
    nnpfs_print_aliases(inode);
    
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode: fetching new node\n"));
    error = new_nnpfs_node(nnpfsp, &message->node, NULL, &n); /* VN_HOLD's */
    if (error != 0) {
	NNPFSDEB(XDEBMSG,
		 ("nnpfs_message_installnode: no new node (%d)\n", error));

	if (error == -EISDIR)
	    return 0;

	return error;
    }

    NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode: inode: %p tokens: 0x%x aliases: ",
		       XNODE_TO_VNODE(n), n->tokens));
    nnpfs_print_aliases(XNODE_TO_VNODE(n));
    sqstr.name = message->name;
    sqstr.len  = strlen(message->name);
    sqstr.hash = full_name_hash(sqstr.name, sqstr.len);
    
    /*
     * for all parent dentries
     *   if node with name
     *     if empty
     *       d_instantiate
     *     else if `check inode'
     *       complain
     *     else
     *       already inserted
     *   else
     *     allocate node
     *
     */
    
    alias = inode->i_dentry.next;
    while (alias != &inode->i_dentry) {
	parent = list_entry(alias, struct dentry, d_alias);
	spin_lock(&dcache_lock);
	dget_locked(parent);
	spin_unlock(&dcache_lock);
	dentry = d_lookup(parent, &sqstr);
	NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode: alias %p\n", parent));
	NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode: lookup %p\n", dentry));
	
	if (dentry) {
	    if (dentry->d_inode == NULL) {
		nnpfs_iref(XNODE_TO_VNODE(n));
		d_instantiate(dentry, XNODE_TO_VNODE(n));
		DENTRY_TO_XDENTRY(dentry)->xd_flags =
		    (NNPFS_XD_ENTRY_VALID|NNPFS_XD_NAME_VALID);
	    } else if (dentry->d_inode != XNODE_TO_VNODE(n)) {
		/* if the name was invalid */
		if ((DENTRY_TO_XDENTRY(dentry)->xd_flags & NNPFS_XD_NAME_VALID) != 0)
		    printk(KERN_EMERG
			   "NNPFS SoftAssert: existing inode "
			   "(%p, fid %d.%d.%d.%d) != "
			   "installing %s(%p, fid %d.%d.%d.%d)\n",
			   dentry->d_inode,
			   VNODE_TO_XNODE(dentry->d_inode)->handle.a,
			   VNODE_TO_XNODE(dentry->d_inode)->handle.b,
			   VNODE_TO_XNODE(dentry->d_inode)->handle.c,
			   VNODE_TO_XNODE(dentry->d_inode)->handle.d,
			   message->name,
			   XNODE_TO_VNODE(n),
			   n->handle.a,
			   n->handle.b,
			   n->handle.c,
			   n->handle.d);
		if (nnpfs_dcount(dentry) == 0)
		    nnpfs_d_remove(dentry);
		else
		    d_drop(dentry);
		goto insert_name;
	    } else {
		DENTRY_TO_XDENTRY(dentry)->xd_flags =
			(NNPFS_XD_ENTRY_VALID|NNPFS_XD_NAME_VALID);
	    }
	} else {
	    /* unprovoked installnode, ie bulkstatus) */
	insert_name:
	    dentry = d_alloc(parent, &sqstr);
	    NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode: "
			       "allocated new entry: %p\n",
			       dentry));
	    error = nnpfs_d_init(dentry);
	    if (error == 0) {
		DENTRY_TO_XDENTRY(dentry)->xd_flags =
		    (NNPFS_XD_ENTRY_VALID|NNPFS_XD_NAME_VALID);
		nnpfs_iref(XNODE_TO_VNODE(n));
		d_add(dentry, XNODE_TO_VNODE(n));
	    }
	}
	
	dput(dentry);
	dentry = NULL;
	alias = alias->next;
	dput(parent);
    }
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode: done installing\n"));
    
    iput(XNODE_TO_VNODE(n));
    
    return error;
}

int
nnpfs_message_installattr(struct nnpfs *nnpfsp,
			  struct nnpfs_message_installattr *message,
			  u_int size)
{
    struct nnpfs_node *t;
    struct inode *inode;
    struct dentry *dentry;
    int i, error;
    
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installattr (%d.%d.%d.%d)\n",
		       message->node.handle.a,
		       message->node.handle.b,
		       message->node.handle.c,
		       message->node.handle.d));
    
    error = nnpfs_node_find(nnpfsp, &message->node.handle, &t);
    if (error) {
	NNPFSDEB(XDEBMSG,
		 ("nnpfs_message_installattr: no such node (%d)\n", error));
	return error;
    }

    inode = XNODE_TO_VNODE(t);
    dentry = list_entry(inode->i_dentry.next, struct dentry, d_alias);
    
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installattr name:%s\n",
		       dentry->d_name.name));
    
    /*
     * Paranoid checks
     */
    
    t->tokens = message->node.tokens;
    if (NNPFS_TOKEN_GOT(t, NNPFS_DATA_R) && DATA_FROM_XNODE(t) == NULL) {
#if 0
	printk(KERN_EMERG 
	       "NNPFS Panic: nnpfs_message_installattr: "
	       "got token w/o data!\n");
#else
	NNPFSDEB(XDEBMSG, ("NNPFS Panic: nnpfs_message_installattr: "
			   "got token w/o data!\n"));
#endif
	NNPFS_TOKEN_CLEAR (t, NNPFS_DATA_R|NNPFS_DATA_W , NNPFS_DATA_MASK);
    }
    
    if (!NNPFS_TOKEN_GOT(t, NNPFS_DATA_R) && DATA_FROM_XNODE(t)) {
	printk(KERN_EMERG 
	       "NNPFS SoftAssert: nnpfs_message_installattr: "
	       "got data w/o token!\n");
    }
    
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installattr: tokens: 0x%x\n", t->tokens));
    t->attr = message->node.attr;
    nnpfs_attr2inode (&t->attr, inode, 0);
    if (t->offset > inode->i_size)
	t->offset = inode->i_size;
    
    memmove(t->id, message->node.id, sizeof(t->id));
    memmove(t->rights, message->node.rights, sizeof(t->rights));
    for (i = 0; i < MAXRIGHTS; i++) {
	NNPFSDEB(XDEBMSG, ("rights %d:", t->id[i]));
	NNPFSDEB(XDEBMSG, (t->rights[i]&NNPFS_RIGHT_R?"r":"-"));
	NNPFSDEB(XDEBMSG, (t->rights[i]&NNPFS_RIGHT_W?"w":"-"));
	NNPFSDEB(XDEBMSG, (t->rights[i]&NNPFS_RIGHT_X?"x":"-"));
	NNPFSDEB(XDEBMSG, ("\n"));
    }
    t->anonrights = message->node.anonrights;
    
    return 0;
}

int
nnpfs_message_installdata(struct nnpfs *nnpfsp,
			  struct nnpfs_message_installdata *message,
			  u_int size)
{
    struct nnpfs_node *t;
    struct dentry *dentry;
    struct inode *inode;
    int error = 0;
    
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installdata\n"));
    
    error = nnpfs_node_find(nnpfsp, &message->node.handle, &t);
    if (error) {
	printk(KERN_EMERG "NNPFS Panic: "
	       "nnpfs_message_installdata didn't find node (%d)!\n",
	       error);
	return error;
    }

    inode = XNODE_TO_VNODE(t);
    
    message->cache_name[sizeof(message->cache_name)-1] = '\0';
    NNPFSDEB(XDEBMSG, ("cache_name '%s'\n", message->cache_name));
    if (message->flag & NNPFS_ID_HANDLE_VALID) {
	dentry = nnpfs_fh_to_dentry(&message->cache_handle);
	
	if (IS_ERR(dentry))
	    printk(KERN_EMERG "NNPFS Panic: "
		   "nnpfs_message_installdata failed "
		   "nnpfs_fh_to_dentry = %s, errno: %ld\n",
		   message->cache_name, PTR_ERR(dentry));
    } else
	dentry = ERR_PTR(-EINVAL);
    
    NNPFSDEB(XDEBMSG, ("nnpfs_message_installdata dentry: %p\n",
		       dentry));
    
    if (IS_ERR(dentry)) {
	struct nameidata nd;
	
#ifdef LINUX2_5
	error = path_lookup(message->cache_name, 0, &nd);
#else
	if (path_init (message->cache_name, LOOKUP_POSITIVE, &nd))
	    error = path_walk (message->cache_name, &nd);
#endif
	if (error == 0) {
	    dentry = nd.dentry;
	    nd.dentry = NULL;
	    path_release(&nd);
	}
	if (error) {
	    printk(KERN_EMERG "NNPFS Panic: "
		   "nnpfs_message_installdata failed open_namei, "
		   "errno: %d\n", error);
	    return error;
	}
    }
    if (!IS_ERR(dentry)) {
	inode->i_mapping = &dentry->d_inode->i_data;
	
	if (DATA_FROM_XNODE(t)) {
	    dput(DATA_FROM_XNODE(t));
	}
	DATA_FROM_XNODE(t) = dentry;
	if (message->flag & NNPFS_ID_INVALID_DNLC)
	    clear_all_children (inode, 0);
	
	/*
	 * Paranoid checks
	 */
	
	t->tokens = message->node.tokens;
	if (!NNPFS_TOKEN_GOT(t, NNPFS_DATA_R)) {
	    printk(KERN_EMERG 
		   "NNPFS SoftAssert: nnpfs_message_installdata: "
		   "got data w/o token!\n");
	}
	
	NNPFSDEB(XDEBMSG, ("nnpfs_message_installdata: tokens: 0x%x\n",
			   t->tokens));
	
	t->attr = message->node.attr;
	NNPFSDEB(XDEBMSG, ("nnpfs_message_installdata size before: %ld\n",
			   (long) inode->i_size));
	nnpfs_attr2inode (&t->attr, inode, 1);
	t->offset = message->offset;
	NNPFSDEB(XDEBMSG, ("nnpfs_message_installdata size after: %ld\n",
			   (long) inode->i_size));
	memmove(t->id, message->node.id, sizeof(t->id));
	memmove(t->rights, message->node.rights, sizeof(t->rights));
	t->anonrights = message->node.anonrights;
	error = 0;
    }

    return error;
}

static void
clear_all_children (struct inode *inode, int parent)
{
    struct list_head *alias;

    if (inode == NULL)
	return;

 again:
    spin_lock(&dcache_lock);
    alias = inode->i_dentry.next;
    while (alias != &inode->i_dentry) {
	struct dentry *dentry;
	struct list_head *subdirs;
	struct nnpfs_dentry_data *xd;
	
	dentry = list_entry(alias, struct dentry, d_alias);
	if (dentry == NULL) {
	    printk(KERN_EMERG "NNPFS Panic: dentry in alias list is null\n");
	    break;
	}
	xd = DENTRY_TO_XDENTRY(dentry);
	if (parent)
	    xd->xd_flags &= ~NNPFS_XD_NAME_VALID;
	NNPFSDEB(XDEBMSG, ("clear_all_children parent: %.*s\n",
			   (int)dentry->d_name.len, dentry->d_name.name));
	
	subdirs = dentry->d_subdirs.next;
	while (subdirs != &dentry->d_subdirs) {
	    struct list_head *tmp = subdirs;
	    struct dentry *child = list_entry(tmp, struct dentry, d_child);
	    subdirs = tmp->next;
	    NNPFSDEB(XDEBMSG, ("clear_all_children child: %.*s inode: %p/%p "
			       "dcount: %d aliases:\n",
			       (int)child->d_name.len, child->d_name.name,
			       inode, child->d_inode, nnpfs_dcount(child)));
	    if (d_unhashed(child))
		continue;
		    
	    if (child->d_inode) {
		nnpfs_print_aliases(child->d_inode);
		if (DENTRY_TO_XDENTRY(child) == NULL)
		    printk(KERN_EMERG "NNPFS Panic: xdentry is null!\n");
		else
		    DENTRY_TO_XDENTRY(child)->xd_flags &= ~NNPFS_XD_NAME_VALID;
	    }
	    /* can't throw ref:ed negative dentries */

	    /* Throw immediately */
	    if (nnpfs_dcount(child) == 0) {
	        spin_unlock(&dcache_lock);
		nnpfs_d_remove(child);
		goto again;
	    }
	}
	alias = alias->next;
    }
    spin_unlock(&dcache_lock);
}

void
nnpfs_invalid_xnode(struct nnpfs_node *xnode)
{
    struct inode *inode = XNODE_TO_VNODE(xnode);
    int num_users;

    /* last close wins */
    if (nnpfs_iwritecount(inode) > 0)
	return;
    
    num_users = nnpfs_node_users(xnode);
    
    NNPFSDEB(XDEBNODE, ("nnpfs_invalid_xnode: used dentries: %d\n",
			num_users));
    
    if (num_users == 0 || S_ISDIR(inode->i_mode)) {
	struct nnpfs *nnpfsp = NNPFS_FROM_XNODE(xnode);
 	nnpfs_force_invalid_xnode(xnode);

	down(&nnpfsp->inactive_sem);
	nnpfs_queue_inactive(xnode);
	up(&nnpfsp->inactive_sem);
    } else {
	xnode->flags |= NNPFS_STALE;
    }
}

void
nnpfs_force_invalid_xnode(struct nnpfs_node *xnode)
{
    NNPFSDEB(XDEBNODE, ("nnpfs_force_invalid_xnode: %p\n", xnode));
    
    /* 
     * XXX Really need to put back dirty data first.
     * XXXRACE set DATA_FROM_XNODE(xnode) before dput() ?
     */
    NNPFS_TOKEN_CLEAR(xnode, ~0,
		      NNPFS_OPEN_MASK | NNPFS_ATTR_MASK |
		      NNPFS_DATA_MASK | NNPFS_LOCK_MASK);
    if (DATA_FROM_XNODE(xnode)) {
	NNPFS_RESET_I_MAPPING(XNODE_TO_VNODE(xnode));
#ifdef LINUX2_5
	invalidate_inode_pages(DATA_FROM_XNODE(xnode)->d_inode->i_mapping);
	dput(DATA_FROM_XNODE(xnode));
	DATA_FROM_XNODE(xnode) = NULL;
	invalidate_inode_pages(XNODE_TO_VNODE(xnode)->i_mapping);
#else
	invalidate_inode_pages(DATA_FROM_XNODE(xnode)->d_inode);
	dput(DATA_FROM_XNODE(xnode));
	DATA_FROM_XNODE(xnode) = NULL;
	invalidate_inode_pages(XNODE_TO_VNODE(xnode));
#endif
    }
    clear_all_children(XNODE_TO_VNODE(xnode), 1);
}

int
nnpfs_message_invalidnode(struct nnpfs *nnpfsp,
			  struct nnpfs_message_invalidnode *message,
			  u_int size)
{
    int error;
    struct nnpfs_node *t;
    
    NNPFSDEB(XDEBMSG, ("nnpfs_message_invalidnode\n"));
    error = nnpfs_node_find(nnpfsp, &message->handle, &t);
    if (error) {
	NNPFSDEB(XDEBMSG, ("nnpfs_message_invalidnode: didn't find node!"
		 " (%d.%d.%d.%d) (%d)\n",
		 message->handle.a,
		 message->handle.b,
		 message->handle.c,
		 message->handle.d, error));
	return error;
    }

    nnpfs_invalid_xnode(t);
    return 0;
}

int
nnpfs_message_updatefid(struct nnpfs *nnpfsp,
			struct nnpfs_message_updatefid * message,
			u_int size)
{
    struct nnpfs_node *t;
    int error;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_updatefid\n"));
    error = nnpfs_node_find (nnpfsp, &message->old_handle, &t);
    if (error) {
 	printk(KERN_EMERG
	       "NNPFS Panic: nnpfs_message_updatefid: no node! (%d)\n", error);
	return error;
    }

    t->handle = message->new_handle;
    return 0;
}

void
gc_vnode(struct inode *inode)
{
    NNPFSDEB(XDEBMSG,("nnpfs_message_gc: inode: %p count: %d",
		      inode, nnpfs_icount(inode)));
    nnpfs_iref(inode);
    d_prune_aliases(inode);
    NNPFSDEB(XDEBMSG, ("\nnnpfs_message_gc: i_count after gc: %d\n",
		       nnpfs_icount(inode)));
    iput(inode);
}

int
nnpfs_message_gc_nodes(struct nnpfs *nnpfsp,
		       struct nnpfs_message_gc_nodes *message,
		       u_int size)
{
    struct nnpfs_node *t;
    int i, error;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_gc\n"));
    
    for (i = 0; i < message->len; i++) {
	error = nnpfs_node_find(nnpfsp, &message->handle[i], &t);
	if (error)
	    continue;
	
	gc_vnode(XNODE_TO_VNODE(t));
    }
    
    return 0;
}

/*
 * Probe what version of the interface this nnpfs supports
 */

int
nnpfs_message_version(struct nnpfs *nnpfsp,
		      struct nnpfs_message_version *message,
		      u_int size)
{
    struct nnpfs_message_wakeup msg;
    int ret;

    ret = NNPFS_VERSION;

    msg.header.opcode = NNPFS_MSG_WAKEUP;
    msg.sleepers_sequence_num = message->header.sequence_num;
    msg.error = ret;

    return nnpfs_message_send(nnpfs, 
			      (struct nnpfs_message_header *) &msg, 
			      sizeof(msg));
}

/*
 * daemon ACKs deletion of node, free it for real
 */

int
nnpfs_message_delete_node(struct nnpfs *nnpfsp,
			  struct nnpfs_message_delete_node *message,
			  u_int size)
{
    struct nnpfs_node *t;
    int error;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_delete_node\n"));

    down(&nnpfsp->inactive_sem);
    error = nnpfs_node_find(nnpfsp, &message->handle, &t);
    if (error == -ENOENT) {
	printk(KERN_EMERG "nnpfs_message_delete_node: node not found\n");
    } else {
	NNPFSDEB(XDEBMSG,
		 ("nnpfs_message_delete_node: %p, flags 0x%x\n", t, t->flags));

	if (error == -EISDIR) {
	    NNPFSDEB(XDEBMSG,
		     ("nnpfs_message_delete_node: free node\n"));
	    nnpfs_free_node(t);
	    error = 0;
	} else {
	    NNPFSDEB(XDEBMSG, ("nnpfs_message_delete_node: not deleted"));
	}
    }
    up(&nnpfsp->inactive_sem);
	
    return error;
}
