/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __FILEWRITER_H__
#define __FILEWRITER_H__

#include <qstack.h>

#include <kurl.h>

#include "tempfile.h"
#include "xmlwriter.h"
#include "docwidget.h"

/*========================================================*/

class ArsonDocWidget;
class ArsonListItem;
class QListViewItem;

class ArsonBaseWriter : public ArsonTreeVisiter
{
public:
	ArsonBaseWriter (void) : m_pd(NULL) { }
	ArsonBaseWriter (ArsonDocWidget *pd, const KURL &url)
		: m_pd(pd), m_url(url) { }
	virtual ~ArsonBaseWriter (void) { }

	virtual void beginDocument (void) { }
	virtual void endDocument (void) { }
	virtual bool valid (void) const = 0;

	const KURL &url (void) const { return m_url; }
	ArsonDocWidget *document (void) { return m_pd; }
	
private:
	ArsonBaseWriter (const ArsonBaseWriter &) { }
	
	ArsonDocWidget *m_pd;
	KURL m_url;
};

/*========================================================*/

class QTextStream;

class ArsonFileWriter : public ArsonBaseWriter
{
public:
	ArsonFileWriter (ArsonDocWidget *pd, const KURL &url);
	virtual ~ArsonFileWriter (void);

	virtual bool valid (void) const { return (m_pt != NULL); }
	QTextStream &stream (void) { return *m_pt; }

	void writeLine (const QString &text);

private:
	ArsonFileWriter (const ArsonFileWriter &) { }
	
	ArsonTempFile m_file;
	QTextStream *m_pt;
};

/*========================================================*/

class ArsonXmlTag;

class ArsonXmlFileWriter : public ArsonFileWriter
{
	typedef QStack<ArsonXmlTag> TAGSTACK;

public:
	ArsonXmlFileWriter (const QString &docname, ArsonDocWidget *pd, const KURL &url);
	virtual ~ArsonXmlFileWriter (void);

protected:
	virtual void beginDocument (void);
	virtual void endDocument (void);

	void pushTag (ArsonXmlTag *tag);
	void popTag (void);
	
	ArsonXmlWriter &xml (void) { return m_xml; }

private:
	ArsonXmlWriter m_xml;
	TAGSTACK m_tags;
};

/*========================================================*/

class ArsonFileListDoc;

class ArsonXmlFileListWriter : public ArsonXmlFileWriter
{
public:
	ArsonXmlFileListWriter (ArsonFileListDoc *pd, const KURL &url);

	virtual bool beginBranch (QListViewItem *pi, ArsonListItem *pl);
	virtual bool visit (QListViewItem *pi, ArsonListItem *pl);
	virtual void endBranch (void);

	ArsonFileListDoc *document (void)
	{ return (ArsonFileListDoc *) ArsonXmlFileWriter::document(); }
};

/*========================================================*/

class ArsonTextFileWriter : public ArsonFileWriter
{
public:
	ArsonTextFileWriter (ArsonDocWidget *pd, const KURL &url);

	virtual bool beginBranch (QListViewItem *pi, ArsonListItem *pl);
	virtual bool visit (QListViewItem *pi, ArsonListItem *pl);
	virtual void endBranch (void);

	QString indent (void) const;

private:
	int m_level;
};

/*========================================================*/

class ArsonM3uFileWriter : public ArsonFileWriter
{
public:
	ArsonM3uFileWriter (ArsonFileListDoc *pd, const KURL &url);

	virtual bool visit (QListViewItem *pi, ArsonListItem *pl);
};

/*========================================================*/
#endif	/*	__FILEWRITER_H__	*/
