/*
 * (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Progress dialog class implementation
 *
 *	by Tony Sideris	(05:54AM Aug 12, 2001)
 *================================================*/
#include "arson.h"

#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qregexp.h>

#include <kfiledialog.h>
#include <klocale.h>
#include <kapp.h>

#include "progressdlg.h"
#include "configdlg.h"
#include "cdwriter.h"
#include "logwnd.h"
#include "infprogbar.h"

/*========================================================*/
/*	ArsonProgressDlg-specific implemetation of the
 *	ArsonWriterUI interface
 *========================================================*/

class progressDlgUI : public ArsonProcessUI
{
public:
	progressDlgUI (ArsonProgressDlg *pDlg)
		: m_pDlg(pDlg) {}

	virtual void setMaxProgress (long maxp)
	{
		ArsonProcessUI::setMaxProgress(maxp);
		m_pDlg->setMaxProgress(maxp);
	}

	virtual void setProgress (long prog)
	{
		ArsonProcessUI::setProgress(prog);
		m_pDlg->setProgress(prog);
	}

	virtual void output (const QString &str, bool error) {
		m_pDlg->output(str, error);
	}

	virtual void setText (const QString &text) {
		m_pDlg->setText (text);
	}

	virtual void begin (void) {
		m_pDlg->beginWrite();
	}

	virtual void end (void) {
		m_pDlg->endWrite();
	}

private:
	ArsonProgressDlg *m_pDlg;
};

/*========================================================*/

class pushButtonLogUI : public ArsonLogState
{
public:
	pushButtonLogUI (QPushButton *pb) : m_pb(pb) { }

	virtual void setState (bool enable) {
		m_pb->setOn(enable);
	}

protected:
	QPushButton *m_pb;
};

/*========================================================*/
/*	Progress dialog class implementation
 *========================================================*/

#include <qfontmetrics.h>
#include <qpainter.h>

ArsonProgressDlg::ArsonProgressDlg (QWidget *parent, const char *name)
	: ArsonProgressDlgBase(parent ? parent : kapp->mainWidget(), name, TRUE),
	m_pWriter(NULL), m_pBtnLayout(NULL), m_pUI(NULL),
	m_pBtnGroup(NULL), m_baseCaption(i18n("Progress"))
{
	ArsonLogWindow::Wnd()->addUiItem((m_pLogUI = new pushButtonLogUI(cmdlog)));

	setProgressMode(arsonProgressNormal);
	out->setSelectionMode(QListBox::NoSelection);
	cmdlog->setToggleButton(true);

	const QFontMetrics fm (QPainter(out).fontMetrics());
	out->setMinimumHeight((fm.height() + fm.leading()) * 6);
}

ArsonProgressDlg::~ArsonProgressDlg (void)
{
	ArsonLogWindow::Wnd()->delUiItem(m_pLogUI);
	delete m_pWriter;
}

/*========================================================*/

ArsonProcessUI *ArsonProgressDlg::ui (void)
{
	if (!m_pUI)
		m_pUI = new progressDlgUI(this);

	return m_pUI;
}

/*========================================================*/

void ArsonProgressDlg::setCaption (const QString &text)
{
	m_baseCaption = text;
	_setCaption();
}

void ArsonProgressDlg::_setCaption (void)
{
	const int max = progress->totalSteps();

	if (progress->mode() != arsonProgressNormal)
		ArsonProgressDlgBase::setCaption(m_baseCaption);

	else if (max != 0)
	{
		const double percent =
			double(progress->progress()) /
			double(max);
		const int n = int(percent * 100);

		ArsonProgressDlgBase::setCaption(
			i18n("%1 - %2%")
			.arg(m_baseCaption)
			.arg((n < 0) ? 0 : n));
	}
}

/*========================================================*/

void ArsonProgressDlg::enableControls (bool enable)
{
	int index;
	QWidget *ctrls[] = {
		config,
		start,
	};

	for (index = 0; index < ARRSIZE(ctrls); ++index)
		ctrls[index]->setEnabled(enable);

	for (index = 0; index < m_widgets.count(); ++index)
		m_widgets[index]->setEnabled(enable);
}

/*========================================================*/

void ArsonProgressDlg::closeEvent (QCloseEvent *close)
{
	reject();
}

/*========================================================*/

void ArsonProgressDlg::config_clicked (void)
{
	ArsonConfigDlg dlg (ACONFIG, ArsonConfigDlg::BURNER, this);

	if (dlg.exec() == QDialog::Accepted)
		reconfigure();
}

void ArsonProgressDlg::start_clicked (void)
{
	Assert(m_pWriter == NULL);

	try
	{
		if ((m_pWriter = createProcessMgr()))
		{
			ArsonProcessOpts opts;
			ArsonCtrlState state (this);

			emit saveState(state);
			processOpts(opts);

			m_pWriter->begin(opts);
		}
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/

int ArsonProgressDlg::exec (void)
{
	ArsonCtrlState state (this);
	emit restoreState(state);
	reconfigure();

	return ArsonProgressDlgBase::exec();
}

/*========================================================*/
/*	Write the contents of the output window to file.
 *========================================================*/

void ArsonProgressDlg::log_clicked (void)
{
	arsonSaveLogWindow(out);
}

/*========================================================*/

void ArsonProgressDlg::show_cmd_log (void)
{
	if (!ArsonLogWindow::Wnd()->isVisible())
		ArsonLogWindow::Wnd()->show();
	else
		ArsonLogWindow::Wnd()->hide();
}

/*========================================================*/

void ArsonProgressDlg::reject (void)
{
	if (!m_pWriter || !m_pWriter->isWriting())
		ArsonProgressDlgBase::reject();

	else if (cancelWarning())
		m_pWriter->abort();
}

/*========================================================*/

bool ArsonProgressDlg::cancelWarning (void)
{
	return KMessageBox::questionYesNo(kapp->mainWidget(),
		i18n("Interrupting a read/write operation on some CDR(W) devices can lead to system instability. Are you sure you wish to cancel?"),
		i18n("Read/Write Abort?")) == KMessageBox::Yes;
}

/*========================================================*/

void ArsonProgressDlg::setText (const QString &text)
{
	message->setText(text);
}

void ArsonProgressDlg::output (const QString &str, bool error)
{
	QString text (str);
	const int index = out->count();
	QScrollBar *ps = out->verticalScrollBar();

	text.replace(QRegExp("[\t]"), "  ");
	out->insertItem(text);

	//	Auto scroll the log window
	if (!ps || (ps->value() == ps->maxValue()))
	{
		out->setSelected(index, true);
		out->setCurrentItem(index);
		out->setBottomItem(index);
	}
}

/*========================================================*/

void ArsonProgressDlg::setMaxProgress (long maxp)
{
	switch (maxp)
	{
	case -1:
		setProgressMode(arsonProgressInfinite);
		break;

	case 0:
		setProgressMode(arsonProgressBlank);
		break;

	default:
		setProgressMode(arsonProgressNormal);
		progress->setTotalSteps(maxp);
	}
}

/*========================================================*/

void ArsonProgressDlg::setProgress (long prog)
{
	if (prog > progress->totalSteps())
		prog = progress->totalSteps();

	progress->setProgress(prog);
	_setCaption();
}

/*========================================================*/

void ArsonProgressDlg::setProgressMode (int mode)
{
	if (progress->mode() != mode)
	{
		progress->setMode(mode);
		_setCaption();
	}
}

/*========================================================*/

void ArsonProgressDlg::beginWrite (void)
{
	enableControls(false);
}

void ArsonProgressDlg::endWrite (void)
{
	setProgressMode(arsonProgressBlank);
	cancel->setText(i18n("Close"));
}

/*========================================================*/
/*	Adds and manages the widgets in the center
 *========================================================*/

QCheckBox *ArsonProgressDlg::addCheckbox (const QString &text, const char *name, bool state)
{
	QCheckBox *ptr = new ArsonProgressCheckbox(text, ctrlParent(), name);

	ptr->setChecked(state);
	widgetMgr() << ptr;
	return ptr;
}

bool ArsonProgressDlg::isCheckboxChecked (const char *name)
{
	if (QCheckBox *ptr = (QCheckBox *) child(name))
		return ptr->isChecked();

	Assert(0 && "Checkbox not found");
	return false;
}

int ArsonProgressDlg::addWidget (QWidget *widget, bool align)
{
	m_widgets.append(widget);

	if (align)
		ctrlLayout()->addWidget(widget);

	return m_widgets.count();
}

/*========================================================*/

QBoxLayout *ArsonProgressDlg::ctrlLayout (void)
{
	if (!m_pBtnLayout)
	{
		m_pBtnGroup = new QGroupBox(i18n("Options"), this);
		m_pBtnGroup->setColumnLayout(0, Qt::Vertical);
		m_pBtnGroup->layout()->setSpacing(0);
		m_pBtnGroup->layout()->setMargin(0);

		m_pBtnLayout = new QVBoxLayout(m_pBtnGroup->layout());
		m_pBtnLayout->setAlignment(Qt::AlignTop);
		m_pBtnLayout->setSpacing(6);
		m_pBtnLayout->setMargin(11);
		
		ctrl_b0x->add(m_pBtnGroup);
	}

	return m_pBtnLayout;
}

QWidget *ArsonProgressDlg::ctrlParent (void)
{
	ctrlLayout();
	return m_pBtnGroup;
}

/*========================================================*/
/*	Geometry Helperz
 *========================================================*/

ArsonProgressDlg::WidgetMgr::WidgetMgr (ArsonProgressDlg *pd)
	: m_pd(pd)
{
	//	Nothing...
}

ArsonProgressDlg::WidgetMgr &ArsonProgressDlg::WidgetMgr::operator<< (QWidget *pw)
{
	m_pd->addWidget(pw, true);
	return *this;
}

/*========================================================*/

ArsonProgressDlg::LayoutRow::LayoutRow (ArsonProgressDlg *pd, bool indent)
	: m_pd(pd), m_row(NULL)
{
	m_row = m_pd->makeLayoutRow(indent);
}

ArsonProgressDlg::LayoutRow &ArsonProgressDlg::LayoutRow::operator<< (QWidget *pw)
{
	m_row->addWidget(pw);
	m_pd->addWidget(pw, false);

	return *this;
}

ArsonProgressDlg::LayoutRow &ArsonProgressDlg::LayoutRow::operator<< (QLayoutItem *pl)
{
	m_row->addItem(pl);
	return *this;
}

QLayoutItem *ArsonProgressDlg::spacer (int width)
{
	return new QSpacerItem(width, 12,
		QSizePolicy::Fixed,
		QSizePolicy::Minimum);
}

/*========================================================*/

QHBoxLayout *ArsonProgressDlg::makeLayoutRow (bool indent)
{
	QHBoxLayout *ph = new QHBoxLayout(ctrlParent());
	ph->setSpacing(1);

	if (indent)
		ph->addItem(
			new QSpacerItem(14, 12,
				QSizePolicy::Fixed,
				QSizePolicy::Minimum));

	ctrlLayout()->addLayout(ph);
	return ph;
}

/*========================================================*/
/*	Simple progress dialog with no additional controls
 *========================================================*/

ArsonSimpleProgressDlg::ArsonSimpleProgressDlg (const QString &text,
	const QString &caption, QWidget *parent, const char *name)
	: ArsonProgressDlg(parent, name)
{
	setCaption(caption);
	message->setText(text);
}

/*========================================================*/
/*	Used to save, and restore the state of controls
 *========================================================*/

ArsonCtrlState::ArsonCtrlState (ArsonProgressDlg *pd)
	: m_pk(kapp->config())
{
	QString group ("progress:");
	const char *name = pd->name();

	Assert(name && *name);
	
	group.append(pd->name());
	m_pk->setGroup(group);
}

/*========================================================*/

ArsonCtrlState &ArsonCtrlState::operator>> (QCheckBox *pc)
{
	if (pc->name(0) && m_pk->readEntry(pc->name()) != QString::null)
		pc->setChecked(m_pk->readBoolEntry(pc->name()));

	return *this;
}

ArsonCtrlState &ArsonCtrlState::operator>> (QComboBox *pc)
{
	if (const char *name = pc->name(0))
	{
		if (pc->editable())
		{
			const QString val = m_pk->readEntry(name);

			if (!val.isEmpty())
				pc->setEditText(val);
		}
		else
		{
			int index;

			if ((index = m_pk->readNumEntry(name, -1)) != -1)
				pc->setCurrentItem(index);
		}
	}

	return *this;
}

ArsonCtrlState &ArsonCtrlState::operator>> (QLineEdit *pe)
{
	QString str;
	
	if (pe->name(0) && (str = m_pk->readEntry(pe->name())) != QString::null)
		pe->setText(str);

	return *this;
}

/*========================================================*/

ArsonCtrlState &ArsonCtrlState::operator<< (QCheckBox *pc)
{
	if (pc->name(0))
		m_pk->writeEntry(pc->name(), (bool) pc->isChecked());

	return *this;
}

ArsonCtrlState &ArsonCtrlState::operator<< (QComboBox *pc)
{
	if (const char *name = pc->name(0))
	{
		if (pc->editable())
			m_pk->writeEntry(name, pc->currentText());
		else
			m_pk->writeEntry(name, pc->currentItem());
	}

	return *this;
}

ArsonCtrlState &ArsonCtrlState::operator<< (QLineEdit *pe)
{
	//	We don't wanna save password boxes
	if (pe->name(0) && pe->echoMode() == QLineEdit::Normal)
		m_pk->writeEntry(pe->name(), pe->text());

	return *this;
}

/*========================================================*/
/*	Fancy widget types that support save/restore
 *========================================================*/

ArsonProgressCheckbox::ArsonProgressCheckbox (const QString &label, QWidget *parent, const char *name)
	: QCheckBox(label, parent, name)
{
	ARSON_CONNECT_PROGRESSCTRL(parent, this);
}

ArsonProgressCombobox::ArsonProgressCombobox (QWidget *parent, const char *name)
	: QComboBox(parent, name)
{
	ARSON_CONNECT_PROGRESSCTRL(parent, this);
}

ArsonProgressLineedit::ArsonProgressLineedit (QWidget *parent, const char *name)
	: QLineEdit(parent, name)
{
	ARSON_CONNECT_PROGRESSCTRL(parent, this);
}

IMPLEMENT_PROGRESSCTRL_SLOTS(ArsonProgressCheckbox)
IMPLEMENT_PROGRESSCTRL_SLOTS(ArsonProgressCombobox)
IMPLEMENT_PROGRESSCTRL_SLOTS(ArsonProgressLineedit)
	
/*========================================================*/
