#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>

#include "protocol-esound.h"
#include "protocol-esound-unix.h"
#include "unix.h"

static int _fd;
static gboolean _enabled = FALSE;

#define ESD_UNIX_SOCKET_DIR     "/tmp/.esd"
#define ESD_UNIX_SOCKET_NAME    (ESD_UNIX_SOCKET_DIR "/socket")

/*******************************************************************/
/* safely create directory for socket
   Code inspired by trans_mkdir from XFree86 source code
   For more credits see xc/lib/xtrans/Xtransutil.c. */

int safe_mksocketdir()
{
  struct stat dir_stats;

#define ESD_UNIX_SOCKET_DIR_MODE (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH)
  
  if (mkdir(ESD_UNIX_SOCKET_DIR, ESD_UNIX_SOCKET_DIR_MODE) == 0) {
    chmod(ESD_UNIX_SOCKET_DIR, ESD_UNIX_SOCKET_DIR_MODE);
    return 0;
  }
  
  /* If mkdir failed with EEXIST, test if it is a directory with
     the right modes, else fail */
  if (errno == EEXIST) {
    if (lstat(ESD_UNIX_SOCKET_DIR, &dir_stats) != 0) {
      return -1;
    }
    if (S_ISDIR(dir_stats.st_mode)) {
      int updateOwner = 0;
      int updateMode = 0;
      int updatedOwner = 0;
      int updatedMode = 0;
      /* Check if the directory's ownership is OK. */
      if ((dir_stats.st_uid != 0) && (dir_stats.st_uid != getuid()))
	updateOwner = 1;
      /*
       * Check if the directory's mode is OK.  An exact match isn't
       * required, just a mode that isn't more permissive than the
       * one requested.
       */
      if ( ~ESD_UNIX_SOCKET_DIR_MODE & dir_stats.st_mode)
	updateMode = 1;
      
      /*
       * If fchown(2) and fchmod(2) are available, try to correct the
       * directory's owner and mode.  Otherwise it isn't safe to attempt
       * to do this.
       */
      if (updateMode || updateOwner) {
	int fd = -1;
	struct stat fdir_stats;
	if ((fd = open(ESD_UNIX_SOCKET_DIR, O_RDONLY)) != -1) {
	  if (fstat(fd, &fdir_stats) == -1) {
	    close(fd);
	    return -1;
	  }
	  /*
	   * Verify that we've opened the same directory as was
	   * checked above.
	   */
	  if (!S_ISDIR(fdir_stats.st_mode) ||
	      dir_stats.st_dev != fdir_stats.st_dev ||
	      dir_stats.st_ino != fdir_stats.st_ino) {
	    close(fd);
	    return -1;
	  }
	  if (updateOwner && fchown(fd, getuid(), getgid()) == 0)
	    updatedOwner = 1;
	  if (updateMode && fchmod(fd, ESD_UNIX_SOCKET_DIR_MODE) == 0)
	    updatedMode = 1;
	  close(fd);
	}
      }
      
      if (updateOwner && !updatedOwner) {
	return -1;
      }
      
      if (updateMode && !updatedMode) {
	return -1;
      }
      
      return 0;
    }
    
  }
  /* In all other cases, fail */
  return -1;
}


gboolean protocol_esound_unix_init()
{
  g_assert(!_enabled);
  
  if (safe_mksocketdir() != 0)
    return (_enabled = FALSE);

  return (_enabled = ((_fd = unix_server_socket(ESD_UNIX_SOCKET_NAME)) >= 0));
}

void protocol_esound_unix_done()
{
  if (_enabled)
    {
      close(_fd);
      unlink(ESD_UNIX_SOCKET_NAME);
    }

  _enabled = FALSE;
}

int protocol_esound_unix_get_fd()
{
  return _fd;
}

gboolean protocol_esound_unix_enabled()
{
  return _enabled;
}

void protocol_esound_unix_accept()
{  
  int sock;
  g_assert(_enabled);

  if ((sock = unix_server_accept(_fd)) >= 0)
     protocol_esound_new_socket(sock);
}
