// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __InstantiationCandidate__
#define __InstantiationCandidate__

#include "Puma/Array.h"
#include "Puma/ErrorSink.h"
#include "Puma/DeducedArgument.h"

namespace Puma {


class CTree;
class CUnit;
class Token;
class CRecord;
class CTypeInfo;
class CConstant;
class CStructure;
class CT_ExprList;
class CObjectInfo;
class CT_SimpleName;
class CTemplateInfo;
class CT_TemplateName;
class CTranslationUnit;
class CTemplateInstance;
class CT_TemplateArgList;
class CTemplateParamInfo;

class InstantiationCandidate {
  typedef Array<CTree*> ArgumentList;
  typedef Array<DeducedArgument*> DeducedArgumentList;

  ArgumentList         arguments;
  DeducedArgumentList  darguments;
  CObjectInfo         *obj_info;
  CTemplateInfo       *tpl_info;
  CTree               *poi;
  ErrorSink           *err;
  CTranslationUnit    *trans_unit;
  CStructure          *inst_scope;
  char                 anon_name_buf[100];

public:
  InstantiationCandidate ();
  ~InstantiationCandidate ();

  void reset ();
  void initialize (ErrorSink *e);
  void initialize (CTree *p, CObjectInfo *o);
  void initialize (CTree *p, CObjectInfo *o, CTemplateInfo *t, ErrorSink *e = 0);
  void addArgument (CTree *);
  void addDeducedArg (DeducedArgument *);
  void forgetDeducedArgs ();

  bool deduceArguments (bool real_inst);
  bool deduceArgumentsFromFctCall (unsigned skip = 0, bool exact_match = false);
  bool match ();
  int compare (InstantiationCandidate &);

  bool canBeInstantiated ();

  CObjectInfo *ObjectInfo () const;
  CTemplateInfo *TemplateInfo () const;
  CTree *PointOfInstantiation () const;
  CTranslationUnit *TranslationUnit () const;
  CStructure *InstanceScope () const;

  void forgetTranslationUnit ();

  CStructure *createInstanceScope();

  unsigned Arguments () const;
  unsigned DeducedArgs () const;
  CTree *Argument (unsigned) const;
  DeducedArgument *DeducedArg (unsigned) const;

  int getPosition (CTemplateParamInfo *) const;
  Token* getPointOfInstantiationToken(CTree* point_of_inst = 0) const;

  void printArgumentList (std::ostream &out, bool print_default_args = false) const;
  bool insertCodeForInstance (CUnit &);
  bool maxInstDepthReached ();

private:
  int deduceArguments (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match = false);
  int deduceFromQualifiedType (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match);
  int deduceFromArray (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match);
  int deduceFromFunction (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match);
  int deduceFromMemberPointer (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match);
  int deduceFromRecord (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match);
  int deduceTemplateParam (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match);
  int deduceArgumentsFromTemplateArgList (CTemplateInstance *fi, CTemplateInstance *ai, DeducedArgumentList &dargs);
  bool matchDirectArguments (unsigned pos, unsigned numargs, CT_TemplateArgList *args);
  bool parseDirectArguments (unsigned pos, unsigned numargs, CT_TemplateArgList *args);
  bool insertCodeForArguments(unsigned pos, unsigned numargs, CT_TemplateArgList *args, CUnit& unit);
  bool deduceFromParsedArguments(unsigned pos, unsigned numargs, CT_TemplateArgList *args);
  bool joinDeducedArguments (DeducedArgumentList &dargs, DeducedArgumentList &curr_dargs, bool &matching_args, bool &ambiguous_type);
  void getTemplateInstances(CRecord* c, Array<CTemplateInstance*>& instances);
  bool matchArgument (CTree *pexpr, CTree *aexpr);
  bool matchArgument (CTypeInfo *ptype, CTypeInfo *atype, CTree *aexpr);
  bool sameValue (int pos, CConstant *value);
  bool sameType (int pos, CTypeInfo *type);
  bool setDeducedArgument (DeducedArgumentList &args, int pos, DeducedArgument *arg);
  bool matchingTemplateParameters (CTemplateParamInfo *param, CObjectInfo *ainfo);
  void extendTemplateName(CUnit &unit, CT_TemplateName *name);
  void extendClassName(CUnit &unit, CT_SimpleName *name);
  CTemplateInfo *getTemplateInfo (CObjectInfo *info);
  CT_SimpleName *findName (CTree *node);
  CT_SimpleName *findPrivateName (CTree *node);
  void listParameters(CUnit &unit, CTemplateInfo *ti);
  void listParameterNames(CUnit &unit, CTemplateInfo *ti);
  void copyTree(CUnit &unit, CTree *tree, Token* end_token = 0, CTree *sn = 0, const char *name = 0, bool make_const = false, Array<CTree*> *desturbing = 0);
  bool insertNonTypeTemplateArgument(CTemplateParamInfo* param, CTree* tree, CConstant* value, const char* name, CUnit& unit);
  bool insertTypeTemplateArgument(CTemplateParamInfo* param, CTree* tree, CTypeInfo* type, const char* name, CUnit& unit);
  bool insertTemplateTemplateArgument(CTemplateParamInfo* param, CTree* tree, CTypeInfo* type, const char* name, CUnit& unit);
  const char* getParameterName(CTemplateParamInfo* param);
  CTemplateParamInfo* getMatchingParameter(CTemplateParamInfo *param);
  void dumpInstanceCode(CUnit& unit);
  void calculateDesturbing(Array<CTree*>& desturbing);
};

inline CObjectInfo *InstantiationCandidate::ObjectInfo () const
 { return obj_info; }
inline CTemplateInfo *InstantiationCandidate::TemplateInfo () const
 { return tpl_info; }
inline CTree *InstantiationCandidate::PointOfInstantiation () const
 { return poi; }
inline CTranslationUnit *InstantiationCandidate::TranslationUnit () const
 { return trans_unit; }
inline CStructure *InstantiationCandidate::InstanceScope () const
 { return inst_scope; }
inline void InstantiationCandidate::forgetTranslationUnit ()
 { trans_unit = 0; }

inline unsigned InstantiationCandidate::Arguments () const 
 { return arguments.length (); }
inline unsigned InstantiationCandidate::DeducedArgs () const
 { return darguments.length (); }
inline CTree *InstantiationCandidate::Argument (unsigned i) const
 { return arguments.lookup (i); }
inline DeducedArgument *InstantiationCandidate::DeducedArg (unsigned i) const
 { return darguments.lookup (i); }

inline void InstantiationCandidate::addArgument (CTree *a)
 { arguments.append (a); }
inline void InstantiationCandidate::addDeducedArg (DeducedArgument *darg)
 { darguments.append (darg); }
inline void InstantiationCandidate::forgetDeducedArgs () 
 { darguments.reset (); }


} // namespace Puma

#endif /* __InstantiationCandidate__ */
