/*
 * Chan_Misdn -- Channel Driver for Asterisk
 *
 * Interface to mISDN
 *
 * Copyright (C) 2004, Christian Richter
 *
 * Christian Richter <crich@beronet.com>
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License
 */



#include <../i4lnet/net_l2.h>
#include <tone.h>

#include "te_lib.h"

extern int misdn_debug;


extern void chan_misdn_log(char *tmpl, ...);

msg_t *create_l3msg(int prim, int mt, int dinfo , int size, int nt);
bchannel_te_t *find_bc_by_cr(cr_list_t *list, unsigned long l3id);

#include "isdn_msg.c"

#define ISDN_PID_L3_B_USER 0x430000ff
#define ISDN_PID_L4_B_USER 0x440000ff


#define MISDN_MYJITTER 1
#define MISDN_IBUF_SIZE 1024

#define MISDN_USE_L3 1


// Fine Tuning of Inband  Signalling time
#define TONE_ALERT_CNT 41 // 1 Sec 
#define TONE_ALERT_SILENCE_CNT 200 // 4 Sec

#define TONE_BUSY_CNT 20 // ?
#define TONE_BUSY_SILENCE_CNT 48 // ?

static int  entity;
static int  entity_nt;


static manager_te_t *glob_mgr =NULL;

unsigned char tone_425_flip[TONE_425_SIZE];
unsigned char tone_silence_flip[TONE_SILENCE_SIZE];


static void *manager_te_event_thread(void *arg);
static int stack2manager_nt(void *dat, void *arg);


void stack_holder_add(stack_te_t *stack, bchannel_te_t *holder);
void stack_holder_remove(stack_te_t *stack, bchannel_te_t *holder);
bchannel_te_t *stack_holder_find(stack_te_t *stack, unsigned long l3id);


msg_t *create_l3msg(int prim, int mt, int dinfo, int size, int ntmode)
{
  int i = 0;
  msg_t *dmsg;
  Q931_info_t *qi;
  iframe_t *frm;
  
  if (!ntmode)
    size = sizeof(Q931_info_t)+2;
  
  while(i < 10) {
    if (ntmode) {
      dmsg = prep_l3data_msg(prim, dinfo, size, 256, NULL);
      if (dmsg) {
	return(dmsg);
      }
    } else {
      dmsg = alloc_msg(size+256+IFRAME_HEAD_SIZE+DEFAULT_HEADROOM);
      if (dmsg)
	{
	  memset(msg_put(dmsg,size+IFRAME_HEAD_SIZE), 0, size+IFRAME_HEAD_SIZE);
	  frm = (iframe_t *)dmsg->data;
	  frm->prim = prim;
	  frm->dinfo = dinfo;
	  qi = (Q931_info_t *)(dmsg->data + IFRAME_HEAD_SIZE);
	  qi->type = mt;
	  return(dmsg);
	}
    }
    
    if (!i) printf("cannot allocate memory, trying again...\n");
    i++;
    usleep(300000);
  }
  printf("cannot allocate memory, system overloaded.\n");
  exit(-1);
}


int send_msg (int midev, bchannel_te_t *bc_te, msg_t *dmsg)
{
  iframe_t *frm;
  int r;
  frm = (iframe_t *)dmsg->data;

  frm->addr = bc_te->stack->upper_id & IF_ADDRMASK | IF_DOWN;
  if (bc_te->cr) {
    frm->dinfo = bc_te->cr->l3_id;
  }  
  
  frm->len = (dmsg->len) - IFRAME_HEAD_SIZE;
  
  //printf("Writing Frame\n");
  mISDN_write(midev, dmsg->data, dmsg->len, TIMEOUT_1SEC);
  
  free_msg(dmsg);
}


static int mypid=0;

cr_list_t * add_cr(manager_te_t *mgr, bchannel_te_t *bc,  iframe_t *frm)
{
  cr_list_t *newcr=malloc(sizeof(cr_list_t));
  cr_list_t* cr;
  int port;
  
  bc->state = STATE_CALL_INIT;
  
  if (!newcr) {
    printf("Malloc in add_cr failed\n");
    return NULL;
  }
  memset(newcr,0,sizeof(cr_list_t));
  
  newcr->bc = bc;
  bc->cr = newcr;
  
  newcr->next=NULL;
  newcr->l3_id=frm->dinfo;
  
  //ALERT: here might be a mysterius bug!!
  if (mypid>5000) mypid=0;
  newcr->pid=mypid++;
  
  //printf("ADDCR: L3id: %x Pid: %d\n",newcr->l3_id, newcr->pid);

  pthread_mutex_lock(&bc->stack->cr_lock);

  if (bc->stack->cr_list == NULL) {
    bc->stack->cr_list = newcr;
    pthread_mutex_unlock(&bc->stack->cr_lock);
    return newcr;
  }
  
  for (cr=bc->stack->cr_list; cr; cr=cr->next)
    if (cr->next == NULL) break;
  cr->next=newcr;
  pthread_mutex_unlock(&bc->stack->cr_lock);
  
  //printf("## added addr %x with pid to cr list%d\n",frm->addr,newcr->pid);
  
  //port=IF_CONTRMASK & frm->addr;
  return newcr;
}

void empty_bc(bchannel_te_t *bc)
{
  //bc->addr=0;
  bc->channel = 0;
  bc->in_use = 0;
  bc->cr=NULL;
  bc->digital=0;
  bc->tone=TONE_NONE;
}

void delete_cr(manager_te_t *mgr, bchannel_te_t *bc, cr_list_t *cr)
{
  cr_list_t *hcr, *cr2;
  
  if (!bc || !cr) return ;
  
  bc->state = STATE_NULL;
  
  if (bc->channel >0 ) {
    empty_chan_in_stack(bc->stack, bc->channel);
  }
  
  if (!bc->stack->cr_list) return;
  
  if (cr->bc) {
    //Free the BC Obj
    empty_bc(cr->bc);
  }
  

  // free procid
  //bc->stack->procids[bc->channel]=0;
  
  
  pthread_mutex_lock(&bc->stack->cr_lock);
  if (bc->stack->cr_list == cr) {
    bc->stack->cr_list=bc->stack->cr_list->next;
    free(cr); 
    pthread_mutex_unlock(&bc->stack->cr_lock);
    return ;
  }
  
  // check wether our cr exists 
  for (hcr=bc->stack->cr_list; hcr ; hcr=hcr->next)
    if (hcr== cr ) break;
  
  if (!hcr)  {
    printf("Error: not found l3id %x in cr list @ handle_cr\n", cr->l3_id);
    return ;
  }

  //remove it from our list
  for (cr2=bc->stack->cr_list; cr2 ; cr2=cr2->next)
    if (cr2->next == cr) break;
  
  // dequeue
  cr2->next=cr->next;

  free(cr);
  pthread_mutex_unlock(&bc->stack->cr_lock);
}


int find_free_chan_in_stack(stack_te_t *stack)
{
  int i;
  
  for (i=0; i <MAX_BCHANS; i++) {
    if (!stack->channels[i]) {
      stack->channels[i]=1;
      return i +1;
    }
  }
  
  return 0;
}

int empty_chan_in_stack(stack_te_t *stack, int channel)
{
  stack->channels[channel-1] = 0;
  
  return 0;
}



static int newteid=0;

static int create_process (manager_te_t *mgr, int midev, bchannel_te_t *bc_te) {
  iframe_t ncr;
  int ret;
  int l3_id;
  iframe_t frm;
  int i;
  stack_te_t *stack=bc_te->stack;
  int free_chan;
  
  if (stack->mode == NT_MODE) {
    free_chan = find_free_chan_in_stack(stack);
    
    if (!free_chan) return -1;
    
    bc_te->channel=free_chan;

    if (stack->pri )
      setup_bc(bc_te, stack->b_stids[bc_te->channel-2] );
    else
      setup_bc(bc_te, stack->b_stids[bc_te->channel-1] );
    
    l3_id = 0xfff0 | i;
    //l3_id=i;
    
    ncr.prim = CC_NEW_CR | REQUEST; 
    ncr.addr = stack->upper_id & IF_ADDRMASK | IF_DOWN;
    ncr.dinfo = l3_id;
    ncr.len = 0;
    
    {cr_list_t *cr = add_cr(mgr,bc_te, &ncr); }
  } else { 
    /* if we are in te-mode, we need to create a process first */
    if (newteid++ > 0xffff)
      newteid = 0x0001;
    
    l3_id = (entity<<16) | newteid;
    /* preparing message */
    ncr.prim = CC_NEW_CR | REQUEST; 
    ncr.addr = stack->upper_id & IF_ADDRMASK | IF_DOWN;
    ncr.dinfo =l3_id;
    ncr.len = 0;
    /* send message */
    if (bc_te->stack->pri)
      bc_te->channel=0 ; //ANY_CHANNEL on E1
    else
      bc_te->channel=3 ; //ANY_CHANNEL on S0
    if (misdn_debug > 1) chan_misdn_log("--> new_l3id %x\n",l3_id);
    
    { cr_list_t *cr = add_cr(mgr,bc_te, &ncr); }
    
    mISDN_write(midev, &ncr, IFRAME_HEAD_SIZE+ncr.len, TIMEOUT_1SEC);
  }
  
  return l3_id;
}



int setup_bc(bchannel_te_t *bc, int b_stid)
{
  unsigned char buff[1025];
  iframe_t *frm = (iframe_t *)buff;
  
  mISDN_pid_t pid;
  int ret;
  int midev=bc->stack->midev;
  
  
  if (bc->upset) {
    if (misdn_debug > -1) chan_misdn_log("$$$ bc already upsetted stid :%x\n", b_stid);
    return -1;
  }
  
  if (misdn_debug > -1) chan_misdn_log("$$$ Setting up bc with stid :%x\n", b_stid);
  
  bc->b_stid = b_stid;

  {
    layer_info_t li;
    memset(&li, 0, sizeof(li));
    //bchannel_te_t *bc=bc_te;
    li.object_id = -1;
    li.extentions = 0;
    //li.st = stack->b_stids[bidx]; //bc->b_stid; // given idx
    li.st = bc->b_stid; // given idx
    
#if MISDN_USE_L3
    strcpy(li.name, "B L3");
    li.pid.layermask = ISDN_LAYER((3));
    //li.pid.protocol[3] = ISDN_PID_L3_B_USER;
    li.pid.protocol[3] = ISDN_PID_L3_B_TRANS;
#else
      strcpy(li.name, "B L4");
      li.pid.layermask = ISDN_LAYER((4));
      //li.pid.protocol[4] = ISDN_PID_L4_B_USER;
      li.pid.protocol[4] = ISDN_PID_L4_B_TRANS;
#endif
      
      ret = mISDN_new_layer(midev, &li);
      if (ret <= 0) {
	printf("New Layer Err: %d %s\n",ret,strerror(errno));
	return(-EINVAL);
      }
      
      bc->layer_id = ret;
  }
  
  memset(&pid, 0, sizeof(pid));
  
  bc->addr =  bc->layer_id & IF_ADDRMASK | IF_DOWN;
  chan_misdn_log("--> Got Adr %x\n", bc->addr);
  chan_misdn_log("--> Channel is %d\n", bc->channel);
  
  
  
  if (!bc->digital) {
    pid.protocol[1] = ISDN_PID_L1_B_64TRANS;
    pid.protocol[2] = ISDN_PID_L2_B_TRANS;
  } else {
    pid.protocol[1] = ISDN_PID_L1_B_64HDLC;
    pid.protocol[2] = ISDN_PID_L2_B_TRANS;
  }
  //pid.protocol[1] = ISDN_PID_L1_B_64TRANS;
  //pid.protocol[2] = ISDN_PID_L2_B_TRANS;
  //pid.protocol[2] = ISDN_PID_L2_B_USER;
  
#if MISDN_USE_L3
  //pid.protocol[3] = ISDN_PID_L3_B_USER;
  pid.protocol[3] = ISDN_PID_L3_B_TRANS;
  pid.layermask = ISDN_LAYER((1)) | ISDN_LAYER((2)) | ISDN_LAYER((3));
#else
  pid.protocol[3] = ISDN_PID_L3_B_DSP;
  //pid.protocol[4] = ISDN_PID_L4_B_USER;
  pid.protocol[4] = ISDN_PID_L4_B_TRANS;
  pid.layermask = ISDN_LAYER((1)) | ISDN_LAYER((2)) | ISDN_LAYER((3)) | ISDN_LAYER((4));
#endif

  //pid.global=1;
  
  ret = mISDN_set_stack(midev, bc->b_stid, &pid);
  
  if (ret){
    if (misdn_debug > -1) chan_misdn_log("$$$ Set Stack Err: %d %s\n",ret,strerror(errno));
    
    mISDN_write_frame(midev, buff, bc->addr, MGR_DELLAYER | REQUEST, 0, 0, NULL, TIMEOUT_1SEC);
    /*  bc->addr = 0; */
    return(-EINVAL);
  }

  /*{
    iframe_t act;
    act.prim = PH_ACTIVATE | REQUEST; 
    act.addr = bc->b_stid & IF_ADDRMASK | IF_DOWN;
    act.dinfo = 0;
    act.len = 0;
    mISDN_write(midev, &act, IFRAME_HEAD_SIZE+act.len, TIMEOUT_1SEC);
    } */

  bc->upset=1;
  
  return 0;
}


int clean_up_bc(bchannel_te_t *bc)
{
  int ret;
  unsigned char buff[32];
  if (!bc || !bc->stack) return -1;


  if (!bc->upset) {
    if (misdn_debug > -1) chan_misdn_log("$$$ Already cleaned up bc with stid :%x\n", bc->b_stid);
    return -1;
  }
  
  if (misdn_debug > -1)  chan_misdn_log("$$$ Cleaning up bc with stid :%x\n", bc->b_stid);
  

#if 1
  //mISDN_write_frame(bc->stack->midev, buff, bc->addr, MGR_DELLAYER | REQUEST, 0, 0, NULL, TIMEOUT_1SEC);
  ret=mISDN_clear_stack(bc->stack->midev, bc->b_stid);
  if (ret < 0) {
    if (misdn_debug > -1) chan_misdn_log("$$$ Clear Stack Err: %x %s\n",ret,strerror(errno));
    return(-EINVAL);
  }
  
  mISDN_write_frame(bc->stack->midev, buff, bc->layer_id, MGR_DELLAYER | REQUEST, 0, 0, NULL, TIMEOUT_1SEC);
  

  //#else
  
#endif
    
  
  bc->b_stid = 0;
  
  bc->upset=0;
  
  return ret;
}

/** IFACE **/
int init_bc_te(stack_te_t *stack,  bchannel_te_t *bc_te, int midev, int port, int bidx,  char *msn, int firsttime)
{
  unsigned char buff[1025];
  iframe_t *frm = (iframe_t *)buff;
  stack_info_t *stinf;
  layer_info_t li;
  interface_info_t ii;
  mISDN_pid_t pid;
  int ret;
  
  if (!bc_te) return -1;
  
  if (misdn_debug > 1) chan_misdn_log("Init.BC %d on port %d\n",bidx, port);

  //if (firsttime) {
    memset(bc_te, 0,sizeof(bchannel_te_t));
    
    if (msn) 
      strcpy(bc_te->msn,msn);
    
    //bc_te->channel =  bidx;
    bc_te->channel =  0;
    bc_te->in_use=0;
    bc_te->active=0;
    bc_te->upset=0;
    bc_te->cr=NULL;
    bc_te->tone=TONE_NONE;
    bc_te->tone_cnt2 = bc_te->tone_cnt=0;
    bc_te->cause = 16; //Normal CC
    bc_te->digital = 0; //Voice Call
    
    bc_te->astbuf= init_ibuffer(MISDN_IBUF_SIZE);
    clear_ibuffer(  bc_te->astbuf);
    bc_te->astbuf->rsem=malloc(sizeof(sem_t));
    sem_init(bc_te->astbuf->rsem,1,0);
    
    bc_te->misdnbuf= init_ibuffer(MISDN_IBUF_SIZE);
    clear_ibuffer(  bc_te->misdnbuf);
    bc_te->misdnbuf->rsem=malloc(sizeof(sem_t));
    sem_init(bc_te->misdnbuf->rsem,1,0);
    //}
  
    printf(" Bidx %d\n",bidx);
    bc_te->stack=stack;

    {
      stack_info_t *stinf;
      ret = mISDN_get_stack_info(midev, bc_te->stack->port, buff, sizeof(buff));
      if (ret < 0) {
	printf("%s: Cannot get stack info for port %d (ret=%d)\n", __FUNCTION__, port, ret);
	return -1;
      }
      
      stinf = (stack_info_t *)&frm->data.p;
      
      chan_misdn_log(" --> Child %x\n",stinf->child[bidx]);
    }

    //setup_bc(bc_te, stack->b_stids[bidx]);
    
  return 0;
}


void destroy_bc_te ( bchannel_te_t *bc )
{
  char buf[1024];
  iframe_t *frm=(iframe_t*)buf;

  if (!bc) return;

  printf ("! Destroying BC addr: %x\n", bc->addr);
  
  mISDN_write_frame(bc->stack->midev, buf, bc->addr, MGR_DELLAYER | REQUEST, 0, 0, NULL, TIMEOUT_1SEC);
  bc->addr = 0;
}

stack_te_t * stack_nt_init(stack_te_t *stack, int midev, int port)
{
  int ret;
  int i, cnt;
  layer_info_t li;
  interface_info_t ii;
  mISDN_pid_t pid;
  
  
  if (misdn_debug > 1) chan_misdn_log("Init. Stack on port %d\n",port);
  stack->mode = NT_MODE;
  
  stack->lower_id = mISDN_get_layerid(midev, stack->d_stid, 1); 
  if (stack->lower_id <= 0) {
    printf("%s: Cannot get layer(%d) id of port %d\n", __FUNCTION__, 1, port);
    return(NULL);
  }
  
  msg_queue_init(&stack->downqueue);
  stack->busy=0;
  
  memset(&li, 0, sizeof(li));
  strcpy(&li.name[0],"net l2");
  li.object_id = -1;
  li.extentions = 0;
  li.pid.protocol[2] = ISDN_PID_L2_LAPD_NET;
  li.pid.layermask = ISDN_LAYER((2));
  li.st = stack->d_stid;
  
  
  stack->upper_id = mISDN_new_layer(midev, &li);
  if (stack->upper_id <= 0) {
    printf("%s: Cannot add layer %d of port %d\n", __FUNCTION__, 2, port);
    return(NULL);
  }

  if (misdn_debug > 1) chan_misdn_log("NT Stacks upper_id %x\n",stack->upper_id);
   
  memset(&ii, 0, sizeof(ii));
  ii.extentions = EXT_IF_EXCLUSIV;
  ii.owner = stack->upper_id;
  ii.peer = stack->lower_id;
  ii.stat = IF_DOWN;
  ret = mISDN_connect(midev, &ii);
  if (ret) {
    printf("%s: Cannot connect layer %d of port %d exclusively.\n", __FUNCTION__, 2, port);
    return(NULL);
  }

  /* create nst (nt-mode only) */
  {
    memset(&stack->nst, 0, sizeof(net_stack_t));
    memset(&stack->mgr, 0, sizeof(manager_t));
    
    stack->mgr.nst = &stack->nst;
    stack->nst.manager = &stack->mgr;
    
    stack->nst.l3_manager = stack2manager_nt;
    stack->nst.device = midev;
    stack->nst.cardnr = port;
    stack->nst.d_stid = stack->d_stid;
    
    stack->nst.l1_id = stack->lower_id;
    stack->nst.l2_id = stack->upper_id;
    
    msg_queue_init(&stack->nst.down_queue);
    
    Isdnl2Init(&stack->nst);
    Isdnl3Init(&stack->nst);
  }
  /* if ntmode, establish L1 to send the tei removal during start */
  {
    iframe_t act;
    act.prim = PH_ACTIVATE | REQUEST; 
    act.addr = stack->upper_id & IF_ADDRMASK | IF_DOWN;
    act.dinfo = 0;
    act.len = 0;
    mISDN_write(midev, &act, IFRAME_HEAD_SIZE+act.len, TIMEOUT_1SEC);
  }
  
  return stack;
}


stack_te_t* stack_te_init( int midev, int port )
{
  int ret;
  unsigned char buff[1025];
  iframe_t *frm = (iframe_t *)buff;
  stack_info_t *stinf;
  int i, cnt;
  layer_info_t li;
  interface_info_t ii;
  mISDN_pid_t pid;
  stack_te_t *stack = malloc(sizeof(stack_te_t));
  if (!stack ) return NULL;

  
  if (misdn_debug > 1) chan_misdn_log("Init. Stack on port %d\n",port);
  
  memset(stack,0,sizeof(stack_te_t));
  
  for (i=0; i<MAX_BCHANS + 1; i++ ) stack->channels[i]=0;
  
  stack->port=port;
  stack->midev=midev;
  stack->cr_list=NULL;
  stack->holding=NULL;
  stack->pri=0;
  
  pthread_mutex_init(&stack->cr_lock, NULL);
  
  /* query port's requirements */
  ret = mISDN_get_stack_info(midev, port, buff, sizeof(buff));
  if (ret < 0) {
    printf("%s: Cannot get stack info for port %d (ret=%d)\n", __FUNCTION__, port, ret);
    return(NULL);
  }
  
  stinf = (stack_info_t *)&frm->data.p;

  stack->d_stid = stinf->id;
  stack->b_num = stinf->childcnt;

  for (i=0; i<stinf->childcnt; i++)
    stack->b_stids[i] = stinf->child[i];
  
  switch(stinf->pid.protocol[0] & ~ISDN_PID_FEATURE_MASK) {
  case ISDN_PID_L0_TE_S0:
    if (misdn_debug > 1) chan_misdn_log("TE Stack\n");
    stack->mode = TE_MODE;
    break;
  case ISDN_PID_L0_NT_S0:
    if (misdn_debug > 1) chan_misdn_log("NT Stack\n");

    return stack_nt_init(stack,midev,port); 
    break;

  case ISDN_PID_L0_TE_U:
    break;
  case ISDN_PID_L0_NT_U:
    break;
  case ISDN_PID_L0_TE_UP2:
    break;
  case ISDN_PID_L0_NT_UP2:
    break;
  case ISDN_PID_L0_TE_E1:
    if (misdn_debug > 1) chan_misdn_log("TE S2M Stack\n");
    stack->mode = TE_MODE;
    stack->pri=1;
    break;
  case ISDN_PID_L0_NT_E1:
    if (misdn_debug > 1) chan_misdn_log("TE S2M Stack\n");
    stack->mode = NT_MODE;
    stack->pri=1;
    break;
  default:
    printf("unknown port(%d) type 0x%08x\n", port, stinf->pid.protocol[0]);
    return(NULL);
  }
  
  if (stinf->pid.protocol[2] & ISDN_PID_L2_DF_PTP) {
    chan_misdn_log("PTP Mode, untested yet\n");
    stack->ptp = 1;
  } else {
   stack->ptp = 0;
  }

  stack->lower_id = mISDN_get_layerid(midev, stack->d_stid, 3); 
  if (stack->lower_id <= 0) {
    printf("No lower Id\n");
    return(NULL);
  }
  
  memset(&li, 0, sizeof(li));
  strcpy(&li.name[0], "user L4");
  li.object_id = -1;
  li.extentions = 0;
  
  li.pid.protocol[4] = ISDN_PID_L4_CAPI20;
  
  li.pid.layermask = ISDN_LAYER((4));
  li.st = stack->d_stid;
  stack->upper_id = mISDN_new_layer(midev, &li);
  
  if (stack->upper_id <= 0) 	{
    printf("No Upper ID \n");
    return(NULL);
  } 
  
  memset(&ii, 0, sizeof(ii));
  ii.extentions = EXT_IF_EXCLUSIV | EXT_IF_CREATE;
  ii.owner = stack->upper_id;
  ii.peer = stack->lower_id;
  ii.stat = IF_DOWN;
  ret = mISDN_connect(midev, &ii);
  if (ret) {
    printf("No Connect\n");
    return NULL;
  }
  

  /* if ptp, establish link */
  if (stack->ptp) {
    iframe_t act;
    act.prim = DL_ESTABLISH | REQUEST; 
    act.addr = stack->upper_id & IF_ADDRMASK | IF_DOWN;
    act.dinfo = 0;
    act.len = 0;
    mISDN_write(midev, &act, IFRAME_HEAD_SIZE+act.len, TIMEOUT_1SEC);
  }
  /* initially, we assume that the link is up */
  stack->l2link = 1;
  
  stack->next=NULL;
  
  return stack;
  
}

void stack_te_destroy(stack_te_t* stack)
{
  char buf[1024];
  iframe_t *frm=(iframe_t*)buf;
  if (!stack) return;
  
  if (stack->lower_id)
    mISDN_write_frame(stack->midev, buf, stack->lower_id, MGR_DELLAYER | REQUEST, 0, 0, NULL, TIMEOUT_1SEC);
}

stack_te_t * find_stack_by_port(manager_te_t * mgr, int port)
{
  stack_te_t *stack;
  
  for (stack=mgr->stack_list;
       stack;
       stack=stack->next) 
    if (stack->port == port) return stack;
  
  return NULL;
}

stack_te_t * find_stack_by_mgr(manager_te_t * mgr, manager_t* mgr_nt)
{
  stack_te_t *stack;
  
  for (stack=mgr->stack_list;
       stack;
       stack=stack->next) 
    if ( &stack->mgr == mgr_nt) return stack;
  
  return NULL;
}

enum layer_e get_layer(unsigned long prim) {
  
  if (0x030000 ==(prim&LAYER_MASK)) return L3;
  if (0x020000 ==(prim&LAYER_MASK)) return L2;
  if (0x010000 ==(prim&LAYER_MASK)) return L1;
  return UNKNOWN;
}

int get_subcmd(unsigned long prim) {
  //printf("Prim %x\n",prim & SUBCOMMAND_MASK);
  
  if ( (prim&SUBCOMMAND_MASK)==REQUEST) return REQUEST;
  if ( (prim&SUBCOMMAND_MASK)==CONFIRM) return CONFIRM;
  if ( (prim&SUBCOMMAND_MASK)==INDICATION) return INDICATION;
  if ( (prim&SUBCOMMAND_MASK)==RESPONSE) return RESPONSE;
  if ( (prim&SUBCOMMAND_MASK)==SUB_ERROR) return SUB_ERROR;
  
 return -1;
}

int handle_event ( manager_te_t *mgr, bchannel_te_t *bc, enum te_event_e event, iframe_t *frm)
{
  switch (event) {
  case EVENT_CONNECT:
  case EVENT_ALERTING:
  case EVENT_PROCEEDING:
  case EVENT_SETUP_ACKNOWLEDGE:
    
    {
      stack_te_t *stack=find_stack_by_port(mgr, frm->addr&IF_CONTRMASK);
      if (!stack) return -1;
      
      //printf("Prehandling the Setup Msg\n");
      if (stack->mode == TE_MODE) {
	if (stack->pri ) 
	  setup_bc(bc, bc->stack->b_stids[bc->channel-2] );
	else
	  setup_bc(bc, bc->stack->b_stids[bc->channel-1] );
      }
      
      //printf("Ready\n");
      
    }
    break;
  case EVENT_RELEASE:
  case EVENT_RELEASE_COMPLETE:
    clean_up_bc(bc);
    break;
  default:
    break;
  }
  
  return 0;
}

int handle_cr ( manager_te_t *mgr, iframe_t *frm)
{
  stack_te_t *stack=find_stack_by_port(mgr, frm->addr&IF_CONTRMASK);

  if (!stack) return -1;
  
  switch (frm->prim) {
  case CC_NEW_CR|INDICATION:
    //printf("cr_ind\n");
    if (misdn_debug>3)chan_misdn_log("lib: NEW_CR Ind with l3id:%x\n",frm->dinfo);
    {
      bchannel_te_t* bc=find_new_bc_in_stack(mgr, frm->addr&IF_CONTRMASK);
      cr_list_t *newcr; 
      
      if (!bc) {
	if (misdn_debug > 3) chan_misdn_log("lib: No free channel!\n");
	return -1;
      }
      
      newcr = add_cr(mgr, bc,frm);
      if (!newcr) return -1;
      
    }
    return 1;
  case CC_NEW_CR|CONFIRM:
    //printf("cr_confirm\n");
    // incoming something
    return 1;
  case CC_NEW_CR|REQUEST:
    //printf("cr_new_req\n");
    return 1;
  case CC_RELEASE_CR|REQUEST:
    //printf("cr_rel_req\n");
    return 1;
  case CC_RELEASE_CR|CONFIRM:
    //printf("RELEASE_CR Conf with l3id:%x\n",frm->dinfo);
    break;
  case CC_RELEASE_CR|INDICATION:
    if (misdn_debug > 3) chan_misdn_log("lib: RELEASE_CR Ind with l3id:%x\n",frm->dinfo);
    {
      bchannel_te_t *bc=find_bc_by_cr(stack->cr_list, frm->dinfo);
      
      if (bc) {
	if (misdn_debug > 3) chan_misdn_log("lib: CLEANING UP l3id: %x\n",frm->dinfo);
	delete_cr(mgr, bc, bc->cr);
	
	glob_mgr->cbEvent(glob_mgr, EVENT_CLEANUP, bc, glob_mgr->user_data);
      }
      else {
	if (stack->mode == NT_MODE) 
	  chan_misdn_log("BC with dinfo: %x  not found.. (prim was %x and addr %x)\n",frm->dinfo, frm->prim, frm->addr);
      }
      
      return 1;
    }
    break;
  }
  
  return 0;
}

bchannel_te_t *find_bc_by_cr(cr_list_t *list, unsigned long l3id)
{
  cr_list_t *cr;
  for (cr=list;
       cr;
       cr=cr->next) {
    if (cr->l3_id == l3id) return cr->bc;
  }
  return NULL;
}


bchannel_te_t *find_bc_by_addr(manager_te_t *mgr, unsigned long addr)
{
  bchannel_te_t *bc;
  int port = addr & IF_CONTRMASK;
  stack_te_t* stack;
  int i;
  
  for (stack=mgr->stack_list;
       stack;
       stack=stack->next) {
    
    if (stack->port == port) {
      for (i=0; i< stack->b_num; i++)
	if (stack->bc_te[i].addr==addr) {
	  return &stack->bc_te[i];
	}
    }
  }
  
  return NULL;
}



bchannel_te_t * find_new_bc_in_stack(manager_te_t * mgr, int port)
{
  stack_te_t *stack;
  int i;
  
  for (stack=mgr->stack_list;
       stack;
       stack=stack->next) {
    
     if (stack->port == port) {
       for (i=0; i< stack->b_num; i++)
	 if (!stack->bc_te[i].in_use) {
	   stack->bc_te[i].in_use=1;
	   return &stack->bc_te[i];
	 }
     }
   }
  
   return NULL;
}

int
stack2manager_nt(void *dat, void *arg)
{
  manager_t *mgr = (manager_t *)dat;
  msg_t *msg = (msg_t *)arg;
  mISDN_head_t *hh=(mISDN_head_t*)msg->data;
  stack_te_t *stack=find_stack_by_mgr(glob_mgr, mgr);
  
  if (misdn_debug >4 ) chan_misdn_log("lib: prim %x dinfo %x\n",hh->prim, hh->dinfo );
  
  {
    // create callref out of SETUP or RESUME
    switch(hh->prim){
    case CC_SETUP|CONFIRM:
      {
	bchannel_te_t *bc=find_bc_by_cr(stack->cr_list, hh->dinfo);
	int l3id = *((int *)(((u_char *)msg->data)+ mISDN_HEAD_SIZE));
	
	if (misdn_debug > 3)chan_misdn_log("lib: Event_ind:SETUP CONFIRM [NT] : new L3ID  is %x\n",l3id );
	
	if (!bc) { printf("Bc Not found\n"); return 0; }
	
	bc->cr->l3_id=l3id;
	
      }
      free_msg(msg);
      return 0;
      
    case CC_SETUP|INDICATION:
      {
	iframe_t frm; //fake te frm to add callref to global callreflist
	frm.dinfo = hh->dinfo;
	frm.addr=stack->upper_id;
	frm.prim = CC_NEW_CR|INDICATION;
	
	if (handle_cr(glob_mgr, &frm)< 0) {
	  msg_t *dmsg;
	  chan_misdn_log("Patch from MEIDANIS:Sending RELEASE_COMPLETE %x (No free Chan for you..)\n", hh->dinfo);
	  dmsg = create_l3msg(CC_RELEASE_COMPLETE | REQUEST,MT_RELEASE_COMPLETE, hh->dinfo,sizeof(RELEASE_COMPLETE_t), 1);
	  stack->nst.manager_l3(&stack->nst, dmsg);
	  free_msg(msg);
	  return 0;
	}
      }
      break;
      
    case CC_CONNECT|INDICATION:
      break;
    case CC_DISCONNECT|INDICATION:
      break;
    case CC_RELEASE_COMPLETE|INDICATION:
      break;
    case CC_RESUME|INDICATION:
      break;

    case CC_RELEASE|INDICATION:
      /*
	Hm, very confusing, when I deliver RELEASE to chan_misdn.c, it sends
	RELEASE_COMPLETE (ISDN standard!), than I get
	CC_RELEASE_CR|INDICATION, which is very fine for me. The funny Thing
	is, that i get a second CC_RELEASE_CR|INDICATION ?? after trying to
	handle it net_l3.c:2239 segfaults !?!? .. so lets ignore RELEASE ..
	--> Ignoring, release means that Phone gets no release_complete which
	isn't good at all, so we make it like jolly and queue the
	release_complete message, to handle it later, after all stack2manager
	calls are done, seems to work look at send_event
	(EVENT_RELEASE_COMPLETE)
      */
#if 0
      if (misdn_debug >0 ) chan_misdn_log("I IND:RELEASE mode:%s addr:%x\n",stack->mode==NT_MODE?"NT":"TE", stack->upper_id);
      free_msg(msg);
      return 0 ;
#endif
      break;
    case CC_RELEASE|CONFIRM:
      chan_misdn_log("!!!! REL_CONF\n");
      break;
      
    case CC_RELEASE_CR|INDICATION:
      {
	
	iframe_t frm; //fake te frm to remove callref from global callreflist
	frm.dinfo = hh->dinfo;
	frm.addr=stack->upper_id;
	frm.prim = CC_RELEASE_CR|INDICATION;
	handle_cr(glob_mgr, &frm);
	free_msg(msg);
	
	return 0 ;
      }
      
      break;

    case CC_NEW_CR|INDICATION:
      // Got New CR for bchan, for now I handle this one in
      // connect_ack, Need to be changed
      {
	bchannel_te_t *bc=find_bc_by_cr(stack->cr_list, hh->dinfo);
	int l3id = *((int *)(((u_char *)msg->data)+ mISDN_HEAD_SIZE));
	if (!bc) {printf("In NEW_CR: didn't found bc ??\n"); return -1;};
	//l3id = ( (bc->cr->l3_id &0xff0000) | bc->cr->ces ) & 0xff00ff;
	if (misdn_debug > 2) chan_misdn_log("lib: Event_ind:CC_NEW_CR : very new L3ID  is %x\n",l3id );

	if (bc->cr) {
	  bc->cr->l3_id =l3id;
	} else {
	  chan_misdn_log("!! New CR ?? in NT STack, somethings foul here I thing\n");
	}

	free_msg(msg);
	return 0;
      }
    }
  }
  
  //return 0;
  {
    // Parse Events and fire_up to App.
    bchannel_te_t *bc; 
    enum te_event_e event = isdn_msg_get_event(msgs_g, msg, 1);
    
    bc=find_bc_by_cr(stack->cr_list, hh->dinfo);
    
    if (bc ) {
      isdn_msg_parse_event(msgs_g,msg,bc, 1);
      
      if(!isdn_get_info(msgs_g,event,1))
	chan_misdn_log("Unknown Event Ind: prim %x dinfo %x\n",hh->prim, hh->dinfo);
      glob_mgr->cbEvent(glob_mgr, event, bc, glob_mgr->user_data);
    }
    free_msg(msg);
  }
  
  return 0;
}


int handle_timers(manager_te_t *mgr, msg_t* msg)
{
  iframe_t *frm= (iframe_t*)msg->data;
  stack_te_t *stack; 
  
  //Timer STuff
  switch (frm->prim) {
  case MGR_INITTIMER | CONFIRM:
  case MGR_ADDTIMER | CONFIRM:
  case MGR_DELTIMER | CONFIRM:
  case MGR_REMOVETIMER | CONFIRM:
    free_msg(msg);
    return(1);
  }
  
  
  
  if (frm->prim==(MGR_TIMER | INDICATION) ) {
    for (stack = mgr->stack_list;
	 stack;
	 stack = stack->next) {
      itimer_t *it;
      
      if (stack->mode != NT_MODE) continue;
      
      it = stack->nst.tlist;
      /* find timer */
      for(it=stack->nst.tlist;
	  it;
	  it=it->next) {
	if (it->id == (int)frm->addr)
	  break;
      }
      if (it) {
	int ret;
	ret = mISDN_write_frame(stack->midev, msg->data, frm->addr,
				MGR_TIMER | RESPONSE, 0, 0, NULL, TIMEOUT_1SEC);
	test_and_clear_bit(FLG_TIMER_RUNING, (long unsigned int *)&it->Flags);
	ret = it->function(it->data);
	
	free_msg(msg);
	return 1;
      }
    }
    
    chan_misdn_log("Timer Msg without Timer ??\n");
    free_msg(msg);
    return 1;
  }
  
  return 0;
}




static int do_tone(bchannel_te_t *bc, int len)
{
  char buf[4096 + IFRAME_HEAD_SIZE];
  iframe_t *frm= (iframe_t*)buf;
  int  r;
  
  if (bc->tone == TONE_NONE) return 0;

  frm->prim = DL_DATA|REQUEST;
  frm->dinfo = 0;
  frm->addr = bc->addr | IF_DOWN;
  
  bc->tone_cnt+=len;

  if (bc->tone_cnt < TONE_425_SIZE) return 1;

  //chan_misdn_log("Sending Tone\n");
  
  switch(bc->tone) {
  case TONE_DIAL:
    {
      frm->len = TONE_425_SIZE;
      memcpy(&buf[IFRAME_HEAD_SIZE], tone_425_flip,TONE_425_SIZE);
      
      r=mISDN_write(bc->stack->midev, buf, frm->len + IFRAME_HEAD_SIZE, TIMEOUT_1SEC);
      if (r<frm->len) {
	perror("Error written less than told bytes :(\n");
      }
    }
    break;

  case TONE_ALERTING:
    bc->tone_cnt2++;
    
    if (bc->tone_cnt2 <= TONE_ALERT_CNT) {
      frm->len = TONE_425_SIZE;
      memcpy(&buf[IFRAME_HEAD_SIZE], tone_425_flip,TONE_425_SIZE);
      r=mISDN_write(bc->stack->midev, buf, frm->len + IFRAME_HEAD_SIZE, TIMEOUT_1SEC);
      if (r<frm->len) {
	perror("Error written less than told bytes :(\n");
      }
    } else if (bc->tone_cnt2 <= (TONE_ALERT_SILENCE_CNT)) {
      frm->len = TONE_SILENCE_SIZE;
      memcpy(&buf[IFRAME_HEAD_SIZE], tone_silence_flip ,TONE_SILENCE_SIZE);
      r=mISDN_write(bc->stack->midev, buf, frm->len + IFRAME_HEAD_SIZE, TIMEOUT_1SEC);
    } else {
      bc->tone_cnt2=-1;
    }
    break;
  case TONE_BUSY:
    bc->tone_cnt2++;
    
    if (bc->tone_cnt2 <= TONE_BUSY_CNT) {
      frm->len = TONE_425_SIZE;
      memcpy(&buf[IFRAME_HEAD_SIZE], tone_425_flip,TONE_425_SIZE);
      r=mISDN_write(bc->stack->midev, buf, frm->len + IFRAME_HEAD_SIZE, TIMEOUT_1SEC);
      if (r<frm->len) {
	perror("Error written less than told bytes :(\n");
      }
    } else if (bc->tone_cnt2 <= (TONE_BUSY_SILENCE_CNT)) {
      frm->len = TONE_SILENCE_SIZE;
      memcpy(&buf[IFRAME_HEAD_SIZE], tone_silence_flip ,TONE_SILENCE_SIZE);
      r=mISDN_write(bc->stack->midev, buf, frm->len + IFRAME_HEAD_SIZE, TIMEOUT_1SEC);
    } else {
      bc->tone_cnt2=-1;
    }
    break;
  case TONE_FILE:
    break;
  case TONE_NONE:
    return 0;
  }
  
  bc->tone_cnt -= TONE_425_SIZE ;
  return 1;
}


void send2misdn(bchannel_te_t *bc, int sendlen)
{
  char buf[4096 + IFRAME_HEAD_SIZE];
  iframe_t *frm= (iframe_t*)buf;
  int len, r;
  
  len = ibuf_usedcount(bc->misdnbuf);
  
  if (len < sendlen) {
    // Send silence to avoid tx/rx underrun

    return ;

    /*
      if (sendlen <4096+IFRAME_HEAD_SIZE)
      memset(buf,0xff, sendlen);
      else {chan_misdn_log("Too much data to send %d\n",sendlen); return; }
      
      frm->prim = DL_DATA|REQUEST;
      frm->dinfo = 0;
      frm->addr = bc->addr; // | IF_DOWN;
      frm->len = sendlen;
      
      r=mISDN_write(bc->stack->midev, buf, frm->len + IFRAME_HEAD_SIZE, 8000 );
      return;
    */
  }
  
  frm->prim = DL_DATA|REQUEST;
  frm->dinfo = 0;
  frm->addr = bc->addr; // | IF_DOWN;
  frm->len = sendlen;
  ibuf_memcpy_r(&buf[IFRAME_HEAD_SIZE], bc->misdnbuf,sendlen);
  
  r=mISDN_write(bc->stack->midev, buf, frm->len + IFRAME_HEAD_SIZE, 8000 );
}


int handle_bchan(manager_te_t *mgr, msg_t *msg)
{
  iframe_t *frm= (iframe_t*)msg->data;
  stack_te_t *stack;
  bchannel_te_t *bc=find_bc_by_addr(mgr, frm->addr);
  stack=find_stack_by_port(mgr, frm->addr &IF_CONTRMASK);
  
  if (!stack) return 0;
  
  //if ((frm->addr&FLG_CHILD_STACK) && ((((unsigned int)stack->bc_te[0].addr)&(~IF_CHILDMASK)) == ((frm->addr)&(~IF_CHILDMASK)))){
  
  //chan_misdn_log("Got B-Data, addr %x ?? \n", frm->addr);
  
  //if ((frm->addr&FLG_CHILD_STACK) && ((((unsigned int) 51300101)&(~IF_CHILDMASK)) == ((frm->addr)&(~IF_CHILDMASK)))){
  if (bc) { 
    //chan_misdn_log("Got B-Data 2 ?? \n");
    
    switch (frm->prim) {
    case PH_ACTIVATE | INDICATION:
    case DL_ESTABLISH | INDICATION:
      if (misdn_debug > 2) chan_misdn_log("Bchan ACT Ind\n");
      break;
    case PH_ACTIVATE | CONFIRM:
    case DL_ESTABLISH | CONFIRM:
      if (misdn_debug > 2) chan_misdn_log("Bchan ACT Confirm\n");
      break;
    case PH_DEACTIVATE | INDICATION:
    case DL_RELEASE | INDICATION:
      if (misdn_debug> 2)chan_misdn_log ("Bchan DeACT Ind\n");
      break;
    case PH_DEACTIVATE | CONFIRM:
    case DL_RELEASE | CONFIRM:
      if (misdn_debug > 2) chan_misdn_log("Bchan DeACT Conf\n");
      break;
      
    case PH_CONTROL|INDICATION:
    case PH_DATA|INDICATION:
    case DL_DATA|INDICATION:
      {
	bchannel_te_t *bc=find_bc_by_addr(mgr, frm->addr);
	if (bc) {
	  bc->bframe = (void*)&frm->data.i;
	  bc->bframe_len = frm->len;
#if MISDN_DEBUG
	  printf("DL_DATA INDICATION Len %d\n", frm->len);
#endif
	  if (frm->len > 0) {
	    
	    if (!do_tone(bc, frm->len)) {
	      mgr->cbEvent(mgr, EVENT_BCHAN_DATA, bc, mgr->user_data);
	      //CRICH !!!
#if MISDN_MYJITTER
	      send2misdn(bc, frm->len);
#endif
	    }
	  }
	  free_msg(msg);
	  return 1;
	}
      }
    case PH_DATA | CONFIRM:
    case DL_DATA|CONFIRM:
#if MISDN_DEBUG
      printf("Data confirmed\n");
#endif
      free_msg(msg);
      return 1;
      break;
    case DL_DATA|RESPONSE:
#if MISDN_DEBUG
      printf("Data response\n");
#endif
      break;
      
    case DL_DATA | REQUEST:
      break;
    }
  }
  return 0;
}

int handle_downqueue(manager_te_t *mgr, stack_te_t *stack)
{
  msg_t *msg;
  
  if (stack->mode == NT_MODE) {
    
    while( (msg=msg_dequeue(&stack->downqueue)) ) {
      if (stack->nst.manager_l3(&stack->nst, msg)) chan_misdn_log("Error @ Queuing message in NT-Stack\n");  
    }
  }
  
  return 0;
}


int handle_frm_nt(manager_te_t *mgr, msg_t *msg)
{
  iframe_t *frm= (iframe_t*)msg->data;
  stack_te_t *stack;
  int err=0;
  stack=find_stack_by_port(mgr, frm->addr &IF_CONTRMASK);
  
  if (!stack) return 0;
  
  if ((frm->addr&IF_ADDRMASK) != (unsigned int)stack->upper_id) return 0;
  
  if (stack->mode != NT_MODE) return 0;

  if ((err=stack->nst.l1_l2(&stack->nst,msg))) {
    
    chan_misdn_log("NT Stack sends us error: %d, maybe we've handled something wrong.\n", err);
    free_msg(msg);
    return 1;
  }
  
  handle_downqueue(mgr,stack);
  
  return 1;
}


int handle_frm(manager_te_t *mgr, msg_t *msg)
{
  iframe_t *frm = (iframe_t*) msg->data;
  stack_te_t *stack=find_stack_by_port(mgr, frm->addr&IF_CONTRMASK);

  if (!stack) return 0;
  
  if ((frm->addr&IF_ADDRMASK) != (unsigned int)stack->upper_id) return 0;
  
  if (stack->mode != TE_MODE) return 0;

  {
    int subcmd=get_subcmd(frm->prim);
    bchannel_te_t *bc;
    enum te_event_e event = isdn_msg_get_event(msgs_g, msg, 0);
    
    if(handle_cr(mgr, frm)) {
      free_msg(msg);
      return 1;
    }
    
    if (!stack) return -1;
    
    bc=find_bc_by_cr(stack->cr_list, frm->dinfo);
    
    if (bc ) {
      enum te_event_response_e response;
      
      isdn_msg_parse_event(msgs_g,msg,bc, 0);
      
      handle_event(mgr, bc, event, frm);
      // shoot up event to App:
      //printf("L3id %d\n",frm->dinfo);
      if (misdn_debug > 4) chan_misdn_log("lib Got Prim: Addr %x prim %x dinfo %x\n",frm->addr, frm->prim, frm->dinfo);
      response=mgr->cbEvent(mgr, event, bc, mgr->user_data);
      switch (response) {
      case RESPONSE_IGNORE_SETUP:
	if (bc->cr) //ignore the l3 Inst
	  delete_cr(mgr, bc, bc->cr);
	/* I think we should send CC_RELEASE_CR, but am not sure*/
	break;
      case RESPONSE_OK:
	break;
      }
      
    } else {
      //printf("ERROR: No bc for cr %x addr %x prim %x !\n",frm->dinfo, frm->addr, frm->prim);
    }
  }
  return 0;
}


int handle_l2(manager_te_t *mgr, msg_t *msg)
{
  iframe_t *frm = (iframe_t*) msg->data;
  
  if (get_layer(frm->prim) == L2) {
    switch(frm->prim) {
    case DL_ESTABLISH | INDICATION:
    case DL_ESTABLISH | CONFIRM:
      {
	stack_te_t *stack=find_stack_by_port(mgr, frm->addr&IF_CONTRMASK);
	if (!stack) return 0;
	stack->l2link=1;
	free_msg(msg);
	return 1;
      }
      //printf(" B DATA ACT - IND\n");
      break;
      
    case DL_RELEASE | INDICATION:
    case DL_RELEASE | CONFIRM:
      {
	stack_te_t *stack=find_stack_by_port(mgr, frm->addr&IF_CONTRMASK);
	if (!stack) return 0;
	stack->l2link=0;
	free_msg(msg);
	return 1;
      }
      //printf(" B DATA DEACT - IND\n");
      break;
    }
  }
  
  return 0;
}




int handle_mgmt(manager_te_t *mgr, msg_t *msg)
{
  iframe_t *frm = (iframe_t*) msg->data;
  
  if ( (frm->prim & 0x0f0000) ==  0x0f0000) {
    if (misdn_debug > 2) chan_misdn_log("$$$ MGMT FRAME: %x\n",frm->prim) ;
    free_msg(msg);
    return 1;
  }
    
  return 0;
}


msg_t *fetch_msg(int midev) 
{
  msg_t *msg=alloc_msg(MAX_MSG_SIZE);
  int r;
  fd_set rdfs;

  FD_ZERO(&rdfs);
  FD_SET(midev,&rdfs);
  
  mISDN_select(FD_SETSIZE, &rdfs, NULL, NULL, NULL);
  
  if (FD_ISSET(midev, &rdfs)) {
    
    r=mISDN_read(midev,msg->data,MAX_MSG_SIZE,0);
    msg->len=r;
    
    if (r==0) {
      free_msg(msg); //danger, cauz usualy freeing in main_loop
      return NULL;
    }
    
    return msg;
  }
  
  return NULL;
}


static void *manager_te_event_thread(void *arg)
{
  manager_te_t *mgr = arg;
  char buffer[1024];
  
  int r;
  int midev= mgr->midev;
  
  while (1) {
    msg_t *msg = fetch_msg(midev); 
    iframe_t *frm;
    if (!msg) continue;
    
    frm = (iframe_t*) msg->data;
    
    
    
    //printf("Read something %x \n",frm->prim);
    if (handle_timers(mgr,msg)) continue; 
    
    if (handle_mgmt(mgr,msg)) continue; 
    
    if (handle_l2(mgr,msg)) continue;
    
    if (handle_frm_nt(mgr,msg)) continue;
    
    if (handle_frm(mgr, msg)) continue;
    
    if (handle_bchan(mgr,msg)) continue;
    
    chan_misdn_log("Unhandled Message: %x\n",frm->prim);
    
    free_msg(msg);
    
  }
}


/** App Interface **/

int te_lib_init() {
  char buff[1025];
  iframe_t *frm=(iframe_t*)buff;
  int midev=mISDN_open();
  int ret;

  memset(buff,0,1025);
  
  if  (midev<=0) return midev;
  
  /* create entity for layer 3 TE-mode */
  mISDN_write_frame(midev, buff, 0, MGR_NEWENTITY | REQUEST, 0, 0, NULL, TIMEOUT_1SEC);
  ret = mISDN_read_frame(midev, frm, sizeof(iframe_t), 0, MGR_NEWENTITY | CONFIRM, TIMEOUT_1SEC);
  
  if (ret < mISDN_HEADER_LEN) {
  noentity:
    fprintf(stderr, "cannot request MGR_NEWENTITY from mISDN: %s\n",strerror(errno));
    exit(-1);
  }
  
  entity = frm->dinfo & 0xffff ;
  
  if (!entity)
    goto noentity;

  return midev;
  
}

void te_lib_destroy(int midev)
{
  char buf[1024];
  mISDN_write_frame(midev, buf, 0, MGR_DELENTITY | REQUEST, entity, 0, NULL, TIMEOUT_1SEC);
  mISDN_close(midev);
}

bchannel_te_t *manager_find_bc_by_pid(manager_te_t *mgr, int pid)
{
  cr_list_t *cr;
  stack_te_t *stack;
  
  for (stack=mgr->stack_list;
       stack;
       stack=stack->next) {
    for (cr=stack->cr_list;
	 cr;
	 cr=cr->next) {
      if (cr->pid == pid) return cr->bc;
    }
  }
  return NULL;
}


bchannel_te_t* manager_te_get_free_bc(manager_te_t *mgr, int port)
{
  return find_new_bc_in_stack(mgr, port);  
}



int manager_te_send_event(manager_te_t *mgr, bchannel_te_t *bc, enum te_event_e event )
{
  msg_t *msg; 
  int l3_id;

  if (!bc) return -1;
  
  if (misdn_debug > 0) chan_misdn_log("I SEND:%s\tpid:%d\tmode:%s\taddr:%x\n",isdn_get_info(msgs_g, event, 0), bc->cr?bc->cr->pid:-1, bc->stack->mode==NT_MODE?"NT":"TE", bc->addr);
  if (misdn_debug >1) chan_misdn_log(" --> dad: %s oad %s channel %d port %d\n",bc->dad,bc->oad,bc->channel, bc->stack->port);
  
  switch (event) {
  case EVENT_SETUP:
    if (create_process(mgr, mgr->midev, bc)<0) {
      chan_misdn_log(" No free channel at the moment @ send_event\n");
      return -ENOCHAN;
    }
    if (misdn_debug > 1) chan_misdn_log("--> pid %d\n",bc->cr?bc->cr->pid:-1);
    
    break;

  case EVENT_PROCEEDING:
  case EVENT_SETUP_ACKNOWLEDGE:
    
    if (bc->stack->mode == NT_MODE) {
      bc->channel = find_free_chan_in_stack(bc->stack);
      if (!bc->channel) {
	chan_misdn_log(" No free channel at the moment\n");
	return -ENOCHAN ;
      }

      
    } 
    
    if (bc->stack->pri )
      setup_bc(bc, bc->stack->b_stids[bc->channel-1] );
    else
      setup_bc(bc, bc->stack->b_stids[bc->channel-1] );
    
    
    
    break;
  case EVENT_HOLD_ACKNOWLEDGE:
    {
      bchannel_te_t *hold_chan=malloc(sizeof(bchannel_te_t));
      memcpy(hold_chan, bc, sizeof(bchannel_te_t));
      hold_chan->state=STATE_HOLD;
      stack_holder_add(bc->stack, hold_chan);
    }
    break;

  case EVENT_RELEASE:
    clean_up_bc(bc);
    break;
    
  case EVENT_RELEASE_COMPLETE:
    
    if (bc->stack->mode == NT_MODE) {
      msg = isdn_msg_build_event(msgs_g, bc, event, bc->stack->mode==NT_MODE?1:0);
      msg_queue_tail(&bc->stack->downqueue, msg);

      clean_up_bc(bc);
      return 0;
    }
    clean_up_bc(bc);
    
    break;
  default:
    break;
  }
  
  
  msg = isdn_msg_build_event(msgs_g, bc, event, bc->stack->mode==NT_MODE?1:0);
  
  
  if (bc->stack->mode == NT_MODE ){
    
    if (bc->stack->busy) {
      chan_misdn_log("Stack Is busy so queing msg\n");
      msg_queue_tail(&bc->stack->downqueue, msg);
    }
    else {
      bc->stack->busy=1;
      if (bc->stack->nst.manager_l3(&bc->stack->nst, msg)) chan_misdn_log("Error@ Queuing Message in NT-Stack.\n");
      bc->stack->busy=0;
    }
    
    
  } else {
    if (msg)
      send_msg(mgr->midev, bc, msg);
  }
  
  if(event == EVENT_HOLD_ACKNOWLEDGE) {
    if (bc->stack->mode == NT_MODE) {
      
      empty_chan_in_stack(bc->stack, bc->channel);
      empty_bc(bc);
    }
  }
  
  return 0;
}


manager_te_t*
manager_te_init(char *portlist, void *cbEvent, void *user_data)
{
  manager_te_t *mgr=malloc(sizeof(manager_te_t));
  char *tok, *tokb;
  char plist[1024];
  int midev; 
  if (!portlist | strlen(portlist)==0) return NULL;
  
  strcpy(plist,portlist);

  if (!mgr) return NULL;
  memset(mgr,0,sizeof(manager_te_t));

  glob_mgr = mgr;
  
  memcpy(tone_425_flip,tone_425,TONE_425_SIZE);
  manager_flip_buf_bits(tone_425_flip,TONE_425_SIZE);

  memcpy(tone_silence_flip,tone_SILENCE,TONE_SILENCE_SIZE);
  manager_flip_buf_bits(tone_silence_flip,TONE_SILENCE_SIZE);
  
  midev=te_lib_init();
  mgr->midev=midev;
  /*midev_nt=te_lib_init(1);
    mgr->midev_nt=midev_nt; */
  
  mgr->cbEvent = cbEvent;
  
  /** Stopped work here **/
  for (tok=strtok_r(plist," ,",&tokb );
       tok; 
       tok=strtok_r(NULL," ,",&tokb)) {
    int port = atoi(tok);
    stack_te_t *stack=stack_te_init(midev, port);
    static int first=1;
    
    if (!stack) {
      perror("init_stack");
      exit(1);
    }
    
    if (stack && first) {
      mgr->stack_list=stack;
      first=0;
      stack->mgr_te = mgr;
      {
	int i;
	for(i=0;i<stack->b_num; i++) {
	  int r;
	  if ((r=init_bc_te(stack, &stack->bc_te[i], stack->midev,port,i, "", 1))<0) {
	    chan_misdn_log("Got Err @ init_bc :%d\n",r);
	    exit(1);
	  }
	  //stack->bc_te[i].stack=stack;
	}
      }
      
      continue;
    }
    
    if (stack) {
      stack_te_t * help;
      for ( help=mgr->stack_list; help; help=help->next ) 
	if (help->next == NULL) break;
      
      
      stack->mgr_te = mgr;
      help->next=stack;

      {
	int i;
	for(i=0;i<stack->b_num; i++) {
	  int r;
	  if ((r=init_bc_te(stack, &stack->bc_te[i], stack->midev,port,i, "",1 ))<0) {
	    chan_misdn_log("Got Err @ init_bc :%d\n",r);
	    exit(1);
	  } 
	  //stack->bc_te[i].stack = stack;
	}
      }
    }
    
  }
  
  //mgr->cr_list=NULL;
  //printf("Starting Event Handler\n");
  pthread_create( &mgr->event_thread, NULL,manager_te_event_thread, mgr);
  
  return mgr;
}



void manager_te_destroy(manager_te_t * mgr_te)
{
  stack_te_t *help;
  int i;
  
  for ( help=mgr_te->stack_list; help; help=help->next ) {
    for(i=0;i<help->b_num; i++) 
      destroy_bc_te(&help->bc_te[i]);

    
    stack_te_destroy(help);
  }
  
}


char *manager_isdn_get_info(enum te_event_e event)
{
  return isdn_get_info(msgs_g , event, 0);
}




void manager_bchannel_activate(bchannel_te_t *bc)
{
  iframe_t act;
  char buf[1024+1];
  iframe_t *frm=(iframe_t*)buf;
  /* we must activate if we are deactivated */
  
  
  if (misdn_debug > 2)  chan_misdn_log("lib: Bchan Activated addr %x\n", bc->addr);
  
  if (bc->active) return;

  if (1) {
    /* activate bchannel */
    
    act.prim = DL_ESTABLISH | REQUEST;
    //act.prim = PH_ACTIVATE | REQUEST; 
    //act.addr = bc->addr;
    act.addr = bc->addr ;
    act.dinfo = 0;
    act.len = 0;
    
    chan_misdn_log("B_addr is addr %x \n",bc->addr);
    {
      int ret=mISDN_write(bc->stack->midev, &act, IFRAME_HEAD_SIZE+act.len, TIMEOUT_1SEC);
    }
    if (misdn_debug > 3) chan_misdn_log("lib: Bchan Activated\n");
    
#if MISDN_USE_L3
#else
    //manager_ph_control(bc, CMX_RECEIVE_OFF, 0);
    //manager_ph_control(bc, CMX_MIX_ON, 0);
    //manager_ph_control(bc, CMX_CONF_SPLIT, 0);
    /*manager_ph_control(bc, VOL_CHANGE_TX, 0);
      manager_ph_control(bc, VOL_CHANGE_RX, 0); */
    
    /*act.prim = DL_DATA | REQUEST; 
      act.addr = bc->addr;
      act.dinfo = 0;
      act.len = 0;
      
      mISDN_write(bc->stack->midev, &act, IFRAME_HEAD_SIZE+act.len, TIMEOUT_1SEC);
    */
#endif
    //printf("First Activate addr %p !\n",bc->addr);
    
    bc->active=1;
    
    return ;
    
  }
  
  return ;
}


void manager_bchannel_deactivate(bchannel_te_t * bc)
{
  iframe_t dact;

  
  if (misdn_debug > 2) chan_misdn_log("$$$ Bchan deActivated addr %x\n", bc->addr);
  
  if (!bc->active) return;

  bc->tone=TONE_NONE;
  
  dact.prim = DL_RELEASE | REQUEST;
  //dact.prim = PH_DEACTIVATE | REQUEST; 
  //dact.addr = bc->addr;
  dact.addr = bc->addr;
  dact.dinfo = 0;
  dact.len = 0;
  
  mISDN_write(bc->stack->midev, &dact, IFRAME_HEAD_SIZE+dact.len, TIMEOUT_1SEC);
  clear_ibuffer(bc->misdnbuf);
  bc->active=0;
  
  return;
}


int manager_send_frame(bchannel_te_t *bc, void *data, int len)
{

#if MISDN_MYJITTER
  int r;

  if (!bc->active) return -1; 
  
  if (len > ibuf_freecount(bc->misdnbuf)) {
    len=ibuf_freecount(bc->misdnbuf);
  }
  ibuf_memcpy_w(bc->misdnbuf, (unsigned char*)data, len);
#else 
  {
    char buf[4096 + IFRAME_HEAD_SIZE];
    iframe_t *frm= (iframe_t*)buf;
    int  r;
    
    frm->prim = DL_DATA|REQUEST;
    frm->dinfo = 0;
    frm->addr = bc->addr | IF_DOWN;
    frm->len = len;
    memcpy(&buf[IFRAME_HEAD_SIZE], data,len);
    
    r=mISDN_write(bc->stack->midev, buf, frm->len + IFRAME_HEAD_SIZE, TIMEOUT_INFINIT);
  }
#endif

  return 0;
}




void manager_send_tone ( manager_te_t *mgr, bchannel_te_t *bc, enum tone_e tone)
{

#ifdef MISDN_USE_L3
  if (tone != TONE_NONE) manager_bchannel_activate(bc);
  bc->tone=tone;
  bc->tone_cnt2=-1;
  bc->tone_cnt=0;
#else
  
  
  switch ( tone ) {
  case TONE_DIAL:
    manager_bchannel_activate(bc);
    manager_ph_control(bc,TONE_PATT_ON, TONE_GERMAN_DIALTONE);
    break;
  case TONE_NONE:
    manager_bchannel_deactivate(bc);
    manager_ph_control(bc,TONE_PATT_OFF, TONE_GERMAN_DIALTONE);
  }
#endif
}


unsigned char * manager_flip_buf_bits ( unsigned char * buf , int len)
{
  int i,k;
  char * start = buf;
  
  for (i = 0 ; i < len; i++) {
    unsigned char sample = 0 ;
    for (k = 0; k<8; k++) {
      if ( buf[i] & 1 << k ) sample |= 0x80 >>  k;
    }
    buf[i] = sample;

  }

  return start;
}

/*
 * send control information to the channel (dsp-module)
 */
void manager_ph_control(bchannel_te_t *bc, int c1, int c2)
{
	unsigned char buffer[IFRAME_HEAD_SIZE+sizeof(int)+sizeof(int)];
	iframe_t *ctrl = (iframe_t *)buffer; /* preload data */
	unsigned long *d = (unsigned long *)&ctrl->data.p;
	
	ctrl->prim = PH_CONTROL | REQUEST;
	ctrl->addr = bc->addr;
	ctrl->dinfo = 0;
	ctrl->len = sizeof(unsigned long)*2;
	*d++ = c1;
	*d++ = c2;
	mISDN_write(bc->stack->midev, ctrl, IFRAME_HEAD_SIZE+ctrl->len, TIMEOUT_1SEC);
}

void manager_clean_bc(manager_te_t *mgr, bchannel_te_t *bc )
{
  if (bc->state == STATE_CONNECTED)
    manager_te_send_event(mgr,bc,EVENT_DISCONNECT);
  
  manager_te_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
  delete_cr(mgr, bc, bc->cr);
}


void stack_holder_add(stack_te_t *stack, bchannel_te_t *holder)
{
  bchannel_te_t *help;

  if (!stack ) return ;
  
  holder->next=NULL;
  
  if (!stack->holding) {
    stack->holding = holder;
    return;
  }
  
  for (help=stack->holding;
       help;
       help=help->next) {
    if (!help->next) {
      help->next=holder;
    }
  }
  
}

void stack_holder_remove(stack_te_t *stack, bchannel_te_t *holder)
{
  bchannel_te_t *h1, *h2;

  if (!stack || ! stack->holding) return;
  
  if (holder == stack->holding) {
    stack->holding = stack->holding->next;
    return;
  }
  
  for (h1=stack->holding;
       h1;
       h1=h1->next) {
    if (h1->next == holder) {
      h1->next=h1->next->next;
      return ;
    }
  }
}


bchannel_te_t *stack_holder_find(stack_te_t *stack, unsigned long l3id)
{
  bchannel_te_t *help;

  if (!stack) return NULL;
  
  for (help=stack->holding;
       help;
       help=help->next) {
    if (help->cr->l3_id == l3id) return help;
  }
  
  return NULL;
}
