#ifndef LINESEQ_DEFINE
#define LINESEQ_DEFINE

/*
atanks - obliterate each other with oversize weapons
Copyright (C) 2003  Thomas Hudson

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
TODO:
 + Add support for comments
 + Give LINESEQ const_iterator's (is this useful?) and other such
   stl dodads
*/

#include <vector>
#include <ctype.h>
#include <stdio.h>
#include <string.h>


/*############################################################################
LINESEQ

Provides parsing, storage for text strings.  Text strings are loaded from
file, each line is stored as a seperate string.  Once constructed, the object
acts like an immutable array of pointers to text strings.
############################################################################*/
class LINESEQ	/* line sequence */
  {
    typedef std::vector< char* > cont_t;
    cont_t	l;

    /*************************************************************************
    readLine

    Reads a text line (of any length) from a file.  Lines stop when a new
    line character is read or when EOF is reached.

    The line is returned.  The caller is responsible for deleting [] the
    returned character array.  A null return value indicates EOF.
    *************************************************************************/
    static char* readLine( FILE* src )
    {
      std::vector<char>	str;
      std::vector<char>::size_type i;
      char *rv;
      int ch;

      /* Parse */
      while ( ((ch=fgetc(src)) != EOF) && (ch != '\n')  )
        if (ch != '\r') //ignore if exist in file
          str.push_back( ch );

      /* Did we fail? */
      if ( str.empty() && (ch==EOF) )
        return 0;

      /* Allocate */
      rv = new char[ str.size()+1 ];
      for ( i=0; i!=str.size(); ++i )
        rv[i] = str[i];
      rv[i] = 0;

      /* Return */
      return rv;
    }


    /*************************************************************************
    allBlank

    Returns true iff the given text string =~ m/^[ ]*$/
    *************************************************************************/
    bool allBlank( const char* p )
    {
      while ( *p && isspace(*p) )
        ++ p;
      return *p == 0;
    }


    /*************************************************************************
    =

    Assignment is not provided.
    *************************************************************************/
    LINESEQ& operator=( const LINESEQ& obj );

  public:

    enum blank_act
    {
      skip_blanks, keep_blanks
    };

    /*************************************************************************
    LINESEQ

    The class provides a default constructor, a copy constructor, and a
    constructor that loads text string from a text file.
    *************************************************************************/
    LINESEQ( ) {}

    LINESEQ( const LINESEQ& o )
    {
      cont_t::const_iterator	at, end;
      cont_t::iterator		out ;

      /* Size the array accordingly */
      l.resize( o.size() );

      /* Copy strings */
      end = o.l.end();
      out = l.begin();
      for ( at = o.l.begin(); at != end; ++at, ++out )
        {
          int cb = strlen( *at ) + 1;
          *out = new char[ cb ];
          memcpy( *out, *at, cb );
        }
    }

    explicit LINESEQ( const char *fname, blank_act bl )
    {
      FILE 	*f;
      char	*str;

      /* Open the file */
      if ( !(f = fopen( fname, (char *)"rb" )) )
        {
          l.push_back( new char[128] );
          sprintf( l[0], (char *)"LINESEQ failed to load file %.64s", fname );
          perror( l[0] );
          return ;
        }

      /* Append the lines one at a time */
      while ( (str = readLine(f)) )
        {
          if ( (bl==skip_blanks) && allBlank(str) )
            delete [] str ;
          else
            l.push_back( str );
        }

      /* finish */
      fclose(f);
    }


    /*************************************************************************
    ~LINESEQ

    Just what you would expect.
    *************************************************************************/
    ~LINESEQ( )
    {
      cont_t::iterator	at, end( l.end() );

      for ( at = l.begin(); at != end; ++at )
        {
          delete *at;
        }
    }


    /*************************************************************************
    empty

    Returns true iff the object contains no text strings.
    *************************************************************************/
    bool empty() const
      {
        return l.empty();
      }


    /*************************************************************************
    size

    Returns the actual number of text strings contained in the object
    *************************************************************************/
    int size( ) const
      {
        return l.size();
      }


    /*************************************************************************
    []

    Returns a pointer to the desired text string.  No bound cheching is
    performed.
    *************************************************************************/
    const char* operator[](int i) const
      {
        return l[i];
      }


    /*************************************************************************
    random

    Returns a pointer to a randomly chosen text string.
    *************************************************************************/
    const char* random( ) const
      {
        return l[ rand() % l.size() ];
      }
  };

#endif //LINESEQ_DEFINE
