# -*- mode: python; coding: utf-8 -*-

import os, sys, string, stat
import glob
import subprocess as subp
import optparse
import logging #, logging.handlers
import time
import select
import signal
import copy
import exceptions, traceback
import socket
import StringIO
from functools import partial

logging.basicConfig()

from atheist.const import *
from atheist.utils import *

"""
Verbosity level:

0:
[  OK  ]( 0) samples/test1.test              ls
[ FAIL ]( 2) samples/test1.test              ls na
Total: 0.03s - 1/2

1: INFO
samples/test1.test: 2 tests
[  OK  ]( 0) samples/test1.test              ls
[ FAIL ]( 2) samples/test1.test              ls na
Total: 0.54s - 1/2

2: DEBUG
samples/test1.test loading
samples/test1.test: 2 tests
Executing <Test 'test1.test' - 'ls'>
<Test 'test1.test' - 'ls'> process ends
Executing <Test 'test1.test' - 'ls na'>
<Test 'test1.test' - 'ls na'> process ends
[  OK  ]( 0) samples/test1.test              ls
[ FAIL ]( 2) samples/test1.test              ls na
Total: 0.03s - 1/2
"""

class LoggingFormatter(logging.Formatter):
    def format(self, record):
        record.levelinitial = record.levelname[0]*2
        return logging.Formatter.format(self, record)


def check_type(val, cls):
    if not isinstance(val, cls):
        raise TypeError(("A %s is required, not %s." % \
              (cls.__name__, val.__class__.__name__)))


def pretty(value):
    term = os.getenv('TERM')
    cterm = os.getenv('COLORTERM')

    if cterm or 'color' in term or 'xterm' == term:
        just = 12
        RESULT_STR = {\
            FAIL:    IRED        + "FAIL" + NORM,
            OK:      GREEN       + " OK " + NORM,
            NOEXEC:  GREY        + " -- " + NORM,
            ERROR:   BOLD + IRED + " !! " + NORM,
            UNKNOWN: GREY        + " ?? " + NORM,
            TODO:    PURPLE      + "ToDo" + NORM,
            mild_FAIL: RED + "fail" + NORM}
    else:
        just = 4
        RESULT_STR = {\
            FAIL:            "FAIL",
            OK:              " OK ",
            NOEXEC:          " -- ",
            ERROR:           " !! ",
            UNKNOWN:         " ?? ",
            TODO:            "ToDo",
            mild_FAIL: "fail"}

    return "[%s]" % RESULT_STR[value].center(just)


def file_template():
    # FIXME: Construir a partir de 'task_attrs'
    return '''# -*- mode:python; coding:utf-8 -*-

# Test(cmd,
#      cwd         = '.',
#      delay       = 0,
#      desc        = 'test template',
#      detach      = False,
#      env         = {},
#      expected    = 0,
#      must_fail   = False,
#      path        = '.',
#      template    = [],
#      timeout     = 5,
#      save_stderr = False,
#      save_stdout = False,
#      signal      = signal.SIGKILL)
'''



class Public:
    "Base class for classes available for the client tasks"
    pass


class PlainReporter:

    #FIXME: No debería depender de ConsoleSummaryReporter
    @staticmethod
    def report(task):
        retval = ConsoleSummaryReporter().task_summary(task)
        for i in [GREEN, RED, NORM, GREY]:
            retval = retval.replace(i,'')
        return retval.strip()


class SummaryReporter:
    def __init__(self, name, handler):
        self.log = logging.getLogger(name)
        self.log.addHandler(handler)
        self.log.propagate = 0
        # FIXME: formato de este handler

    def _make_report(self, mgr):
        retval = '''
## Atheist report ##

SOME TASKS FAILED!

user: %s@%s
date: %s
argv: "%s"

''' % (os.getlogin(), socket.gethostname(),
       time.asctime(), str.join(' ', sys.argv))
        for t in mgr.itertasks():
            retval += PlainReporter.report(t) + '\n'

        return retval

    def update(self, task_list):

        Log.debug("%s: sending report to %s" % \
                       (self.__class__.__name__, self.dst))
        report = self._make_report(task_list)
        if report:
            self.log.error(report)



class ConsoleSummaryReporter(SummaryReporter):
    "Create a task summary representation for the console"
    def __init__(self, config):
        self.dst = 'console'
        self.config = config
        self.width = config.screen_width - 21
        SummaryReporter.__init__(self, "Console",
                                 logging.StreamHandler())

    def _make_report(self, mgr):
        self.report = []
        if self.config.report_detail == 0: return ''
        for taskset in mgr.itercases():
            self.build_taskcase(taskset)

        return str.join('\n', self.report)

    def build_taskcase(self, taskcase):

        TESTCASE_FORMAT = '{result} Test case: {path}'

        self.report.append(
            TESTCASE_FORMAT.format(
                result = pretty(taskcase.result),
                path   = compath(taskcase.fname)
                ))

        if taskcase.result in [OK,NOEXEC] and self.config.report_detail < 2:
            return

        tree_draw(taskcase.tasks, cb=self.build_task)


    def build_task(self, task, conn, tree):
        retcode = int(task.retcode) if task.retcode != None else '-'

        cmd = task.cmd if hasattr(task, 'cmd') else ''

        if task.desc:
            cmd_desc = '%s | %s' % (ellipsis(cmd, self.width*0.25, True),
                                    task.desc)
        else:
            cmd_desc = ellipsis(cmd, self.width*0.75)


        TASK_SUMMARY = u"{result} {tree} {type}{mode}{index:<3} ({retcode:>2}:{expected:2})  {desc}"

        self.report.append(
            TASK_SUMMARY.format(
                result   = pretty(task.result),
                tree     = tree,
                retcode  = retcode,
                expected = task.expected,
                mode     = task.get_mode(),
                type     = task.initial,
                index    = task.indx,
                desc     = cmd_desc
                ))

        if (self.config.report_detail > 2 or task.result != OK) and \
                isinstance(task, CompositeTask):
            tree_draw(task.children, cb=self.build_task, conn=conn)

        if not task.conditions_fail and (self.config.report_detail < 4):
            return

        tree_draw(task.pre + task.post,
                  cb=self.build_condition,
                  conn=conn,
                  args=[task])


    def build_condition(self, cond, conn, tree, task):

        CONDITION_SUMMARY = u"{value} {tree} {kind:<4} {name} {info}"

        value_len = len(pretty(cond.value))

        self.report.append(
            CONDITION_SUMMARY.format(
                value  = pretty(cond.value),
                tree   = tree,
                kind   = 'pre: ' if cond in task.pre else 'post:',
                name   = cond.name,
                info   = cond.basic_info()
                ))


def tree_draw(container, cb, conn='', args=[]):

    lon = len(container)
    for i,item in enumerate(container):
        brother = u'├─'
        parent  = u'│  '

        if lon == i+1:
            brother = u'└─'
            parent  = '   '

        cb(item,
           conn + parent,
           conn + brother, *args)



class JabberSummaryReporter(SummaryReporter):
    def __init__(self, dst):
        self.dst = dst

        handler = utils.JabberHandler(\
            (usercfg['jabber.user'],
             usercfg['jabber.pass']),
            dst)

        SummaryReporter.__init__(self,
                              'Jabber-%s' % dst,
                              handler)

    def update(self, mgr):
        if not mgr.ALL(): SummaryReporter.update(self, mgr)


class MailSummaryReporter(SummaryReporter):
    def __init__(self, dst):
        self.dst = dst

        handler = utils.SMTPHandler(\
            (usercfg['smtp.host'],
             usercfg.get_item('smtp.port', 25)),
            usercfg['smtp.user'],
            dst,
            'Atheist report',
            (usercfg['smtp.user'],
             usercfg['smtp.pass']))

        SummaryReporter.__init__(self,
                              'Email-%s' % dst,
                              handler)

    def update(self, mgr):
        if not mgr.ALL(): SummaryReporter.update(self, mgr)


class TypedList(list):
    def __init__(self, cls):
        self._cls = cls

    def append(self, val):
        Log.error("Use operator += instead")

    def __iadd__(self, val):
        if isinstance(val, list):
            for i in val: self += i
            return self

        check_type(val, self._cls)
        list.append(self, val)
        return self

    def copy(self):
        retval = TypedList(self._cls)
        retval.extend(self)
        return retval

    def prepend(self, val):
        self.insert(0, val)

    @classmethod
    def isA(cls, val, klass):
        "checks 'val' is a TypedList(cls)"
        assert isinstance(val, list), val
        for i in val:
            assert isinstance(i, klass)


class ConditionList(TypedList):
    def __init__(self, values=[]):
        TypedList.__init__(self, Condition)
        for i in values:
            self += i

    def __iadd__(self, val):
        if val in self:
            Log.error("Trying to add a duplicate condition")
            return self
#        print "adding", id(val), val
        check_type(val, self._cls)
        return TypedList.__iadd__(self, val.clone())

    def contains(self, cond):
        """ Find a condition in the list, even decorated """
        if cond in self:
            return True

        return any([c.contains(cond) for c in self
                    if isinstance(c, ConditionDecorator)])

    def clone(self):
        return ConditionList(self)

    def remove_dups(self):
        return ConditionList(remove_dups(self))


class Condition:
    def __init__(self):
        self._value = UNKNOWN
        self.task = None

    @property
    def value(self):
        return self._value

    def before(self, task):
        """ Executed just BEFORE task execution """

        print "%s - %s" % (self, task)
        if self.task is not None and task is not self.task:
            Log.error("Conditions can not be shared: %s in %s" % (self, self.task))
            sys.exit(1)

        self._value = UNKNOWN
        self.task = task


    def after(self):
        """ Exectuted just AFTER task execution """

    def run(self):
        raise NotImplemented

    def evaluate(self):
        self._value = self.run()
        assert self._value is not None
        return self._value

    def clone(self):
        retval = copy.copy(self)
        retval._value = UNKNOWN
        retval.task = None
        return retval

    def __repr__(self):
        return str(self)

    def __str__(self):
        return "<%s%s %s>" % (pretty(self._value),
                              self.name, self.basic_info())

#    def summary(self):
#        return "%s %s>" % (str(self)[:-1], self.more_info())

    @property
    def name(self):
        return self.__class__.__name__

    def basic_info(self):
        return ''

    def more_info(self):
        return ''


class Callback(Condition, Public):
    def __init__(self, func, args=()):
        self.func = func
        assert isinstance(args, tuple)
        self.args = args
        Condition.__init__(self)

    def run(self):
        try:
            retval = self.func(*self.args)
            if retval is None:
                Log.warning("%s() returns None" % self.func.__name__)
            return bool(retval)

        except Exception, e:
            Log.error("%s: %s" % (self.__class__.__name__, e))
            print_exc(traceback.format_exc())

        return False

    def basic_info(self):
        return "%s: '%s'" % (self.func.__name__, ellipsis(self.args))



class EnvVarDefined(Condition, Public):
    def __init__(self, varname, varvalue=None):
        Condition.__init__(self)
        self.varname = varname
        self.varvalue = varvalue

    def run(self):
        retval = self.varname in os.environ.keys()
        if retval and self.varvalue:
            retval = (os.environ[self.varname] == self.varvalue)
        return retval

    def basic_info(self):
        return "'%s'" % self.varname


class FileExists(Condition, Public):
    def __init__(self, fname):
        self.fname = fname
        Condition.__init__(self)

    def __eq__(self, other):
        return self.__class__ == other.__class__ and self.fname == other.fname

    def run(self):
        assert isinstance(self.fname, str), "Not a valid filename"
        return os.path.exists(self.fname)

    def basic_info(self):
        return "'%s'" % self.fname


class DirExists(FileExists, Public):
    def run(self):
        return FileExists.run(self) and os.path.isdir(self.fname)


class FileContains(Condition, Public):
    def __init__(self, data, fname=None, times=1):
        self.fname = fname
        self.times = times
        self.contains = self.check_contains(data)
        Condition.__init__(self)

    def before(self, task):
#        print "setting task <<<<<< %s" % self

        Condition.before(self, task)
        if self.fname == None:
            task.setup_outs(True)
            self.fname = task.stdout
            return

        # auto-prepend a FileExists condition
#        print "añadiendo un FileExists para", self.fname
        filecond = FileExists(self.fname)

        for conds in [task.pre, task.post]:
#            print "self in conds:", self in conds
#            print "not filecond in cods:", not filecond in conds
#            if self in conds and not filecond in conds:
            if conds.contains(self) and \
                    not conds.contains(filecond):
                conds[0:0] = [filecond]


    def check_contains(self, data):
        if isinstance(data, str):
            return [data]
        elif isinstance(data, list):
            assert all([isinstance(x, str) for x in data]),\
                "FileContains args must be strings"
            return data

        raise exceptions.TypeError(data)


    def run(self):
        if self.fname is None:
            Log.critical("Condition is not initialized!")
            return ERROR

        if not os.path.exists(self.fname):
            Log.error("'%s' does not exists" % compath(self.fname))
            return ERROR

        try:
            fd = open(self.fname)
            fcontent = fd.read()
            fd.close()
        except IOError, e:
            Log.error(e)
            return ERROR

        return all([fcontent.count(x) >= self.times for x in self.contains])
        #return all([x in fcontent for x in self.contains])

    def basic_info(self):
        return "'%s' (%s) content:'%s'" % (self.fname, self.times,
                                         ellipsis(self.contains))


class FileEquals(Condition, Public):
    def __init__(self, fname1, fname2=None):
        self.fname1 = fname1
        self.fname2 = fname2
        Condition.__init__(self)

    def before(self, task):
        Condition.before(self, task)
        if self.fname2 == None:
            task.setup_outs(True)
            self.fname2 = task.stdout

        # auto-prepend a FileExists condition
        for fname in [self.fname1, self.fname2]:
            filecond = FileExists(fname)
            for conds in [task.pre, task.post]:
                if self in conds and not filecond in conds:
                    i = conds.index(self)
                    conds[i:i+1] = [filecond, self]

    def run(self):
        return not os.system('diff %s %s > /dev/null' % (self.fname1, self.fname2))

    def basic_info(self):
        return "'%s' == '%s'" % (self.fname1, self.fname2)


class FileIsNewerThan(Condition, Public):
    def __init__(self, fname1, fname2):
        self.fname1 = fname1
        self.fname2 = fname2
        Condition.__init__(self)

    def run(self):
        time1 = os.stat(self.fname1)[stat.ST_MTIME]
        time2 = os.stat(self.fname2)[stat.ST_MTIME]
        return int(time1 > time2);

    def basic_info(self):
        return "'%s' is newer than '%s'" % (self.fname1, self.fname2)


class OpenPort(Condition, Public):
    def __init__(self, port, host='localhost', proto='tcp'):
        assert proto in ['tcp','udp']
        assert isinstance(port , int) and 0 < port < 65535
        Condition.__init__(self)

        self.proto = proto
        self.port = port
        self.host = host

    def run(self):
        if self.host == 'localhost':
            self.cmd = 'fuser -n %s %s > /dev/null 2> /dev/null' % \
                       (self.proto, self.port)
        else:
            self.cmd = 'nmap -p%s %s | grep "%s/%s open" > /dev/null' % \
                       (self.port, self.host, self.port, self.proto)
        return os.system(self.cmd) == 0

    def basic_info(self):
        return "'%s %s/%s'" % (self.host, self.proto, self.port)

    def more_info(self):
        return "cmd: %s" % self.cmd


class ProcessRunning(Condition, Public):
    def __init__(self, pid):
        assert isinstance(pid, int)
        self.pid = pid
        Condition.__init__(self)

    def run(self):
        return os.system("ps ef %s > /dev/null" % self.pid) == 0

    def basic_info(self):
        return "'%s'" % self.pid


class AtheistVersion(Condition, Public):
    def __init__(self, version):
        self.version = float(version)
        Condition.__init__(self)

    def run(self):
        Log.debug("%s: inst:%s req:%s" %
                   (self.__class__.__name__, VERSION, self.version))
        return float(VERSION) >= self.version


#FIXME: Clonación de Condition decoradas
class ConditionDecorator(Condition, Public):
    def __init__(self, *conds):
        Condition.__init__(self)
        self.conds = ConditionList()
        for c in conds:
            check_type(c, Condition)
            self.conds += c

    def clone(self):
        retval = Condition.clone(self)
        retval.conds = self.conds.clone()
        return retval

    def before(self, task):
        for c in self.conds:
            c.before(task)
        Condition.before(self, task)

    def run(self):
        raise NotImplemented

    def contains(self, cond):
        return self.conds.contains(cond)

    def __str__(self):
        return "<%s%s %s)>" % \
            (pretty(self._value),
             self.name, self.basic_info())

    @property
    def name(self):
        return "%s (%s" % (self.__class__.__name__,
                          str.join(' ', [x.name for x in self.conds]))

    def basic_info(self):
        return str.join(' ', [x.basic_info() for x in self.conds])

    def more_info(self):
        return str.join(' ', [x.more_info() for x in self.conds])


class Not(ConditionDecorator, Public):
    def run(self):
        val = self.conds[0].run()
        if isinstance(val, bool):
            return not(val)

        return ERROR


# # FIXME:test this
# class Or(ConditionDecorator, Public):
#    def run(self):
#        return any([c.run() for c in self.conds])


class Poll(ConditionDecorator, Public):
    def __init__(self, condition, interval=1, timeout=5):
        ConditionDecorator.__init__(self, condition)
        self.interval = interval
        self.timeout = timeout

    def run(self):
        enlapsed = 0
        while 1:
#            if self.task.mgr.abort: break
            Log.debug("Polling condition: %s" % self.conds[0])
            if self.conds[0].run(): return True
            if self.timeout is not None and enlapsed > self.timeout: break
            time.sleep(self.interval)
            enlapsed += self.interval

        return False


class RemoteTestFactory(Public):
    def __init__(self, account):
        self.account = account
        try:
            self.user, self.host = self.account.split('@')
        except ValueError:
            self.host = self.account
            self.user = os.getlogin()

    def __call__(self, cmd, **values):
        values.update({'shell':True})
        return Test('ssh %s "%s"' % (self.account, cmd), **values)



DFL_TIMEOUT = 5
DFL_SIGNAL = signal.SIGKILL


class BufFile:

    def __init__(self, fd):
        self.data = ''
        self.fileno = fd.fileno()

    def readline(self):
        new = os.read(self.fileno, 2048)
        if not new:
            yield self.data
            raise StopIteration

        self.data += new
        if not '\n' in self.data:
            yield ''
            raise StopIteration

        for line in self.data.split()[:-1]:
            yield line



class FileWrapper:

    def __init__(self, console_out, file_handler, tag, only_on_fail=False):

        def file_write_tag(text, fd=sys.__stdout__):
            # FIXME: imprime como lineas textos que no acaban en \n.
            # Necesitamos un readline()
            for line in text.split('\n'):
                if not line: continue
                fd.write("%s| %s\n" % (tag, line))
                fd.flush()

        def file_write(text, fd):
            fd.write(text)
            fd.flush()

        #--

        self.out_funcs = []
        self.last_task_out = None

        if console_out:
            self.out_funcs.append(file_write_tag)

        elif only_on_fail:
            self.last_task_out = StringIO.StringIO()
            self.out_funcs.append(partial(file_write_tag, fd=self.last_task_out))

        if file_handler:
            self.out_funcs.append(partial(file_write, fd=file_handler))

    def write(self, data):
        for func in self.out_funcs: func(data)

    def flush(self):
        pass

    def print_fail_out(self):
        if not self.last_task_out:
            return # it was not asked

        print self.last_task_out.getvalue(),
        sys.__stdout__.flush()


class Task(Record):

    allows = ['tid', 'cwd', 'delay', 'desc', 'expected', 'must_fail', 'template',
              'save_stdout', 'save_stderr', 'stderr', 'stdout', 'todo']
    forbid = []

    @classmethod
    def set_mgr(cls, mgr):
        cls.mgr = mgr

    @classmethod
    def validate_kargs(cls, _dict):
        for k,v in _dict.items():
            assert task_attrs.has_key(k), "'%s' is not a valid keyword" % k
            assert k in cls.allows, \
            "'%s' is not a allowed keyword for '%s'" % (k, cls.__name__)

            check_type(v, task_attrs[k].type_)

            if v == task_attrs[k].default:
                Log.info("Default value for '%s' is already '%s'. Remove it." % (k, v))
        return _dict


    def __init__(self, **kargs):
        #current_ids = [x.id for x in TC] + [x.id for x in ts]  # All
        current_ids = [x.tid for x in self.mgr.ts]    # Only this testcase

        assert 'tid' not in kargs.keys() or \
            kargs['tid'] not in current_ids, "Duplicate task id: '%s'" % kargs['tid']

        try:
            self.ts = kargs['parent'].children
        except KeyError:
            self.ts = self.mgr.ts

        self.ts.append(self)
        self.kargs = kargs

        defaults = {}
        for k,v in task_attrs.items(): defaults[k] = v.default
        self.__dict__.update(defaults)

        if kargs.has_key('template'):
            check_type(kargs['template'], list)
            for i in kargs['template']:
                self.__dict__.update(self.validate_kargs(i))

        self.__dict__.update(self.validate_kargs(kargs))

        self._indx = self.mgr.next()
        self.fname = ''
        self.gen = TypedList(str)
        self.pre =  ConditionList()
        self.post = ConditionList()
        self.terminates = TypedList(Task)

        self._mode = None               # normal, setup, teardown
        self.result = NOEXEC
        self.conditions_fail = False

        self.retcode = None
        self.thread = None
        self.time_start = None
        self.time_end = None

        self.already_setup = False
        self.stderr_fd = None
        self.stdout_fd = None
        self.last_task_out = None

        self.setup_outs(self.save_stdout, self.save_stderr)

        #- logger
        self.log = logging.getLogger("%s" % (self.name))
        if self.log.handlers:
            return # the logger already exists

        term = logging.StreamHandler()
        term.setFormatter(\
            LoggingFormatter(\
                self.mgr.config.timetag + '[%(levelinitial)s] %(name)s: %(message)s',
                LOG_DATEFORMAT))
        term.setLevel(self.mgr.config.loglevel)
        self.log.setLevel(self.mgr.config.loglevel)
        self.log.addHandler(term)
        self.log.propagate = 0


    def get_mode(self):
        return self._mode

    def set_mode(self, mode):
        self._mode = mode


    def setup_outs(self, enable_stdout=False, enable_stderr=False):
        if not (enable_stdout or enable_stderr): return
        if self.already_setup == True: return
        self.already_setup = True

        if enable_stdout and not self.stdout:
            self.stdout = '/tmp/atheist/%s_%s.out' % (self.name, os.getpid())

        if enable_stderr and not self.stderr:
            self.stderr = '/tmp/atheist/%s_%s.err' % (self.name, os.getpid())

        if self.stdout:
            self.save_stdout = True
            cond = FileExists(self.stdout)
#            if not cond in self.post:
            self.post[0:0] = [cond]

#            if not self.stdout in self.gen:
            self.gen += self.stdout

        if self.stderr:
            self.save_stderr = True
            cond = FileExists(self.stderr)
#            if not cond in self.post:
            self.post[0:0] = [cond]

#            if not self.stdout in self.gen:
            self.gen += self.stderr


    # FIXME: Debería dejar la Task en el estado inicial, necesario para --until-failure
    def setup(self):
        self.result = UNKNOWN

        for c in self.pre + self.post:
            c.before(self)

        self.pre = self.pre.remove_dups()
        self.post = self.post.remove_dups()
        self.gen = remove_dups(self.gen)

        if self.pre or self.post:
            self.result = OK


    @property
    def indx(self):
        return self._indx

    @property
    def description(self):
        return  "%s:%s" % (self.name, self.desc)

    def describe(self, pre_keys=[]):
        self.setup()

        dic = self.__dict__.copy()
        dic['fname'] = compath(dic['fname'])

        keys = pre_keys[:] + ['fname']
        if self.result is not NOEXEC: keys.append('result')

        # add standard attrs with no-default value
        for k,v in task_attrs.items():
            if getattr(self,k) != v.default:
                keys.append(k)

        attrs = []
        for i in keys:
            value = dic[i]
            if isinstance(value, str):
                value = "'%s'" % value.replace('\n', '\n' + ' '*16)

            attrs.append("%s: %s" % (i.rjust(14), value))

        # multivalued attributes
        for a in ['pre','post','gen']:
            for i in getattr(self, a):
                attrs.append("%s %s" % ('%s:' % a.rjust(14), i))

        return '%3s:%s\n%s\n' % (self.indx, self.__class__.__name__,
                                 str.join('\n', attrs))

    def __repr__(self):
        return str(self)

    def __str2__(self):
        return "%s: <%s '%s' - '%s'>" % \
            (high(self.name), self.__class__.__name__,
             os.path.splitext(os.path.basename(self.fname))[0],
             self.cmd.split('\n')[0])

    def __str__(self):
        return "%s: <%s(%s)>" % \
            (high(self.name), self.__class__.__name__, self.str_param())

    def str_param(self):
        return ''


    @property
    def name(self):
        return self.initial + str(self.indx)

    @property
    def initial(self):
        retval = self.__class__.__name__[0]
        return retval


    def clone(self):
        d = self.kargs
        d['desc'] = '(copy of %s) %s' % (self.name, d['desc'])
        retval = Task(**d)  #FIXME: esto debe especializarse
        for c in self.pre:
            retval.pre += c
        for c in self.post:
            retval.post += c
        retval.gen = self.gen.copy()
        return retval


    def do_exec_task(self):
        self.outwrap = FileWrapper(self.mgr.config.stdout,
                                   self.stdout_fd,
                                   tag='%s:out' % self.name,
                                   only_on_fail=self.mgr.config.stdout_on_fail)

        self.errwrap = FileWrapper(self.mgr.config.stderr,
                                   self.stderr_fd,
                                   tag='%s:err' % self.name)

        self.exec_task()

        if self.result != OK:
            self.outwrap.print_fail_out()

        if self.result == FAIL and self.todo:
            self.result = TODO

        if self.result == FAIL and not self.check:
            self.result = mild_FAIL

        self.outwrap = None
        self.errwrap = None


    def exec_task(self):
        raise NotImplementedError

    def close(self):
        pass


class Subprocess(Task):

    allows = Task.allows + ['detach', 'env', 'path', 'timeout', 'shell', 'signal']

    def __init__(self, cmd, **kargs):
        self.cmd = '' if cmd is None else cmd

        assert isinstance(self.cmd, str) or self.cmd is None

        Task.__init__(self, **kargs)

        # add the user variables to dft environ
        environ = os.environ.copy()
        for k,v in self.env.items():
            environ[k] = string.Template(v).safe_substitute(environ)

        self.env = environ
        self.ps = None

    def exec_task(self):
        if not self.cmd: return

        if self.path:
            os.environ['PATH'] += ':'+self.path

#        self.tini = time.time() # lo hace run_task

        if self.shell:
#            cmd = ('/bin/bash -c \"%s\"' % self.cmd.replace('\"', '\\"')).split()
            cmd = ['/bin/bash', '-c', "%s" % self.cmd]
        else:
            cmd = self.cmd.split()

#        print cmd
#        cmd = 'setsid /bin/sh -c \"%s\"' % self.cmd.replace('\"', '\\"')
#        cmd = self.cmd
        try:
            self.ps = subp.Popen(cmd,
                                 close_fds  = True,
                                 stdin      = subp.PIPE,
                                 stdout     = subp.PIPE,
                                 stderr     = subp.PIPE,
                                 shell      = False,
                                 bufsize    = 0,
                                 cwd        = self.cwd,
                                 env        = self.env,
                                 preexec_fn = os.setsid)
        except OSError, e:
            self.log.error("%s: '%s'" % (e, self.cmd))
            self.result = ERROR
            return

        self.log.info("starts (pid: %s)" % self.ps.pid)

        read_fds = [self.ps.stdout, self.ps.stderr]
        read_ready = select.select(read_fds,[],[], 0.05)[0]

        while self.ps.poll() is None or read_fds:
            for fd in read_ready:
# http://mail.python.org/pipermail/python-bugs-list/2001-March/004491.html
                time.sleep(0.01)
                data = os.read(fd.fileno(), 2048)
                if not data:
                    read_fds.remove(fd)
                elif fd is self.ps.stdout:
                    self.outwrap.write(data)
                elif fd is self.ps.stderr:
                    self.errwrap.write(data)

            try:
                read_ready = select.select(read_fds,[],[], 0.2)[0]
            except select.error:
                self.mgr.abort = True
                self.result = ERROR
                return

            if self.timeout == 0: continue

            if time.time()-self.tini > self.timeout:
                self.log.debug("timeout expired (%.1f>%s), sending signal %s to %s" % \
                                   (time.time() - self.tini, self.timeout,
                                    self.signal, self.ps.pid))
                self.kill()

        self.outwrap.write(self.ps.stdout.read())
        self.errwrap.write(self.ps.stderr.read())

        self.retcode = self.ps.returncode
        self.log.debug("finish with %s" % self.retcode)

        if self.shell and self.retcode == 127:
            self.log.error("No such file or directory: '%s'" % self.cmd)
            self.result = ERROR
            return

        self.result = (self.retcode == self.expected)

        if self.must_fail:
            self.result = self.retcode != 0

    def str_param(self):
        return "'%s'" % self.cmd.split('\n')[0]

    def describe(self):
        return Task.describe(self, ['cmd'])

    def clone(self):
        cls = self.__class__
        return cls(self.cmd, **self.kargs)

    def kill(self, n=None):
        if n is None:
            n = self.signal

        try:
            os.killpg(self.ps.pid, n)
        except OSError, e:
            Log.debug("%s (%s)" % (e, self.ps.pid))
        except AttributeError:
            self.log.warning("did not even starts")
            return

        while 1:
            if self.ps.poll() is not None: return
            time.sleep(0.5)

    def close(self):
        if not self.ps: return
        self.ps.stdin.close()
        self.ps.stdout.close()
        self.ps.stderr.close()



class Test(Subprocess, Public):
    pass


class TestFunc(Task, Public):

    allows = ['cwd', 'delay', 'desc', 'expected', 'must_fail', 'template', 'tid']

    def __init__(self, func, args=(), **kargs):
        assert callable(func)
        self.func = func
        self.args = args
        self.check = True

        Task.__init__(self, **kargs)


    def exec_task(self):
        sys.stdout = self.outwrap
        sys.stderr = self.errwrap

        try:
            current = os.getcwd()
            if self.cwd: os.chdir(self.cwd)

            self.retcode = self.func(*self.args)

            if not isinstance(self.retcode, int):
                Log.error("Function '%s' should return an integer" % self.func.__name__)

            if self.retcode is None:
                self.result = UNKNOWN
            else:
                self.result = (self.retcode == self.expected)

        except Exception, e:
            self.log.info(e)
            self.result = ERROR

        finally:
            sys.stdout = sys.__stdout__
            sys.stderr = sys.__stderr__

        if self.cwd: os.chdir(current)

        if self.must_fail and self.result == FAIL:
            self.result = OK


class Command(Subprocess, Public):
    def __init__(self, cmd, **kargs):
        Subprocess.__init__(self, cmd, **kargs)
        self.check = False


class CompositeTask(Task, Public):

    allows = ['detach']

    """ The CompositeTask let you specify a predicate to evalue a set of tasks:

    CompositeTask(all, t1, t2, t3) is OK if all the tasks are OK.

    #--

    def at_least_two(tasks):
        return [bool(x) for x in tasks].count(True) >= 2

    CompositeTask(at_least_two, t1, t2, t3) is OK when 2 or more tasks are OK.
    """

    def __init__(self, oper, *tasks, **kargs):
        self.oper = oper
        self.children = list(tasks)
        Task.__init__(self, **kargs)

        for t in tasks:
            t.parent = self
            self.ts.remove(t)

        self.desc = "%s %s" % (oper.__name__, [x.name for x in tasks])

    def set_mode(self, mode):
        self._mode = mode
        for t in self.children:
            t.set_mode(mode)

    def exec_task(self):
        self.gen =  merge_uniq(*[t.gen for t in self.children])

        for t in self.children:
            run_task(t)
        self.result = self.oper([t.result is OK for t in self.children if t.check])


task_attrs = SortedDict({
    'tid':         Record(type_=str,  default=None),
    'check':       Record(type_=bool, default=True),
    'cwd':         Record(type_=str,  default=None),
    'delay':       Record(type_=int,  default=0),
    'desc':        Record(type_=str,  default=''),
    'detach':      Record(type_=bool, default=False),
    'env':         Record(type_=dict, default=os.environ.copy()),
    'expected':    Record(type_=int,  default=0),
    'must_fail':   Record(type_=bool, default=False),
    'parent':      Record(type_=CompositeTask, default=None),
    'path':        Record(type_=str,  default=''),
    'template':    Record(type_=list, default=[]),
    'timeout':     Record(type_=int,  default=DFL_TIMEOUT),
    'save_stderr': Record(type_=bool, default=False),
    'save_stdout': Record(type_=bool, default=False),
    'shell':       Record(type_=bool, default=False),
    'signal':      Record(type_=int,  default=DFL_SIGNAL),
    'stderr':      Record(type_=str,  default=''),
    'stdout':      Record(type_=str,  default=''),
    'todo':        Record(type_=bool, default=False),
    })



class Daemon(Command):
    forbid = ['timeout', 'detach', 'expected']

    def __init__(self, cmd, **kargs):
        Command.__init__(self, cmd, **kargs)
        self.timeout = 0
        self.detach = True
        self.check = False
        self.expected = -9


class Template(dict, Public):
    def __init__(self, **kargs):
        self.update(kargs)


def run_task(task, keep_going=False, recurse=False, end_callback=lambda:None):
    if task.detach and recurse==False:
        task.log.debug("detaching")
        task.thread = ThreadFunc(run_task, (task,),
                                 {'keep_going':keep_going, 'recurse':True,
                                  'end_callback':end_callback})
        time.sleep(0.0)
        return None

    task.setup()

    # pre-conditions
    for c in task.pre:
        value = c.evaluate()
        task.log.info("Pre:  %s" % c)
        if value != OK:
            task.conditions_fail = True
            task.result = FAIL
            if not keep_going:
                return False

    if not task.conditions_fail:
        if task.stdout:
            task.stdout_fd = open(task.stdout, 'w')

        if task.stderr:
            task.stderr_fd = open(task.stderr, 'w')

        time.sleep(task.delay)

        task.tini = time.time()
        try:
            task.do_exec_task()
        except select.error:
            task.mgr.abort = True

        if task.stdout_fd:
            task.stdout_fd.close()

        if task.stderr_fd:
            task.stderr_fd.close()

        task.close()


    # post-conditions
    for c in task.post:
        c.after()
        value = c.evaluate()
        task.log.info("Post: %s" % c)
        if value != OK:
            task.conditions_fail = True
            task.result = False
            if not keep_going:
                return False

    # terminates
    for t in task.terminates:
        Log.error("'terminates' attribute is deprecated. Use TaskTerminator instead.")
        sys.exit(-1)

    end_callback()

    return task.result


class TestCase(object):
    def __init__(self, fname, mgr, template={}):
        fname = compath(fname)
        self.fname = fname
        self.mgr = mgr
        self.ts = mgr.ts

        aux_fname = os.path.splitext(compath(fname))[0]

        self.template = template
        testdir =  os.path.dirname(fname)
        self.template.update({
                'basedir':  compath(),
                'dirname':  compath(testdir),
                'fname':    aux_fname,
                'testname': os.path.basename(aux_fname),
                'atheist':  './athcmd.py'})


        #for i in self.ts[:]: self.ts.remove(i)  # empty the list :-S
        del self.ts[:]

        if not self.mgr.config.skip_setup:
            setup = os.path.join(testdir, SETUP)
            if os.path.exists(setup ):
                self.process_file(setup, task_mode.SETUP)

        self.process_file(fname, task_mode.MAIN)

        if not self.mgr.config.skip_setup:
            teardown = os.path.join(testdir, TEARDOWN)
            if os.path.exists(teardown):
                self.process_file(teardown, task_mode.TEARDOWN)

        for t in self.ts:
            for i in t.gen:
                t.pre.prepend(Not(FileExists(i)))
                t.post.prepend(FileExists(i))

#            if isinstance(t, CompositeTask):
#                for j in t.tasks:
#                    self.ts.remove(j)

        self.tasks = self.ts[:]

        self.result = NOEXEC


    def process_file(self, fname, mode):
        Log.debug("%s loading" % compath(fname))

        # ---
        before = self.ts[:]
        env = self.mgr.exec_env.copy()
        env['ts'] = self.ts
        env['__file__'] = os.path.abspath(fname)

        try:
            exec(self.load_file(fname)) in env
        except Exception, e:
            print_exc(traceback.format_exc())
            Log.error("Errors in the task definition '%s'" % fname)
            sys.exit(1)

        for t in [x for x in self.ts if x not in before]:
            t.fname = fname
        # ---

        for t in self.ts:
#            if t.mode is None: t.mode = mode
            t.set_mode(mode)



    def load_file(self, fname):
        def include_error(n, line):
            Log.error("Syntax error in: '%s:%s'" % (fname, n+1))
            Log.error("               : %s" % line)

        script = open(fname)
        lines = [string.Template(x).safe_substitute(self.template) \
                     for x in script.readlines()]
        script.close()

        for n, line in enumerate(lines):
            if not line.startswith('include('): continue

            #print '-', line
            indx = lines.index(line)
            #print indx
            lines.remove(line)

            try:
                line = line.strip()
                included_file = INCLUDE_RE.match(line).group(1)
            except AttributeError:
                include_error(n, line)
                sys.exit(1)

            try:
                Log.debug("included file: '%s'" % included_file)
                fd = open(included_file)
                ilines = fd.readlines()
                fd.close()
                lines[indx:indx] = ilines
            except IOError, e:
                include_error(n, line)
                Log.error(e)
                sys.exit(1)

        return str.join('', lines)


    def run(self, ob):
        Log.info("Test case %s" % (compath(self.fname)+' ').ljust(80, '-'))

        if not self.tasks:
            return

        for t in self.tasks:
            if self.mgr.abort: break

            Log.info(t)
            result = run_task(t,
                              keep_going=self.mgr.config.keep_going,
                              end_callback=ob)

            if result == False and t.check == True \
                    and not self.mgr.config.keep_going:
                t.log.info("FAIL, skipping remaining tasks ('keep-going mode' disabled)")
                break


        while 1:
            unfinished = [x for x in self.tasks if x.thread \
                              and x.thread.isAlive() \
                              and x.timeout != 0]

            if not unfinished: break

            for t in unfinished:
                t.log.debug("waiting to finish detached task")

                if self.mgr.abort:
                    t.kill(signal.SIGKILL)

                t.thread.join(1)

        daemons = [x for x in self.tasks if x.thread and x.thread.isAlive() \
                       and x.timeout == 0]
        if daemons:
            Log.debug("-- Killing remaining daemons...")
            for t in daemons:
                Log.debug("sending signal %s to %s" % (t.signal, t.name))
                t.kill()
                t.thread.join(1)

        self.result = all([x.result==OK for x in self.tasks if x.check])


        #-- cleanning
        generated = []

        for t in self.tasks:
            if t.result == NOEXEC: continue
            generated.extend(remove_dups(t.gen))

        if not generated:
            return

        if self.mgr.config.clean:
            clean_generated(generated)
        else:
            self.create_gen_list(generated)


    def create_gen_list(self, filelist):
        gen_fname = os.path.basename(self.fname).replace('.test', '').replace('.', '').strip('_')
        gen_fname = os.path.join(ATHEIST_TMP, '%s_%s.gen' % (gen_fname, os.getpid()))
        gen_fd = open(gen_fname, 'w')
        filelist.append(gen_fname)
        Log.debug("Storing generated files in %s:\n%s" % (gen_fname, filelist))
        gen_fd.writelines([x+'\n' for x in filelist])
        gen_fd.flush()
        gen_fd.close()



#    def status(self):
#        return all([t.result for t in self.tasks if t.check])


class task_mode:
    MAIN,SETUP,TEARDOWN = ['-', 's', 't']


def clean_generated(filelist=None):

    if filelist is None:
        filelist = []
        for fname in glob.glob('/tmp/atheist/*.gen'):
            Log.debug("reading generated files from '%s'" % fname)

            try:
                fd = open(fname)
                filelist.extend([x.strip('\n') for x in fd.readlines()])
            except IOError, e:
                Log.error(e)
                continue
            finally:
                fd.close()

    basedir = compath()

    for f in filelist:
        if not (f.startswith('/tmp') or f.startswith(basedir)):
            Log.warning('Removing files out of /tmp or %s is forbbiden: %s' % (basedir, f))
            continue

        if os.path.isdir(f):
            Log.warning("- removing directory '%s'" % compath(f))
            os.system('rm -r %s' % f)

        elif os.path.isfile(f):
            Log.debug("- removing file '%s'" % compath(f))
            try:
                os.remove(f)
            except OSError, e:
                Log.debug(e)
        else:
            Log.warning("%s is not a file nor directory!" % compath(f))


#def clean_generated(list_fname=None):
#    "remove generated files"
#    basedir = compath()
#
#    if list_fname:
#        files = [list_fname]
#    else:
#        files = glob.glob('/tmp/atheist/*.gen')
#
#    delfiles = []
#    for fname in files:
#        Log.debug("reading generated files from '%s'" % fname)
#
#        try:
#            fd = open(fname)
#            delfiles.extend(fd.readlines())
#            fd.close()
#        except IOError, e:
#            Log.error(e)
#            continue
#
#    #delfiles.extend(files)
#
#    Log.info("removing %s generated files" % len(delfiles))
#
#    for f in delfiles:
#        f = f.strip('\n')
#
#        if not (f.startswith('/tmp') or f.startswith(basedir)):
#            Log.warning('Removing files out of /tmp or %s is forbbiden: %s' % (basedir, f))
#            continue
#
#        if os.path.isdir(f):
#            Log.warning("- removing directory '%s'" % compath(f))
#            os.system('rm -r %s' % f)
#
#        elif os.path.isfile(f):
#            Log.debug("- removing file '%s'" % compath(f))
#            try:
#                os.remove(f)
#            except OSError, e:
#                Log.debug(e)
#        else:
#            Log.warning("%s is not a file nor directory!" % compath(f))


# globals
usercfg = IniParser(ATHEIST_CFG)  # FIXME: debería tenerlo el manager
Log = logging.getLogger('atheist')




LOG_DATEFORMAT = '%d/%m/%y %H:%M:%S'

VERSION = '0.'+'$Date: 2010-04-16 22:15:44 +0200 (Fri, 16 Apr 2010) $'[7:17].replace('-','')

USAGE = '''atheist [options] [file|directory]

atheist is a testing tool.
This is version %s, Copyright (C) 2009,2010 David Villa Alises
atheist comes with ABSOLUTELY NO WARRANTY; This is free software, and you are
welcome to redistribute it under certain conditions; See COPYING for details.''' % VERSION

parser = optparse.OptionParser(usage=USAGE)

parser.add_option('-a', '--task-args', dest='task_args',
                  default='',
                  help='colon-separated options for the tasks')

parser.add_option('-b', '--base-dir', dest='base_dir',
                  default=os.getcwd(),
                  help='set working directory')

parser.add_option('-C', '--clean-only', dest='clean_only',
                  action='store_true',
                  help="execute nothing, only remove generated files")

parser.add_option('--cols', dest='screen_width', default='80',
                  type=int, help="terminal width (in chars)")

parser.add_option('-d', '--describe', dest='describe',
                  action='store_true',
                  help='execute nothing, only describe tasks')

parser.add_option('--dirty', dest='clean',
                  action='store_false', default=True,
                  help="do not remove generated files after task execution")

parser.add_option('-e', '--stderr', dest='stderr',
                   action='store_true',
                   help='print task process stderr')

parser.add_option('-f', '--stdout-on-fail', dest='stdout_on_fail',
                  action='store_true',
                  help='print task output but only if it fail')

parser.add_option('-g', '--gen-template', dest='gen_template',
                  action='store_true',
                  help='generate a taskcase file template with default values')

# FIXME: probar esto
parser.add_option('-i', '--report-detail', default=1, type=int,
             help='Report verbosity (0:nothing, [1:case], 2:task, 3:composite, 4:condition)')

parser.add_option('-j', '--skip-setup', dest='skip_setup',
                  action='store_true',
                  help='skip _setup and _teardown files')

parser.add_option('-k', '--keep-going', dest='keep_going',
                  action='store_true', default=False,
                  help='continue even with failed tasks')

parser.add_option('-l', '--list', dest='list_only',
                  action='store_true',
                  help='list tasks but do not execute them')

parser.add_option('--log', default=None,
                  help="Log to specified filename")

parser.add_option('--notify-jabber', dest='jabber',
                  action='append', default=[],
                  help='notify failed tasks to the given jabber account')

parser.add_option('--notify-smtp', dest='mail',
                  action='append', default=[],
                  help='notify failed taks to the given email address')

parser.add_option('-o', '--stdout', dest='stdout',
                  action='store_true',
                  help='print task stdout')

# FIXME: probar esto
parser.add_option('-p', '--plugin_dir', dest='pluginpath',
                  action='append', default=[],
                  help='a directory containing plugins')

parser.add_option('-q', '--quiet', dest='quiet',
                  action='store_true',
                  help='do not show result summary nor warnings, only totals')

parser.add_option('-r', '--random', dest='random', type='int',
                  help='run taskcases in random order using the specified seed')

parser.add_option('-s', '--script', dest='inline',
                  action='append', default=[],
                  help='specifies command line script')

parser.add_option('-t', '--time-tag', dest='timetag',
                  action='store_true',
                  help='include time info in the logs')

parser.add_option('-u', '--until-failure', dest='until_failure',
                  action='store_true',
                  help='Repeat test until it fails')

parser.add_option('-v', '--verbose', dest='verbosity',
                  action='count', default=0,
                  help='incresse verbosity')

parser.add_option('-w', '--workers', default=1, type=int,
                  help='number of simultaneous tasks (deafult:1) 0:auto-select.')

parser.add_option('-x', '--extension', dest='ext',
                  default='.test',
                  help='file extension for the task files')


#FIXME: -k y -u son incompatibles
