# -*- mode: python; coding: utf-8 -*-

"""logging handlers

.. moduleauthor:: Arco Research Group

"""
import string
import types
import logging

# Minimal modificaciton from the standard SMTPHandler from python2.6
# FIXME: inherit from logging.handlers.SMTPHandler (when python2.6 be default)
class SMTPHandler(logging.Handler):
    """
    A handler class which sends an SMTP email for each logging event.
    """
    def __init__(self, mailhost, fromaddr, toaddrs,
                 subject, credentials=None):
        """
        Initialize the handler.

        Initialize the instance with the from and to addresses and subject
        line of the email. To specify a non-standard SMTP port, use the
        (host, port) tuple format for the mailhost argument. To specify
        authentication credentials, supply a (username, password) tuple
        for the credentials argument.
        """
        logging.Handler.__init__(self)
        if type(mailhost) == types.TupleType:
            self.mailhost, self.mailport = mailhost
        else:
            self.mailhost, self.mailport = mailhost, None
        if type(credentials) == types.TupleType:
            self.username, self.password = credentials
        else:
            self.username = None
        self.fromaddr = fromaddr
        if type(toaddrs) == types.StringType:
            toaddrs = [toaddrs]
        self.toaddrs = toaddrs
        self.subject = subject

    def getSubject(self, record):
        """
        Determine the subject for the email.

        If you want to specify a subject line which is record-dependent,
        override this method.
        """
        return self.subject

    weekdayname = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun']

    monthname = [None,
                 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
                 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']

    def date_time(self):
        """
        Return the current date and time formatted for a MIME header.
        Needed for Python 1.5.2 (no email package available)
        """
        year, month, day, hh, mm, ss, wd, y, z = time.gmtime(time.time())
        s = "%s, %02d %3s %4d %02d:%02d:%02d GMT" % (
                self.weekdayname[wd],
                day, self.monthname[month], year,
                hh, mm, ss)
        return s

    def emit(self, record):
        """
        Emit a record.

        Format the record and send it to the specified addressees.
        """
        try:
            import smtplib
            try:
                from email.utils import formatdate
            except ImportError:
                formatdate = self.date_time
            port = self.mailport
            if not port: port = smtplib.SMTP_PORT

            logging.debug("SMTP Connecting: %s:%s" % (self.mailhost, port))

            smtp = smtplib.SMTP(self.mailhost, port)
            msg = self.format(record)
            msg = "From: %s\r\nTo: %s\r\nSubject: %s\r\nDate: %s\r\n\r\n%s" % (
                            self.fromaddr,
                            string.join(self.toaddrs, ","),
                            self.getSubject(record),
                            formatdate(), msg)

            smtp.ehlo()
            smtp.starttls()
            smtp.ehlo()

            if self.username:
                smtp.login(self.username, self.password)
            smtp.sendmail(self.fromaddr, self.toaddrs, msg.encode('utf-8'))
            smtp.close()
        except (KeyboardInterrupt, SystemExit):
            raise
        except:
            self.handleError(record)



class JabberHandler(logging.Handler):
     def __init__(self, account, to_id, log=None, lazy=True):
        ''' log: is a creator logger '''

        self.from_id, self.password = account
        self.to_id = to_id
        self.log = log if log is not None else logging.getLogger("JabberHandler")
        self.connected = False

        logging.Handler.__init__(self)

        if not lazy:
            self.connect()


     def connect(self):
         if self.connected: return

         self.xmpp = __import__('xmpp')

         jid = self.xmpp.protocol.JID(self.from_id)
         self.user = jid.getNode()
         self.server = jid.getDomain()

         self.log.debug("Connecting %s@%s" % (self.user, self.server))

         self.conn = self.xmpp.Client(self.server, debug=[])
         conres = self.conn.connect()

         if not conres:
             self.log.error("Unable to connect to server %s!" % self.server)
             return

         if conres != 'tls':
             self.log.warning("Unable to estabilish secure connection - TLS failed!")

         authres = self.conn.auth(self.user, self.password, resource=self.server)

         if not authres:
             self.log.error("Unable to authorize on %s - check login/password." % self.server)
             return

         if authres != 'sasl':
             self.log.warning("Unable to perform SASL auth os %s. Old authentication method used!" % self.server)

         self.conn.sendInitPresence(requestRoster=0)
         self.connected = True

     def emit(self, record):
         if not self.connected:
             self.connect()

#        try:
         mid = self.conn.send(\
             self.xmpp.protocol.Message(to   = self.to_id,
                                  body = self.format(record)))
#        except:
#            self.handleError(record)

class NotifyHandler(logging.Handler):
    'A logging Handler to show using notification daemon'
    ICONS = {
        logging.CRITICAL: 'gtk-cancel',
        logging.ERROR:    'gtk-dialog-error',
        logging.WARNING:  'gtk-dialog-warning',
        logging.INFO:     'gtk-dialog-info',
        logging.DEBUG:    'gtk-execute',
        }

    def emit(self, record):
        import pynotify
        try:
            icon = NotifyHandler.ICONS[record.levelno]

        except KeyError:
            icon = 'gtk-dialog-question'

        n = pynotify.Notification(record.name, message=record.getMessage(),
                                  icon=icon)
        n.show()

class ColorFormatter(logging.Formatter):
    "A formatter wich add support on logging for colors."

    codes = {\
        None:       (0, 0,   0), # default
        'DEBUG':    (0, 0,   2), # gray
        'INFO':     (0, 0,   0), # normal
        'WARNING':  (38,5, 208), # orange
        'ERROR':    (0, 1,  31), # red
        'CRITICAL': (0, 1, 101), # black with red background
        }

    def color(self, level=None):
        return (chr(27)+'[%d;%d;%dm') % self.codes[level]


    def format(self, record):
        retval = logging.Formatter.format(self, record)
        return self.color(record.levelname) + retval + self.color()
