# -*- mode: python; coding: utf-8 -*-

"""
Command line User Interface utilities

.. moduleauthor:: Arco Research Group
"""

import sys, os, string
import time
import logging

from pyarco.Conio import *
from pyarco.Pattern import Singleton

Log = logging.getLogger('pyarco.UI')

class ProgressBar:
    def __init__(self, max_val=100, width=80, label='', disable=False):
        self.max = max_val+1     # total item amount
        self.label = label       # prefix for the bar
        self.width = min(100, width - 34)  # bar width

        disable = disable or max_val<4
        if disable:
            self.__disable()

        self.alfa = 0.2
        self.logger = None

        self.reset()

    def reset(self):
        self.val = 0
        self.blk = 0
        self.pct = 0

        self.tinit = time.time()
        self.tlast = self.tinit
        self.eta = '~'
        self.lasteta = 0
        self.inc()

    def __disable(self):
        self._render = lambda x:None
        self.clean = lambda:None
        self.inc =  lambda:None
        self.abort = lambda:None


    def inc(self, val=1, cad=''):
        self.val += val
        if self.val > self.max:
            self.reset()

        self.blk = self.val * self.width / self.max
        self.pct = self.val * 100 / self.max
        current = time.time()
        if self.val > 3 and (current - self.lasteta > 1):
            per_item = (1-self.alfa) * ((current - self.tinit) / self.val) \
                + self.alfa * (current-self.tlast)
            remain = self.max  - self.val
            self.eta = "%ss" % (1 + int(1.1 * remain * per_item))
            self.lasteta = current
        self._render(cad)
        self.tlast = current


    def _render(self, ustr):
        cad = ':%4s [ %s' % (self.label[:4], self.blk * '#')
        cad += (self.width - self.blk) * '-'
        cad += ' ] %s/%s (%3d%%) %s %s\r' % \
            (str(self.val).rjust(len(str(self.max))),
             self.max, self.pct, self.eta,
             ellipsis(ustr))

        sys.__stdout__.write(cad)
        sys.__stdout__.flush()


    def clean(self):
        """clean the whole bar from screen"""
        print (' ' * (self.width + 30 + len(self.label))) + '\r',
        sys.__stdout__.flush()


    def listen_logger(self, logger, level):
        '''
        Register a monitor handler in the given logger.
        If it is invoqued, progress bar is aborted.
        '''
        class DummyHandler(logging.Handler):
            def __init__(self, pb, level):
                self._pb = pb
                self.level = level

            def handle(self, *args):
                self._pb.abort()

        if self.logger is None:
            self.logger = logger
            logger.addHandler(DummyHandler(self, level))


    def abort(self):
        print "progress-bar canceled by logging events."
        self.__disable()


def ellipsis(param, width=40, just=False, char=u'…'):
    """
    >>> ellipsis("this is a sentence", width=10, char='_')
    u'this is a_'
    >>> ellipsis("this is a ", width=12, just=True, char='_')
    u'this is a   '
    """
    width = int(width)

    if not param: param = ''
    cad = str(param).strip()
    first_line = cad.split('\n')[0][:width-1]
    #retval = unicode(first_line, 'utf-8')
    retval = first_line.decode('utf-8')
    if first_line != cad: retval += char
    if just: retval = retval.ljust(width)
    return retval


def high(val):
    """
    >>> high('foo')
    '\\x1b[1mfoo\\x1b[m'
    """
    return cout(HIGH, str(val), NORM)


def cout(*args):
    if not hasattr(cout, 'color') or not cout.color:
        args = [x for x in args if x[0] != ESC]

    return "%s" % str.join('', args)


def cout_config(color=None):
    if color is not None:
        cout.color = color
        return

    term = os.getenv('TERM')
    cterm = os.getenv('COLORTERM')
    cout.color = (cterm != '' or 'color' in term or 'xterm' == term)
    Log.debug("UI: Color out set to: %s" % cout.color)
