/* NAS layer of libaudiooss. Large portions stolen from the XMMS NAS plugin,
   which has the credits listed below. Adapted by Jon Trulson, further
   modified by Erik Inge Bols. Largely rewritten afterwards by
   Tobias Diedrich. Modifications during 2000-2002. */

/*\  XMMS - Cross-platform multimedia player
|*|  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas,
|*|                           Thomas Nilsson and 4Front Technologies
|*|
|*|  Network Audio System driver by Willem Monsuwe (willem@stack.nl)
\*/

/* buffer handling:
 * 
 * The buffer is split into a server part and a client part (1:1 split).
 * The fragment size used by the server seems to be 4096 because
 * even if I go lower than that the Events num_bytes fields are
 * always multiples of 4096 (at least on my system)
 *
 * It will still work with lower values but there is no gain from
 * that.
 * 
 * The LOW_WATERMARK value is set to one fragment below the
 * buffer size. This should be the highest possible watermark value.
 * 
 * The ElementNotify events num_bytes value tells us how much
 * bytes we have to refill. If we always refill that much this
 * seems to guarantee the next event will be a LOW_WATERMARK
 * event again.
 **/

#define DSP_DEBUG 0

#include "nasaudio.h"

#define FRAG_SIZE 4096
#define FRAG_COUNT 8
#define STARTUP_THRESHOLD	FRAG_SIZE * 4
#define MIN_BUFFER_SIZE FRAG_SIZE * FRAG_COUNT

int bytes_written = 0;

static AuServer  *aud = NULL;
static AuFlowID   flow;
static AuDeviceID dev;
static pthread_mutex_t nas_mutex = PTHREAD_MUTEX_INITIALIZER;

static int do_pause = 0, paused = 0, close = 0;
static int bps;
static struct timeval last_tv;
static unsigned char format;
static int int_rate;
static int int_nch;
static unsigned char format;

static int frag_size = FRAG_SIZE;
static char *client_buffer = NULL;
static int client_buffer_size = 0;
static int client_buffer_used = 0;
static int server_buffer_size = MIN_BUFFER_SIZE;
static int server_buffer_used = 0;
static pthread_mutex_t buffer_mutex = PTHREAD_MUTEX_INITIALIZER; 

static void check_event_queue();
static void wait_for_event();
static int buffer_get_size();
static int buffer_get_used();
static void buffer_resize(int newlen);
static void writeBuffer(void *data, int len);
static int readBuffer(int num);
static AuDeviceID find_device(int nch);
static AuBool event_handler(AuServer* aud, AuEvent* ev, AuEventHandlerRec* hnd);
static AuBool error_handler(AuServer* aud, AuErrorEvent* ev);
  
static void
print_error(char *prefix, AuStatus as)
{
	char s[100];
	AuGetErrorText(aud, as, s, 100);
	fprintf(stderr, "libaudiooss: %s returned status %d (%s)\n",
		prefix, as, s);
	fflush(stderr);
}

static void
check_event_queue()
{
	AuEvent ev;
	AuBool result;

	do {
		result = AuScanForTypedEvent(aud, AuEventsQueuedAfterFlush,
				AuTrue, AuEventTypeElementNotify, &ev);
		if (result == AuTrue)
			AuDispatchEvent(aud, &ev);
	} while (result == AuTrue);
}

static void
wait_for_event()
{
	AuEvent ev;

	AuNextEvent(aud, AuTrue, &ev);
	AuDispatchEvent(aud, &ev);
}

static int
buffer_get_size()
{
	if (client_buffer_size == 0)
		buffer_resize(MIN_BUFFER_SIZE);

	return client_buffer_size + server_buffer_size;
}

static int
buffer_get_used()
{
	return client_buffer_used + server_buffer_used;
}

void
stopflow(void)
{
	AuStatus as;

	AuStopFlow(aud, flow, &as);
	if (as != AuSuccess)
		print_error("nas_close: AuStopFlow", as);

	client_buffer_used = 0;
	server_buffer_used = 0;
	flow = 0;
}

int
startflow(void)
{
	AuStatus as;
	AuElement elms[3];
	int bytes_per_sample = MAX(1, int_nch * AuSizeofFormat(format));
	int frag_samples = frag_size / bytes_per_sample;

	if ((dev = find_device(int_nch)) == AuNone) {
		fprintf(stderr, "libaudiooss: find_device failed in startflow\n");
		return 0;
	}

	flow = AuCreateFlow(aud, &as);
	if (as != AuSuccess) {
		print_error("startflow: AuCreateFlow", as);
		return 0;
	}
	if (!flow) {
		fprintf(stderr, "libaudiooss: startflow: flow==NULL!\n");
		return 0;
	}

	AuMakeElementImportClient(elms, int_rate, format, int_nch, AuTrue,
				frag_samples * FRAG_COUNT, 
				frag_samples * (FRAG_COUNT - 1), 0, NULL);
	AuMakeElementExportDevice(elms+1, 0, dev, int_rate,
				AuUnlimitedSamples, 0, NULL);
	AuSetElements(aud, flow, AuTrue, 2, elms, &as);
	if (as != AuSuccess) {
		print_error("nas_open: AuSetElements", as);
		return 0;
	}
	AuRegisterEventHandler(aud, AuEventHandlerIDMask |
				    AuEventHandlerTypeMask,
			       AuEventTypeElementNotify,
			       flow, event_handler,
			       (AuPointer) NULL);
	AuSetErrorHandler(aud, error_handler);

	gettimeofday(&last_tv, 0);
	paused = do_pause = 0;
	close = 0;
	AuStartFlow(aud, flow, &as);
	if (as != AuSuccess) {
		print_error("nas_open: AuStartFlow", as);
		return 0;
	}
	AuSync(aud, AuTrue);

	return 1;
}

void
update_bps(void)
{
	bps = int_rate * int_nch * AuSizeofFormat(format);
	if (flow) {
		stopflow();
		AuSync(aud, AuFalse);
	}
}

int
nas_frag_count()
{
	int frag_count;
	
	pthread_mutex_lock(&buffer_mutex);
	frag_count = buffer_get_size() / frag_size;
	pthread_mutex_unlock(&buffer_mutex);
	
	return frag_count;
}

int
nas_frag_size()
{
	return frag_size;
}

int
nas_free_frags()
{
	int free_frags;

	nas_mutex_lock();
	check_event_queue();

	pthread_mutex_lock(&buffer_mutex);
	free_frags = (buffer_get_size() - buffer_get_used()) / frag_size;
	pthread_mutex_unlock(&buffer_mutex);

	DPRINTF("nas_free_frags(): %d\n", free_frags);

	nas_mutex_unlock();
	return free_frags;
}

static void
writeBuffer(void *data, int len)
{
	pthread_mutex_lock(&buffer_mutex);
	if (len > client_buffer_size)
		buffer_resize(len);
	
	DPRINTF("writeBuffer(): len=%d client=%d/%d\n",
			len, client_buffer_used, client_buffer_size);

	memcpy(client_buffer + client_buffer_used, data, len);
	client_buffer_used += len;
	pthread_mutex_unlock(&buffer_mutex);

	if ((server_buffer_used < server_buffer_size) &&
	    ((server_buffer_used != 0) ||
	     (client_buffer_used > STARTUP_THRESHOLD))) {
		AuSync(aud, AuFalse);
		check_event_queue();
		readBuffer(server_buffer_size - server_buffer_used);
	}
}

static int
readBuffer(int num)
{
	AuStatus as;

	pthread_mutex_lock(&buffer_mutex);
	if (!client_buffer) 
		buffer_resize(MIN_BUFFER_SIZE);

	DPRINTF("readBuffer(): num=%d client=%d/%d server=%d/%d\n",
			num,
			client_buffer_used, client_buffer_size,
			server_buffer_used, server_buffer_size);

	if (client_buffer_used < num)
		num = client_buffer_used;

	if (client_buffer_used == 0) {
		pthread_mutex_unlock(&buffer_mutex);
		return 0;
	}

	AuWriteElement(aud, flow, 0, num, client_buffer, AuFalse, &as);
	if (as == AuSuccess) {
		client_buffer_used -= num;
		server_buffer_used += num;
		gettimeofday(&last_tv, 0);
		memmove(client_buffer, client_buffer + num, client_buffer_used);
		bytes_written += num;
	} else {
		print_error("readBuffer: AuWriteElement", as);
		num = 0;
	}
	AuFlush(aud);

	pthread_mutex_unlock(&buffer_mutex);

	return num;
}

int
nas_write(char *ptr, int len)
{
	int writelen = 0;
	int numwritten = 0;
	if (!aud) { errno = EINVAL; return -1; }

	DPRINTF("nas_write(): len=%d client=%d/%d server=%d/%d\n",
			len, client_buffer_used, client_buffer_size,
			server_buffer_used, server_buffer_size);
	nas_mutex_lock();
	
	if (client_buffer_size == 0)
		buffer_resize(MIN_BUFFER_SIZE);
	
	if (!flow)
		startflow();

	if (paused) {
		nas_mutex_unlock();
		return 0;
	}

	while (numwritten < len) {
		while (client_buffer_size == client_buffer_used) 
			wait_for_event();
		if ((len - numwritten) > (client_buffer_size - client_buffer_used))
			writelen = client_buffer_size - client_buffer_used;
		else writelen = (len - numwritten);
		writeBuffer(ptr + numwritten, writelen);
		numwritten += writelen;
	}

	AuSync(aud, AuFalse);
	nas_mutex_unlock();
	return numwritten;
}

void
nas_reset(void)
{
	DPRINTF("nas_reset()\n");

	nas_mutex_lock();
	if (flow) {
		close = 1;
		stopflow();
	}
	AuSync(aud, AuTrue);
	nas_mutex_unlock();
}

void
nas_close(void)
{
	DPRINTF("nas_close()\n");
	if (!aud) return;

	nas_mutex_lock();
	if (flow) {
		close = 1;
		while (buffer_get_used() > 0) {
			readBuffer(server_buffer_size - server_buffer_used);
			wait_for_event();
		}
		stopflow();
	}
	
	AuCloseServer(aud);
	aud = 0;
	if (client_buffer) {
		free(client_buffer);
		client_buffer = NULL;
	}
	client_buffer_size = 0;
	nas_mutex_unlock();
}

static AuDeviceID
find_device(int nch)
{
	int i;
	for (i = 0; i < AuServerNumDevices(aud); i++) {
		if ((AuDeviceKind(AuServerDevice(aud, i)) ==
				AuComponentKindPhysicalOutput) &&
			AuDeviceNumTracks(AuServerDevice(aud, i)) == nch) {
			return AuDeviceIdentifier(AuServerDevice(aud, i));
		}
	}
	return AuNone;
}

static AuBool
event_handler(AuServer* aud, AuEvent* ev, AuEventHandlerRec* hnd)
{
	switch (ev->type) {
	case AuEventTypeElementNotify: {
		AuElementNotifyEvent* event = (AuElementNotifyEvent *)ev;

		DPRINTF("event_handler(): kind %s state %s->%s reason %s numbytes %ld\n",
			nas_elementnotify_kind(event->kind),
			nas_state(event->prev_state),
			nas_state(event->cur_state),
			nas_reason(event->reason),
			event->num_bytes);

		server_buffer_used -= event->num_bytes;
		if (server_buffer_used < 0)
			server_buffer_used = 0;

		switch (event->kind) {
		case AuElementNotifyKindLowWater:
			readBuffer(event->num_bytes);
			break;
		case AuElementNotifyKindState:
			if ((event->cur_state == AuStatePause) &&
			    (event->reason == AuReasonUnderrun))
				/* buffer underrun -> refill buffer 
				   we may have data in flight on its way
				   to the server, so do not write a full
				   buffer, only what the server asks for */
				
				if (!close) /* don't complain if we expect it */
					fprintf(stderr,
					        "audiooss: buffer underrun\n");
				readBuffer(event->num_bytes);
			break;
		}
		break;
	}
	default:
		fprintf(stderr,
			"audiooss: event_handler: unhandled event type %d\n",
			ev->type);
		fflush(stderr);
		break;
	}
	return AuTrue;
}

static AuBool
error_handler(AuServer* aud, AuErrorEvent* ev)
{
	char s[100];
	AuGetErrorText(aud, ev->error_code, s, 100);
	fprintf(stderr,"libaudiooss: error [%s]\n"
		"error_code: %d\n"
		"request_code: %d\n"
		"minor_code: %d\n", 
		s,
		ev->error_code,
		ev->request_code,
		ev->minor_code);
	fflush(stderr);
	
	return AuTrue;
}

int
nas_getdelay(AFormat fmt, int rate, int nch)
{
	static struct timeval now_tv;
	static int temp, temp2;
	int retval = 0;

	nas_mutex_lock();
	check_event_queue();
	
	gettimeofday(&now_tv, 0);
	temp = now_tv.tv_sec - last_tv.tv_sec;
	temp *= bps;

	temp2 = now_tv.tv_usec - last_tv.tv_usec;
	temp2 /= 1000;
	temp2 *= bps;
	temp2 /= 1000;
	temp += temp2;

	retval = buffer_get_used() - temp; 

	DPRINTF("nas_getdelay(): %d\n", retval);

	if (retval < 0) retval = 0;
	AuSync(aud, AuFalse);
	nas_mutex_unlock();

	return (retval);
}

static void
buffer_resize(int newlen)
{
	void *newbuf = malloc(client_buffer_size = MAX(newlen, MIN_BUFFER_SIZE));
	void *oldbuf = client_buffer;
	
	memcpy(newbuf, oldbuf, client_buffer_used);
	client_buffer = newbuf;
	if (oldbuf != NULL)
		free(oldbuf);
}

int
nas_set_volume (int volume)
{
    AuDeviceAttributes attr;
    AuStatus status;
    AuDeviceAttributes *d;

    if (!aud) return -1;
    nas_mutex_lock();

    if ((dev = find_device(int_nch)) == AuNone) {
	fprintf(stderr, "libaudiooss: find_device failed in nas_set_volume\n");
	return -1;
    }

    d = AuGetDeviceAttributes (aud, dev, &status);
    if (status != AuSuccess) {
	print_error("nas_set_volume: AuGetDeviceAttributes", status);
	nas_mutex_unlock();
	return -1;
    }

    if (!d) {
	    nas_mutex_unlock();
	    return -1;
    }
    if (!(AuDeviceValueMask(d) & AuCompDeviceGainMask)) {
	    AuFreeDeviceAttributes(aud, 1, d);
	    nas_mutex_unlock();
	    return -1;
    }

    attr.device.gain = AuFixedPointFromSum (volume, 0);
    AuSetDeviceAttributes (aud, dev, AuCompDeviceGainMask, &attr, &status);
    if (status != AuSuccess) {
	AuFreeDeviceAttributes(aud, 1, d);
	print_error("nas_set_volume: AuSetDeviceAttributes", status);
	nas_mutex_unlock();
	return -1;
    }
    AuFreeDeviceAttributes(aud, 1, d);
    AuSync(aud, AuFalse);
    nas_mutex_unlock();
    return 0;
}

int
nas_get_volume ()
{
    AuStatus status;
    AuDeviceAttributes *d;
    int retval = -1;

    if (!aud) return -1;
    nas_mutex_lock();

    if ((dev = find_device(int_nch)) == AuNone) {
	fprintf(stderr, "libaudiooss: find_device failed in nas_get_volume\n");
	return -1;
    }

    d = AuGetDeviceAttributes(aud, dev, &status);
    if (status != AuSuccess) {
	print_error("nas_get_volume: AuGetDeviceAttributes", status);
    }
	
    if (!d) {
	    nas_mutex_unlock();
	    return -1;
    }
    if (!(AuDeviceValueMask(d) & AuCompDeviceGainMask)) {
	    AuFreeDeviceAttributes (aud, 1, d);
	    nas_mutex_unlock();
	    return -1;
    }
    retval = AuFixedPointRoundDown(AuDeviceGain(d));

    AuFreeDeviceAttributes (aud, 1, d);
    nas_mutex_unlock();
    return retval;
}

void
nas_set_format(AFormat fmt)
{
	if (format == fmt) return;
	nas_mutex_lock();

	format = fmt;
	update_bps();

	nas_mutex_unlock();
}

void
nas_set_rate(int rate)
{
	if (int_rate == rate) return;
	nas_mutex_lock();

	int_rate = rate;
	update_bps();

	nas_mutex_unlock();
}

void
nas_set_nch(int nch)
{
	if (int_nch == nch) return;
	nas_mutex_lock();

	int_nch = nch;
	update_bps();

	nas_mutex_unlock();
}

int
nas_open(AFormat fmt, int rate, int nch)
{
	DPRINTF("nas_open(fmt %d, rate %d, nch %d)\n", fmt, rate, nch);
	nas_mutex_lock();

	format = fmt;
	int_rate = rate;
	int_nch = nch;
	update_bps();

	if (!(aud = AuOpenServer(NULL, 0, NULL, 0, NULL, NULL))) {
		fprintf(stderr, "libaudiooss: could not open nas audio server\n");
		nas_mutex_unlock();
		return 0;
	};

	nas_mutex_unlock();
	return 1;
}
