/*
 *  playlist.c
 *
 *  Copyright 2009 Arnaud Soyez <weboide@codealpha.net>
 *
 *  This file is part of AudioPreview.
 *
 *  AudioPreview is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  AudioPreview is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with AudioPreview.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n-lib.h>
#include "playlist.h"
#include "functions.h"
#include "stream.h"
#include "ap_config.h"

/**
 * Add a path to the playlist.
 */
gboolean playlist_add (const gchar* path)
{
  Stream *st;
  
  st = stream_new ( g_file_new_for_commandline_arg (path) );
  if (st == NULL)
    return FALSE;
  
  g_debug ("%s: Adding: %s", __FUNCTION__, path);
  ap_config.playlist = g_list_prepend (ap_config.playlist, st);
  return TRUE;
}

/**
 * Shuffle the playlist.
 */
void playlist_shuffle ()
{
  ap_config.playlist = g_list_shuffle (ap_config.playlist);
}

/**
 * Free the playlist and its content.
 */
void playlist_free ()
{
  g_list_foreach (ap_config.playlist, (GFunc) stream_free, NULL);
  g_list_foreach (ap_config.playlist, (GFunc) g_free, NULL);
  g_list_free (ap_config.playlist);
}

/**
 * Load a playlist file (path).
 */
gboolean playlist_add_from_file (const gchar* path)
{
  GError *err = NULL;
  gchar *buffer = NULL;
  
  g_debug ("%s: called with: %s", __FUNCTION__, path);
  
  GIOChannel* channel = g_io_channel_new_file (path, "r", &err);
  
  if (!channel || err)
  {
    g_debug ("%s: g_io_channel_new_file FAILED.", __FUNCTION__);
    if (err)
    {
      g_warning ("Unable to read playlist file: %s", err->message);
      g_error_free (err);
    }
    return FALSE;
  }
  
  g_debug ("Reading file: %s", path);
  
  err = NULL;
  while (g_io_channel_read_line (channel, &buffer, NULL, NULL, &err))
  {
    if (err != NULL)
    {
      g_debug ("%s: g_io_channel_read_line FAILED.", __FUNCTION__);
      g_critical ("Error while reading file: %s", err->message);
      g_error_free (err);
      break;
    }
    
    if (buffer == NULL)
    {
      break;
    }
    g_strstrip (buffer);
    g_debug ("%s: buffer=%s.", __FUNCTION__, buffer);
    playlist_add (buffer);
    g_free (buffer);
  }
  
  g_io_channel_unref (channel);
  return TRUE;
}
