/*
    This file is part of audtty, copyright 2008-2009 Chris Taylor.

    audtty is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    audtty is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with audtty; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <audacious/audctrl.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>
#include <ncursesw/curses.h>
#include <sys/stat.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <unistd.h>
#include "main.h"
#include "dbus.h"
#include "settings.h"
#include "kbd_constants.h"
#include "curses_printf.h"
#include "browse.h"

win browser;
controller cont;

void get_contents( void );
void sort( void );
void swap(gchar** tmp1, gchar** tmp2);
void browser_paint( void );
gboolean check_dir( void );
void add_file(gboolean dir);
void remove_win( void );
void display_error(gchar *message, gchar *name, gboolean type);


void file_browser(gint height)
{
	cont.height=height;
	gint c;
        cont.files=(files*)malloc(sizeof(files)*2000);
	cont.location=g_get_current_dir();
        g_chdir(cont.location);
        cont.selector=0;
	cont.length = 1;
	cont.first=0;
	cont.pos_height=0;
	clear();
	refresh();

	browser.location = newwin(1, 0, 0, 0);
	browser.title = newwin(1, 0, 1, 0);
	browser.list = newwin(cont.height, 0, 3, 0);


	wcolor_set(browser.location, 1, NULL);
	wcolor_set(browser.title, 1, NULL);
	wcolor_set(browser.list, 1, NULL);

	mvwtitledhline(browser.title, 0, "File Browser");
	wnoutrefresh(browser.title);
	wnoutrefresh(browser.list);

	doupdate();


	get_contents();
	browser_paint();

	while((c=getch()))
	{
		switch(c)
		{
			case 'q':
			case 'k':
			case ESCAPE:
				remove_win();
				return;
			case KEY_DOWN:
				if( cont.selector < (cont.length - 1) )
				{
					++(cont.selector);
					browser_paint();
				}
				break;
			case KEY_UP:
				if ( cont.selector > 0 )
				{
					--(cont.selector);
					browser_paint();
				}
				break;
			case KEY_NPAGE:
				cont.selector+=cont.height;
				cont.pos_height=0;
				if(cont.selector>=cont.length)
					cont.selector = cont.length-1;
				cont.first=cont.selector;
				browser_paint();
				break;
			case KEY_PPAGE:
				cont.selector-=cont.height;
				cont.pos_height=0;
				if(cont.selector<0)
					cont.selector = 0;
				cont.first=cont.selector;
				browser_paint();
				break;
			case KEY_HOME:
				cont.pos_height=0;
				cont.selector=0;
				cont.first=0;
				browser_paint();
				break;
			case KEY_END:
				cont.pos_height=0;
				cont.selector=cont.length-1;
				cont.first=cont.selector;
				browser_paint();
				break;
			case 'a':
			case 'A':
				add_file(TRUE);
				break;
			case ENTER:
				add_file(FALSE);
				get_contents();
				browser_paint();
				break;
			default:
				break;
		}
	}
	return;
}

void get_contents( void )
{
        gint count=2;
        gint i=0;
        gchar *name;
        GDir *dir;
	GError *open_error=NULL;
        GError *convert_error=NULL;
        
	cont.first=0;
	cont.pos_height=0;

	werase(browser.location);
	doupdate();
	waddstr(browser.location, "Current Directory: ");
	waddstr(browser.location, g_get_current_dir());
	wnoutrefresh(browser.location);

        for(i=0;i<=cont.length;i++)
        {
            cont.files[i].file=NULL;
        }

	dir = g_dir_open(cont.location,0,&open_error);
        if(!dir)
        {
            display_error(open_error->message, cont.location, true);
            return;
        }
        else
        {
            cont.files[0].file=".";
            cont.files[1].file="..";
            while((name=(gchar*)g_dir_read_name(dir)))
            {
                    cont.files[count].file=g_filename_to_utf8(name,-1,NULL,NULL,&convert_error);
                    if(convert_error)
                    {
                        display_error(convert_error->message, name, false);
                        convert_error=NULL;
                    }
                    count++;
            }
            cont.length=count;
            sort();
            g_dir_close(dir);
        }
        browser_paint();
        g_free(name);
        if(open_error)
            g_error_free(open_error);
        if(convert_error)
            g_error_free(convert_error);
}

void sort( void )
{
	gint i, j;
	gint min;
	for (i = 0; i <= cont.length; i++)
	{
		min = i;
		for (j = i + 1; j < cont.length; j++)
		{
			if(cont.files[min].file==NULL||cont.files[j].file==NULL)
				continue;
                        else if (g_strcmp0(cont.files[min].file,cont.files[j].file) > 0)
			{
					min = j;
			}
		}
		swap(&cont.files[i].file, &cont.files[min].file);
	}
}

void swap(gchar** tmp1, gchar** tmp2)
{
	gchar *tmp;

	tmp = *tmp1;
	*tmp1 = *tmp2;
	*tmp2 = tmp;
}

gboolean check_dir( void )
{
	struct stat statbuf;
	gchar *temp;

	temp = g_strjoin("/",cont.location,cont.files[cont.selector].file,NULL);
	stat(temp,&statbuf);

	if(S_ISDIR(statbuf.st_mode)) {
		g_free(temp);
		return TRUE;
	}
	else {
		g_free(temp);
		return FALSE;
	}
}

void add_file(gboolean dir)
{
	if(dir)
	{
		if(g_strcmp0(cont.files[cont.selector].file,".")==0)
			flash();
		else
		{
			gchar *temp;
			gchar *uri = NULL;

			temp = g_strjoin("/",cont.location,cont.files[cont.selector].file,NULL);

			uri = g_filename_to_uri(temp, NULL, NULL);
			audacious_remote_playlist_add_url_string(dbus_proxy,uri);
			g_free(temp);
                        g_free(uri);
                        return;
		}
	}
	else
	{
		if(g_strcmp0(cont.files[cont.selector].file,".")==0)
			flash();
                else if(g_strcmp0(cont.files[cont.selector].file,"..")==0)
		{
			g_chdir("..");
                        cont.location=g_get_current_dir();
                        cont.selector=1;
			get_contents();
		}
		else if(check_dir())
		{
			cont.location = g_strjoin("/",cont.location,cont.files[cont.selector].file,NULL);

			g_chdir(cont.location);
			cont.selector=1;
			get_contents();
		}
		else
		{
			gchar *temp;
			gchar *uri = NULL;

			temp = g_strjoin("/",cont.location,cont.files[cont.selector].file,NULL);

			uri = g_filename_to_uri(temp, NULL, NULL);
			audacious_remote_playlist_add_url_string(dbus_proxy,uri);
			g_free(temp);
                        g_free(uri);
		}
	}
}

void remove_win( void )
{
        g_free(cont.files);
	delwin(browser.title);
	delwin(browser.list);
	delwin(browser.location);
	doupdate();
	mainwin_repaint();
}

void browser_paint( void )
{
	gint i,last;
	last=cont.first+(cont.height-1);

	if(cont.selector<cont.prev_pos)
	{
		--cont.pos_height;
		if(cont.selector<cont.first)
		{
			cont.pos_height=0;
			--cont.first;
			if(cont.first<0)
				cont.first=0;
		}
	}
	else if(cont.selector>cont.prev_pos)
	{
		++cont.pos_height;
		if(cont.selector>last)
		{
			++cont.first;
			cont.pos_height=cont.height;
		}
	}
	for(i=0;i<cont.height;i++)
	{
                gint show_file=cont.first+i;
		if(cont.files[show_file].file!=NULL)
		{
                    mvwaddstr(browser.list,i,0,cont.files[show_file].file);
                    wclrtoeol(browser.list);

                    if(i==0)
                        wclrtobot(browser.list);

                    if (show_file == cont.selector)
                        mvwchgat(browser.list, i, 0, -1, A_REVERSE, 2, NULL);
                    else
			mvwchgat(browser.list, i, 0, -1, A_NORMAL, 1, NULL);
		}
	}
	cont.prev_pos=cont.selector;
	wnoutrefresh(browser.list);
	doupdate();
}

void display_error(gchar *message, gchar *name, gboolean type)
{
    gint c,i;
    werase(browser.list);
    for(i=0;i<(cont.height/2)-3;i++)
        waddstr(browser.list,"\n");
    waddstr(browser.list,message);
    waddstr(browser.list,"\n");
    waddstr(browser.list,"\n");
    if(type)
    {
        waddstr(browser.list,"Failed to open directory: ");
        waddstr(browser.list,name);
    }
    else
    {
        waddstr(browser.list,"Failed to convert filename: ");
        waddstr(browser.list,name);
    }
    waddstr(browser.list,"\n");
    waddstr(browser.list,"\n");
    if(type)
            waddstr(browser.list,"Hit enter twice to move up one directory.");
    else
            waddstr(browser.list,"Hit enter twice to continue.");
    wnoutrefresh(browser.list);
    doupdate();
    while((c=getch()))
    {
        switch(c)
        {
                    case ENTER:
                           if(type)
                            {
                                g_chdir("..");
                                cont.location=g_get_current_dir();
                            }
                            werase(browser.list);
                            wnoutrefresh(browser.list);
                            doupdate();
                            return;
        }
    }
    return;
}