#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <glib.h>
#include "main.h"
#include "settings.h"

/*
    This file is part of xmms-curses, copyright 2003-2005 Knut Auvor Grythe.

    xmms-curses is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    xmms-curses is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with xmms-curses; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

extern prefs settings;

static void usage(void) 
{
	fprintf(stderr, "Usage: %s [ options ]\n\n", PACKAGE);
	fprintf(stderr, "OPTIONS:\n"
			"  To negate a short option, use '+' instead of '-' (e.g. +a)\n"
			"  Long-options are negated with 'no-', (e.g. --no-read-all)\n\n"
			"  -n, --numbers  Show numbers in playlist (default on)\n"
			"  -t, --title    Read the title from ID3/ogg-tags (SLOW) (default on)\n"
			"  -l, --length   Show length of track in playlist (SLOW) (default on)\n"
			"  -a, --read-all Read all titles and lengths on startup (EXTREMELY SLOW)\n"
			"                 (default off)\n"
			"  -V, --version  Display version information\n"
			"  -h, --help     Print this message\n"
		   	"\n");
	fprintf(stderr, "Tip: Things going slow? Try %s +tl\n\n", PACKAGE);
}

static void unknown_option(char *option)
{
	fprintf(stderr, "ERROR: Unrecognized option: '%s'\n\n", option);
	usage();
	exit(EXIT_FAILURE);
}

static void version(void)
{
	printf("%s version %s,\n%s\n\n",
			PACKAGE, VERSION, COPYRIGHT);
	printf("This program is released under the GNU Public License.\n"
			"See the COPYING file for details.\n");
	exit(EXIT_SUCCESS);
}

void set_defaults(void) {
	settings.show_numbers = true;
	settings.read_all = false;
	settings.read_title = true;
	settings.read_length = true;
}

static void shortoptions(char cmd, int state)
{
	char *tmp;

	switch (cmd) {
		case 'n': // Show numbers in playlist
			settings.show_numbers = state;
			break;
		case 't': // Read title from ID3/ogg-tags
			settings.read_title = state;
			break;
		case 'l': // Read song length
			settings.read_length = state;
			break;
		case 'a': // Read all tags on startup
			settings.read_all = state;
			break;
		case 'V': // Show version information
			if (state) version();
			break;
		case 'h': // Help
			if (state) {
				usage();
				exit(EXIT_SUCCESS);
			}
			break;
		default:
			tmp = (char*)malloc(sizeof(char)*3);
			sprintf(tmp, "-%c", cmd);
			unknown_option(tmp);
	}
}

static void longoptions(char *input)
{
	int state = true;
	if (strncmp(input, "no-", 3) == 0) {
		state = false;
		input += 3;
	}

	if (strcmp(input, "numbers") == 0) {
		shortoptions('n', state);
	} else if (strcmp(input, "title") == 0) {
		shortoptions('t', state);
	} else if (strcmp(input, "length") == 0) {
		shortoptions('l', state);
	} else if (strcmp(input, "read-all") == 0) {
		shortoptions('a', state);
	} else if (strcmp(input, "version") == 0) {
		shortoptions('V', state);
	} else if (strcmp(input, "help") == 0) {
		shortoptions('h', state);
	} else {
		unknown_option(input);
	}
}

static void parse_line(char *file, char *line)
{
	char *end = strchr(line, '#');
	if (end == NULL) 
		end = line+strlen(line)-1;
	else
		*end-- = '\0';
	while (*end == ' ' || *end == '\t' || *end == '\n') *end-- = '\0';
	while (*line == ' ' || *line == '\t') ++line;
	if (*line == '\0') return;
	longoptions(line);
}

static void read_file(char *file)
{
	struct stat buf;
	if (stat(file, &buf) == -1) {
		if(errno != ENOENT) {
			printf("Error %d while reading %s: %s\n", 
					errno, file, strerror(errno));
			exit(1);
		}
		return;
	}
	
	FILE *fd = fopen(file, "r");
	if (fd != NULL) {
		char *buffer = (char*) malloc(sizeof(char) * CONFIG_MAX_LINE_LENGTH);
		while (fgets(buffer, CONFIG_MAX_LINE_LENGTH, fd) != NULL) {
			parse_line(file, buffer);
		}
	} else {
		fprintf(stderr, "Warning: Error %d while opening %s: %s\n", 
				errno, file, strerror(errno));
	}
}

void read_conffiles(void)
{
	char *homedir = getenv("HOME");
	
	read_file("/etc/audtty.conf");
	
	if (homedir != NULL) {
		int length = strlen(homedir) + strlen("/.audtty") + 1;
		char *homefile = malloc(sizeof(char) * length);
		snprintf(homefile, length, "%s/.audtty", homedir);
		read_file(homefile);
		free(homefile);
	}
}

void arguments(int argc, char *argv[])
{
	int i;
	for (i=1; i<argc; i++) {
		int state = true;
		char *input = argv[i];
		
		// Short options
		if ((*input == '-' || *input == '+') && *(input+1) != '-') {
			char *pos = input;
			state = *pos++ == '-';
			while (*pos != '\0')
				shortoptions(*pos++, state);
		}
		
		// Long options
		else if (strncmp(input, "--", 2) == 0) {
			input += 2;
			longoptions(input);
		} else {
			unknown_option(input);
		}
	}
}

