from __future__ import print_function

import subprocess, os, tempfile, time

class Container:
    def __init__(self, name):
        if not os.getuid() == 0:
            raise Exception("You must be root")

        self.lxc_name = name

    def load_config(self, alt_file = None):
        raise NotImplemented("Not implemented")

    def save_config(self, alt_file = None):
        raise NotImplemented("Not implemented")

    def set_config_item(self, key, value):
        raise NotImplemented("Not implemented")

    def get_config_item(self, key):
        raise NotImplemented("Not implemented")

    def read_cgroups(self):
        raise NotImplemented("Not implemented")

    def apply_cgroups(self):
        raise NotImplemented("Not implemented")

    def create(self, template, args):
        if self.is_defined():
            self.stop()
            subprocess.call(["lxc-destroy", "-n", self.lxc_name], universal_newlines=True)

        template_args = []
        for item in args.items():
            template_args.append("--%s" % item[0])
            template_args.append("%s" % item[1])

        subprocess.call(['lxc-create', '-t', template, '-n', self.lxc_name, '--'] + template_args, universal_newlines=True)

    def is_defined(self):
        if os.path.exists('/var/lib/lxc/%s' % self.lxc_name):
            return True
        return False

    def is_running(self):
        if not self.is_defined():
            print("container doesn't exist")
            return False

        subp = subprocess.Popen(['lxc-info', '-s', '-n', self.lxc_name], stdout=subprocess.PIPE, universal_newlines=True)
        subp.wait()
        status = subp.stdout.read().strip()
        if status.endswith('STOPPED'):
            return False
        return True

    def get_ips(self, timeout = 60):
        if not self.is_defined():
            print("container doesn't exist")
            return False

        if not self.is_running():
            print("container isn't running")
            return True

        subp = subprocess.Popen(['lxc-info', '-p', '-n', self.lxc_name], stdout=subprocess.PIPE, universal_newlines=True)
        subp.wait()
        pid = subp.stdout.read().split()[-1]
        if not pid.isdigit() and int(pid) > 1:
            return False

        try:
            os.makedirs("/run/netns")
        except:
            pass

        path = tempfile.mktemp(dir="/run/netns")

        os.symlink("/proc/%s/ns/net" % pid, path)

        ips = []

        count = 0
        while not ips and count < timeout:
            # Get IPv6
            ip = subprocess.Popen(["ip", "netns", "exec", path.split("/")[-1], "ip", "-6", "addr", "show", "scope", "global"], stdout=subprocess.PIPE, universal_newlines=True)
            ip.wait()
            for line in ip.stdout.read().split("\n"):
                fields = line.split()
                if len(fields) > 2 and fields[0] == "inet":
                    ips.append(fields[1].split('/')[0])

            # Get IPv4
            ip = subprocess.Popen(["ip", "netns", "exec", path.split("/")[-1], "ip", "-4", "addr", "show", "scope", "global"], stdout=subprocess.PIPE, universal_newlines=True)
            ip.wait()
            for line in ip.stdout.read().split("\n"):
                fields = line.split()
                if len(fields) > 2 and fields[0] == "inet":
                    ips.append(fields[1].split('/')[0])

            time.sleep(1)
            count+=1

        os.remove(path)
        return ips

    def destroy(self):
        if not self.is_defined():
            print("container doesn't exist")
            return False

        self.stop()
        subprocess.call(["lxc-destroy", "-n", self.lxc_name], universal_newlines=True)
        return True

    def stop(self):
        if not self.is_defined():
            print("container doesn't exist")
            return False

        if not self.is_running():
            print("already stopped")
            return True

        subprocess.call(["lxc-stop", "-n", self.lxc_name], universal_newlines=True)
        subprocess.call(["lxc-wait", "-n", self.lxc_name, "-s", "STOPPED"], universal_newlines=True)
        return True

    def start(self):
        if not self.is_defined():
            print("container doesn't exist")
            return False

        if self.is_running():
            print("already running")
            return True

        subprocess.call(["lxc-start", "-n", self.lxc_name, "-d"], universal_newlines=True)
        subprocess.call(["lxc-wait", "-n", self.lxc_name, "-s", "RUNNING"], universal_newlines=True)
        return True

    def restart(self):
        if not self.is_defined():
            print("container doesn't exist")
            return False

        self.stop()
        self.start()
        return True

    def run_command(self, *args):
        raise NotImplemented("Not implemented")

if __name__ == "__main__":
    my_container = Container("test-precise-i386")
    print(my_container.create("ubuntu", {"arch": "i386", "release": "precise"}))
    print(my_container.start())
    print(my_container.get_ips())
    print(my_container.stop())
    print(my_container.destroy())
