/*--------------------------------------------------------------------------*
 * AUTOPROFILE                                                              *
 *                                                                          *
 * A Gaim away message and profile manager that supports dynamic text       *
 *                                                                          *
 * AutoProfile is the legal property of its developers.  Please refer to    *
 * the COPYRIGHT file distributed with this source distribution.            *
 *                                                                          *
 * This program is free software; you can redistribute it and/or modify     *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation; either version 2 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA *
 *--------------------------------------------------------------------------*/

#include "autoprofile.h"

#include "gtkimhtml.h"
#include "gtkimhtmltoolbar.h"
#include "gtkdialogs.h"

static GtkWidget *msg_window = NULL;
static GtkWidget *msgtext = NULL;
static GtkWidget *away_view;
static GtkWidget *added_text;
static GtkListStore *away_list;

struct edit_message {
	GtkWidget *window;
	GtkWidget *toolbar;
	GtkWidget *text;
  char *orig_title;
  char *orig_text;
};

static void destroy_dialog (GtkWidget *w, GtkWidget *w2)
{
  GtkWidget *dest;

  if (!GTK_IS_WIDGET(w2))
    dest = w;
  else
    dest = w2;

  gtk_widget_destroy(dest);
}

/* --------------------"Away" menu ------------------------------------*/

/* Executes the away-ness */
static void use_away (GtkButton *button, gpointer window)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  char *title;
  int i;
  GList *cur_title, *start_node;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(away_view));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, 0, &title, -1);

    cur_title = gaim_prefs_get_string_list (
      "/plugins/gtk/autoprofile/message_titles");
    start_node = cur_title;

    i = 0;

    /* Search for the associated message */
    while (TRUE) {
      if (cur_title == NULL) {
        free_string_list (start_node);
        return;
      }
      if (!strcmp (title, (char *)cur_title->data)) break;
      cur_title = cur_title->next;
      i++;
    }
    free_string_list (start_node);

    gaim_prefs_set_int ("/plugins/gtk/autoprofile/current_away", i);
    gaim_prefs_set_string ("/plugins/gtk/autoprofile/added_text",
      gtk_entry_get_text (GTK_ENTRY(added_text)));
    status_away (); 
    destroy_dialog ((GtkWidget *)window, NULL);
  }
}


/* Displays the dialog */
static void go_away_menu (GaimPluginAction *action)
{
  GtkWidget *vbox, *hbox;
  GtkWidget *window;
  GtkWidget *sw;
  GtkWidget *button;

  GtkTreeSelection *selection;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *col;
  GtkTreeIter iter, default_iter;

  gboolean default_found;

  GList *cur_node, *start_node;
  int i;

  char *title;
  const char *isdefault;

  /* Set up window */
  GAIM_DIALOG(window);
  gtk_widget_set_size_request(window, -1, 250);
  gtk_container_set_border_width(GTK_CONTAINER(window), 6);
  gtk_window_set_role(GTK_WINDOW(window), "away_mess_list");
  gtk_window_set_title(GTK_WINDOW(window), _("Choose an away message"));
  g_signal_connect(G_OBJECT(window), "delete_event",
    G_CALLBACK(destroy_dialog), window);

  vbox = gtk_vbox_new (FALSE, 6);
  gtk_container_add (GTK_CONTAINER(window), vbox);

  /* List of away messages */
  sw = gtk_scrolled_window_new (0, 0);
  gtk_box_pack_start (GTK_BOX(vbox), sw, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
                                  GTK_POLICY_NEVER,
                                  GTK_POLICY_NEVER);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
    GTK_SHADOW_IN);

  away_list = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  away_view = gtk_tree_view_new_with_model (
    GTK_TREE_MODEL (away_list));
  gtk_container_add (GTK_CONTAINER (sw), away_view);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW(away_view), FALSE);
  selection = gtk_tree_view_get_selection (
    GTK_TREE_VIEW (away_view));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

  col = gtk_tree_view_column_new_with_attributes (
    _("Away message"), renderer, "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (away_view), col);
  col = gtk_tree_view_column_new_with_attributes (
    _("Default"), renderer, "text", 1, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (away_view), col);

  i = 0;
  cur_node = gaim_prefs_get_string_list (
    "/plugins/gtk/autoprofile/message_titles");
  start_node = cur_node;
  default_found = FALSE;
  while (cur_node) {
    title = (char *)cur_node->data;
    isdefault = _("");
 
    if (i == gaim_prefs_get_int (
        "/plugins/gtk/autoprofile/default_away")) {
      isdefault = _("(Default)");
    }

    gtk_list_store_append (away_list, &iter);
    gtk_list_store_set (away_list, &iter,
                        0, title,
                        1, isdefault, -1);
    if (i == gaim_prefs_get_int (
        "/plugins/gtk/autoprofile/default_away")) {
      default_iter = iter;
      default_found = TRUE;
    }

    cur_node = cur_node->next;
    i++;
  }
  free_string_list (start_node);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(hbox), gtk_label_new (_("%a:")),
    FALSE, FALSE, 0);
  added_text = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX(hbox), added_text, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  button = gtk_button_new_with_label(_("Use"));
  g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(use_away), window);
  gtk_box_pack_end(GTK_BOX(hbox), button, TRUE, TRUE, 0);

  button = gtk_button_new_with_label(_("Cancel"));
  g_signal_connect(G_OBJECT(button), "clicked", 
    G_CALLBACK(destroy_dialog), window);
  gtk_box_pack_end(GTK_BOX(hbox), button, TRUE, TRUE, 0);

  /* Shake it, baby! */
  gtk_widget_show_all (window);

  gtk_tree_selection_unselect_all (selection);

  if (default_found) {
    gtk_tree_selection_select_iter (selection, &default_iter);
  }
}

/* --------------------"Set user info" box ----------------------------*/

static void mess_destroy (GtkWidget *widget, struct edit_message *ca)
{
  destroy_dialog(NULL, ca->window);
  g_free (ca->orig_title);
  g_free (ca->orig_text);
  g_free (ca);
}

static void mess_destroy_ca (GtkWidget *widget, 
  GdkEvent *event, struct edit_message *ca)
{
  mess_destroy(NULL, ca);
}

static void save_mess(GtkWidget *widget, struct edit_message *ca)
{
  gchar *new_message, *old_title;
  GList *cur_title, *titles_start, *cur_message, *messages_start;
  int i;

  new_message = strdup (gtk_imhtml_get_markup (GTK_IMHTML(ca->text)));
  old_title = ca->orig_title;

  cur_title = gaim_prefs_get_string_list (
    "/plugins/gtk/autoprofile/message_titles");
  titles_start = cur_title;
  cur_message = gaim_prefs_get_string_list (
    "/plugins/gtk/autoprofile/message_texts");
  messages_start = cur_message;

  i = 0;

  /* Search for the associated message */
  while (TRUE) {
    if (cur_title == NULL) {
      auto_debug ("message-editor", "ERROR: message not found");
      gaim_notify_error(NULL, NULL,
        _("Could not find the original message"),
        _("It might have been deleted, please recreate it"));
      mess_destroy(NULL, ca);
    }
  
    if (!strcmp (old_title, (char *)cur_title->data)) {
      break;
    } else {
      cur_title = cur_title->next;
      cur_message = cur_message->next;
      i++;
    }
  }

  /* Update the saved copies */
  cur_message->data = new_message;

  gaim_prefs_set_string_list ("/plugins/gtk/autoprofile/message_texts",
    messages_start);
  free_string_list (messages_start);
  free_string_list (titles_start);

  profile_update (NULL);

  mess_destroy(NULL, ca);
}

static void refresh_view (GtkButton *button, gpointer data)
{
  char *buf;
  struct edit_message *ca = (struct edit_message *)data;

  if (msgtext)
    gtk_container_remove (GTK_CONTAINER(msg_window), msgtext);

  msgtext = gtk_imhtml_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER(msg_window), msgtext);
  gaim_setup_imhtml (msgtext);
  gtk_widget_show (msgtext);
  buf = autoprofile_generate (gtk_imhtml_get_markup (
    GTK_IMHTML(ca->text)));
  gtk_imhtml_append_text (GTK_IMHTML(msgtext), buf, GTK_IMHTML_NO_TITLE |
    GTK_IMHTML_NO_COMMENTS | GTK_IMHTML_NO_SCROLL);
  g_free (buf);
}

/* Display the popup edit dialog */
static void gtk_edit_msg (char *title, char *message)
{
  /* Most code jacked from original Gaim dialog */
  GtkWidget *vbox, *hbox, *big_hbox;
  GtkWidget *label;
  GtkWidget *sw;
  GtkWidget *button;

  gchar *buf;
  GList *cur_comp_node;
  struct component *cur_comp;

  struct edit_message *ca = g_new0(struct edit_message, 1);

  /* Set up window */
  GAIM_DIALOG(ca->window);
  gtk_container_set_border_width(GTK_CONTAINER(ca->window), 6);
  gtk_window_set_role(GTK_WINDOW(ca->window), "away_mess");
  gtk_window_set_title(GTK_WINDOW(ca->window), _("Edit message"));
  g_signal_connect(G_OBJECT(ca->window), "delete_event",
    G_CALLBACK(mess_destroy_ca), ca);

  big_hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER(ca->window), big_hbox);

  /* Main edit window */
  vbox = gtk_vbox_new(FALSE, 6);
  gtk_box_pack_start (GTK_BOX(big_hbox), vbox, TRUE, TRUE, 0);

  /* Toolbar */
  ca->toolbar = gtk_imhtmltoolbar_new();
  gtk_box_pack_start(GTK_BOX(vbox), ca->toolbar, FALSE, FALSE, 0);

	/* Text */
  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw), GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(vbox), sw, TRUE, TRUE, 0);

  ca->text = gtk_imhtml_new(NULL, NULL);
  gtk_imhtml_set_editable(GTK_IMHTML(ca->text), TRUE);

  gtk_imhtmltoolbar_attach(GTK_IMHTMLTOOLBAR(ca->toolbar), ca->text);

  gtk_container_add(GTK_CONTAINER(sw), ca->text);
  gtk_widget_set_size_request(ca->text, -1, 150);

  /* Set the text values */
  gtk_imhtml_append_text_with_images(GTK_IMHTML(ca->text), 
  strdup (message), 0, NULL);
  ca->orig_text = strdup (message);
  ca->orig_title = strdup (title);

  /* Preview window */
  msg_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(msg_window), 
    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(msg_window), 
    GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX(vbox), msg_window, TRUE, TRUE, 0);

  msgtext = gtk_imhtml_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER(msg_window), msgtext);
  gaim_setup_imhtml (msgtext);
  gtk_imhtml_append_text (GTK_IMHTML(msgtext), autoprofile_generate (message),
    GTK_IMHTML_NO_TITLE | GTK_IMHTML_NO_COMMENTS |
    GTK_IMHTML_NO_SCROLL);
  gtk_widget_set_size_request(msgtext, -1, 150);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  button = gaim_pixbuf_button_from_stock(_("Save"), 
    GTK_STOCK_SAVE, GAIM_BUTTON_HORIZONTAL);
  g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(save_mess), ca);
  gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);

  button = gtk_button_new_with_label (_("Refresh view"));
  gtk_box_pack_end (GTK_BOX(hbox), button, FALSE, FALSE, 0);
  g_signal_connect (G_OBJECT(button), "clicked",
                    G_CALLBACK(refresh_view), ca);

  button = gaim_pixbuf_button_from_stock(_("Cancel"), 
    GTK_STOCK_CANCEL, GAIM_BUTTON_HORIZONTAL);
  g_signal_connect(G_OBJECT(button), "clicked", 
    G_CALLBACK(mess_destroy), ca);
  gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);

  label = gtk_label_new (_("Use shift+enter to insert a new line"));
  gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  /* Display component options */
  vbox = gtk_vbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX(big_hbox), vbox, FALSE, FALSE, 0);
  label = gtk_label_new (_("\%\% Percent sign"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);
  label = gtk_label_new (_("\%n Buddy's name (AIM only)"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);
  label = gtk_label_new (_("\%t Current time (AIM only)"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);
  label = gtk_label_new (_("\%d Current date (AIM only)"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);

  cur_comp_node = components;
  while (cur_comp_node) {
    cur_comp = (struct component *)cur_comp_node->data;
    buf = (char *)malloc(101);
    g_snprintf (buf, 100, "%%%c %s", 
      cur_comp->format_char, cur_comp->id);
    label = gtk_label_new (buf);
    gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
    gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);
    cur_comp_node = cur_comp_node->next;
  }

  gtk_widget_show_all(ca->window);
}

/* Gets data for the dialog */
static void edit_profile (GaimPluginAction *action)
{
  GList *cur_title, *cur_message, *titles_start, *messages_start;

  int index = gaim_prefs_get_int ("/plugins/gtk/autoprofile/default_profile");

  if (index >= 0) {
    cur_title = gaim_prefs_get_string_list (
      "/plugins/gtk/autoprofile/message_titles");
    cur_message = gaim_prefs_get_string_list (
      "/plugins/gtk/autoprofile/message_texts");
    titles_start = cur_title;
    messages_start = cur_message;

    /* Search for the associated message */
    while (index-- != 0) {
      cur_title = cur_title->next;
      cur_message = cur_message->next;
    }

    gtk_edit_msg ((char *)cur_title->data, (char *)cur_message->data);
    free_string_list (titles_start);
    free_string_list (messages_start);
  } else {
    gaim_notify_error (NULL, NULL,
      _("No profile currently set"),
      _("To enable this menu select one from the main AutoProfile "
        "preferences menu."));
  }
}

/* Return the actions */
GList *actions (GaimPlugin *plugin, gpointer context)
{
  GaimPluginAction *act;
  GList *list = NULL;

  act = gaim_plugin_action_new (_("Edit User Profile"),
    edit_profile);
  list = g_list_append (list, act);

  act = gaim_plugin_action_new (_("Away"),
    go_away_menu);
  list = g_list_append (list, act);

  return list;
}
