/*
 * resolver.c - resolving stuff
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#include <config.h>

#include "resolver.h"
#include "util.h"
#include "log.h"

/** Set the options that relate to resolving */
int
set_resolver_options(options_t* options, tp_t* trust_point,
	struct ub_ctx* ctx)
{
	int i, retval = 0;

	/* verbosity */
	retval = ub_ctx_set_option(ctx, (char*) "verbosity", (char*) "0");
	retval = ub_ctx_debuglevel(ctx, 0);

	/* resolv.conf */
	if (options->resolvconf)
	{
		verbos(3,"set %s", options->resolvconf);
		retval = ub_ctx_resolvconf(ctx, options->resolvconf);
		if (retval != 0)
			warning("failed to read resolve config '%s': %s",
				options->resolvconf, ub_strerror(retval));
		retval = 0;
        }

	/* unbound.conf */
	if (options->unboundconf)
	{
		verbos(3,"set %s", options->unboundconf);
		retval = ub_ctx_config(ctx, options->unboundconf);
		if (retval != 0)
			warning("failed to read unbound config '%s': %s",
				options->unboundconf, ub_strerror(retval));
		retval = 0;
        }

	/* root hints */
	if (options->roothints)
	{
		verbos(3,"set %s", options->roothints);
		retval = ub_ctx_set_option(ctx, (char*) "root-hints:",
			options->roothints);
		if (retval != 0)
			warning("failed to read root hints '%s': %s",
				options->roothints, ub_strerror(retval));
		retval = 0;
	}

	/* address family */
	if (options->address_family == 1)
		retval = ub_ctx_set_option(ctx, (char*) "do-ip6:", (char*) "no");
	else if (options->address_family == 2)
		retval = ub_ctx_set_option(ctx, (char*) "do-ip4:", (char*) "no");
	if (retval != 0)
		warning("failed to set address family: %s",
			ub_strerror(retval));
	retval = 0;

	/* trust anchors */
	for (i=0; i < (int) trust_point->count; i++)
	{
		char *str =
		  ldns_rr2str(trust_point->anchorlist[i]->rr);

		/* don't use ZSKs for active refresh */
		if (rr_is_zsk(trust_point->anchorlist[i]->rr))
		{
			free(str);
			continue;
		}

		if (trust_point->anchorlist[i]->s == STATE_VALID ||
                    trust_point->anchorlist[i]->s == STATE_MISSING)
		{

			retval = ub_ctx_add_ta(ctx, str);
			if (retval != 0)
			{
				warning("failed to add trust anchor "
					"[%s, DNSKEY, id=%i] for validation: "
					"%s", trust_point->name,
			ldns_calc_keytag(trust_point->anchorlist[i]->rr),
				ub_strerror(retval));
			}
		}

		free(str);
	}

	return STATUS_OK;
}

/** Is rr self-signed revoked key */
static int
rr_is_selfsigned_revoked(ldns_rr* rr, ldns_rr_list* dnskeys,
	ldns_rr_list* rrsigs, int rrsig_count)
{
	int i=0;
	if (!rr_has_flag(rr, LDNS_KEY_REVOKE_KEY))
		return 0;

	for (i=0; i < rrsig_count; i++)
	{
		ldns_rr* rrsig = ldns_rr_list_rr(rrsigs, i);
		if ((ldns_rr_get_type(rrsig) == LDNS_RR_TYPE_RRSIG) &&
			(ldns_verify_rrsig(dnskeys, rrsig, rr) == LDNS_STATUS_OK))
		{
			char* str = ldns_rdf2str(ldns_rr_owner(rr));
			verbos(1, "trust anchor [%s, DNSKEY, id=%i] is "
				"self-signed revoked", str,
				ldns_calc_keytag(rr));
			free(str);
			return 1;
		}
	}
	return 0;
}

/** Set update events */
static void
update_events(struct ub_result* result, tp_t* trust_point)
{
	/* ldns to the rescue! */
	ldns_status lstatus = LDNS_STATUS_OK;
	ldns_pkt* packet = NULL;
	ldns_rr* rrsig_rr = NULL;
	ldns_rdf* rrsig_exp = NULL;
	/* walk through answer list */
	int i = 0, answer_count = 0, rrsig_count = 0;
	/* orig ttl, rrsig exp interval */
	uint32_t orig_ttl = FIFTEEN_DAYS;
	uint32_t rrsig_exp_interval = FIFTEEN_DAYS, rrsig_exp_tmp;

	lstatus = ldns_wire2pkt(&packet, (const uint8_t*) result->answer_packet,
		result->answer_len);
	if (lstatus != LDNS_STATUS_OK)
		error("failed to read DNS packet: %s",
			ldns_get_errorstr_by_id(lstatus));
	else
	{
		ldns_rr_list* answer_list =
			ldns_rr_list_clone(ldns_pkt_answer(packet));
		ldns_rr_list* dnskey_list = ldns_pkt_rr_list_by_type(packet,
			LDNS_RR_TYPE_DNSKEY, LDNS_SECTION_ANSWER);
		ldns_rr_list* rrsig_list = ldns_pkt_rr_list_by_type(packet,
			LDNS_RR_TYPE_RRSIG, LDNS_SECTION_ANSWER);

		answer_count = ldns_rr_list_rr_count(answer_list);
		rrsig_count = ldns_rr_list_rr_count(rrsig_list);
		ldns_pkt_free(packet);

		/* The exp interval of which RRSIG must we take?
		 * This is not defined, so we take the minumum exp interval. */
		for(i=0; i < rrsig_count; i++)
		{
			rrsig_rr = ldns_rr_list_rr(rrsig_list, i);
			rrsig_exp = ldns_rr_rrsig_expiration(rrsig_rr);

			rrsig_exp_tmp = ldns_rdf2native_int32(rrsig_exp) -
				trust_point->last_queried;
			rrsig_exp_interval = min(rrsig_exp_interval, rrsig_exp_tmp);
		}

		for (i=0; i < answer_count; i++)
		{
			ldns_rr* answer_rr = ldns_rr_list_rr(answer_list, i);
			ta_t* anchor = NULL;
			ta_t* answer = create_trustanchor(ldns_rr_clone(answer_rr),
								STATE_START);

			orig_ttl = ldns_rr_ttl(answer_rr);

			if (!rr_is_dnskey_sep(answer_rr))
			{
				free_trustanchor(answer);
				continue;
			}

			/* NewKey? */
			if ((anchor = search_trustanchor(trust_point, answer))
							== NULL)
				anchor = add_trustanchor(trust_point, answer);
			else
				free_trustanchor(answer);

			if (!anchor)
			{
				error("trust anchor [%s, DNSKEY, id=%i] "
					"active refresh failed",
					trust_point->name,
					ldns_calc_keytag(answer_rr));
				continue;
			}

			if (result->secure && !rr_has_flag(answer_rr,
							LDNS_KEY_REVOKE_KEY))
			{
				verbos(1,"trust anchor [%s, DNSKEY, id=%i, ttl=%i] in "
					"DNS response", trust_point->name,
						ldns_calc_keytag(answer_rr), orig_ttl);
				seen_trustanchor(anchor, 1);
			}
			/* a trust anchor might have its revocation bit set; we
			 * must first check if there is an rrsig signed by this
			 * key. If not, we must consider the trust anchor 'not
			 * revoked'.
			 */
			if (rr_is_selfsigned_revoked(answer_rr,
					dnskey_list, rrsig_list, rrsig_count))
				seen_revoked_trustanchor(anchor, 1);
		}

		ldns_rr_list_deep_free(answer_list);
		ldns_rr_list_deep_free(dnskey_list);
		ldns_rr_list_deep_free(rrsig_list);

		trust_point->query_interval = max(ONE_HR, min(FIFTEEN_DAYS,
								min(0.5*orig_ttl, 0.5*rrsig_exp_interval)));
		trust_point->retry_time = max(ONE_HR, min(ONE_DAY,
								min(0.1*orig_ttl, 0.1*rrsig_exp_interval)));
		debug(("orig_ttl for %s: %i",
			trust_point->name, orig_ttl));
		debug(("rrsig_exp_interval for %s: %i",
			trust_point->name, rrsig_exp_interval));
	}
}

/** Query DNSKEYs */
int
query_dnskeys(struct ub_ctx* ctx, tp_t* trust_point)
{
	int retval;
	/* the response */
	struct ub_result* result;

	if (!ctx || !trust_point)
	{
		error("assertion error: !ctx || !trust_point");
		return STATUS_ERR_QUERY;
	}

	/* last queried now */
	trust_point->last_queried = time(NULL);
	if ((retval = ub_resolve(ctx, trust_point->name, LDNS_RR_TYPE_DNSKEY,
		LDNS_RR_CLASS_IN, &result)) != 0)
	{
		error("failed to resolve [DNSKEY for %s]: %s",
			trust_point->name, ub_strerror(retval));
		trust_point->query_failed += 1;
		return STATUS_ERR_QUERY;
	}
	else if (result->secure && !result->bogus)
	{
		verbos(1, "response [DNSKEY for %s] is secure",
			trust_point->name);
		trust_point->query_failed = 0;

		if (!result->havedata)
		{
			if (result->nxdomain)
				warning("response [DNSKEY for %s] is NXDOMAIN",
					trust_point->name);
			else
				warning("response [DNSKEY for %s] has no data",
					trust_point->name);
		}
	}
	else
	{
		trust_point->query_failed += 1;
		warning("response [DNSKEY for %s] is insecure%s",
			trust_point->name, result->bogus ? " (bogus)" : "");
		return STATUS_ERR_QUERY;
	}

	if (result->havedata)
		update_events(result, trust_point);

	ub_resolve_free(result);

	return STATUS_OK;
}

/** Events */
static int
do_newkey(ta_t* anchor)
{
	if (anchor->s == STATE_START)
		set_trustanchor_state(anchor, STATE_ADDPEND);
	return STATUS_OK;
}

static int
do_addtime(ta_t* anchor, options_t* options)
{
	int exceeded = check_holddown(anchor, options->add_holddown);
	if (exceeded && anchor->s == STATE_ADDPEND)
	{
		char *str = ldns_rdf2str(ldns_rr_owner(anchor->rr));
		verbos(1, "trust anchor [%s, DNSKEY, id=%i] add-holddown time "
			  "exceeded %i seconds ago",
			str, ldns_calc_keytag(anchor->rr), exceeded);
		if (anchor->pending_count >= MIN_PENDINGCOUNT)
		{
			set_trustanchor_state(anchor, STATE_VALID);
			free(str);
			return STATUS_CHANGED;
		}

		verbos(1, "trust anchor [%s, DNSKEY, id=%i] add-holddown time "
			  "sanity check failed (pending count: %d)",
			str, ldns_calc_keytag(anchor->rr),
			anchor->pending_count);
		free(str);
	}
	return STATUS_OK;
}

static int
do_remtime(ta_t* anchor, options_t* options)
{
	int exceeded = check_holddown(anchor, options->del_holddown);
	if (exceeded && anchor->s == STATE_REVOKED)
	{
		char *str = ldns_rdf2str(ldns_rr_owner(anchor->rr));
		verbos(1, "trust anchor [%s, DNSKEY, id=%i] del-holddown time "
			  "exceeded %i seconds ago", str,
			ldns_calc_keytag(anchor->rr), exceeded);
		free(str);
		set_trustanchor_state(anchor, STATE_REMOVED);
	}
	return STATUS_OK;
}

static int
do_keyrem(ta_t* anchor)
{
	if (anchor->s == STATE_ADDPEND)
		set_trustanchor_state(anchor, STATE_START);
	else if (anchor->s == STATE_VALID)
		set_trustanchor_state(anchor, STATE_MISSING);
	return STATUS_OK;
}

static int
do_keypres(ta_t* anchor)
{
	if (anchor->s == STATE_MISSING)
		set_trustanchor_state(anchor, STATE_VALID);
	return STATUS_OK;
}

static int
do_revoked(ta_t* anchor)
{
	if (anchor->s == STATE_VALID || anchor->s == STATE_MISSING)
	{
		set_trustanchor_state(anchor, STATE_REVOKED);
		revoke_dnskey(anchor, 0);
		notice("DNSKEY REVOKED: new id is %i",
			ldns_calc_keytag(anchor->rr));
		return STATUS_CHANGED;
	}
	return STATUS_OK;
}

/** Remove missing trust anchors */
static int
remove_missing_trustanchors(tp_t* trust_point, options_t* options)
{
	size_t i;
	int status = STATUS_OK;

	/* if there is no trust point, or keep_missing == 0 (forever),
	 * there is nothing to do.
	 */
	if (!trust_point || !options->keep_missing)
                return status;

	for (i=0; i < (size_t) trust_point->count; i++)
	{
		ta_t* anchor = trust_point->anchorlist[i];
		int exceeded = check_holddown(anchor, options->keep_missing);

		/* Only do KSKs */
		if (!rr_is_dnskey_sep(anchor->rr))
			continue;

		/* Only do MISSING keys */
		if (anchor->s != STATE_MISSING)
			continue;

		/* If keep_missing has exceeded and we still have more than on
			KSK: remove missing trust anchor */
		if (exceeded && trust_point->valid > 0)
		{
			char *str = ldns_rdf2str(ldns_rr_owner(anchor->rr));
			verbos(1, "trust anchor [%s, DNSKEY, id=%i] "
				"keep-missing time exceeded %i seconds ago, "
				"[%i VALID]", str, ldns_calc_keytag(anchor->rr),
				exceeded, trust_point->valid);
			free(str);
			set_trustanchor_state(anchor, STATE_REMOVED);
			return STATUS_CHANGED;
		}
	}

	return status;
}

/** Update trust point */
int
do_statetable(tp_t* trust_point, options_t* options)
{
	size_t i;
	int status = STATUS_OK;
	int ret = status;

	log_assert(!trust_point->query_failed);

	if (!trust_point)
                return status;
	for (i=0; i < (size_t) trust_point->count; i++)
	{
		ta_t* anchor = trust_point->anchorlist[i];

		/* Only do KSKs */
		if (!rr_is_dnskey_sep(anchor->rr))
			continue;

		switch (anchor->s)
                {
			/* START */
			case STATE_START:
				/* NewKey: ADDPEND */
				if (anchor->fetched)
					status = do_newkey(anchor);
				break;
			/* ADDPEND */
			case STATE_ADDPEND:
				/* KeyRem: START */
				if (!anchor->fetched)
					status = do_keyrem(anchor);
				/* AddTime: VALID */
				else
					status = do_addtime(anchor, options);

				if (status == STATUS_CHANGED)
					trust_point->valid ++;
				break;
			/* VALID */
			case STATE_VALID:
				/* RevBit: REVOKED */
				if (anchor->revoked)
					status = do_revoked(anchor);
				/* KeyRem: MISSING */
				else if (!anchor->fetched)
				{
					status = do_keyrem(anchor);
					trust_point->valid --;
					trust_point->missing ++;
				}

				if (status == STATUS_CHANGED)
					trust_point->valid --;
                                break;
			/* MISSING */
			case STATE_MISSING:
				/* RevBit: REVOKED */
				if (anchor->revoked)
					status = do_revoked(anchor);
				/* KeyPres */
				else if (anchor->fetched)
				{
					status = do_keypres(anchor);
					trust_point->missing --;
					trust_point->valid ++;
				}

				if (status == STATUS_CHANGED)
					trust_point->missing --;
				break;
			/* REVOKED */
			case STATE_REVOKED:
				if (anchor->fetched)
					reset_holddown(anchor);
				/* RemTime: REMOVED */
				else
					status = do_remtime(anchor, options);
				break;
			/* REMOVED */
			case STATE_REMOVED:
			default:
				break;
		}

		if (ret == STATUS_OK) /* otherwise, don't update return value */
			ret = status;
	}

	/* setting: keep-missing */
	status = remove_missing_trustanchors(trust_point, options);
	if (ret == STATUS_OK)
		ret = status;

	return ret;
}

/** Signal resolver */
void
signal_resolver(const char* pidfile)
{
	size_t len;
	pid_t resolver_pid = 0;

	if (pidfile && pidfile[0])
	{
		resolver_pid = readpid(pidfile, &len);
		if (resolver_pid > 0)
		{
			debug(("signal resolver %i", resolver_pid));

			if (kill(resolver_pid, SIGHUP) == -1)
				error("failed to signal resolver (pid %i): %s",
					resolver_pid, strerror(errno));
		}
	}
}

/** Reload resolver */
void
reload_resolver(const char* reloadcmd)
{
	if (reloadcmd && reloadcmd[0])
	{
		debug(("execute: %s", reloadcmd));
		if (system(reloadcmd) == -1)
			error("cannot execute: %s: %s", reloadcmd, strerror(errno));
	}
}
