/*
 * trust_anchors.h -library for trust anchors.
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#ifndef TRUST_ANCHORS_H
#define TRUST_ANCHORS_H

#include <config.h>

#include "rbtree.h"

/* Trust anchor states */
typedef enum {
	STATE_START=0,
	STATE_ADDPEND=1,
	STATE_VALID=2,
	STATE_MISSING=3,
	STATE_REVOKED=4,
	STATE_REMOVED=5
} ta_state_t;

/* Default capacity of 8 trust anchors */
#define TA_KEY_CAPACITY_INIT 8

/* Trust anchor */
typedef struct ta_struct ta_t;
struct ta_struct {
	ldns_rr*		rr; 		/* the RR in ldns_rr form */
	ta_state_t		s; 		/* 5011 state */
	time_t			last_change;	/* last update of key */
	uint8_t			pending_count;	/* pending count */
	uint8_t			fetched;	/* fresh TA was seen */
	uint8_t			revoked;	/* revoked TA was seen */
};

/* Trust point, unique by (name, class) */
typedef struct tp_struct tp_t;
struct tp_struct {
	ldns_rdf*		dname;	/* owner name in wire format */
	char*			name;	/* owner name in wire format */
	ldns_rr_class		klass;	/* class */

	const char*		file;	/* file to store the trust point in */

	time_t			last_queried; /* last queried DNSKEY set */
	uint8_t			query_failed; /* how many times did it fail */
	uint32_t		query_interval; /* when to query if !failed */
	uint32_t		retry_time; /* when to retry if failed */

	uint8_t 		valid; /* number of valid DNSKEYs */
	uint8_t 		missing; /* number of missing DNSKEYs */

	/* trust anchors */
	size_t			count;	/* number of trust anchors in set */
	size_t			capacity; /* maximum storage */
	ta_t**			anchorlist;/* list of trust anchors */
};

/* Trust point tree */
typedef struct tp_set_struct tp_set_t;
struct tp_set_struct {
	rbtree_t*		sep;
};

/* Does the dns key have the specified flag:
 * @param rr: The trust anchor (must be a DNSKEY).
 * @param flag: the flags field.
 * @return: 1 on has flag, 0 otherwise.
 */
uint8_t rr_has_flag(ldns_rr* rr, uint16_t flag);

/* Flip a bit in the flag fields:
 * @param ta: The trust anchor (must be a DNSKEY).
 * @param off: off if you want to unrevoke.
 */
void revoke_dnskey(ta_t* ta, int off);

/* Fetch comments from trust anchor (as stored in statefile):
 * @param str: The resource record.
 * @param comment: storage for comment.
 * @param max: max length of the str.
 * @return: 1 on success, 0 otherwise.
 */
int rr_fetch_comments(char* str, char* comment, int max);

/* Parse comments (as stored in statefile):
 * @param ta: The trust anchor.
 * @param str: The comments.
 * @return: 1 on success, 0 otherwise.
 */
int rr_parse_comments(ta_t* ta, char* str);

/* Check if a trust anchor is a SEP DNSKEY:
 * @param rr: The trust anchor.
 * @return: 1 if is DNSKEY SEP, 0 otherwise.
 */
uint8_t rr_is_dnskey_sep(ldns_rr* rr);

/* Check if a trust anchor is a ZSK DNSKEY:
 * @param rr: The trust anchor.
 * @return: 1 if is ZSK, 0 otherwise.
 */
uint8_t rr_is_zsk(ldns_rr* rr);

/* Check if a trust anchor is a DS:
 * @param rr: The trust anchor.
 * @return: 1 if is DS, 0 otherwise.
 */
uint8_t rr_is_ds(ldns_rr* rr);

/* Check if this is a valid trust anchor.
 * @param rr: The trust anchor.
 * @return: 1 if valid, 0 otherwise.
 */
uint8_t rr_is_trust_anchor(ldns_rr* rr);

/* Create a new trust anchor:
 * @param rr: the RR that is used as trust anchor.
 * @param s: the state of the trust anchor.
 * @return: on success, a pointer to the created trust anchor,
 *      on failure (malloc failed), NULL
 */
ta_t* create_trustanchor(ldns_rr* rr, ta_state_t s);

/* Clone a trust anchor:
 * @param orig: the original trust anchor.
 * @return: the clone, NULL if failed to clone.
 */
ta_t* ta_clone(ta_t* orig);

/* Get the human readable trust anchor state:
 * @param s: the state.
 * @return: the string version of state @s.
 */
const char* trustanchor_state2str(ta_state_t s);

/* Check if the holddown time has already exceeded:
 * @param ta: the trust anchor.
 * @param holddown: the holddown timer.
 * @return: 1 if the time is exceeded, 0 else.
 */
int check_holddown(ta_t* ta, unsigned int holddown);

/* If the trust anchor is seen by the resolver, set the fetched value:
 * @param ta: the trust anchor.
 * @param seen: seen value.
 */
void seen_trustanchor(ta_t* ta, int seen);

/* If the trust anchor is seen by the resolver, set the fetched value:
 * @param ta: the trust anchor.
 * @param revoked: revoked value.
 */
void seen_revoked_trustanchor(ta_t* ta, int revoked);

/* Set the state for this trust anchor:
 * @param ta: the trust anchor.
 * @param s: the state.
 */
void set_trustanchor_state(ta_t* ta, ta_state_t s);

/* Get the state of this trust anchor:
 * @param ta: the trust anchor.
 * @return: the state.
 */
ta_state_t get_trustanchor_state(ta_t* ta);

/* Reset the timer by setting lastchange to now:
 * @param ta: the trust anchor.
 */
void reset_holddown(ta_t* ta);

/* Create a trust point:
 * @param tp_dname: owner name of the trust anchor.
 * @param tp_class: length of the domain name.
 * @return: on success, a pointer to the created empty trust point,
 *      on failure (malloc failed), NULL
 */
tp_t* create_trustpoint(char* tp_dname, ldns_rr_class tp_class);

/* Create a trust point set:
 * @return: on success, a pointer to the created empty trust point set,
 *      on failure (malloc failed), NULL
 */
tp_set_t* create_trustpoint_set(void);

/* Check if the trust point contains valid trust anchors:
 * @param tp: the trust point
 * @return: number of valid trust anchors (including missing ones)
 */
int tp_have_valid(tp_t* tp);

/* Add a trust point:
 * @param tp_dname: owner name of the trust anchor.
 * @param tp_class: length of the domain name.
 * @return: on success, a pointer to the created empty trust point,
 *      on failure (malloc failed), NULL
 */
tp_t* tp_add(tp_set_t* tp_set, char* tp_dname, ldns_rr_class tp_class);

/* Get the number of keys in a trust point.
 * @param tp: the trust point.
 * @return: 0, if set==NULL, otherwise the number of keys in set,
 *      as indicated by the count variable.
 */
size_t tp_tacount(tp_t* tp);

/* Get the capacity of a trust point:
 * @param tp: the trust point.
 * @return: 0, if set==NULL, otherwise the capacity of the set,
 *      as indicated by the capacity variable.
 */
size_t tp_tacapacity(tp_t* tp);

/* Add a trust anchor to a trust point:
 * @param tp: the trust point.
 * @param ta: the trust anchor.
 * @return: on failure: NULL, on success: pointer to the the trust anchor.
 */
ta_t* add_trustanchor(tp_t* tp, ta_t* ta);

/* Add a trust anchor in ldns_rr format to a trust point:
 * @param tp_set: the trust point set.
 * @param rr: the trust anchor in ldns_rr format.
 * @param tpt: store the trust point where the trust anchor is added to.
 * @param state: 1 if file is statefile, 0 otherwise.
 * @return: 1 on success, 0 on fail.
 */
ta_t* add_trustanchor_frm_rr(tp_set_t* tp_set, ldns_rr* rr, tp_t** tpt,
	int state);

/* Check if a RR in string format is not empty:
 * @param str: the RR.
 * @param comment: the comment character.
 * @return: 1 on success, 0 on fail.
 */
int str_contains_data(char* str, char comment);

/* Add a trust anchor in string format to a trust point:
 * @param tp_set: the trust point set.
 * @param str: the trust anchor in string format.
 * @param tpt: store the trust point where the trust anchor is added to.
 * @param state: 1 if file is statefile, 0 otherwise.
 * @return: 1 on success, 0 on fail.
 */
ta_t* add_trustanchor_frm_str(tp_set_t* tp_set, char* str, tp_t** tpt,
	int state);

/* Next active refresh:
 * @param tp: the trust point.
 * @return: the next active refresh for tp.
 */
time_t next_refresh(tp_t* tp);

/* [TODO] trust anchor and trust point deletion */

/* Compare two trust anchors:
 * @param a: a trust anchor.
 * @param b: another trust anchor.
 * @return: 0 if equal, -1 if a is "smaller", 1 if a is "bigger".
 */
int ta_compare(const void* a, const void *b);

/* Compare two trust points by (name, class):
 * @param a: a trust point.
 * @param b: another trust point.
 * @return: 0 if equal, -1 if a is "smaller", 1 if a is "bigger".
 */
int tp_compare(const void* a, const void *b);

/* Search a trust anchor in a trust point:
 * @param tp: the trust point.
 * @param ta: the trust anchor.
 * @return: if anchor exists: pointer to the anchor, else: NULL.
 */
ta_t* search_trustanchor(tp_t* tp, ta_t* ta);

/* Search a trust point in the trust point tree:
 * @param tp_set: the trust points.
 * @param rr_dname: the trust point dname.
 * @param rr_class: the trust point class.
 * @return: if trust point exists: pointer to the trust point, else: NULL.
 */
tp_t* tp_search(tp_set_t* tp_set, ldns_rdf* rr_dname,
	ldns_rr_class rr_class);

/* Remove a trust anchor from the memory:
 * @param ta: the trust anchor.
 */
void free_trustanchor(ta_t* ta);

/* Remove a trust point from the memory:
 * @param tp: the trust point.
 */
void tp_free(const void* tp);

/* Remove a trust point set from the memory:
 * @param tp_set: the trust point.
 */
void free_trustpoints(tp_set_t* tp_set);

/* Print a trust point set:
 * @param tp_set: the trust point.
 */
#ifdef TDEBUG
void debug_print_trustpoints(tp_set_t* tp_set);
#endif /* TDEBUG */

#endif /* TRUST_ANCHORS_H */
