/*
    AWFFull - A Webalizer Fork, Full o' features
    
    $Id: preserve.c 250 2006-08-02 07:34:25Z steve $

    Copyright (C) 1997-2001  Bradford L. Barrett (brad@mrunix.net)
    Copyright (C) 2004, 2005, 2006 by Stephen McInerney (spm@stedee.id.au)
    Copyright (C) 2006 by Alexander Lazic (al-awffull@none.at)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

    This software uses the gd graphics library, which is copyright by
    Quest Protein Database Center, Cold Spring Harbor Labs.  Please
    see the documentation supplied with the library for additional
    information and license terms, or visit www.boutell.com/gd/ for the
    most recent version of the library and supporting documentation.
*/

/*********************************************/
/* STANDARD INCLUDES                         */
/*********************************************/

#include "awffull.h"                            /* main header              */

/* local variables */
struct history history_list[MAXHISTLEN];        /* Complete Array for the history */

int hist_month[12], hist_year[12];              /* arrays for monthly total */
u_long hist_hit[12];                            /* calculations: used to    */
u_long hist_files[12];                          /* produce index.html       */
u_long hist_site[12];                           /* these are read and saved */
double hist_xfer[12];                           /* in the history file      */
u_long hist_page[12];
u_long hist_visit[12];

int hist_fday[12], hist_lday[12];               /* first/last day arrays    */

/************************************************
 * cmp_history                                  *
 *   submit to qsort for sorting the history    *
 *   sorts by year & month, oldest first        *
 ************************************************/
int
cmp_history(const void *h1, const void *h2)
{
    struct history *history1 = (struct history *) h1;
    struct history *history2 = (struct history *) h2;
    int cmp_rtn = 0;                            /* Value to return */

    if (history1->year < history2->year) {
        cmp_rtn = -1;
    } else if (history1->year > history2->year) {
        cmp_rtn = 1;
    } else {
        /* Same year, compare for month */
        if (history1->month < history2->month) {
            cmp_rtn = -1;
        } else if (history1->month > history2->month) {
            cmp_rtn = 1;
        }
    }
    return (cmp_rtn);
}


/*********************************************
 * GET_HISTORY - load in history file        *
 *********************************************/

void
get_history()
{
    int i = 0;
    int numfields = 0;                          /* Number of fields returned from each line */

    /*  Used to verify if an older history file */
    int nbr_hist_entries = 0;                   /* How many Lines out of the hitsory file have been read */

    FILE *hist_fp;
    char buffer[BUFSIZE];

    /* first initalize the history array */
    for (i = 0; i < MAXHISTLEN; i++) {
        history_list[i].year = 0;
        history_list[i].month = 0;
        history_list[i].hit = 0;
        history_list[i].file = 0;
        history_list[i].site = 0;
        history_list[i].xfer = 0.0;
        history_list[i].page = 0;
        history_list[i].visit = 0;
        history_list[i].fday = 0;
        history_list[i].lday = 0;
    }

    /* Open History file and Process */
    hist_fp = fopen(hist_fname, "r");
    if (hist_fp) {
        VPRINT(VERBOSE1, "%s %s\n", _("Reading history file..."), hist_fname);
        while ((fgets(buffer, BUFSIZE, hist_fp)) != NULL) {
            /* month# year# requests files sites xfer firstday lastday */
            numfields = sscanf(buffer, "%d %d %lu %lu %lu %lf %d %d %lu %lu",
                               &history_list[nbr_hist_entries].month,
                               &history_list[nbr_hist_entries].year,
                               &history_list[nbr_hist_entries].hit,
                               &history_list[nbr_hist_entries].file,
                               &history_list[nbr_hist_entries].site,
                               &history_list[nbr_hist_entries].xfer, &history_list[nbr_hist_entries].fday, &history_list[nbr_hist_entries].lday,
                               &history_list[nbr_hist_entries].page, &history_list[nbr_hist_entries].visit);
            if (numfields == 8) {               /* kludge for reading 1.20.xx history files */
                history_list[nbr_hist_entries].page = 0;
                history_list[nbr_hist_entries].visit = 0;
            }

            nbr_hist_entries++;
            if (nbr_hist_entries > MAXHISTLEN) {
                ERRVPRINT(VERBOSE1, "%s (mth=%d)\n", _("Error: Ignoring invalid history record"), nbr_hist_entries + 1);
                break;
            }
        }
        fclose(hist_fp);
        /* Sort the history array. We only need to sort those entries we've just received */
        qsort(history_list, nbr_hist_entries, sizeof(struct history), cmp_history);
    } else {
        VPRINT(VERBOSE1, "%s\n", _("History file not found..."));
    }
}


/*********************************************
 * PUT_HISTORY - write out history file      *
 *********************************************/

void
put_history()
{
    int i;
    FILE *hist_fp;

    hist_fp = fopen(hist_fname, "w");

    if (hist_fp) {
        VPRINT(VERBOSE1, "%s\n", _("Saving history information..."));
        for (i = 0; i < MAXHISTLEN; i++) {
            if ((history_list[i].month != 0) && (history_list[i].hit != 0)) {
                fprintf(hist_fp, "%d %d %lu %lu %lu %.0f %d %d %lu %lu\n",
                        history_list[i].month, history_list[i].year, history_list[i].hit, history_list[i].file, history_list[i].site, history_list[i].xfer, history_list[i].fday,
                        history_list[i].lday, history_list[i].page, history_list[i].visit);
            }
        }
        fclose(hist_fp);
    } else {
        ERRVPRINT(VERBOSE1, "%s %s\n", _("Error: Unable to write history file"), hist_fname);
    }
}

/*********************************************/
/* SAVE_STATE - save internal data structs   */
/*********************************************/

int
save_state()
{
    HNODEPTR hptr;
    UNODEPTR uptr;
    RNODEPTR rptr;
    ANODEPTR aptr;
    SNODEPTR sptr;
    INODEPTR iptr;
    ENODEPTR eptr;

    FILE *fp;
    int i;

    char buffer[BUFSIZE];

    /* Open data file for write */
    fp = fopen(state_fname, "w");
    if (fp == NULL)
        return 1;

    /* Saving current run data... */
    sprintf(buffer, "%04d/%02d/%02d %02d:%02d:%02d", g_cur_year, g_cur_month, g_cur_day, g_cur_hour, g_cur_min, g_cur_sec);
    VPRINT(VERBOSE1, "%s [%s]\n", _("Saving current run data..."), buffer);

    /* first, save the easy stuff */
    /* Header record */
    sprintf(buffer, "# AWFFull V%s Incremental Data - %02d/%02d/%04d %02d:%02d:%02d\n", version, g_cur_month, g_cur_day, g_cur_year, g_cur_hour, g_cur_min, g_cur_sec);
    if (fputs(buffer, fp) == EOF)
        return 1;                               /* error exit */

    /* Current date/time          */
    sprintf(buffer, "%d %d %d %d %d %d\n", g_cur_year, g_cur_month, g_cur_day, g_cur_hour, g_cur_min, g_cur_sec);
    if (fputs(buffer, fp) == EOF)
        return 1;                               /* error exit */

    /* Monthly totals for sites, urls, etc... */
    sprintf(buffer, "%lu %lu %lu %lu %lu %lu %llu %lu %lu %lu %lu %lu %lu\n", t_hit, t_file, t_site, t_url, t_ref, t_agent, t_xfer, t_page, t_visit, t_user, t_bookmarks, t_bad,
            t_ignored);
    if (fputs(buffer, fp) == EOF)
        return 1;                               /* error exit */

    /* Daily totals for sites, urls, etc... */
    sprintf(buffer, "%lu %lu %lu %d %d\n", dt_site, ht_hit, mh_hit, f_day, l_day);
    if (fputs(buffer, fp) == EOF)
        return 1;                               /* error exit */

    /* Monthly (by day) total array */
    for (i = 0; i < 31; i++) {
        sprintf(buffer, "%lu %lu %llu %lu %lu %lu %lu\n", tm_hit[i], tm_file[i], tm_xfer[i], tm_site[i], tm_page[i], tm_visit[i], tm_bookm[i]);
        if (fputs(buffer, fp) == EOF)
            return 1;                           /* error exit */
    }

    /* Daily (by hour) total array */
    for (i = 0; i < 24; i++) {
        sprintf(buffer, "%lu %lu %llu %lu %lu\n", th_hit[i], th_file[i], th_xfer[i], th_page[i], th_bookm[i]);
        if (fputs(buffer, fp) == EOF)
            return 1;                           /* error exit */
    }

    /* Response codes */
    for (i = 0; i < TOTAL_RC; i++) {
        sprintf(buffer, "%lu\n", response[i].count);
        if (fputs(buffer, fp) == EOF)
            return 1;                           /* error exit */
    }

    /* now we need to save our linked lists */
    /* URL list */
    if (fputs("# -urls- \n", fp) == EOF)
        return 1;                               /* error exit */
    for (i = 0; i < MAXHASH; i++) {
        uptr = um_htab[i];
        while (uptr != NULL) {
            sprintf(buffer, "%s\n%d %lu %lu %llu %lu %lu %lu %llu\n", uptr->string, uptr->flag, uptr->count, uptr->files, uptr->xfer, uptr->entry, uptr->exit, uptr->pcount,
                    uptr->pxfer);
            if (fputs(buffer, fp) == EOF)
                return 1;
            uptr = uptr->next;
        }
    }
    if (fputs("# End Of Table - urls\n", fp) == EOF)
        return 1;                               /* error exit */

    /* daily hostname list */
    if (fputs("# -sites- (monthly)\n", fp) == EOF)
        return 1;                               /* error exit */

    for (i = 0; i < MAXHASH; i++) {
        hptr = sm_htab[i];
        while (hptr != NULL) {
            sprintf(buffer, "%s\n%d %lu %lu %llu %lu %lu %lu\n%s\n", hptr->string, hptr->flag, hptr->count, hptr->files, hptr->xfer, hptr->visit, hptr->pages, hptr->tstamp,
                    (hptr->lasturl == blank_str) ? "-" : hptr->lasturl);
            if (fputs(buffer, fp) == EOF)
                return 1;                       /* error exit */
            hptr = hptr->next;
        }
    }
    if (fputs("# End Of Table - sites (monthly)\n", fp) == EOF)
        return 1;

    /* hourly hostname list */
    if (fputs("# -sites- (daily)\n", fp) == EOF)
        return 1;                               /* error exit */
    for (i = 0; i < MAXHASH; i++) {
        hptr = sd_htab[i];
        while (hptr != NULL) {
            sprintf(buffer, "%s\n%d %lu %lu %llu %lu %lu\n%s\n", hptr->string, hptr->flag, hptr->count, hptr->files, hptr->xfer, hptr->visit, hptr->tstamp,
                    (hptr->lasturl == blank_str) ? "-" : hptr->lasturl);
            if (fputs(buffer, fp) == EOF)
                return 1;
            hptr = hptr->next;
        }
    }
    if (fputs("# End Of Table - sites (daily)\n", fp) == EOF)
        return 1;

    /* Referrer list */
    if (fputs("# -referrers- \n", fp) == EOF)
        return 1;                               /* error exit */
    if (t_ref != 0) {
        for (i = 0; i < MAXHASH; i++) {
            rptr = rm_htab[i];
            while (rptr != NULL) {
                sprintf(buffer, "%s\n%d %lu\n", rptr->string, rptr->flag, rptr->count);
                if (fputs(buffer, fp) == EOF)
                    return 1;                   /* error exit */
                rptr = rptr->next;
            }
        }
    }
    if (fputs("# End Of Table - referrers\n", fp) == EOF)
        return 1;

    /* User agent list */
    if (fputs("# -agents- \n", fp) == EOF)
        return 1;                               /* error exit */
    if (t_agent != 0) {
        for (i = 0; i < MAXHASH; i++) {
            aptr = am_htab[i];
            while (aptr != NULL) {
                sprintf(buffer, "%s\n%d %lu\n", aptr->string, aptr->flag, aptr->count);
                if (fputs(buffer, fp) == EOF)
                    return 1;                   /* error exit */
                aptr = aptr->next;
            }
        }
    }
    if (fputs("# End Of Table - agents\n", fp) == EOF)
        return 1;

    /* Search String list */
    if (fputs("# -search strings- \n", fp) == EOF)
        return 1;                               /* error exit */
    for (i = 0; i < MAXHASH; i++) {
        sptr = sr_htab[i];
        while (sptr != NULL) {
            sprintf(buffer, "%s\n%lu\n", sptr->string, sptr->count);
            if (fputs(buffer, fp) == EOF)
                return 1;                       /* error exit */
            sptr = sptr->next;
        }
    }
    if (fputs("# End Of Table - search strings\n", fp) == EOF)
        return 1;

    /* username list */
    if (fputs("# -usernames- \n", fp) == EOF)
        return 1;                               /* error exit */

    for (i = 0; i < MAXHASH; i++) {
        iptr = im_htab[i];
        while (iptr != NULL) {
            sprintf(buffer, "%s\n%d %lu %lu %llu %lu %lu\n", iptr->string, iptr->flag, iptr->count, iptr->files, iptr->xfer, iptr->visit, iptr->tstamp);
            if (fputs(buffer, fp) == EOF)
                return 1;                       /* error exit */
            iptr = iptr->next;
        }
    }
    if (fputs("# End Of Table - usernames\n", fp) == EOF)
        return 1;

    /* ErrorPages list */
    if (fputs("# -404errors- \n", fp) == EOF)
        return 1;                               /* error exit */

    for (i = 0; i < MAXHASH; i++) {
        eptr = ep_htab[i];
        while (eptr != NULL) {
            sprintf(buffer, "%s\n%lu\n%s\n", eptr->url, eptr->count, eptr->referer);
            if (fputs(buffer, fp) == EOF)
                return 1;
            eptr = eptr->next;
        }
    }
    if (fputs("# End Of Table - 404errors\n", fp) == EOF)
        return 1;                               /* error exit */

    fclose(fp);                                 /* close data file...                            */
    return 0;                                   /* successful, return with good return code      */
}

/*********************************************/
/* RESTORE_STATE - reload internal run data  */
/*********************************************/

int
restore_state()
{
    FILE *fp;
    int i;
    struct hnode t_hnode;                       /* Temporary hash nodes */
    struct unode t_unode;
    struct rnode t_rnode;
    struct anode t_anode;
    struct snode t_snode;
    struct inode t_inode;
    struct enode t_enode;

    char buffer[BUFSIZE];
    char tmp_buf[BUFSIZE], tmp2_buf[BUFSIZE];

    struct tm time_rec;                         /* Gotta convert that string'ed time into a timerec first */

    u_long ul_bogus = 0;

    unsigned int tmp_version_major = 0;         /* Scan in the version of the Current Run Data file. Use int's as a float scan was not consistant */
    unsigned int tmp_version_minor = 0;
    unsigned int tmp_version_micro = 0;
    unsigned long rundata_version = 0;          /* The webalizer version from the file header */
    int ssout;                                  /* sscanf return result for checking */

    fp = fopen(state_fname, "r");
    if (fp == NULL) {
        /* Previous run data not found... */
        VPRINT(VERBOSE1, "%s\n", _("Previous run data not found..."));
        return 0;                               /* return with ok code */
    }

    /* Reading previous run data... */
    VPRINT(VERBOSE1, "%s %s\n", _("Reading previous run data.."), state_fname);

    /* get easy stuff */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) { /* Header record */
        ssout = sscanf(buffer, "# AWFFull V%u.%u.%u", &tmp_version_major, &tmp_version_minor, &tmp_version_micro);
        if (ssout <= 0) {
            /* Possibly Webalizer format header? */
            ssout = sscanf(buffer, "# Webalizer V%u.%u", &tmp_version_major, &tmp_version_minor);
            if (ssout <= 0) {
                return (1);
            }
        }
        /* Version level = Major x 10,000 + Minor x 100 + Micro
         *      Micro is *always* > 0
         *      Gives us 99 levels in Minor and Micro
         *   3.3.1 = 30301
         */
        if (tmp_version_micro == 0) {           /* Old Webalizer Format, early awffull */
            rundata_version = tmp_version_major * 100 + tmp_version_minor;
        } else {
            rundata_version = tmp_version_major * 10000 + tmp_version_minor * 100 + tmp_version_micro;
        }

        VPRINT(VERBOSE1, "%s %lu\n", _("Previous Run Version:"), rundata_version);
        if (rundata_version < 201) {            /* Check for version compatibility */
            return 99;
        }
        if (rundata_version < 301) {
            ERRVPRINT(VERBOSE0, "%s\n", _("Warning: Changes to Referrals & User Agents count methods. See README!"));
        }
    } /* bad magic? */
    else
        return 1;                               /* error exit */

    /* Get current timestamp */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        memset(&time_rec, 0, sizeof(time_rec));
        strptime(buffer, "%Y %m %d %H %M %S", &time_rec);
        /* calculate current timestamp (seconds since epoch) and setup current time globals */
        time_rec.tm_isdst = -1;                 /* stop mktime from resetting for daylight savings */
        cur_tstamp = mktime(&time_rec);
        VPRINT(VERBOSE1, "%s %lu\n", _("Previous Final Timestamp:"), cur_tstamp);
        g_cur_year = time_rec.tm_year + 1900;
        g_cur_month = time_rec.tm_mon + 1;
        g_cur_day = time_rec.tm_mday;
        g_cur_hour = time_rec.tm_hour;
        g_cur_min = time_rec.tm_min;
        g_cur_sec = time_rec.tm_sec;
    } else
        return 2;                               /* error exit */

    /* Get monthly totals */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        if (rundata_version < 302) {
            sscanf(buffer, "%lu %lu %lu %lu %lu %lu %llu %lu %lu %lu", &t_hit, &t_file, &t_site, &t_url, &t_ref, &t_agent, &t_xfer, &t_page, &t_visit, &t_user);
        } else {
            sscanf(buffer, "%lu %lu %lu %lu %lu %lu %llu %lu %lu %lu %lu %lu %lu", &t_hit, &t_file, &t_site, &t_url, &t_ref, &t_agent, &t_xfer, &t_page, &t_visit, &t_user,
                   &t_bookmarks, &t_bad, &t_ignored);
        }
    } else
        return 3;                               /* error exit */

    /* Get daily totals */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        sscanf(buffer, "%lu %lu %lu %d %d", &dt_site, &ht_hit, &mh_hit, &f_day, &l_day);
    } else
        return 4;                               /* error exit */

    /* get daily totals */
    for (i = 0; i < 31; i++) {
        if ((fgets(buffer, BUFSIZE, fp)) != NULL) {
            if (rundata_version < 302) {
                sscanf(buffer, "%lu %lu %llu %lu %lu %lu", &tm_hit[i], &tm_file[i], &tm_xfer[i], &tm_site[i], &tm_page[i], &tm_visit[i]);
            } else {
                sscanf(buffer, "%lu %lu %llu %lu %lu %lu %lu", &tm_hit[i], &tm_file[i], &tm_xfer[i], &tm_site[i], &tm_page[i], &tm_visit[i], &tm_bookm[i]);
            }
        } else
            return 5;                           /* error exit */
    }

    /* get hourly totals */
    for (i = 0; i < 24; i++) {
        if ((fgets(buffer, BUFSIZE, fp)) != NULL) {
            if (rundata_version < 302) {
                sscanf(buffer, "%lu %lu %llu %lu", &th_hit[i], &th_file[i], &th_xfer[i], &th_page[i]);
            } else {
                sscanf(buffer, "%lu %lu %llu %lu %lu", &th_hit[i], &th_file[i], &th_xfer[i], &th_page[i], &th_bookm[i]);
            }
        } else
            return 6;                           /* error exit */
    }

    /* get response code totals */
    for (i = 0; i < TOTAL_RC; i++) {
        if ((fgets(buffer, BUFSIZE, fp)) != NULL)
            sscanf(buffer, "%lu", &response[i].count);
        else
            return 7;                           /* error exit */
    }

    /* Kludge for V2.01-06 TOTAL_RC off by one bug */
    if (!strncmp(buffer, "# -urls- ", 9))
        response[TOTAL_RC - 1].count = 0;
    else {
        /* now do hash tables */

        /* url table */
        if ((fgets(buffer, BUFSIZE, fp)) != NULL) {     /* Table header */
            if (strncmp(buffer, "# -urls- ", 9))
                return 10;
        } /* (url)        */
        else
            return 10;                          /* error exit */
    }

    while ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        if (!strncmp(buffer, "# End Of Table ", 15))
            break;
        strncpy(tmp_buf, buffer, MAXURLH);
        tmp_buf[strlen(tmp_buf) - 1] = 0;

        if ((fgets(buffer, BUFSIZE, fp)) == NULL)
            return 10;                          /* error exit */
        if (!isdigit((int) buffer[0]))
            return 10;                          /* error exit */

        /* load temporary node data */
        if (rundata_version > 30403) {
            sscanf(buffer, "%d %lu %lu %llu %lu %lu %lu %llu", &t_unode.flag, &t_unode.count, &t_unode.files, &t_unode.xfer, &t_unode.entry, &t_unode.exit, &t_unode.pcount,
                   &t_unode.pxfer);
        } else {
            sscanf(buffer, "%d %lu %lu %llu %lu %lu", &t_unode.flag, &t_unode.count, &t_unode.files, &t_unode.xfer, &t_unode.entry, &t_unode.exit);
        }

        /* Good record, insert into hash table */
        if (put_unode(tmp_buf, t_unode.flag, t_unode.count, t_unode.xfer, &ul_bogus, t_unode.entry, t_unode.exit, um_htab)) {
            /* Error adding URL node, skipping ... */
            ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding URL node, skipping"), t_unode.string);
        }
    }

    /* monthly sites table */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) { /* Table header */
        if (strncmp(buffer, "# -sites- ", 10))
            return 8;
    } /* (monthly)    */
    else
        return 8;                               /* error exit */

    while ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        /* Check for end of table */
        if (!strncmp(buffer, "# End Of Table ", 15))
            break;
        strncpy(tmp_buf, buffer, MAXHOST);
        tmp_buf[strlen(buffer) - 1] = 0;

        if ((fgets(buffer, BUFSIZE, fp)) == NULL)
            return 8;                           /* error exit */
        if (!isdigit((int) buffer[0]))
            return 8;                           /* error exit */

        /* load temporary node data */
        if (rundata_version < 300) {
            sscanf(buffer, "%d %lu %lu %llu %lu %lu", &t_hnode.flag, &t_hnode.count, &t_hnode.files, &t_hnode.xfer, &t_hnode.visit, &t_hnode.tstamp);
            t_hnode.pages = 0;
        } else {                                /* version 3.00, aka 300, included t_hnode.pages in the output */
            sscanf(buffer, "%d %lu %lu %llu %lu %lu %lu", &t_hnode.flag, &t_hnode.count, &t_hnode.files, &t_hnode.xfer, &t_hnode.visit, &t_hnode.pages, &t_hnode.tstamp);
        }

        /* get last url */
        if ((fgets(buffer, BUFSIZE, fp)) == NULL)
            return 8;                           /* error exit */
        if (buffer[0] == '-')
            t_hnode.lasturl = blank_str;
        else {
            buffer[strlen(buffer) - 1] = 0;
            t_hnode.lasturl = find_url(buffer);
        }

        /* Good record, insert into hash table */
        /* FIXME!!! */
        if (put_hnode
            (tmp_buf, t_hnode.flag, t_hnode.count, t_hnode.files, t_hnode.xfer, &ul_bogus, t_hnode.visit + 1, t_hnode.pages, t_hnode.tstamp, t_hnode.lasturl, sm_htab, false)) {
            /* Error adding host node (monthly), skipping .... */
            ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Host node (monthly), skipping"), t_hnode.string);
        }
    }

    /* Daily sites table */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) { /* Table header */
        if (strncmp(buffer, "# -sites- ", 10))
            return 9;
    } /* (daily)      */
    else
        return 9;                               /* error exit */

    while ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        /* Check for end of table */
        if (!strncmp(buffer, "# End Of Table ", 15))
            break;
        strncpy(tmp_buf, buffer, MAXHOST);
        tmp_buf[strlen(buffer) - 1] = 0;

        if ((fgets(buffer, BUFSIZE, fp)) == NULL)
            return 9;                           /* error exit */
        if (!isdigit((int) buffer[0]))
            return 9;                           /* error exit */

        /* load temporary node data */
        sscanf(buffer, "%d %lu %lu %llu %lu %lu", &t_hnode.flag, &t_hnode.count, &t_hnode.files, &t_hnode.xfer, &t_hnode.visit, &t_hnode.tstamp);

        /* get last url */
        if ((fgets(buffer, BUFSIZE, fp)) == NULL)
            return 9;                           /* error exit */
        if (buffer[0] == '-')
            t_hnode.lasturl = blank_str;
        else {
            buffer[strlen(buffer) - 1] = 0;
            t_hnode.lasturl = find_url(buffer);
        }

        /* Good record, insert into hash table */
        /* FIXME!!! */
        if (put_hnode(tmp_buf, t_hnode.flag, t_hnode.count, t_hnode.files, t_hnode.xfer, &ul_bogus, t_hnode.visit + 1, 0, t_hnode.tstamp, t_hnode.lasturl, sd_htab, false)) {
            /* Error adding host node (daily), skipping .... */
            ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Host node (daily), skipping"), t_hnode.string);
        }
    }

    /* Referrers table */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) { /* Table header */
        if (strncmp(buffer, "# -referrers- ", 14))
            return 11;
    } /* (referrers) */
    else
        return 11;                              /* error exit */

    while ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        if (!strncmp(buffer, "# End Of Table ", 15))
            break;
        strncpy(tmp_buf, buffer, MAXREFH);
        tmp_buf[strlen(buffer) - 1] = 0;

        if ((fgets(buffer, BUFSIZE, fp)) == NULL)
            return 11;                          /* error exit */
        if (!isdigit((int) buffer[0]))
            return 11;                          /* error exit */

        /* load temporary node data */
        sscanf(buffer, "%d %lu", &t_rnode.flag, &t_rnode.count);

        /* insert node */
        if (put_rnode(tmp_buf, t_rnode.flag, t_rnode.count, &ul_bogus, rm_htab)) {
            ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Referrer node, skipping"), log_rec.refer);
        }
    }

    /* Agents table */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) { /* Table header */
        if (strncmp(buffer, "# -agents- ", 11))
            return 12;
    } /* (agents) */
    else
        return 12;                              /* error exit */

    while ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        if (!strncmp(buffer, "# End Of Table ", 15))
            break;
        strncpy(tmp_buf, buffer, MAXAGENT);
        tmp_buf[strlen(buffer) - 1] = 0;

        if ((fgets(buffer, BUFSIZE, fp)) == NULL)
            return 12;                          /* error exit */
        if (!isdigit((int) buffer[0]))
            return 12;                          /* error exit */

        /* load temporary node data */
        sscanf(buffer, "%d %lu", &t_anode.flag, &t_anode.count);

        /* insert node */
        if (put_anode(tmp_buf, t_anode.flag, t_anode.count, &ul_bogus, am_htab)) {
            ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding User Agent node, skipping"), log_rec.agent);
        }
    }

    /* Search Strings table */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) { /* Table header */
        if (strncmp(buffer, "# -search string", 16))
            return 13;
    } /* (search) */
    else
        return 13;                              /* error exit */

    while ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        if (!strncmp(buffer, "# End Of Table ", 15))
            break;
        strncpy(tmp_buf, buffer, MAXSRCH);
        tmp_buf[strlen(buffer) - 1] = 0;

        if ((fgets(buffer, BUFSIZE, fp)) == NULL)
            return 13;                          /* error exit */
        if (!isdigit((int) buffer[0]))
            return 13;                          /* error exit */

        /* load temporary node data */
        sscanf(buffer, "%lu", &t_snode.count);

        /* insert node */
        if (put_snode(tmp_buf, t_snode.count, sr_htab)) {
            ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Search String Node, skipping"), t_snode.string);
        }
    }

    /* usernames table */
    if ((fgets(buffer, BUFSIZE, fp)) != NULL) { /* Table header */
        if (strncmp(buffer, "# -usernames- ", 10))
            return 14;
    } else
        return 14;                              /* error exit */

    while ((fgets(buffer, BUFSIZE, fp)) != NULL) {
        /* Check for end of table */
        if (!strncmp(buffer, "# End Of Table ", 15))
            break;
        strncpy(tmp_buf, buffer, MAXIDENT);
        tmp_buf[strlen(buffer) - 1] = 0;

        if ((fgets(buffer, BUFSIZE, fp)) == NULL)
            return 14;                          /* error exit */
        if (!isdigit((int) buffer[0]))
            return 14;                          /* error exit */

        /* load temporary node data */
        sscanf(buffer, "%d %lu %lu %llu %lu %lu", &t_inode.flag, &t_inode.count, &t_inode.files, &t_inode.xfer, &t_inode.visit, &t_inode.tstamp);

        /* Good record, insert into hash table */
        if (put_inode(tmp_buf, t_inode.flag, t_inode.count, t_inode.files, t_inode.xfer, &ul_bogus, t_inode.visit + 1, t_inode.tstamp, im_htab, false)) {
            /* Error adding username node, skipping .... */
            ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Username node, skipping"), t_inode.string);
        }
    }

    if (rundata_version > 302) {
        /* 404 table */
        if ((fgets(buffer, BUFSIZE, fp)) != NULL) {     /* Table header */
            if (strncmp(buffer, "# -404errors- ", 14))
                return 15;
        } else
            return 15;                          /* error exit */

        while ((fgets(buffer, BUFSIZE, fp)) != NULL) {
            if (!strncmp(buffer, "# End Of Table ", 15))
                break;
            strncpy(tmp_buf, buffer, MAXURLH);
            tmp_buf[strlen(tmp_buf) - 1] = 0;

            if ((fgets(buffer, BUFSIZE, fp)) == NULL)
                return 15;                      /* error exit */
            if (!isdigit((int) buffer[0]))
                return 15;                      /* error exit */

            /* load temporary node data */
            sscanf(buffer, "%lu", &t_enode.count);

            if ((fgets(buffer, BUFSIZE, fp)) == NULL)
                return 15;                      /* error exit */

            strncpy(tmp2_buf, buffer, MAXURLH);
            tmp2_buf[strlen(tmp2_buf) - 1] = 0;

            /* Good record, insert into hash table */
            if (put_enode(tmp_buf, tmp2_buf, OBJ_REG, t_enode.count, &t_errorp, ep_htab)) {
                /* Error adding URL node, skipping ... */
                ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding URL node, skipping"), t_unode.string);
            }
        }
    }

    fclose(fp);
    check_dup = 1;                              /* enable duplicate checking */
    return 0;                                   /* return with ok code       */
}
