--------------------------- sal_complex.as ----------------------------------
--
-- Fairly general Complex constructor for extending with a root of X^2+1
--
-- Copyright (c) Manuel Bronstein 2001
-- Copyright (c) INRIA 2001, Version 0.1.13
-- Logiciel Salli (c) INRIA 1998, dans sa version du 29/10/1998
-----------------------------------------------------------------------------

#include "aldor"

#if ALDOC
\thistype{Complex}
\History{Manuel Bronstein}{12/7/2001}{created}
\Usage{import from \this~R}
\Params{ {\em R} & \altype{ArithmeticType} & Type to be extended\\ }
\Descr{\this~R implements the algebraic extension of {\em R}
generated by a root of $X^2+1 = 0$.
The coefficient type {\em R} must satisfy 2 additional mathematical
properties for the arithmetic of \this~R to be correct, namely:
\begin{itemize}
\item $X^2 + 1$ is irreducible over {\em R}.
\item The element $\alpha$ such that $\alpha^2 + 1 = 0$ must
      commute with all the elements of {\em R} (which is obviously
      satisfied when the multiplication of {\em R} is commutative).
\end{itemize}
}
\begin{exports}
\category{\altype{ArithmeticType}}\\
\category{\altype{CopyableType}}\\
\category{\altype{LinearCombinationType} R}\\
\alexp{coerce}: & R $\to$ \% & Natural embedding\\
\alexp{complex}: & (R, R) $\to$ \% & create a complex\\
\alexp{conjugate}: & \% $\to$ \% & conjugation\\
\alexp{conjugate!}: & \% $\to$ \% & in-place conjugation\\
\alexp{copy!}: & (\%, R, R) $\to$ \% & in-place copy\\
\alexp{imag}: & \% $\to$ R & imaginary part\\
\alexp{real}: & \% $\to$ R & real part\\
\end{exports}
\begin{exports}[if {\em R} has \altype{FloatType} then]
\alfunc{FloatType}{/}: & (\%, \%) $\to$ \% & Division\\
\end{exports}
\begin{exports}[if {\em R} has \altype{InputType} then]
\category{\altype{InputType}}\\
\end{exports}
\begin{exports}[if {\em R} has \altype{OutputType} then]
\category{\altype{OutputType}}\\
\end{exports}
\begin{exports}[if {\em R} has \altype{SerializableType} then]
\category{\altype{SerializableType}}\\
\end{exports}
#endif

Complex(R: ArithmeticType):
	Join(ArithmeticType, CopyableType, LinearCombinationType R) with {
        if R has InputType then InputType;
        if R has OutputType then OutputType;
        if R has SerializableType then SerializableType;
	if R has FloatType then {
		/: (%, %) -> %;
	}
	coerce: R -> %;
#if ALDOC
\alpage{coerce}
\Usage{\name~x\\ x::\%}
\Signature{R}{\%}
\Params{ {\em x} & R & an element of the base type\\ }
\Retval{Returns the complex $x + 0 \sqrt{-1}$.}
#endif
	complex: (R, R) -> %;
#if ALDOC
\alpage{complex}
\Usage{\name(x,y)}
\Signature{(R, R)}{\%}
\Params{{\em x, y} & R & real and imaginary parts\\ }
\Retval{Returns the complex $x + y \sqrt{-1}$.}
#endif
	conjugate: % -> %;
	conjugate!: % -> %;
#if ALDOC
\alpage{conjugate}
\altarget{\name!}
\Usage{\name~z\\ \name!~z}
\Signature{\%}{\%}
\Params{{\em z} & \% & a complex}
\Descr{Returns $x - y \sqrt{-1}$ where $z = x + y \sqrt{-1}$.
When using \name!, the storage used by {\em z} is allowed to be destroyed
or reused, so {\em z} is lost after this call.}
#endif
	copy!: (%, R, R) -> %;
#if ALDOC
\alpage{copy!}
\Usage{\name(z,x,y)}
\Signature{(\%, R, R)}{\%}
\Params{
{\em z} & \% & a complex\\
{\em x, y} & R & real and imaginary parts\\
}
\Descr{Replaces {\em z} by $x + y \sqrt{-1}$ and return {\em z},
where the storage used by {\em z} is allowed
to be destroyed or reused, so {\em z} is lost after this call.}
\Remarks{This call may cause {\em z} to be destroyed, so do not use it unless
{\em z} has been locally allocated, and is guaranteed not to share space
with other elements. Some functions are not necessarily copying their
arguments and can thus create memory aliases.}
\alseealso{\alfunc{CopyableType}{copy!}}
#endif
	imag: % -> R;
	real: % -> R;
#if ALDOC
\alpage{imag,real}
\altarget{imag}
\altarget{real}
\Usage{\name~z}
\Signature{\%}{R}
\Params{{\em z} & \% & a complex}
\Retval{imag(z) and real(z) return respectively {\em y} and {\em x}
where $z = x + y \sqrt{-1}$.}
#endif
	norm: % -> %;
#if ALDOC
\alpage{norm}
\Usage{\name~z}
\Signature{\%}{\%}
\Params{{\em z} & \% & a complex}
\Retval{Returns $z \bar z$ where $\bar z$ is the conjugate of {\em z}.}
\Remarks{When {\em R} is commutative, the imaginary part of $z \bar z$
is $0$, and the usual complex norm can be computed via {\tt real norm z}.
The imaginary part of $z \bar z$ is not necessarily $0$ when {\em R}
is not commutative.}
#endif
} == add {
	Rep == Record(real: R, imag: R);
	import from R;

	commutative?:Boolean	== commutative?$R;
	complex(r: R, i: R): %	== { import from Rep; per [r, i] }
	real(a: %): R		== { import from Rep; rep(a).real }
	imag(a: %): R		== { import from Rep; rep(a).imag }
	coerce(x: R): %		== complex(x, 0);
	0:  %			== complex(0, 0);
	1:  %			== complex(1, 0);
	zero?(a:%):Boolean	== zero?(real a) and zero?(imag a);
	one?(a:%):Boolean	== one?(real a) and zero?(imag a);
	(a:%) = (b:%): Boolean	== real a = real b and imag a = imag b;
	- (a: %): %		== { zero? a => a; complex(- real a, - imag a) }

	copy!(a:%, r:R, i:R):% == {
		zero? a or one? a => complex(r, i);
		reset!(a, r, i);
	}

	local reset!(a:%, r:R, i:R):% == {
		import from Rep, Boolean;
		assert(~zero? a); assert(~one? a);
		rep(a).real := r;
		rep(a).imag := i;
		a;
	}

	minus!(a:%):% == {
		zero? a => a;
		one? a => -a;
		reset!(a, - real a, - imag a);
	}

	(a: %) + (b: %): % == {
		zero? a => b; zero? b => a;
		complex(real a + real b,  imag a + imag b);
	}

	(a: %) - (b: %): % == {
		zero? b => a;
		zero? a => -b;
		complex(real a - real b,  imag a - imag b);
	}

	add!(a:%, b:%):% == {
		zero? b => a;
		zero? a => copy b;
		one? a => b + 1;
		reset!(a, real a + real b, imag a + imag b);
	}

	(r: R) * (a: %): % == {
		zero? r => 0; one? r => a;
		complex(r * real a, r * imag a);
	}

	times!(r:R, a:%):% == {
		zero? r or zero? a => 0;
		one? r => a; one? a => r::%;
		reset!(a, r * real a, r * imag a);
	}

	add!(a:%, r:R, b:%):% == {
		one? r => add!(a, b);
		zero? r or zero? b => a;
		zero? a => times!(r, copy b);
		one? a => add!(times!(r, copy b), 1);
		reset!(a, r * real a + real b, r * imag a + imag b);
	}

	conjugate(a:%):% == {
		zero? imag a => a;
		complex(real a, - imag a);
	}

	conjugate!(a:%):% == {
		zero? imag a => a;
		reset!(a, real a, - imag a);
	}

	norm(a:%):% == {
		zero? a or one? a => a;
		complex norm0 a;
	}

	local norm0(a:%):(R, R) == {
		r := real a * real a;
		zero? imag a => (r, 0);
		r := r + imag a * imag a;
		commutative?$% => (r, 0);
		(r, imag a * real a - real a * imag a);
	}

	(a: %) * (b: %): % == {
		zero? a or zero? b => 0;
		one? a => b; one? b => a;
		complex(real a * real b - imag a * imag b,
					real a * imag b + imag a * real b);
	}

	times!(a:%, b:%):% == {
		zero? a or zero? b => 0;
		one? a => copy b; one? b => a;
		reset!(a, real a * real b - imag a * imag b,
					real a * imag b + imag a * real b);
	}

	copy(a:%):% == {
		zero? a or one? a => a;
		complex(real a, imag a);
	}

	copy!(a:%, b:%):% == {
		zero? a or one? a => copy b;
		reset!(a, real b, imag b);
	}

	if R has FloatType then {
		(a:%)^(n:MachineInteger):% == {
			import from BinaryPowering(%, MachineInteger);
			n > 0 => binaryExponentiation(a, n);
			1 / binaryExponentiation(a, -n);
		}

		(a:%) / (b:%):% == {
			import from Boolean;
			assert(~zero? b);
			zero? a or one? b => a;
			(r, i, d) := floatdiv(a, b);
			one? d => complex(r, i);
			complex(r / d, i / d);
		}

		local floatdiv(a:%, b:%):(R, R, R) == {
			(d, zr) := norm0 b;
			(real a * real b + imag a * imag b,
				imag a * real b - real a * imag b, d);
		}
	}

	if R has SerializableType then {
		(p:BinaryWriter) << (a:%):BinaryWriter == p << real a << imag a;

		<< (p:BinaryReader):% == {
			r:R := << p;
			complex(r, << p);
		}
	}

	if R has OutputType then {
		(p:TextWriter) << (a:%):TextWriter ==
			p << leftParen << real a <<comma<< imag a << rightParen;
	}

	if R has InputType then {
		import from MachineInteger;

		local comma:Character		== char 44;
		local leftParen:Character	== char 40;
		local rightParen:Character	== char 41;

		<< (p:TextReader):% == {
			import from Character;
			local c:Character;
			while space?(c := << p) or c = newline repeat {};
			c ~= leftParen => {
				push!(c, p);
				throw SyntaxException;
			}
			x:R := << p;
			while space?(c := << p) or c = newline repeat {};
			c ~= comma => {
				push!(c, p);
				throw SyntaxException;
			}
			y:R := << p;
			while space?(c := << p) or c = newline repeat {};
			c = rightParen => complex(x, y);
			push!(c, p);
			throw SyntaxException;
		}
	}
}

