------------------------------- sit_gcd.as ----------------------------------
-- Copyright (c) Manuel Bronstein 1994
-- Copyright (c) INRIA 1999, Version 0.1.12
-- Logiciel Sum^it (c) INRIA 1999, dans sa version 0.1.12
-- Copyright (c) Swiss Federal Polytechnic Institute Zurich, 1994-97
-----------------------------------------------------------------------------

#include "algebra"

#if ALDOC
\thistype{GcdDomain}
\History{Manuel Bronstein}{13/12/94}{created}
\Usage{\this: Category}
\Descr{\this~is the category of commutative Gcd domains.}
\begin{exports}
\category{\astype{IntegralDomain}}\\
\asexp{gcd}: & (\%, \%) $\to$ \% & Gcd of $2$ elements\\
\asexp{gcd}: & \astype{Generator} \% $\to$ \% & Gcd of several elements\\
\asexp{gcd!}: & (\%, \%) $\to$ \% & In-place gcd of $2$ elements\\
\asexp{gcdquo}: & (\%, \%) $\to$ (\%, \%, \%) & Gcd with quotients\\
\asexp{gcdquo}:
& \astype{List} \% $\to$ (\%, \astype{List} \%) & Gcd with quotients\\
\asexp{lcm}: & (\%, \%) $\to$ \% & Lcm of $2$ elements\\
\asexp{lcm}: & \astype{List} \% $\to$ \% & Lcm of several elements\\
\end{exports}
#endif

define GcdDomain: Category == IntegralDomain with {
	gcd: (%, %) -> %;
	gcd!: (%, %) -> %;
	gcd: Generator % -> %;
	lcm: (%, %) -> %;
	lcm: List % -> %;
#if ALDOC
\aspage{gcd,lcm}
\astarget{gcd}
\astarget{gcd!}
\astarget{lcm}
\Usage{gcd($x_1,x_2$)\\ gcd!($x_1,x_2$)\\ gcd~g\\
lcm($x_1,x_2$)\\ lcm~[$x_1,\dots,x_n$]}
\Signatures{
\name: & (\%, \%) $\to$ \%\\
gcd!: & (\%, \%) $\to$ \%\\
gcd: & \astype{Generator} \% $\to$ \%\\
lcm: & \astype{List} \% $\to$ \%\\
}
\Params{
$x_i$ & \% & Elements of the ring\\
{\em g} & \astype{Generator} \% & Generates elements of the ring\\
}
\Retval{gcd($x_1,x_2$) and lcm($x_1,x_2$) return
respectively a greatest common divisor and least common multiple
of $x_1$ and $x_2$, while gcd(g) return a greatest common divisor
of all the elements generated by g and lcm([$x_1,\dots,x_n$])
return a least common multiple of the $x_i$.}
\Remarks{With certain types, for example polynomials, the
generator version can be more efficient than iterating the binary version.
The function gcd! may cause $x_1$ and $x_2$ to be destroyed,
so do not use it unless $x_1$ and $x_2$ have been locally allocated,
and are guaranteed not to share space with other elements.
Some functions are not necessarily copying their
arguments and can thus create memory aliases.}
\alseealso{\asexp{gcdquo}}
#endif
	gcdquo: (%, %) -> (%, %, %);
	gcdquo: List % -> (%, List %);
#if ALDOC
\aspage{gcdquo}
\Usage{\name($x_1,x_2$)\\ \name~[$x_1,\dots,x_n$]}
\Signatures{
\name: & (\%,\%) $\to$ (\%,\%,\%)\\
\name: & \astype{List} \% $\to$ (\%, \astype{List} \%)\\
}
\Params{ {\em $x_i$} & \% & Elements of the ring\\ }
\Retval{
\name($x_1,x_2$) returns $(g, y_1, y_2)$ where $g = \gcd(x_1, x_2)$,
$x_1 = g y_1$ and $x_2 = g y_2$,\\
\name($[x_1,\dots,x_n]$) returns $(g, [y_1,\dots,y_n])$ where
$g = \gcd(x_1,\dots,x_n)$ and $x_i = g y_i$.
}
\Remarks{With certain types, for example polynomials, the n-ary version
can be more efficient than iterating the binary version.}
\alseealso{\asexp{gcd}}
#endif
	default {
		gcd!(a:%, b:%):% == gcd(a, b);

		lcm(a:%, b:%):% == {
			(g, aa, bb) := gcdquo(a, b);
			aa * b;
		}

		-- note: lcm/l * gcd/l is NOT always  */l if #l > 2
		-- however, lcm(a_1,...,a_n) = g lcm(b_1,...,b_n)
		-- where g = gcd(a_1,...,a_n) and b_i = a_i / g
		-- uses iterative lcm for the b_i's
		lcm(l:List %):%	== {
			import from Boolean;
			empty? l => 1;
			empty? rest l => first l;
			empty? rest rest l => lcm(first l, first rest l);
			(g, quotients) := gcdquo l;
			m := first quotients;
			while ~empty?(quotients := rest quotients) repeat
				m := lcm(m, first quotients);
			g * m;
		}

		-- default is compute the gcd, then divide
		-- some gcd algorithms yield the cofactors (e.g. modgcd/heugcd)
		gcdquo(a:%, b:%):(%, %, %) == {
			unit?(g := gcd(a, b)) => (1, a, b);
			(g, quotient(a, g), quotient(b, g));
		}

		gcdquo(l:List %):(%, List %) == {
			unit?(g := gcd generator l) => (1, l);
			(g, [quotient(a, g) for a in l]);
		}

		-- default is iterative gcd
		-- must be coded specially where faster is possible (e.g. K[X])
		gcd(l:Generator %):% == {
			g := 0;
			for x in l repeat {
				g := gcd(g, x);
				unit? g => return 1;
			}
			g;
		}
	}
}
