-------------------------------------------------------------------------------
--
-- milex.as: Lexicographical exponent vectors of machine integers.
--
--------------------------------------------------------------------------------
--  Copyright (c) 1990-2007 Aldor Software Organization Ltd (Aldor.org).
-- Copyright: INRIA, UWO and University of Lille I, 2001
-- Copyright: Marc Moreno Maza
--------------------------------------------------------------------------------

-- ext MachineIntegerLexicographicalExponent

#include "algebra"

macro { 
      Z == MachineInteger;
      MI == MachineInteger;
}

+++ `MachineIntegerLexicographicalExponent(V)' implements the 
+++ exponents of the monomials (or power products) generated by the finite 
+++ set of variables `V'. Such an exponent is represented by a primtive 
+++ array of machine integers with length `dim = n + 1' where `n' is the
+++ number of elements in `V'. Given `e' in `%' and `i' in `1..n' the
+++ degree of `e' in `variable(i)$V' is stored in slot `i'. Slot `0'
+++ is used to store the total degree of `e' that is the sum of the content 
+++ of all the other slots. An exponent `a' is greater than an exponent `b' 
+++ if `a' is greater than `b' w.r.t. the lexicographical 
+++ ordering induced by `V' (by comparing `a.i' and `b.i' for `i' in 
+++ `1..n').
+++ Author: Marc Moreno Maza
+++ Date Created: 09/07/01
+++ Date Last Update: 12/07/01

MachineIntegerLexicographicalExponent(V: FiniteVariableType): 
  MachineIntegerExponentVectorCategory(V) ==
   MachineIntegerDegreeReverseLexicographicalExponent(V) add { 
      Rep == PrimitiveArray(MI);
      import from V, Rep;
      local n__V: MI == #$V;
      (x: %) < (y: %) : Boolean == {
             xx := rep x;
             yy := rep y;
             import from Boolean;
             for i in 1..n__V repeat {
                 xx.i < yy.i => return true;
                 xx.i > yy.i => return false;
             }
             false;
      }
}


#if ALDOC
\thistype{MachineIntegerLexicographicalExponent}
\History{Marc Moreno Maza}{08/07/01}{created}
\Usage{import from \this~V}
\Params{
{\em V} & \altype{FiniteVariableType} & The type of the variables \\
}
\Descr{\this~V implements the 
 exponents of the monomials (or power products) generated by the finite 
 set of variables $V.$ Such an exponent is represented by a primtive 
 array of machine integers with length $dim = n + 1$ where $n$ is the
 number of elements in $V.$ Given $e$ in \% and $i$ in $1 \cdots n$ the
 degree of $e$ w.r.t. the $i$-th variable of $V$ is stored in slot $i$. Slot $0$
 is used to store the total degree of $e$, that is the sum of the content 
 of all the other slots. An exponent $a$ is greater than an exponent $b$ 
 if $a$ is greater than $b$ w.r.t. the lexicographical 
 ordering induced by $V$ (by comparing $a_i$ and $b_i$ for $i$ running
 from $1$ to $n$). }
\begin{exports}
\category{\astype{MachineIntegerExponentVectorCategory} V} \\
\end{exports}
#endif
