/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.util.Iterator;

import nz.net.catalyst.Log;
import org.dom4j.Document;
import org.dom4j.Element;


/**
 * Helper class for XML sessions.
 * 
 * These help the input of xml.  For output of xml please see the Transmission
 * class.
 * @see Transmission
 */

public class XMLHelper implements IPackage {

	/**
	 * Header used to describe an error.
	 */
	private static final String ERROR = "Error";

  public Transmission getTransmission(Document doc) {
		long sessionStartTime = System.currentTimeMillis();
		
		String line = null; //current line of input.
	  	Transmission transmission = null;
	  	
	  	ECommand command = null;

		Log.debug("Root element: " + doc.getRootElement().getName());

		if (doc.getRootElement().getName().equals("LuceneQueryRequest")) {
			command = ECommand.getByName("QUERY");

			transmission = new Transmission(command);
			Element root = doc.getRootElement();
			for (Iterator i = root.elementIterator(); i.hasNext(); ) {
				Element header = (Element) i.next();
				String name = header.getQualifiedName();
				Log.debug("Element name:" + name + ":\t\tValue:" + nz.net.catalyst.Util.clean(header.getText()) + ":");
				
				if (name.equalsIgnoreCase("Sort")) {
					transmission.add(name, header.getText());
					transmission.add("Sort-Limit", header.valueOf("@limit"));
					Log.debug("\t\tSort-Limit set to:" + header.valueOf("@limit") + ":");
				} else if (name.equalsIgnoreCase("Offset")) {
					//setup Offset as a synonym for First.
					transmission.add("First", header.getText());
				} else if (name.equalsIgnoreCase("Range")) {
					String r_field = header.valueOf("@field");
					String r_to    = header.valueOf("To");
					String r_from  = header.valueOf("From");
					Log.debug("Range-Field:" + r_field + ": Range-From:" + r_from + ": Range-To:" + r_to + ":");
					if (r_field.equals("")) {
						Log.warn("Range specified without field.  Need to specify field in range query.");
						return error("Range specified without field.  Need to specify field in range query.");
					} else {
						transmission.add("Range-Field", r_field);
						if (!r_from.equals("")) transmission.add("Range-From", r_from);
						if (!r_to.equals(""))   transmission.add("Range-To", r_to);
					}
				} else if (name.equalsIgnoreCase("Fields")) {
					getFieldDefinitions(doc, transmission, false);
				} else if (name.equalsIgnoreCase("Query")) {
					transmission.add(name, header.getText());
					String defaultField    = header.valueOf("@default-field");
					if (!defaultField.equals("")) {
						transmission.add("Default-Field", defaultField);
						Log.debug("\tDefault-Field: " + defaultField);
					}
				} else {
					//default handler for all other headers
					transmission.add(name, header.getText());
				}
			}
		} else if (doc.getRootElement().getName().equals("LuceneIndexRequest")) {
				command = ECommand.getByName("INDEX");
				transmission = new Transmission(command);
				Element root = doc.getRootElement();
				for (Iterator i = root.elementIterator(); i.hasNext(); ) {
					Element header = (Element) i.next();
					String name = header.getQualifiedName();
					Log.debug("Element name:" + name + ":\t\tValue:" + nz.net.catalyst.Util.clean(header.getText()) + ":");

					if (name.equalsIgnoreCase("Fields")) {
						getFieldDefinitions(doc, transmission, true);
					} else {
						//for id, application, serial, stop-list, domain etc.
						transmission.add(header.getQualifiedName(), header.getText());
					}
				}
		} else if (doc.getRootElement().getName().equals("LuceneUnIndexRequest")) {
				command = ECommand.getByName("UNINDEX");
				transmission = new Transmission(command);
				Element root = doc.getRootElement();
				String name = "";
				for (Iterator i = root.elementIterator(); i.hasNext(); ) {
					Element header = (Element) i.next();
					name = header.getQualifiedName();
					Log.debug("Element name:" + name + ":\t\tValue:" + header.getText() + ":");

					if (name.equalsIgnoreCase("Purge")) {
						transmission.add(name, "true");
						Log.debug("\t\t" + name + "->true");
					} else {
						//for id, application, serial, stop-list, domain etc.
						transmission.add(name, header.getText());
					}
				}
		} else if (doc.getRootElement().getName().equals("LuceneUtilityRequest")) {
				command = ECommand.getByName("CONTROL");
				transmission = new Transmission(command);
				Element root = doc.getRootElement();
				String name = "";
				for (Iterator i = root.elementIterator(); i.hasNext(); ) {
					Element header = (Element) i.next();
					name = header.getQualifiedName();
					Log.debug("Element name:" + name + ":\t\tValue:" + header.getText() + ":");

					if (name.equalsIgnoreCase("Utility")) {
						transmission.add("Sub-Command", header.getText());
						Log.debug("\t\tSub-Command ->" + header.getText());
					} else if (name.equalsIgnoreCase("Extra")) {
						//do nothing at the moment
						Log.debug("\t\tWe got extra info.");
					} else {
						//for id, application, serial, stop-list, domain etc.
						transmission.add(name, header.getText());
					}
				}
		} else {
			//process in the budget way.					
			String commandString = doc.selectSingleNode("//LuceneRequest/@type").getText();
			command = ECommand.getByName(commandString.trim());

			transmission = new Transmission(command);
			Element root = doc.getRootElement();
			for (Iterator i = root.elementIterator(); i.hasNext(); ) {
				Element header = (Element) i.next();
				Log.debug("Element name:" + header.getQualifiedName() + ":\t\tValue:" + header.getText() + ":");
				transmission.add(header.getQualifiedName(), header.getText());
			}
		}

		long endTime = System.currentTimeMillis();
		Log.debug("Time to parse LuceneRequest type="	+ command.getName() + ": " + (endTime - sessionStartTime) + " ms");
		
		if (transmission != null) transmission.setXML(true);
		return transmission;
  } //getTransmission
  

	static boolean isEndTag(String xml) {
		xml = xml.trim();
		if (xml.endsWith("</LuceneIndexRequest>")) return true;
		if (xml.endsWith("</LuceneRequest>")) return true;
		if (xml.endsWith("</LuceneQueryRequest>")) return true;
		if (xml.endsWith("</LuceneUnIndexRequest>")) return true;
		if (xml.endsWith("</LuceneUtilityRequest>")) return true;
		return false;
	}
























	/**
	 * return an error transmission if something bad has happened.
	 * @param message The error message to wrap in the transmission
	 * @return Transmission the error transmission ready to be written.
	 */
  static Transmission error(String message) {
	Transmission transmission = new Transmission(ECommand.ERROR);
	transmission.add(ERROR, message);
	transmission.setXML(true);
	return transmission;
  }


	/**
	 * Internal method for extracting the field definitions from the XML.
	 * 
	 * Please note that not all attributes need to be supplied. Name is the only
	 * compulsory attribute.  It is unnecesary to specify name by itself as an
	 * undefined field will inherit the default attributes which are:
	 * -Tokenised (i.e. text type)
	 * -Indexed
	 * -NOT-Stored
	 * 
	 * @param doc
	 * @param transmission
	 */
	private void getFieldDefinitions(Document doc, Transmission transmission, boolean getText) {
		for (Iterator i2 = doc.selectNodes("//Fields/Field").iterator(); i2.hasNext(); ) {
			Element element = (Element) i2.next();

			if (element.valueOf("@name")    != "") {
				transmission.add("Field-Name",    element.valueOf("@name"));
				//only add other details if a name exists.
				if (element.valueOf("@type")    != "") transmission.add("Field-Type",    element.valueOf("@type"));
				if (element.valueOf("@indexed") != "") {
					if (element.valueOf("@indexed").equalsIgnoreCase("true") || element.valueOf("@indexed").equalsIgnoreCase("yes")) {
						transmission.add("Field-Indexed", "yes");
					} else {
						transmission.add("Field-Indexed", "no");
					}
				}
				if (element.valueOf("@stored")  != "") {
					if (element.valueOf("@stored").equalsIgnoreCase("true") || element.valueOf("@stored").equalsIgnoreCase("yes")) {
						transmission.add("Field-Stored", "yes");
					} else {
						transmission.add("Field-Stored", "no");	
					}
				}

				transmission.add(element.valueOf("@name"), element.getText());
			}

			Log.debug("\tField-Def: " + element.valueOf("@name") + ", " + element.valueOf("@type") + ", " + element.valueOf("@indexed") + ", " +element.valueOf("@stored") + " ->" + nz.net.catalyst.Util.clean(element.getText()));
		}
	}
}