/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;

import nz.net.catalyst.Log;

/**
 * A Dialogue is an abstraction of the communication between the client and 
 * the server.
 * Each dialogue performs ONE task.
 */
public class Dialogue implements IPackage, Constants
{
  private final Transmission incomingTransmission;
  private 	    Transmission result;
  private final BufferedReader input;
  private final PrintWriter output;

  /**
   * Static Factory Method.
   * Create a dialogue based on a command string.
   * The string is checked if its a valid command and if so then a new
   * Transmission is created.
   */
  public static Dialogue create(String line, BufferedReader input, PrintWriter output)
  {
    ECommand command = ECommand.getByName(line.trim());
    if (command == null)
      return null;

    Transmission transmission = new Transmission(command);
    return new Dialogue(new Transmission(command), input, output);
  }

  /**
   * Static factory method for non-stream based dialogues
   */
  public static Dialogue create(Transmission transmission)
  {
    return new Dialogue(transmission, null, null);
  }

	/**
	 * Static Factory Method.
	 * Creates a dialogue when a transmission has already been created.
	 */
  public static Dialogue create(Transmission transmission, BufferedReader input, PrintWriter output)  {
  	return new Dialogue(transmission, input, output);
  }

  /**
   * Private Constructor. See static create methods to create a dialogue
   * instance.
   * @param transmission
   * @param input
   * @param output
   */
  private Dialogue(Transmission transmission, BufferedReader input, PrintWriter output)	{
	  incomingTransmission = transmission;
	  this.input = input;
	  this.output = output;
  }


 /**
  * This method processes the dialog.  That is it calls the command
  * that this dialog encapsulates. 
  * 
  * <br><b>TODO:</b> Make the xxxTransmission classes use inheritance
  * so that the below method can call an polymorphism method .execute on
  * a Command superclass.
  * 
  * @throws IOException
  */
  public void process() throws IOException  {
    //added by hamish for xml dialogues.
    if (!incomingTransmission.isXML()) 
      incomingTransmission.read(input);

    ECommand command = incomingTransmission.getCommand();
  	Log.debug("Transmission Command is \"" + command.getName() + "\"");

    if (command == ECommand.QUERY) {
	  	Query query = new Query(incomingTransmission);
	  	result = query.execute();
    } else if (command == ECommand.INDEX) {
    	Index index = new Index(incomingTransmission);
    	result = index.execute();
    } else if (command == ECommand.CONTROL) {
    	Control control = new Control(incomingTransmission);
    	result = control.execute();
    } else if (command == ECommand.UNINDEX) {
    	UnIndex unIndex = new UnIndex(incomingTransmission);
    	result = unIndex.execute();
    } else {
    	String msg = "Unknown command, cannot process \"" + command.getName() + "\"";
    	Log.error(msg);
    	Session.error(output, msg);
    	return;
    }
    
  	result.setXML(incomingTransmission.isXML());
  	result.write(output);

  }

}

