/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;


import java.io.IOException;
import java.io.Reader;

/**
 * A reader which only allows a finite number of characters to be read
 * before signalling EOF.  <p>
 *
 * When the reader is closed, all characters up to the finite number
 * are read and discarded.
 */

public class FiniteReader extends Reader implements IPackage
{
  private long remaining;
  private Reader in;

  public FiniteReader(Reader in, long contentLength)
  {
    this.in = in;
    this.remaining = contentLength >= 0 ? contentLength : 0;
  }

  /**
   * Read characters into a portion of an array.  This method will block
   * until some input is available, an I/O error occurs, or the end of the
   * stream is reached.
   *
   * @param      cbuf  Destination buffer
   * @param      off   Offset at which to start storing characters
   * @param      len   Maximum number of characters to read
   *
   * @return     The number of characters read, or -1 if the end of the
   *             stream has been reached
   *
   * @exception  IOException  If an I/O error occurs
   */
  public synchronized int read(char cbuf[], int off, int len) throws IOException
  {
    if (remaining == -1)
      throw new IOException("Illegal operation after close()");

    if (remaining == 0)
      return -1;

    if (len > remaining)
      len = (int) remaining;

    int result = in.read(cbuf, off, len);
    if (result == -1 || result >= remaining)
      remaining = 0;
    else
      remaining -= result;

    return result;
  }
  /**
   * Tell whether this stream is ready to be read.
   *
   * @return True if the next read() is guaranteed not to block for input,
   * false otherwise.  Note that returning false does not guarantee that the
   * next read will block.
   *
   * @exception  IOException  If an I/O error occurs
   */
  public synchronized boolean ready() throws IOException
  {
    if (remaining == -1)
      throw new IOException("Illegal operation after close()");

    return remaining == 0 || in.ready();
  }

  /**
   * Close the stream.  Once a stream has been closed, further read(),
   * ready(), mark(), or reset() invocations will throw an IOException.
   * Closing a previously-closed stream, however, has no effect.  <p>
   *
   * If the stream has not been exhausted, then all remaining
   * characters are retrieved, even if that means that the close
   * operation blocks.
   *
   * @exception  IOException  If an I/O error occurs
   */
  public synchronized void close() throws IOException
  {
    if (remaining > 0)
    {
      skip(remaining);
      // Assert: remaining == 0
      remaining = -1;
    }
  }
}

