/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst;

//import java.io.BufferedReader;
//import java.io.BufferedWriter;
import java.io.IOException;
//import java.io.InputStreamReader;
import java.io.InterruptedIOException;
//import java.io.OutputStreamWriter;
//import java.io.PrintWriter;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;

/**
 * Lucene Server Socket Listener Class.  <p>
 *
 * This class listens for incomming socket connections and then spawns a client
 * to process the connection.  This has been hardcoded for the lucene server.
 *
 * @see nz.net.catalyst.SocketClient
 */
public class SocketListener implements Runnable, IPackage
{
  protected static final int SUSPEND = 1;
  protected static final int QUIT = 2;
  protected static final int RUN = 3;

  protected int state = RUN;

  protected ServerSocket listenerSocket;
  private final int port;
  protected Thread listenerThread;
  protected int failureLimit = 0;
  protected int failureCount = 0;
  private final ISocketClientFactory clientFactory;

  public SocketListener(int port, ISocketClientFactory clientFactory)  {
    this.port = port;
    this.clientFactory = clientFactory;
  }

  public void bind() throws IOException  {
    listenerSocket = new ServerSocket(port);
  }

  public void startListener()  {
    listenerThread = new Thread(this, getListenerThreadName());
    listenerThread.start();
  }

  public int getPort()  {
    return port;
  }

  public ServerSocket getListenerSocket()  {
    return listenerSocket;
  }

  public String getListenerThreadName()  {
    return "TCP listener on port " + port;
  }

  public Thread getListenerThread()  {
    return listenerThread;
  }

  public void run()  {
    // We want to poll to check whether it is time to shut down
    // periodically.

    try    {
      listenerSocket.setSoTimeout(1000);
    } catch (SocketException e) {
      throw new RtException("Error trying to set Socket timeout in thread " +
                            getListenerThreadName(), e);
    }

	//loop forever
    for (;;) {
      if (terminating())
        break;

      try {
        Socket clientSocket = listenerSocket.accept();
        //clientSocket.setSoTimeout(0); 	//timeout disabled (stay open forever)
      	clientSocket.setSoTimeout(30000); 	//30 seconds
        newConnection(clientSocket);
      } catch(InterruptedIOException e) {
        acceptTimeout();
      } catch(IOException e) {
        connectionFailed(e);
      }
    }
  }
  /** used when socket throughs an exception */
  protected void connectionFailed(IOException e)
  {
    reportConnectionFailure(e);
    signalConnectionFailure(e);
  }

	/** used when exception happens*/
  protected void signalConnectionFailure(IOException e)
  {
    if (failureLimit >= 0 && ++failureCount > failureLimit)
      throw new RtException(e);
  }


  protected void acceptTimeout()  {
  	//do nothing
  }

	/** used when exception happens*/
  protected void reportConnectionFailure(IOException e)
  {
    Log.log(ELog.ERROR, getListenerThreadName() +
                       ": Error accepting socket connection:" + e);
  }

	/** not used */
  protected String getConnectionName(Socket clientSocket)
  {
    return Util.name(clientSocket);
  }

  /**
   * Internal method to call the communicate method on the session object for
   * the socket.
   * @param socketClient
   */
  protected void newConnection(final Socket socketClient)
  {
    String name = getConnectionName(socketClient);

    Log.debug("Client Connected. About to process request.");  

    //THIS IS WHERE TO ADD THREAD POOLING CODE.
    /*
     * get thread from pool
     * set variables
     * thread.start();
     */
  	
  	new Thread(new SocketClient(socketClient), name).start();

     /*
    new Thread(new Runnable(){
        public void run()
        {
          try
          {
            //clientFactory.createSocketClient().communicate(socketClient);
 
              //these lines added by hamish to try and simplify things.
              String name = Util.name(socketClient);
              try {
                //get the input
                BufferedReader input = new BufferedReader(new InputStreamReader(socketClient.getInputStream(), "UTF-8"));
                PrintWriter output = new PrintWriter(new BufferedWriter(new OutputStreamWriter(socketClient.getOutputStream(), "UTF-8")));

              	new nz.net.catalyst.lucene.server.Session().communicate(input, output, name);

                input.close();
                output.close();
                socketClient.close();
                input = null;
                output=null;
              } catch (IOException e) {
                throw new RtException("Error initialising new client connection " + name, e);
              }              
              //end added by hamish.    
          } finally {
            try {
              socketClient.close();
            } catch (IOException e) {
              // Ignore
            } finally {
            	//Log.debug("Socket - request finished. Socket Closed.");
            }
          }
        }
      }, name).start();
      */
  }

	/** not used */  
  protected ISocketClientFactory getClientFactory()  {
    return clientFactory;
  }
	/** not used */
  public synchronized void shutDown()  {
    state = QUIT;
  }
	/** not used */
  protected synchronized boolean terminating()  {
    while (getState() == SUSPEND)
    {
      try
      {
        wait();
      }
      catch (InterruptedException e)
      {
        // ignore
      }
    }
    return getState() == QUIT;
  }

  /** not used */
  protected int getState() {
    return state;
  }
  
}
