#!/bin/sh
# Upgrade/patch a POSTGRESQL database for an Axyl website.

# NOTE: This is not normally run standalone. The main DB upgrade
# script 'install/upgrade-axyl-databases.sh normally runs this.
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# INCOMING PARAMETERS
#  $1  - DB_NAME
#  $2  - DB_USER
#  $3  - DB_PASSWD ('none' means a blank password)
#  $4  - DB_HOST ('direct' means a local database server)
#  $5  - DB_PORT
#  $6  - AXYL_VERSION (Axyl version, '' or 'n.n.n'
#  $7  - MODE ('upgrade' or 'stamp-only')
DB_NAME=""
DB_USER=""
DB_PASSWD=""
DB_HOST=""
DB_PORT=5432
AXYL_VERSION="unknown"
MODE=upgrade

POS=0
while [ $# -gt 0 ] ; do
  POS=`expr $POS + 1`
  case $POS in
    1)  DB_NAME=$1
        ;;
    2)  DB_USER=$1
        ;;
    3)  DB_PASSWD=$1
        ;;
    4)  DB_HOST=$1
        ;;
    5)  DB_PORT=$1
        ;;
    6)  AXYL_VERSION=$1
        ;;
    7)  MODE=$1
        ;;
  esac
  shift
done

# Cater for standalone running of this script. Normally we are called
# with AXYL_HOME et al already defined..
if [ -z $AXYL_HOME ] ; then
	CONFDIR=/etc/axyl
	CONF=${CONFDIR}/axyl.conf
	if [ ! -f $CONF ] ; then
	  echo "Axyl configuration file $CONF not found!"
	  exit 2
	else
	  . $CONF
	  if [ ! -d $AXYL_HOME ] ; then
	    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
	    exit 6
	  fi
	  . ${AXYL_HOME}/install/axyl-common.sh
	fi
fi

# We require Postgres to be locally installed, at least as postgresql-client
# even if no servers are created locally. This is based on the standard
# Debian location, with a few likely fallbacks..
find_location_of directory bin containing psql in /usr/lib/postgresql /usr /usr/local
[ "$LOC" != "" ] && PG_BIN=$LOC

# Echo and log messages..
LOGFILE=${AXYL_LOGS}/upgrade-db.log
function logit() {
  echo $1
  echo `date` $1 >> $LOGFILE
}

# Upgrade the database contents (tables and data)
if [ "$PG_BIN" != "" ] ; then

    PSQL=${PG_BIN}/psql
    if [ -x $PSQL ] ; then
		# Optional host settings for remotely accessed databases..
		HOSTOPTS=""
		[ "$DB_HOST" != "direct" ] && HOSTOPTS="--host $DB_HOST --port $DB_PORT"
		
		# Make sure our database connection user is present..
		USER=`$PSQL --tuples-only --username postgres --dbname template1 $HOSTOPTS --command "SELECT usename FROM pg_user WHERE usename='$DB_USER'" | tr -d ' '`
		if [ "$USER" != "$DB_USER" ] ; then
			logit "Creating database user $DB_USER"
			$PSQL --username postgres --dbname template1 $HOSTOPTS --command "CREATE USER $DB_USER WITH PASSWORD '$DB_PASSWD' CREATEDB"
		fi

		# Acquire application version from ax_control table which must exist
		# or else we simply skip the rest of the process.
		AXYL_DB_VERSION=
		AX_CONTROL=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT relname FROM pg_class WHERE relname='ax_control'" | tr -d ' '`
		if [ "$AX_CONTROL" = "ax_control" ] ; then
			AXYL_DB_VERSION=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT site_axyl_version FROM ax_control" | tr -d ' '`
			if [ "$MODE" != "stamp-only" -a "$AXYL_DB_VERSION" != "" ] ; then
				if [ "$AXYL_DB_VERSION" = "$AXYL_VERSION" ] ; then
					logit "Database ${DB_NAME} is up to date."
				else
					# Ok, we know the registered version, and we know the Axyl installation
					# version, so let's see if we have a patch file series for this upgrade.
					# No patch files, no upgrade.
					UPGRADEROOT=${AXYL_HOME}/db/postgres/upgrade
					cd ${UPGRADEROOT}
					patchfiles=`ls ${UPGRADEROOT}`
					if [ "$patchfiles" != "" ] ; then
						patchSQL=`tempfile --prefix dbupgrade` 
						patchver=$AXYL_DB_VERSION
						gotpatches=0
						for patchfile in $patchfiles ; do
							if [ "$patchver" = "$AXYL_VERSION" ] ; then
								break
							fi
							patchnos=`echo $patchfile | sed "s;.sql;;"`
							if [ "$patchnos" != "" ] ; then
								fromver=`echo $patchnos | cut -d'_' -f2`
								nextver=`echo $patchnos | cut -d'_' -f4`
								if [ "$patchver" = "$fromver" ] ; then
									cat $patchfile >> $patchSQL
									patchver=$nextver
									gotpatches=1
								fi
							fi
						done
						if [ $gotpatches -eq 1 ] ; then
							logit "patching ${DB_NAME} ${AXYL_DB_VERSION} --> ${patchver}"
							$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --file $patchSQL >>$LOGFILE 2>&1
							if [ $? -ne 0 ] ; then
								logit "Errors occurred during the patch process"
							fi
						fi
					else
						logit "no patches were provided for ${DB_NAME} ${AXYL_DB_VERSION}"
					fi
					# Leave the application at the correct version for next time..
					$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "UPDATE ax_control set site_axyl_version='$AXYL_VERSION'" >>$LOGFILE 2>&1
					# Remove temporary patch file..
					rm -f $patchSQL
                fi
            else
                # All we do in this case is assume everything is up to date..
                if [ "$MODE" = "stamp-only" ] ; then
                  logit "Stamping database $DB_NAME with version $AXYL_VERSION"
                fi
                N=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT COUNT(*) from ax_control" | tr -d ' '`
                if [ $N -eq 0 ] ; then
                  $PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "INSERT INTO ax_control (site_axyl_version) VALUES('$AXYL_VERSION')" >>$LOGFILE 2>&1
                else
                  $PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "UPDATE ax_control set site_axyl_version='$AXYL_VERSION'" >>$LOGFILE 2>&1
                fi
            fi
        else
            # No ax_control..
            $PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "CREATE TABLE ax_control (app_prefix text,site_axyl_version text,site_docroot text)" >>$LOGFILE 2>&1
            logit "Stamping database with version $AXYL_VERSION"
            $PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "INSERT INTO ax_control (site_axyl_version) VALUES('$AXYL_VERSION')" >>$LOGFILE 2>&1
        fi
    else
        logit "Postgres client binaries (psql) not found."
        exit 0
    fi
	
else
    logit "Could not determine postgresql bin location."
    exit 0
fi
	
# END