<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    application-defs.php                                    */
/* Author:      Paul Waite                                              */
/* Description: Basic definitions pertaining to application config.     */
/*                                                                      */
/* ******************************************************************** */
/** @package core */

/** XML classes */
include_once("xml-defs.php");
/** Filesystem access */
include_once("file-defs.php");
// ----------------------------------------------------------------------
/**
* A parameter class to contain setting parameter information. Every
* parameter has an idetifying name, and a type. The value of a
* parameter can be an array of name=>value pairs, or a scalar type
* such as a string, integer or boolean.
* @package core
*/
class parameter {
  /** Parameter name */
  var $name = "";
  /** Parameter type */
  var $type = "";
  /** Parameter value */
  var $value = "";
  // .....................................................................
  /**
  * Make a new parameter.
  * @param string $name Name of this parameter
  * @param string $type Parameter type eg: 'array', 'string' etc.
  */
  function parameter($name, $type) {
    $this->name = $name;
    $this->type = $type;
    if ($type == "array") {
      $this->value = array();
    }
  }
  // .....................................................................
  /**
  * Set the parameter to given value. If the parameter has an
  * array of values, the name is also given.
  * @param mixed $value Value to assign to this parameter
  * @param string $ename Name of this value (arrayed parameters only)
  */
  function setvalue($value, $ename="") {
    if ($this->type == "array") {
      $this->value[$ename] = $value;
    }
    elseif ($this->type == "boolean") {
      $this->value = (($value === "true"||$value === "t"||$value === true) ? true : false);
    }
    else {
      $this->value = $value;
    }
  }
  // .....................................................................
  /**
  * Get the parameter value. If the element name is given (for an
  * array type parameter) then we return the value of that element.
  * @param string $ename Name of this value (arrayed parameters only)
  */
  function getvalue($ename="") {
    if ($this->type == "array") {
      return $this->value[$ename];
    }
    else {
      return $this->value;
    }
  }
  // .....................................................................
  /**
  * Return decoded value (or array of values) so that applications can use
  * the data unencumbered with URL encoding etc.
  * @return mixed Value or array of values which have been urldecoded
  */
  function get_decodedvalue() {
    switch ($this->type) {
      case "array":
        $rar = array();
        foreach ($this->value as $name => $value) {
          $rar[$name] = rawurldecode($value);
        }
        return $rar;
        break;

      case "boolean":
        return $this->value;
        break;

      default:
        return rawurldecode($this->value);
    }
  }
  // .....................................................................
  /** Dump this parameter */
  function htmldump() {
    $s = "";
    $s .= "parameter: $this->name ($this->type) ";
    if ($this->type == "array") {
      $s .= "values:";
      foreach ($this->value as $ename => $val) {
        $s .= " [$ename = '$val']";
      }
    }
    else {
      $s .= "value = '$this->value'";
    }
    return $s . "<br>";
  }
} // parameter class

// .......................................................................
/**
* Application settings class. This class contains one or more parameters.
* A setting can also have an 'agent' defined which will action it. This
* can, for example, be the name of a function, if used.
* @package core
*/
class setting {
  /** Name of this setting */
  var $name = "";
  /** Agent which will action this setting */
  var $agent = "";
  /* Array of parameters for this setting */
  var $parameters = array();
  // .....................................................................
  /** Make a new setting */
  function setting($name="", $agent="") {
    $this->name = $name;
    $this->agent = $agent;
  }
  // .....................................................................
  /**
  * Add the given parameter to this setting.
  * @param string $name Name of the parameter to add
  * @param object $parm Parameter object to add to this setting
  */
  function addparameter($name, $parm) {
    $this->parameters[$name] = $parm;
  }
  // .....................................................................
  /**
  * Set the value of a parameter which already exists in this
  * setting.
  * @param string $pname Name of the parameter to set value of
  * @param mixed $value Value to set in the parameter
  * @param string $ename Name of this value (arrayed parameters only)
  */
  function setparameter($pname, $value, $ename="") {
    if (isset($this->parameters[$pname])) {
      $this->parameters[$pname]->setvalue($value, $ename);
    }
  }
  // .....................................................................
  /**
  * Get the value of a named parameter.
  * @param string $name Name of the parameter to get value of
  * @param string $ename Name of element (arrayed parameters only)
  */
  function getparameter($name, $ename="") {
    $val = false;
    if (isset($this->parameters) && isset($this->parameters[$name])) {
      $parm = $this->parameters[$name];
      $val = $parm->getvalue($ename);
    }
    return $val;
  }
  // .....................................................................
  /** Dump setting */
  function htmldump() {
    $s = "";
    $s .= "Setting: agent = '$this->agent'<br>";
    foreach ($this->parameters as $parm) {
      $s .= $parm->htmldump();
    }
    return $s;
  }
} // setting class

// ----------------------------------------------------------------------
/**
* Class comprising the functionality of an application. This is used
* to contain and manage the basic configuration properties of an
* application. This class knows how to read the configuration in,
* store the values, and write it out again.
* @package core
*/
class application extends xmlparser  {
  /** Path to application configuration file */
  var $configpath = "";
  /** State of processing */
  var $state = "";
  /** Current/last tag opened */
  var $tag = "";
  /** Attributes array for current/last tag */
  var $attr = array();
  /** True if response was valid, ie. no errors */
  var $valid = false;
  // .....................................................................
  // Application configuration..
  var $definitions = array();
  var $globals = array();
  var $settings = array();
  // .....................................................................
  /**
  * Construct a new application. Creating the application will also attempt
  * to read in the XML configuration file as specified (or defaulted). If
  * the file is read successfully, then the valid flag is set true.
  * @param string $configpath Path to XML configuration file for application
  */
  function application($configpath="application.xml") {
    $this->configpath = $configpath;
    $this->xmlparser();
    if (file_exists($this->configpath)) {
      $xmlfile = new inputfile($this->configpath);
      if ($xmlfile->opened) {
        $xmlfile->readall();
        $xmlfile->closefile();
        $this->parse($xmlfile->content);
        $this->valid = $this->valid_xml;
      }
    }
  }
  // .....................................................................
  /**
  * Make this current application object the same structure as the
  * given application. This process checks that the definitions, globals,
  * and settings of this application match those of the given one. If
  * a given item is missing, it is created. If an item is not present
  * in the given application, it is deleted in this one. Existing items
  * retain their current values - only structure is checked.
  * @param object $refapp The reference application to synchronize to
  * @return boolean True if changes were made, else false.
  */
  function synchronize($refapp) {
    if ($refapp->valid) {
      $synced = false;
      // Definitions..
      foreach ($refapp->definitions as $name => $val) {
        if (!isset($this->definitions[$name])) {
          $this->definitions[$name] = $val;
          $synced = true;
        }
      }
      foreach ($this->definitions as $name => $val) {
        if (!isset($refapp->definitions[$name])) {
          unset($this->definitions[$name]);
          $synced = true;
        }
      }

      // Globals..
      foreach ($refapp->globals as $name => $val) {
        if (!isset($this->globals[$name])) {
          $this->globals[$name] = $val;
          $synced = true;
        }
      }
      foreach ($this->globals as $name => $val) {
        if (!isset($refapp->globals[$name])) {
          unset($this->globals[$name]);
          $synced = true;
        }
      }

      // Settings..
      foreach ($refapp->settings as $ref_setting) {
        if ($ref_setting->name != "database") {
          if ($this->get_setting($ref_setting->name) === false) {
            $this->settings[] = $ref_setting;
            error_log("adding new setting $ref_setting->name");
            $synced = true;
          }
        }
      }
      $newsettings = array();
      for ($ix = 0; $ix < count($this->settings); $ix++) {
        $setting = $this->settings[$ix];
        if ($setting->name == "database") {
          $newsettings[] = $setting;
        }
        else {
          if ($setting->name != "") {
            if ($refapp->get_setting($setting->name) !== false) {
              $newsettings[] = $setting;
            }
            else {
              error_log("removing setting $setting->name");
              $synced = true;
            }
          }
        }
      }
      $this->settings = $newsettings;
    }
    return $synced;
  }
  // .....................................................................
  /**
  * Return setting by name. NB: some settings can have multiple entries
  * under the same name, eg. 'database'. In this case we return an
  * array of setting objects, otherwise the single setting object.
  * We return false if not found.
  * @param string $name Name of the setting(s) to return
  * @return mixed Single setting object, array of settings, or false
  */
  function get_setting($name) {
    $got = array();
    foreach ($this->settings as $setting) {
      if ($setting->name == $name) {
        $got[] = $setting;
      }
    }
    if (count($got) == 0) {
      return false;
    }
    elseif (count($got) == 1) {
      return $got[0];
    }
    else {
      return $got;
    }
  }
  // .....................................................................
  /**
  * Get the value of a named parameter from a named setting. This only
  * works for settings which are unique - ie. it won't work well for
  * parms which can occur multiple times, eg: 'database'. Option to
  * specify the element name for arrayed parameters.
  * @param string $name Name of the setting which contains the parameter
  * @param string $pname Name of the parameter to get value of
  * @param string $ename Name of element (arrayed parameters only)
  */
  function getparameter($name, $pname, $ename="") {
    $parm = "";
    if (isset($this->settings)) {
      if ($setting = $this->get_setting($name)) {
        $parm = $setting->getparameter($pname, $ename);
      }
    }
    return $parm;
  }
  // .....................................................................
  /**
  * Set the value of a named parameter for a named setting. Optionally
  * provide the element name for arrayed parameters.
  * @param mixed $value Value of the parameter setting
  * @param string $name Name of the setting which contains the parameter
  * @param string $pname Name of the parameter to get value of
  * @param string $ename Name of element (arrayed parameters only)
  */
  function setparameter($value, $name, $pname, $ename="") {
    $parm = "";
    if (isset($this->settings)) {
      $pix = -1;
      for ($ix = 0; $ix < count($this->settings); $ix++) {
        if ($this->settings[$ix]->name == $name) {
          $pix = $ix;
          break;
        }
      }
      if ($pix > -1) {
        $this->settings[$pix]->setparameter($pname, $value, $ename);
      }
    }
    return $parm;
  }
  // .....................................................................
  /**
  * Return dump of the application content as a string. Useful for
  * diagnostics mainly.
  * @return string Dump of the application content as html string.
  */
  function htmldump() {
    $s = "APPLICATION<br>";
    $s .= "Definitions:<br>";
    foreach ($this->definitions as $name => $val) {
      $s .= "$name = '$val'<br>";
    }
    $s .= "Globals:<br>";
    foreach ($this->globals as $name => $val) {
      $s .= "$name = '$val'<br>";
    }
    $s .= "Settings:<br>";
    foreach ($this->settings as $setting) {
      $s .= $setting->htmldump();
    }
    return $s;
  }
  // .....................................................................
  /**
  * Save the application as XML file back to the same filename it was
  * read in from, ie. after changes have been made. This generates
  * the whole file as fresh XML, and writes it out.
  */
  function save() {
    if ($this->valid) {
      $xmlfile = new outputfile($this->configpath);
      if ($xmlfile->opened) {
        $xml = new xmltag("application");

        // DEFINITIONS
        $xmldefs = new xmltag("definitions");
        foreach ($this->definitions as $name => $val) {
          $xmldef = new xmltag("definition", $val);
          $xmldef->setattribute("name", $name);
          $xmldefs->childtag($xmldef);
        }
        $xml->childtag($xmldefs);

        // GLOBALS
        $xmlglobs = new xmltag("globals");
        foreach ($this->globals as $name => $val) {
          $xmlglob = new xmltag("global", $val);
          $xmlglob->setattribute("name", $name);
          $xmlglobs->childtag($xmlglob);
        }
        $xml->childtag($xmlglobs);

        // RESPONSE SETTINGS
        $xmlsettings = new xmltag("settings");
        foreach ($this->settings as $setting) {
          $xmlsetting = new xmltag("setting");
          $xmlsetting->setattribute("name", $setting->name);
          $xmlsetting->setattribute("agent", $setting->agent);
          foreach ($setting->parameters as $parameter) {
            $xmlparm = new xmltag("parameter");
            $xmlparm->setattribute("name", $parameter->name);
            $xmlparm->setattribute("type", $parameter->type);
            if ($parameter->type == "array") {
              if (is_array($parameter->value) && count($parameter->value) > 0) {
                foreach ($parameter->value as $ename => $evalue) {
                  $xmlelem = new xmltag("element", $evalue);
                  $xmlelem->setattribute("name", $ename);
                  $xmlparm->childtag($xmlelem);
                }
              }
            }
            else {
              $xmlparm->value = $parameter->value;
            }
            $xmlsetting->childtag($xmlparm);
          }
          $xmlsettings->childtag($xmlsetting);
        }
        $xml->childtag($xmlsettings);

        // Write the file content..
        $xmlfile->writeln( xmlheader() );
        $xmlfile->writeln("<!-- Axyl XML Configuration File - Please do not edit this file directly. -->");
        $xmlfile->writeln("<!-- The settings in this file determine how your application will run. -->");
        $xmlfile->writeln("<!-- Generation timestamp: " . timestamp_to_displaydate(NICE_FULLDATETIME, time()) . " -->");
        $xmlfile->write( $xml->render() );
        $xmlfile->closefile();
      } // if file opened
    } // if valid
  }
  // .....................................................................
  /** Method invoked when a tag is opened */
  function tag_open($parser, $tag, $attributes) {
    $this->tag = $tag;
    if (is_array($attributes) && count($attributes) > 0) {
      foreach ($attributes as $key => $value ) {
        $this->attr[$key] = $value;
      }
    }
    switch ($this->tag) {
      case "definition":
        $this->definitions[ $this->attr["name"] ] = "";
        break;
      case "global":
        $this->globals[ $this->attr["name"] ] = "";
        break;
      case "setting":
        $this->setting = new setting( $this->attr["name"], $this->attr["agent"] );
        break;
      case "parameter":
        $this->parameter = new parameter( $this->attr["name"], $this->attr["type"] );
        break;
    } // switch
  }
  // .....................................................................
  /** Method invoked when character data is available */
  function cdata($parser, $cdata) {
    switch ($this->tag) {
      case "definition":
        $this->definitions[ $this->attr["name"] ] = $cdata;
        break;
      case "global":
        $this->globals[ $this->attr["name"] ] = $cdata;
        break;
      case "parameter":
        if (isset($this->parameter) && $this->attr["type"] != "array") {
          $this->parameter->setvalue( $cdata, $this->attr["name"] );
        }
        break;
      case "element":
        if (isset($this->parameter)) {
          $this->parameter->setvalue( $cdata, $this->attr["name"] );
        }
        break;
    } // switch
  }
  // .....................................................................
  /** Method invoked when a tag is closed */
  function tag_close($parser, $tag) {
    switch ($tag) {
      case "setting":
        $this->settings[] = $this->setting;
        unset($this->setting);
        break;
      case "parameter":
        if (isset($this->setting) && isset($this->parameter)) {
          $this->setting->addparameter($this->parameter->name, $this->parameter);
          unset($this->parameter);
        }
        break;
    } // switch
    $this->tag = "";
    $this->attr = array();
  }
  // .....................................................................
  /**
  * Parse the application XML which is provided.
  * @param string $xml The XML content to parse for the application.
  */
  function parse($xml) {
    xmlparser::parse($xml);
    if (!$this->valid_xml) {
      $this->valid = false;
    }
  }

} // application class

// ----------------------------------------------------------------------
/**
* Return the index of the given named setting, or -1 if not found.
* @param object $app The application object to scan
* @param string $settingname The name of the setting in $app
* @return integer The index of the setting in $app
*/
function get_settingindex($app, $settingname) {
  $theix = -1;
  for ($ix = 0; $ix < count($app->settings); $ix++) {
    $setting = $app->settings[$ix];
    if (is_object($setting)) {
      if ($setting->getparameter("name") == $settingname) {
        $theix = $ix;
        break;
      }
    }
  }
  return $theix;
}
// ----------------------------------------------------------------------
?>