<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    block-defs.php                                          */
/* Author:      Paul Waite                                              */
/* Description: Definitions for content block management in webpages.   */
/*                                                                      */
/* ******************************************************************** */
/** @package cm */

/** Form classes */
include_once("form-defs.php");

// ......................................................................
// DEFINITIONS
/** Identity value signifying new block
@access private */
define("NEW_BLOCK", -1);
/** Identity value signifying new blocklet
@access private */
define("NEW_BLOCKLET", -1);

/** Block/layout version undefined
@access private */
define("VERSION_UNDEFINED", -1);
/** Block/layout version is pending
@access private */
define("VERSION_PENDING", 0);
/** Block/layout version is live
@access private */
define("VERSION_LIVE", 1);
/** Block/layout version is previous
@access private */
define("VERSION_PREVIOUS", 2);

/** Layout cell content: empty
@access private */
define("EMPTY_CELL", "" );
/** Layout cell content: standard block
@access private */
define("BLOCK_CONTENT", "b");
/** Layout cell content: HTMLArea wysiwyg editor
@access private */
define("WYSIWYG_EDITOR", "w");
/** Layout cell content: plain cell
@access private */
define("PLAIN_CELL", "p");

// ......................................................................
/**
* Block
* We define a class called a 'block'. This can contain multiple
* 'blocklet' elements which fill up a block top-to-bottom. A block
* can be divided into multiple columns, and blocklets fill these
* left-to-right.
* Blocklets can be of several defined blocklet_types:
*  Text, List, Ordered list, Bullets, and Table
* A blocklet can also have a heading and a ruler defined for it,
* with various formatting properties. Provision is also made for
* inserting special 'tags' into the blocklet content which are
* translated by the system accordingly. These are:
*   Data      - tags to access database information
*   Images    - reference to a resident image
*   Links     - A clickable link (url)
*   Doc links - A link to a resident document file
* @package cm
*/
class block extends RenderableObject {
  // Public
  /** Whether the block exists in database or not */
  var $exists = false;
  /** The id of the current block */
  var $blockid = 0;
  /** The description of the current block */
  var $block_desc;
  /** The language of the block (0 = default) */
  var $language = 0;
  /** The language encoding code */
  var $lang_encoding = "";
  /** The language text direction */
  var $lang_direction = "";
  /** The number of columns in this block */
  var $cols = 1;
  /** Width of inter-column gutter in pixels */
  var $gutter_width = 0;
  /** Colour of inter-column gutter */
  var $gutter_colour = "";
  /** Vertical separation of blocklets in pixels */
  var $blocklet_sep = 0;
  /** Width of border in pixels */
  var $border_width = 0;
  /** Colour of border */
  var $border_colour = "";
  /** Colour of block background */
  var $background_colour = "";
  /** Background image - ID from ax_catalog table, if defined */
  var $background_img = NULLVALUE;
  /** Justification: 'left', 'center', 'right' */
  var $justify = "";
  /** Vertical alignment: 'top', 'middle', 'bottom' */
  var $valign = "";
  /** Manual style entered by user */
  var $block_style = "";
  /** If true, then an EXPORT button will be provided for CSV dump */
  var $exportable = false;
  /** Array of blocklet objects in this block */
  var $blocklets = array();
  /** The layout this block belongs to */
  var $layoutid;
  /** Version of layout this block belongs to (optional). If present
      then the version is included in forms as a hidden field. */
  var $layout_version = VERSION_UNDEFINED;
  /** Count of layout versions in existence. */
  var $layout_version_count = 0;
  /** The language encoding for the containing layout */
  var $layout_lang_encoding = "";
  /** The language direction for the containing layout */
  var $layout_lang_direction = "";
  /** Type of block, ""=empty, "b"=block content, "w"=wysiwyg, "p"=plain cell */
  var $block_type = "";

  // Private
  /** The form name for the current block @access private */
  var $blockfm = "";
  /** Mode of operation, 'viewing', 'editing', 'saving' @access private */
  var $mode = "viewing";
  // ....................................................................
  /**
  * Constructor
  * Create a new block object.
  * Blocks are self-contained entities, and so this constructor is written
  * to return the rendered block content. This just allows you to avoid
  * having to make the render() call, and use the constructor to return
  * the block content in one hit.
  * @param string $id The unique name/identity of the block.
  * $return string The block content, whatever that may be.
  */
  function block($id=NEW_BLOCK) {
    // Deal with new block requirement..
    if ($id == NEW_BLOCK) {
      $id = get_next_sequencevalue("seq_block_id", "ax_block", "block_id");
    }

    // Save block ID..
    $this->blockid = $id;

    // Define a unique form name..
    $this->blockfm = "blockfm_$id";

    // Process anything POSTed via form..
    $this->POSTprocess();

    // Read it all from disk
    $this->get($id);

  } // block
  // ....................................................................
  /**
  * Provide a blockeditor. This is used to instantiate a blockeditor
  * object for when we need to change this block somewhow. We only
  * need one, so we check if it's already been done first.
  */
  function activate_editing() {
    if (!isset($this->blockeditor)) {
      global $RESPONSE, $LIBDIR;
      include_once("block-editor-defs.php");
      $this->blockeditor = new blockeditor($this);
    }
  } // activate_editing
  // ....................................................................
  /**
  * Set the layout info for the layout which contains this block. If
  * defined the version is included in the block edit form as a hidden
  * field so that the layout can be kept across block form submission.
  * Both bits of data are also used to determine what happens regarding
  * for example indexing saved block content etc.
  * @param string $lver The version of the layout containing this block
  * @param integer $vercount The # versions of the layout
  * @param integer $lang The language ID of the layout
  * @param string $lang_enc The language encoding of the layout
  * @param string $lang_dir The language direction of the layout
  * @access private
  */
  function set_layout_info($ver=VERSION_UNDEFINED, $vercount=0, $lang=0, $lang_enc="", $lang_dir="") {
    $this->layout_version = $ver;
    $this->layout_version_count = $vercount;
    $this->layout_language = $lang;
    $this->layout_lang_encoding = $lang_enc;
    $this->layout_lang_direction = $lang_dir;
  } // set_layout_info
  // ....................................................................
  /**
  * Return true if the current user is permitted to edit block details.
  * We allow editing only for versions VERSION_PENDING and VERSION_LIVE
  * and the latter only for Editors.
  * @return boolean True if editing is permitted by current user.
  */
  function user_can_edit($required_version=VERSION_UNDEFINED) {
    global $RESPONSE;
    if ($this->layout_version == VERSION_UNDEFINED) {
      return true;
    }
    $perm = false;
    if ($required_version == VERSION_UNDEFINED ||
        $required_version == $this->layout_version) {
      // Pending version
      if ($this->layout_version == VERSION_PENDING ||
          $this->layout_version_count == 1) {
        if ($RESPONSE->ismemberof_group_in("Editor,Author")) {
          $perm = true;
        }
      }
      // Live version
      elseif ($this->layout_version == VERSION_LIVE) {
        if ($RESPONSE->ismemberof_group_in("Editor")) {
          $perm = true;
        }
      }
    }
    return $perm;
  } // user_can_edit
  // ....................................................................
  /**
  * Get the block.
  * Retrieves the specified block from database.
  * @param string $id The unique name/identity of the block to get
  */
  function get($id="") {
    global $RESPONSE;
    debug_trace($this);
    $this->exists = false;

    // Assign the ID if given..
    if ($id != "") $this->blockid = $id;

    // Try and find it..
    if ($RESPONSE->multilang) {
      $q  = "SELECT * FROM ax_block, ax_language";
      $q .= " WHERE ax_block.block_id='" . addslashes($this->blockid) . "'";
      $q .= "   AND ax_language.lang_id=ax_block.lang_id";
    }
    else {
      $q  = "SELECT * FROM ax_block";
      $q .= " WHERE ax_block.block_id='" . addslashes($this->blockid) . "'";
    }
    $frq = dbrecordset($q);
    if ($frq->hasdata) {
      $this->layoutid = $frq->field("layout_id");
      $this->block_desc = $frq->field("block_desc");
      $this->cols = $frq->field("cols");
      if ($this->cols == "" || $this->cols <= 0) {
        $this->cols = 1;
      }
      if ($RESPONSE->multilang) {
        $this->language          = $frq->field("lang_id");
        $this->lang_encoding     = $frq->field("char_encoding");
        $this->lang_direction    = $frq->field("direction");
      }
      $this->gutter_width      = $frq->field("gutter_width");
      $this->gutter_colour     = $frq->field("gutter_colour");
      $this->blocklet_sep      = $frq->field("blocklet_sep");
      $this->background_colour = $frq->field("background_colour");
      $this->background_img    = $frq->field("background_img");
      if ($this->background_img == "") {
        $this->background_img = NULLVALUE;
      }
      $this->justify           = $frq->field("justify");
      $this->valign            = $frq->field("valign");
      $this->border_width      = $frq->field("border_width");
      $this->border_colour     = $frq->field("border_colour");
      $this->block_style       = $frq->field("block_style");
      $this->block_type        = $frq->field("block_type");
      $this->exportable        = $frq->istrue("exportable");
      $this->wysiwyg           = $frq->istrue("wysiwyg");
      $this->exists = true;

      // Get any defined blocklets. We do this here in one query
      // rather than using the blocklet class get() method, which
      // would be very inefficient..
      $q  = "SELECT *";
      $q .= "  FROM ax_block_blocklet bb, ax_blocklet b";
      $q .= " WHERE bb.block_id='" . addslashes($this->blockid) . "'";
      $q .= "   AND b.blocklet_id=bb.blocklet_id";
      $q .= " ORDER BY bb.display_order";
      $frq = dbrecordset($q);
      if ($frq->hasdata) {
        do {
          $blockletid = $frq->field("blocklet_id");
          $blocklet = new blocklet();
          $blocklet->blockletid        = $blockletid;
          $blocklet->visible           = $frq->istrue("visible");
          $blocklet->display_order     = $frq->field("display_order");
          $blocklet->blocklet_desc     = $frq->field("blocklet_desc");
          $blocklet->type              = $frq->field("blocklet_type");
          $blocklet->width             = $frq->field("blocklet_width");
          $blocklet->justify           = $frq->field("justify");
          $blocklet->heading           = $frq->field("heading");
          $blocklet->heading_level     = $frq->field("heading_level");
          $blocklet->heading_colour    = $frq->field("heading_colour");
          $blocklet->ruler             = $frq->field("ruler");
          $blocklet->ruler_width       = $frq->field("ruler_width");
          $blocklet->ruler_size        = $frq->field("ruler_size");
          $blocklet->ruler_colour      = $frq->field("ruler_colour");
          $blocklet->content           = $frq->field("content");
          $blocklet->content_size      = $frq->field("content_size");
          $blocklet->content_colour    = $frq->field("content_colour");
          $blocklet->blocklet_style    = $frq->field("blocklet_style");
          $blocklet->table_style       = $frq->field("table_style");
          $blocklet->table_autojustify = $frq->istrue("table_autojustify");
          $blocklet->table_rowstripes  = $frq->istrue("table_rowstripes");
          // Stash blocklet..
          $this->blocklets[$blockletid] = $blocklet;
        } while ($frq->get_next());
      }
    }
    debug_trace();
    // Return true if at least the block exists..
    return $this->exists;
  } // get
  // ....................................................................
  /**
  * Save the block.
  * Save this block to the database. Create a new one if it
  * doesn't already exist.
  */
  function put() {
    debug_trace($this);
    // Deal with brand new block..
    start_transaction();
    if ($this->exists) {
      $frq = new dbupdate("ax_block");
      $frq->where("block_id=$this->blockid");
    }
    else {
      $frq = new dbinsert("ax_block");
      $frq->set("block_id", $this->blockid);
    }
    $frq->set("layout_id",         $this->layoutid);
    $frq->set("block_desc",        $this->block_desc);
    $frq->set("lang_id",           $this->language);
    $frq->set("cols",              $this->cols);
    $frq->set("gutter_width",      $this->gutter_width);
    $frq->set("gutter_colour",     $this->gutter_colour);
    $frq->set("background_colour", $this->background_colour);
    $frq->set("background_img",    $this->background_img);
    $frq->set("justify",           $this->justify);
    $frq->set("valign",            $this->valign);
    $frq->set("border_width",      $this->border_width);
    $frq->set("border_colour",     $this->border_colour);
    $frq->set("blocklet_sep",      $this->blocklet_sep);
    $frq->set("block_style",       $this->block_style);
    $frq->set("block_type",        $this->block_type);
    $frq->set("exportable",        $this->exportable);
    $frq->set("last_modified",     'now()');
    $this->exists = $frq->execute();

    // Save any blocklets..
    if (isset($this->blocklets)) {
      foreach ($this->blocklets as $blocklet) {
        $blockletexists = $blocklet->exists;
        $blocklet->put();
        if ($blockletexists) {
          $frq = new dbupdate("ax_block_blocklet");
          $frq->where("block_id=$this->blockid");
          $frq->where("AND blocklet_id=$blocklet->blockletid");
        }
        else {
          $frq = new dbinsert("ax_block_blocklet");
          $frq->set("block_id", $this->blockid);
          $frq->set("blocklet_id", $blocklet->blockletid);
        }
        $frq->set("visible",       $blocklet->visible);
        $frq->set("display_order", $blocklet->display_order);
        $frq->execute();
      } // foreach
    }
    commit();
    debug_trace();
  } // put
  // ....................................................................
  /**
  * Replicate this block into a new block with a new set of blocklets
  * as a complete content copy of this original block.
  * NOTE: We end up with this current block as the replicated one.
  */
  function replicate($layoutname="") {
    $this->activate_editing();
    $this->blockeditor->replicate($layoutname);
  } // replicate
  // ....................................................................
  /**
  * Add a new blocklet to the block.
  * This adds a new blocklet to the list, but does not add any records
  * to the database.
  */
  function add_blocklet() {
    debug_trace($this);
    $blocklet = new blocklet();
    $blocklet->blockletid = get_next_sequencevalue("seq_blocklet_id", "ax_blocklet", "blocklet_id");
    $blocklet->display_order = 999;
    $this->blocklets[$blocklet->blockletid] = $blocklet;
    debug_trace();
  } // add_blocklet
  // ....................................................................
  /**
  * Remove blocklet from the block.
  * We remove the entry from the block_blocklet link table, and, if it
  * is the last link involving the blocklet we delete the blocklet record.
  */
  function remove_blocklet($id) {
    debug_trace($this);
    if (isset($this->blocklets[$id])) {
      $q  = "DELETE FROM ax_block_blocklet";
      $q .= " WHERE block_id=$this->blockid";
      $q .= "   AND blocklet_id=$id";
      dbcommand($q);
      $chkq = "SELECT * FROM ax_block_blocklet WHERE blocklet_id=$id";
      if (!$chkq->hasdata) {
        $blocklet = $this->blocklets[$id];
        $blocklet->delete();
      }
      // Remove it from our list..
      unset($this->blocklets[$id]);
    }
    debug_trace();

  } // remove_blocklet
  // ....................................................................
  /**
  * Delete this block from the database. NB: we do not rely on RI to do
  * this since various versions of Postgres don't support this nicely.
  * All related entities are explicitly deleted in a transaction.
  */
  function delete() {
    $this->activate_editing();
    $this->blockeditor->delete();
  } // delete
  // ....................................................................
  /**
  * Index the block.
  * Index all blocklets of this block using Lucene, if Lucene indexing is
  * enabled (via the configvalue 'Lucene Site Indexing'). This has no effect
  * unless you are using the Axyl framework.
  * Notes: This method indexes the blocklets in this block which are at
  * the present time in the database, ie. not the blocklets as defined in
  * this block object. This method is usually called from the POSTprocess()
  * method, just after saving any changes to the database.
  * @param string $path The relative path to the webpage this block is in
  * @param string $title The title of the webpage this block is in
  * @param string $category The category string to index content with
  * @param string $metadata Metadata object containing metadata to index
  */
  function index($path, $title, $category="", $metadata=false) {
    global $RESPONSE, $CONTEXT;
    debug_trace($this);
    if ( isset($CONTEXT) && $CONTEXT->configvalue("Lucene Site Indexing") ) {
      include_once("lucene-defs.php");
      // Read content from DB since it may have just been saved..
      $q  = "SELECT b.heading, b.content";
      $q .= "  FROM ax_block_blocklet bb, ax_blocklet b";
      $q .= " WHERE bb.block_id=$this->blockid";
      $q .= "   AND b.blocklet_id=bb.blocklet_id";
      $bQ = dbrecordset($q);
      $allcontent = array();
      $allcontent[] = $RESPONSE->head->title;
      if ($bQ->hasdata) {
        do {
          $allcontent[] = $bQ->field("heading");
          $allcontent[] = $bQ->field("content");
        } while ($bQ->get_next());
      }
      $I = new lucene_indexmsg();
      $I->index_field("category:Text", ($category == "" ? "sitecontent" : $category));
      $I->index_field("path:Text", $path);
      $I->index_field("title:Text", $title);
      $I->index_field("lastmodified:Date", time());
      if ($RESPONSE->multilang && $this->language > 0) {
        $I->index_field("language:Text", $this->language);
      }
      // Add in any metadata fields being supplied..
      if ($metadata !== false && count($metadata->metadata_elements) > 0) {
        foreach ($metadata->metadata_elements as $element_id => $element) {
          if ($element->tag_name != "" && $element->tag_value != "") {
            $I->index_field($element->tag_name . ":Text", $element->tag_value);
          }
        }
      }
      $I->index_content("AXYLBLOCKID_$this->blockid", implode(" ", $allcontent));
      $I->send();
    }
    debug_trace();
  } // index
  // ....................................................................
  /**
  * Un-index the block.
  * Un-index this block from Lucene, if Lucene indexing is enabled (via
  * the configvalue 'Lucene Site Indexing'). This has no effect if you are
  * not using the Axyl framework.
  */
  function unindex() {
    global $CONTEXT;
    debug_trace($this);
    // Deal with Lucene indexing if enabled. In this case we then
    // use the unique block_id as the index ID, and unindex from
    // Lucene so that references to this block are removed..
    if ( isset($CONTEXT) && $CONTEXT->configvalue("Lucene Site Indexing") ) {
      include_once("lucene-defs.php");
      $UI = new lucene_unindexmsg();
      $UI->unindex("AXYLBLOCKID_$this->blockid");
      $UI->send();
    }
    debug_trace();
  } // unindex
  // ....................................................................
  /**
  * Render the Wysiwyg editing suite.
  * @return string The HTML for the editing suite form etc.
  * @access private
  */
  function wysiwyg_editform($lang_encoding, $lang_direction) {
    $this->activate_editing();
    return $this->blockeditor->wysiwyg_editform($lang_encoding, $lang_direction);
  } // wysiwyg_editform
  // ....................................................................
  /**
  * Render the block editing suite.
  * @return string The HTML for the editing suite form etc.
  * @access private
  */
  function block_editform($lang_encoding, $lang_direction) {
    $this->activate_editing();
    return $this->blockeditor->block_editform($lang_encoding, $lang_direction);
  } // block_editform
  // ....................................................................
  /**
  * Render the block content.
  * @return string The HTML
  * @access private
  */
  function blockcontent($lang_encoding, $lang_direction) {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;
    $editing = false;

    $s = "";

    // Render the blocklets in a table..
    $Tvw = new table($this->blockid);

    // Apply supplemental style, if given..
    if ($this->block_style != "") {
      $Tvw->setstyle($this->block_style);
    }

    // Apply background image if there is one..
    if ($this->background_img != NULLVALUE) {
      $qQ = dbrecordset("SELECT * FROM ax_catalog WHERE cat_id=$this->background_img");
      if ($qQ->hasdata) {
        $src = $qQ->field("filepath");
        $Tvw->setbackground($src);
      }
    }

    // Apply border styles, if we have a border..
    $bdrstyle = "";
    if ($this->border_width > 0) {
      $bdrstyle  = "border-width:" . $this->border_width  . "px;";
      $bdrstyle .= "border-color:" . $this->border_colour . ";";
      $bdrstyle .= "border-style:solid;";
      $Tvw->setstyle($bdrstyle);
    }

    if ($this->mode != "previewing" && $RESPONSE->ismemberof_group_in("Editor,Author")) {
      $editor = true;
      if ($bdrstyle == "") {
        $Tvw->setstyle("border-width:1px;border-style:dotted;border-color:#ff0000;");
      }
      // Tools for the toolbar
      $toolbar = array();
      if ($this->user_can_edit()) {
        $toolbar[] = new form_imagebutton("_edit", "", "", "$LIBDIR/img/_edit.gif", "Edit block", 42, 15);
      }

      // Toolbar table
      $Tbar = new table("toolbar");
      $Tbar->tr("axtitle");
      $Tbar->th("[B$this->blockid]", "axtitle");
      $tools = "";
      foreach ($toolbar as $tool) {
        $tools .= $tool->render();
      }
      $Tbar->th($tools, "axtitle");
      $Tbar->th_css("text-align:right");

      // Put toolbar into table..
      $Tvw->tr("axtitle");
      $Tvw->td( $Tbar->render(), "axtitle" );
    }

    // Generate an ordered array of our visible blocklets..
    if (isset($this->blocklets)) {
      $ordblocklets = array();
      foreach ($this->blocklets as $blocklet) {
        if ($blocklet->visible) {
          $ordblocklets[$blocklet->blockletid] = $blocklet->display_order;
        }
      }
      asort($ordblocklets, SORT_NUMERIC);
      $visible_blocklets = array();
      foreach($ordblocklets as $blockletid => $order) {
        $visible_blocklets[] = $blockletid;
      }

      // Ascertain blocklet display metrics..
      $tot_blocklets = count($visible_blocklets);
      $tot_rows = ceil($tot_blocklets / $this->cols);

      // Render the blocklets..
      $Tvw->tr();
      $Tvw->td();
      $Tvw->td_alignment("", "top");

      $Tbl = new table("B" . $this->blockid);
      if ($this->background_colour != "") {
        $Tbl->setbgcolor($this->background_colour);
      }
      $Tbl->tr();
      $bc = 0;
      $col = 1;
      $widthpct = floor(100 / $this->cols);
      foreach ($visible_blocklets as $blockletid) {
        $blocklet = $this->blocklets[$blockletid];
        if ($bc++ % $tot_rows == 0) {
          if (isset($Tcol)) {
            // Put the last blocklets column in..
            $Tbl->td_content($Tcol->render());
            unset($Tcol);

            // If multi-column and guttering enabled, make a
            // gutter column here..
            if ($this->cols > 1 && $this->gutter_width > 0) {
              $gutc = (($this->gutter_colour != "") ? $this->gutter_colour : "black");
              $Tbl->td();
              //$Tbl->td_metrics($this->gutter_width);
              $guts .= "border-right:" . $this->gutter_width . "px ";
              $guts .= " solid";
              $guts .= " $this->gutter_colour;";
              $Tbl->td_css($guts);
            }
          }
          // Make new cell for next column..
          $Tbl->td();
          $Tbl->td_alignment("", "top");
          if ($bc == $tot_blocklets) $widthpct += 1;
          $Tbl->td_metrics("$widthpct%");
          $Tcol = new table("B" . $this->blockid . "_" . $col++);
        }
        $Tcol->tr();
        $Tcol->td($blocklet->html());
        if ($this->blocklet_sep > 0 && $tot_blocklets > 1 && $bc < $tot_blocklets) {
          $Tcol->tr();
          $Tcol->td();
          $Tcol->td_height($this->blocklet_sep);
        }
      }
      if (isset($Tcol)) {
        $Tbl->td_content($Tcol->render());
      }
      $Tvw->td_content( $Tbl->render() );
    }

    // Render the content table..
    $s .= $Tvw->render();

    debug_trace();
    // Return the html..
    return $s;
  } // blockcontent

  // ....................................................................
  /**
  * Render the block content as a CSV formatted stream. This is designed
  * to facilitate the exporting of complex tables of data as CSV format
  * for importing into spreadsheets, or databases etc.
  * @return string The content in CSV format.
  */
  function csv() {
    debug_trace($this);
    $s = "";
    // Generate an ordered array of our visible blocklets..
    if (isset($this->blocklets)) {
      $ordblocklets = array();
      foreach ($this->blocklets as $blocklet) {
        if ($blocklet->visible) {
          $ordblocklets[$blocklet->blockletid] = $blocklet->display_order;
        }
      }
      asort($ordblocklets, SORT_NUMERIC);
      $visible_blocklets = array();
      while (list($blockletid, $order) = each($ordblocklets)) {
        $visible_blocklets[] = $blockletid;
      }
      foreach ($visible_blocklets as $blockletid) {
        $blocklet = $this->blocklets[$blockletid];
        $s .= $blocklet->csv() . "\n";
      }
    }
    debug_trace();
    // Return the csv stream..
    return $s;
  } // csv

  // ....................................................................
  /**
  * Render the block content according to the mode of operation
  * we are in. Possible modes: 'viewing', 'editing', 'saving'.
  * @return string The HTML
  */
  function html() {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;
    global $edit_blockid;

    // Language details..
    $lang_encoding = "";
    $lang_direction = "";
    $language = $this->language;
    if ($RESPONSE->multilang) {
      if ($this->language != 0) {
        $lang_encoding = $this->lang_encoding;
        $lang_direction = $this->lang_direction;
      }
      else {
        if ($this->layout_language != 0) {
          $language = $this->layout_language;
          $lang_encoding = $this->layout_lang_encoding;
          $lang_direction = $this->layout_lang_direction;
        }
      }
      // Make sure RESPONSE takes note of language setting. If this
      // language was already added, it doesn't matter..
      if ($language != 0) {
        $RESPONSE->add_language($language);
      }
    }

    $s = "";
    // Start form for editing..
    if ($this->exportable || $RESPONSE->ismemberof_group_in("Editor,Author")) {
      $s .= "<form name=\"$this->blockfm\" method=\"post\" enctype=\"multipart/form-data\">";
    }

    switch($this->mode) {
      case "editing":
        // Make sure edit request is meant for us..
        if (!isset($edit_blockid) || $edit_blockid != $this->blockid) {
          return "";
        }
        // Deal with first block edit. In this case it won't yet
        // exist in the database, so we create it here..
        if (!$this->exists) {
          $this->put();
        }
        $this->mode = "saving";
        // Show the appropriate editing form..
        switch ($this->block_type) {
          case WYSIWYG_EDITOR:
            $s .= $this->wysiwyg_editform($lang_encoding, $lang_direction);
            break;
          default:
            $s .= $this->block_editform($lang_encoding, $lang_direction);
        } // switch
        break;

      default:
        if ($this->mode != "viewing" && $this->mode != "previewing") {
          $this->mode = "viewing";
        }
        $s .= $this->blockcontent($lang_encoding, $lang_direction);
        if ($this->exportable) {
          $expbtn = new form_imagebutton("_exportblock");
          $expbtn->setimage("$LIBDIR/img/_export.gif", "Export this information in CSV format");
          $s .= "<p>" . $expbtn->render() . "</p>";
        }
    } // switch

    // Finish the form..
    if ($this->exportable || $RESPONSE->ismemberof_group_in("Editor,Author")) {
      // Include action hidden field, and block ID
      $mode  = new form_hiddenfield("blockmode", $this->mode);
      $blid  = new form_hiddenfield("edit_blockid", $this->blockid);
      $s .= $mode->render() . $blid->render();
      // Layout version hidden field
      if ($this->layout_version != VERSION_UNDEFINED) {
        $lvers = new form_hiddenfield("layout_version", $this->layout_version);
        $lvcnt = new form_hiddenfield("layout_version_count", $this->layout_version_count);
        $s .= $lvers->render() . $lvcnt->render();
      }
      $s .= "</form>\n";
    }

    return $s;
  } // html
  // ....................................................................
  /**
  * Process a block edit form POST.
  * Assume that the fields have been submitted in a form as named
  * in the config, and grab the POSTed values. This method is executed
  * from the constructor usually, before anything is read in from
  * the database. We get first shot to change data here.
  * @param text $id The identity of the set to update from POST
  * @access private
  */
  function POSTprocess() {
    debug_trace($this);
    global $CONTEXT, $RESPONSE, $LIBDIR;
    global $edit_blockid;

    if (isset($edit_blockid) && $edit_blockid == $this->blockid) {
      global $_done_x, $_edit_x, $_exportblock_x, $blockmode;
      $this->mode = $blockmode;
      switch ($this->mode) {
        case "viewing":
          // Check for editing mode..
          if (isset($_edit_x)) {
            $this->mode = "editing";
          }
          // Check for user export button-press. In this
          // case we deliver CSV content, and exit..
          elseif (isset($_exportblock_x)) {
            $exptitle = $RESPONSE->body->title;
            if ($exptitle == "") {
              $exptitle = $this->blockid;
            }
            $this->get();
            $csv = $this->csv();
            $RESPONSE->discard();
            $RESPONSE->set_encoding("", "text/csv");
            header("Content-Disposition: attachment; filename=$exptitle.csv");
            echo $csv;
            exit;
          }
          break;

        case "saving":
          if ($edit_blockid == $this->blockid) {
            global $_csvimport_x, $blocklet_id;
            global $_save_x, $_cancel_x, $_done_x;
            global $_wysiwygpost_form;
            global $_recmaintpost_form;
            global $_recmaintpost_data;
            global $_recmaintpost_flds;
            global $_recmaintpost_dels;
            global $_recmaintpost_order;

            // Let me out of here..
            if (isset($_done_x) || isset($_cancel_x)) {
              // Drop through to viewing..
              $this->mode = "viewing";
            }
            // Posted record maintenance data..
            elseif ( isset($_recmaintpost_form)
                  && $_recmaintpost_form == $this->blockfm) {
              // Posted CSV table data. This data is to be appended to
              // the blocklet content as a table..
              if (isset($_csvimport_x) && isset($blocklet_id)) {
                $up = new fileupload();
                if ($up->uploaded_count >= 1) {
                  // Process a uploaded file(s)..
                  $file_mime = $up->mimetype;
                  if ($file_mime != CONTENT_CSV) {
                    $file_mime = mimetype_from_filename($up->filename);
                  }
                  if ($file_mime == CONTENT_CSV) {
                    // Process the file..
                    $fup = new csv_inputfile($up->filepath);
                    $tablerow = false;
                    $newtable = "";
                    if ($fup->opened) {
                      do {
                        $tablerow = $fup->readln();
                        if ($tablerow) {
                          $newtable .= implode("|", $tablerow) . "\n";
                        }
                      } while ($tablerow);
                      $fup->closefile();
                      if ($newtable != "") {
                        $ob = dbrecordset("SELECT content FROM ax_blocklet WHERE blocklet_id=$blocklet_id");
                        if ($ob->hasdata) {
                          $orig_content = $ob->field("content");
                          if ($orig_content != "") $orig_content .= "\n";
                          $ub = new dbupdate("ax_blocklet");
                          $ub->set("content", $orig_content . $newtable);
                          $ub->set("blocklet_type", "table");
                          $ub->where("blocklet_id=$blocklet_id");
                          $ub->execute();
                        }
                      }
                    }
                    else {
                      $emsg = "failed to open uploaded file: '$up->filename'";
                      $errmsgs[] = $emsg;
                      debugbr($emsg, DBG_DEBUG);
                    }
                  }
                  else {
                    $emsg = "Not a comma-separated variable (CSV) file: '$up->filename'";
                    $errmsgs[] = $emsg;
                    debugbr($emsg, DBG_DEBUG);
                  }
                }
                else {
                  $emsg = "Nothing was uploaded.";
                  $errmsgs[] = $emsg;
                  debugbr($emsg, DBG_DEBUG);
                }
                // Drop through to viewing..
                $this->mode = "editing";

              } // if CSV import

              else {
                // Deal with Save, Delete, Re-ordering etc..
                if (isset($_recmaintpost_dels) && $_recmaintpost_dels != "") {
                  $blocklet_delids = explode(FIELD_DELIM, $_recmaintpost_dels);
                  foreach ($blocklet_delids as $delblockletid) {
                    dbcommand("DELETE FROM ax_blocklet WHERE blocklet_id=$delblockletid");
                  }
                }
                // Blocklet adds and saves..
                if (isset($_recmaintpost_data) && $_recmaintpost_data != "") {
                  $blockletrecs = explode(RECORD_DELIM, $_recmaintpost_data);
                  $blocklet_fields = explode(",", $_recmaintpost_flds);
                  foreach ($blockletrecs as $blockletrec) {
                    $blocklet_values = explode(FIELD_DELIM, $blockletrec);
                    $blockletid = array_shift($blocklet_values);
                    // Cater for new creations..
                    if (strstr($blockletid, "NEW_")) {
                      $savedid = $blockletid;
                      $blockletid = get_next_sequencevalue("seq_blocklet_id", "ax_blocklet", "blocklet_id");
                      $ib = new dbinsert("ax_blocklet");
                      $ib->set("blocklet_id", $blockletid);
                      $ib->execute();
                      $ibb = new dbinsert("ax_block_blocklet");
                      $ibb->set("block_id", $this->blockid);
                      $ibb->set("blocklet_id", $blockletid);
                      $ibb->set("display_order", 99);
                      $ibb->execute();
                      // Fix up potential re-ordering id..
                      if (isset($_recmaintpost_order)) {
                        $_recmaintpost_order = str_replace("$savedid", "$blockletid", $_recmaintpost_order);
                      }
                    }
                    // Update the blocklet data..
                    $bu = new dbupdate("ax_blocklet");
                    $bu->where("blocklet_id=$blockletid");
                    $pos = 0;
                    foreach ($blocklet_fields as $blocklet_field) {
                      if ($blocklet_field == "visible") {
                        $visible_value = $blocklet_values[$pos++];
                      }
                      else {
                        $bu->set($blocklet_field, $blocklet_values[$pos++]);
                      }
                    }
                    $bu->execute();
                    // Now save the visibility flag..
                    if (isset($visible_value)) {
                      $bu = new dbupdate("ax_block_blocklet");
                      $bu->where("block_id=$this->blockid AND blocklet_id=$blockletid");
                      $bu->set("visible", ($visible_value == "t"));
                      $bu->execute();
                    }
                  } // foreach blockletrecs
                }

                // Save block properties..
                global $block_desc, $language, $block_style;
                global $cols, $gutter_width, $gutter_colour;
                global $background_colour, $background_img;
                global $block_justify, $block_valign;
                global $block_border_width, $block_border_colour;
                global $blocklet_sep, $block_exportable;
                if (isset($cols) && $cols > 0) {
                  $bu = new dbupdate("ax_block");
                  $bu->where("block_id=$this->blockid");
                  $bu->set("block_desc",        $block_desc);
                  $bu->set("lang_id",           $language);
                  $bu->set("cols",              $cols);
                  $bu->set("gutter_width",      $gutter_width);
                  $bu->set("gutter_colour",     $gutter_colour);
                  $bu->set("blocklet_sep",      $blocklet_sep);
                  $bu->set("background_colour", $background_colour);
                  $bu->set("background_img",    $background_img);
                  $bu->set("justify",           $block_justify);
                  $bu->set("valign",            $block_valign);
                  $bu->set("border_width",      $block_border_width);
                  $bu->set("border_colour",     $block_border_colour);
                  $bu->set("block_style",       $block_style);
                  $bu->set("exportable",        isset($block_exportable));
                  $bu->execute();
                }

                // Check/save blocklet ordering..
                if (isset($_recmaintpost_order) && $_recmaintpost_order != "") {
                  $ord = 1;
                  $idlist = explode(FIELD_DELIM, $_recmaintpost_order);
                  foreach ($idlist as $blockletid) {
                    $upd = new dbupdate("ax_block_blocklet");
                    $upd->where("block_id=$this->blockid AND blocklet_id=$blockletid");
                    $upd->set("display_order", $ord);
                    $upd->execute();
                    $ord += 1;
                  }
                }

                // Index the block with current webpage path and title..
                global $layout_version, $layout_version_count;
                if (!isset($layout_version) ||
                    $layout_version == VERSION_LIVE ||
                    $layout_version_count == 1) {
                  debugbr("indexing saved block. Layout ver=$layout_version", DBG_DEBUG);
                  $this->index($RESPONSE->requested, $RESPONSE->head->title);
                }

                // Drop through to viewing..
                $this->mode = "viewing";
              }
            }
            // Is it a wysiwyg post..
            elseif ( isset($_wysiwygpost_form)
                  && $_wysiwygpost_form == $this->blockfm) {
              global $wysiwyg_content;
              $this->get();
              $bb = current($this->blocklets);
              if (isset($bb) && isset($bb->blockletid)) {
                $bbup = new dbupdate("ax_blocklet");
                $bbup->set("content", $wysiwyg_content);
                $bbup->where("blocklet_id=$bb->blockletid");
                $bbup->execute();
                $this->blocklets[$bb->blockletid] = $bb;
              }
              // Also save some of the block properties..
              global $block_desc, $block_style;
              global $background_colour, $background_img;
              global $block_justify, $block_valign;
              global $block_border_width, $block_border_colour;
              $bu = new dbupdate("ax_block");
              $bu->where("block_id=$this->blockid");
              $bu->set("block_desc",        $block_desc);
              $bu->set("background_colour", $background_colour);
              $bu->set("background_img",    $background_img);
              $bu->set("justify",           $block_justify);
              $bu->set("valign",            $block_valign);
              $bu->set("border_width",      $block_border_width);
              $bu->set("border_colour",     $block_border_colour);
              $bu->set("block_style",       $block_style);
              $bu->execute();
            }
          }
          break;
      } // switch
    }
    // If the user preference says so, then set the mode to
    // content preview mode..
    $prefs = new configuration("preferences", $RESPONSE->userid);
    if ($prefs->field_exists("Content Preview Mode")) {
      if ($prefs->value("Content Preview Mode") === true) {
        $this->mode = "previewing";
      }
    }
    debug_trace();
  } // POSTprocess

} // block class

// ----------------------------------------------------------------------
/**
* The blocklet is simply a part of a block. It is almost a 'paragraph' of
* content, excepting that a blocklet might have content of several text
* paragraphs, so it is something slightly different, hence the name. This
* class knows how to get, save and delete blocklets, and how to render
* them as html.
* @package cm
*/
class blocklet extends RenderableObject {
  /** blocklet ID */
  var $blockletid;
  /** Whether blocklet exists in database */
  var $exists = false;
  /** The description of the current blocklet */
  var $desc = "";
  /** The blocklet type: 'text', 'list', 'ordered', 'bullets', or 'table' */
  var $type = "text";
  /** The width % of the blocklet table */
  var $width = 100;
  /** Justification: 'left', 'right', 'centered' */
  var $justify = "left";
  /** blocklet heading */
  var $heading;
  /** blocklet heading level (1-6) */
  var $heading_level = 3;
  /** Heading colour */
  var $heading_colour = "";
  /** blocklet ruler: 'none', 'top', 'bottom' */
  var $ruler = "none";
  /** Ruler width percent: 0-100 */
  var $ruler_width = 100;
  /** Ruler pixel size */
  var $ruler_size = 1;
  /** Ruler colour */
  var $ruler_colour = "";
  /** Blocklet textual content */
  var $content = "";
  /** Content size adjustment -2 thru +2 */
  var $content_size = 0;
  /** Content colour */
  var $content_colour = "";
  /** Manual style to apply to content */
  var $blocklet_style = "";
  /** Table style to apply to content */
  var $table_style = "";
  /** True if we autojustify table */
  var $table_autojustify = false;
  /** True if we autojustify table */
  var $table_rowstripes = false;
  /** Whether blocklet is visible */
  var $visible = true;
  /** Display order of this blocklet */
  var $display_order = 1;
  // ....................................................................
  /**
  * Constructor
  * Create a new blocklet object.
  * @param string $id The unique identity of the blocklet.
  */
  function blocklet($id=NEW_BLOCKLET) {
    $this->blockletid = $id;
    $this->get($id);

  } // blocklet
  // ....................................................................
  /**
  * Get the blocklet.
  * Retrieves the specified blocklet from database.
  * @param string $id The unique integer identity of the blocklet to get.
  */
  function get($id) {
    debug_trace($this);
    $this->exists = false;
    if ($id != NEW_BLOCKLET) {
      // Try and read in existing blocklet info..
      $q  = "SELECT * FROM ax_blocklet";
      $q .= " WHERE blocklet_id=$id";
      $blq = dbrecordset($q);
      if ($blq->hasdata) {
        $this->blocklet_desc     = $blq->field("blocklet_desc");
        $this->type              = $blq->field("blocklet_type");
        $this->justify           = $blq->field("justify");
        $this->heading           = $blq->field("heading");
        $this->heading_level     = $blq->field("heading_level");
        $this->heading_colour    = $blq->field("heading_colour");
        $this->ruler             = $blq->field("ruler");
        $this->ruler_width       = $blq->field("ruler_width");
        $this->ruler_size        = $blq->field("ruler_size");
        $this->ruler_colour      = $blq->field("ruler_colour");
        $this->content           = $blq->field("content");
        $this->content_size      = $blq->field("content_size");
        $this->content_colour    = $blq->field("content_colour");
        $this->blocklet_style    = $blq->field("blocklet_style");
        $this->table_style       = $blq->field("table_style");
        $this->table_autojustify = $blq->istrue("table_autojustify");
        $this->table_rowstripes  = $blq->istrue("table_rowstripes");
        $this->exists = true;
      }
    }
    debug_trace();
    // Return true if at least the blocklet exists..
    return $this->exists;
  } // get
  // ....................................................................
  /**
  * Save the blocklet.
  * Save this blocklet to the database. Create a new one if it
  * doesn't already exist.
  */
  function put() {
    debug_trace($this);
    // Deal with brand new blocklet..
    if (!$this->exists) {
      // If we are in need of a new ID, then get one..
      if ($this->blockletid == NEW_BLOCKLET) {
        $this->blockletid = get_next_sequencevalue("seq_blocklet_id", "ax_blocklet", "blocklet_id");
      }
      $blq = new dbinsert("ax_blocklet");
      $blq->set("blocklet_id", $this->blockletid);
    }
    else {
      $blq = new dbupdate("ax_blocklet");
      $blq->where("blocklet_id=$this->blockletid");
    }
    $blq->set("blocklet_desc",     $this->blocklet_desc);
    $blq->set("blocklet_type",     $this->type);
    $blq->set("justify",           $this->justify);
    $blq->set("heading",           $this->heading);
    $blq->set("heading_level",     $this->heading_level);
    $blq->set("heading_colour",    $this->heading_colour);
    $blq->set("ruler",             $this->ruler);
    $blq->set("ruler_width",       $this->ruler_width);
    $blq->set("ruler_size",        $this->ruler_size);
    $blq->set("ruler_colour",      $this->ruler_colour);
    $blq->set("content",           $this->content);
    $blq->set("content_size",      $this->content_size);
    $blq->set("content_colour",    $this->content_colour);
    $blq->set("blocklet_style",    $this->blocklet_style);
    $blq->set("table_style",       $this->table_style);
    $blq->set("table_autojustify", $this->table_autojustify);
    $blq->set("table_rowstripes",  $this->table_rowstripes);
    $this->exists = $blq->execute();
    debug_trace();
  } // put
  // ....................................................................
  /**
  * Delete the blocklet.
  * Delete this blocklet from the database.
  */
  function delete() {
    debug_trace($this);
    // Don't assume RI wil delete the block_blocklets of this blocklet..
    $del = new dbdelete("ax_block_blocklet");
    $del->where("blocklet_id=$this->blockletid");
    $del->execute();
    $del = new dbdelete("ax_blocklet");
    $del->where("blocklet_id=$this->blockletid");
    $del->execute();
    debug_trace();
  } // delete
  // ....................................................................
  /**
  * Content style.
  * Returns an internal class style string based on the values of
  * properties: content_colour and content_size. Returns the string
  * as bare style specs 'font-size:0.8em;' etc.
  * $access private
  */
  function content_css() {
    $s = "";
    if ($this->content_colour != "") {
      $s .= "color:$this->content_colour;";
    }
    if ($this->content_size != 0) {
      $em = number_format( (10 + $this->content_size)/10, 2);
      $s .= "font-size:$em" . "em;";
    }
    if ($this->blocklet_style != "") {
      $s .= $this->blocklet_style;
      if (substr($this->blocklet_style, -1) != ";") {
        $s .= ";";
      }
    }
    return $s;
  } // content_css
  // ....................................................................
  /**
  * Content style.
  * Returns an internal class style string based on the values of
  * properties: content_colour and content_size. Returns the string
  * as a full 'style="..." style, ready for a <span> tag etc..
  * $access private
  */
  function content_style() {
    $s = $this->content_css();
    if ($s != "") $s = " style=\"$s\"";
    return $s;
  } // content_style
  // ....................................................................
  /**
  * Format Content.
  * Formats the given content according to the given content type.
  * $access private
  */
  function format_content($content, $type="text", $css="") {
    $s = "";
    if (is_numeric($content)) $content = number_format($content);
    switch ($type) {
      // Vanilla list..
      case "list":
        $list = new vl($css);
        $list->items = explode("\n", $content);
        $s = $list->render();
        break;
      // Ordered list..
      case "ordered":
        $list = new ol($css);
        $list->items = explode("\n", $content);
        $s = $list->render();
        break;
      // Bullet points list..
      case "bullets":
        $list = new ul($css);
        $list->items = explode("\n", $content);
        $s = $list->render();
        break;

      case "table":
        $t = new table("BKT_$this->blockletid");
        if ($this->table_autojustify) {
          $t->autojustify();
        }
        if ($this->table_rowstripes) {
          $t->rowstripes("axyl_rowstripe_dark,axyl_rowstripe_lite");
        }
        $t->setpadding(2);
        if ($this->table_style != "") {
          $t->setcss($this->table_style);
        }
        // Assemble table..
        $wprofile = "";
        $rows = explode("\n", $content);
        if (count($rows) > 0) {
          foreach ($rows as $row) {
            if (strstr($row, "~")) {
              $wprofile = trim(str_replace("~", ",", $row));
            }
            else {
              $t->tr();
              // Heading rows use '^' separator..
              if (strstr($row, "^")) {
                $cells = explode("^", $row);
                foreach ($cells as $cell) {
                  $t->th($cell);
                  $t->th_contentcss($css);
                }
              }
              // Normal body rows use '|' separator..
              else {
                $cells = explode("|", $row);
                foreach ($cells as $cell) {
                  $t->td($cell);
                  $t->td_alignment("", "top");
                  $t->td_contentcss($css);
                }
              }
            }
          } // foreach row
        }
        // Apply any width profile we got..
        if ($wprofile != "") {
          $t->set_width_profile($wprofile);
        }
        // Now put together the whole table..
        $s .= $t->render();
        break;

      // Text content..
      case "text":
        // Paragraph content..
        $style = $css;
        if ($style != "") $style = " style=\"$style\"";
        $para = "<p" . $style . ">";
        $content = str_replace("\r\n\r\n", "$para", $content);
        $s .= $sizetags . $para . $content . $sizeuntags;
        break;

      // Other content, raw content
      default:
        $s .= $sizetags . $content . $sizeuntags;
        break;

    } // switch

    // Return content
    return $s;
  } // format_content

  // ....................................................................
  /**
  * Render the blocklet as a CSV stream. We split the content types for
  * this into 'table' and 'other'. For tables we use the "|" delimiter to
  * determine where our commas go. For other content we basically just
  * leave it as-is, only adding quotes.
  * @return string The CSV formatted blocklet content.
  */
  function csv() {
    debug_trace($this);
    global $RESPONSE;

    $content = strip_tags( $this->expanded_content() );
    $content = str_replace("\r", "\n", $content);

    // Format depending on type: 'text', 'list', 'ordered',
    // 'bullets', or 'table'
    switch ($this->type) {
      case "table":
        $rows = explode("\n", $content);
        $content = "";
        if (count($rows) > 0) {
          foreach ($rows as $row) {
            $csvrow = "";
            if ($row != "" && $row != "\n" && !strstr($row, "~")) {
              $row = str_replace("^", "|", $row);
              $row = str_replace("\"", "\"\"", $row);
              $bits = explode("|", $row);
              $csvrow = "";
              foreach ($bits as $bit) {
                $csvrow .= "\"$bit\",";
              }
              $csvrow = substr($csvrow, 0, -1);
            }
            $content .= $csvrow;
            if (substr($content, -1, 1) != "\n") {
              $content .= "\n";
            }
          }
        }
        break;

      default:
        $content = str_replace("\"", "\"\"", $content);
        $content = "\"$content\"";
        break;
    }
    debug_trace();
    // Return the CSV..
    return $content;
  } // csv
  // ....................................................................
  /**
  * Process the current blocklet content for special tags. This effectively
  * expands the special tags into 'real' content from database, or as image
  * references etc. We return the expanded content as a string.
  * @return string The new content, expanded as appropriate to the tags.
  * $access private
  */
  function expanded_content() {
    // Pre-process content for datasources..
    $content = $this->content;
    $matches = array();
    $datapat = "/<object .*?axyl\/embedded-media.*?>|<img .*?axyl\/embedded-media.*?>/i";
    while ( preg_match($datapat, $content, $matches) ) {
      $datatag = $matches[0];
      $attributes = extractAttributes($datatag);
      $new_content = "";
      $type = strtolower($attributes["type"]);
      if (count($attributes) > 0 && $type == "axyl/embedded-media") {
        $mediatype = strtolower($attributes["codetype"]);
        switch ($mediatype) {
          // DATA
          case "data":
            $quid   = $attributes["id"];
            $format = $attributes["format"];
            $where  = $attributes["where"];
            $tableheadings = (isset($attributes["tableheadings"]));
            if ($quid != "") {
              $qQ = dbrecordset("SELECT * FROM ax_query_resource WHERE quid=$quid");
              if ($qQ->hasdata) {
                // Datasource content..
                if ($qQ->field("q_query") != "") {
                  $datasrc = unserialize($qQ->field("q_query"));
                  // Apply any block-level where clause..
                  if ($where != "") {
                    if ($datasrc->where->total > 0) {
                      if (strtolower(substr($where, 0, 3)) != "and") {
                        $where = "and $where";
                      }
                    }
                    $datasrc->where($where);
                  }
                  $raw_content = "";
                  // Handle table headings if required..
                  if ($format == "table" && $tableheadings) {
                    $fieldnames = explode(",", $datasrc->fields->listed());
                    $headings = array();
                    foreach ($fieldnames as $fieldname) {
                      if (strstr($fieldname, ".")) {
                        $bits = explode(".", $fieldname);
                        $fieldname = $bits[1];
                      }
                      $headings[] = ucfirst(str_replace("_", " ", $fieldname));
                    }
                    $raw_content .= implode("^", $headings) . "\n";
                  }
                  // Get datasource data..
                  $datasrc->execute();
                  if ($datasrc->hasdata) {
                    do {
                      $data = array();
                      for ($i=0; $i < $datasrc->fields->total; $i++) {
                        $data[] = $datasrc->current_row[$i];
                      }
                      switch ($format) {
                        case "list":
                        case "ordered":
                        case "bullets":
                          $raw_content .= implode(" ", $data) . "\n";
                          break;
                        case "table":
                          $raw_content .= implode("|", $data) . "\n";
                          break;
                        default:
                          $raw_content .= implode(" ", $data);
                      } // switch
                    } while ($datasrc->get_next());
                    $new_content = $this->format_content(
                                    $raw_content,
                                    $format,
                                    $this->content_css()
                                    );
                  }
                }
                // Script-generated content..
                elseif ($qQ->field("q_script") != "") {
                  debugbr("execute script here.", DBG_DEBUG);
                }
              }
            }
            // case data
            break;

          // IMAGE
          case "image":
            $catid   = $attributes["id"];
            $align   = $attributes["align"];
            $pad     = $attributes["pad"];
            $tooltip = $attributes["title"];
            $width   = $attributes["width"];
            $height  = $attributes["height"];
            $border  = $attributes["border"];
            $bdcolor = $attributes["bordercolor"];
            if ($catid != "") {
              $qQ = dbrecordset("SELECT * FROM ax_catalog WHERE cat_id=$catid");
              if ($qQ->hasdata) {
                $src = $qQ->field("filepath");
                if ($width == 0 || $width == "") $width = $qQ->field("width");
                if ($width == 0 || $width == "") $width = false;
                if ($height == 0 || $height == "") $height = $qQ->field("height");
                if ($height == 0 || $height == "") $height = false;
                $img=new img($src, $catid, $tooltip, $width, $height);
                if ($border != "") {
                  $img->setborder($border);
                  $img->setstyle("border-style:solid;border-width:" . $border . "px;");
                  if ($bdcolor != "") $img->setstyle("border-color:$bdcolor;");
                }
                if ($align != "") $img->setalign($align);
                if ($pad != "") {
                  $img->sethspace($pad);
                  $img->setvspace($pad);
                }
                $new_content = $img->render();
              }
            }
            break;

          // MEDIA or DOCUMENT
          case "media":
          case "document":
            $new_content = "";
            $catid   = $attributes["id"];
            $display = $attributes["display"];
            $width   = $attributes["width"];
            $height  = $attributes["height"];
            if ($mediatype == "media") {
              $autostart    = ($attributes["autostart"] == "yes");
              $loop         = ($attributes["loop"] == "yes");
              $showcontrols = ($attributes["showcontrols"] == "yes");
            }
            if ($width  == 0 || $width  == "") $width = 200;
            if ($height == 0 || $height == "") $height = 200;
            $tooltip = $attributes["title"];
            if ($catid != "") {
              $qQ = dbrecordset("SELECT * FROM ax_catalog WHERE cat_id=$catid");
              if ($qQ->hasdata) {
                $src = $qQ->field("filepath");
                $category = $qQ->field("mime_category");
                switch ($category) {
                  case "movie":
                    $media = new MediaObject($src, $width, $height, $autostart, $loop, $showcontrols);
                    switch ($display) {
                      case "icon": $media->AsIcon($tooltip); break;
                      case "link": $media->AsLink($tooltip); break;
                    }
                    break;
                  case "audio":
                    $media = new MediaObject($src, $width, $height, $autostart, $loop, $showcontrols);
                    switch ($display) {
                      case "icon": $media->AsIcon($tooltip); break;
                      case "link": $media->AsLink($tooltip); break;
                    }
                    break;
                  case "flash":
                    $media = new FlashObject($src, $width, $height, $autostart, $loop);
                    switch ($display) {
                      case "icon": $media->AsIcon($tooltip); break;
                      case "link": $media->AsLink($tooltip); break;
                    }
                    break;
                  case "document":
                    $media = new DocumentObject($src, $width, $height);
                    switch ($display) {
                      case "icon": $media->AsIcon($tooltip); break;
                      case "link": $media->AsLink($tooltip, "_new"); break;
                    }
                    break;
                } // switch
                if (is_object($media)) {
                  $new_content = $media->render();
                }
              }
            }
            break;
        }// switch
      } // got attributes

      // Replace data tag with new content..
      $content = str_replace($datatag, $new_content, $content);
    } // while

    // Return the processed content..
    return $content;

  } // expanded_content
  // ....................................................................
  /**
  * Render the blocklet.
  * @return string The HTML
  */
  function html() {
    debug_trace($this);

    $s = "";

    // Main Blocklet Table
    $btable = new table("BKT_$this->blockletid");

    // Set table width if specified..
    if ($this->width != "") {
      $btable->setwidth($this->width . "%");
    }

    if ($RESPONSE->browser != BROWSER_NETSCAPE) {
      $btable->setalign($this->justify);
    }

    // TOP RULER
    if ($this->ruler == "top") {
      $btable->tr();
      $rul = new hr("$this->ruler_width%", $this->ruler_size, $this->ruler_colour);
      $btable->td( $rul->render() );
      $btable->td_alignment($this->justify);
    }

    // HEADING
    if ($this->heading != "") {
      $HTAG = "h$this->heading_level";
      $btable->tr();
      $btable->td("<$HTAG>" . $this->heading . "</$HTAG>");
      $btable->td_alignment($this->justify);
      if ($this->heading_colour != "") {
        $btable->td_contentcss("color:$this->heading_colour");
      }
    }

    // CONTENT
    // First, expand any special tags..
    $content = $this->expanded_content();

    // Now, render the content..
    if ($content != "") {
      $btable->tr();
      // Format the content..
      $scell = $this->format_content(
                  $content,
                  $this->type,
                  $this->content_css()
                  );
      // Add the blocklet content to the table..
      $btable->td($scell);
      $btable->td_alignment($this->justify);
    } // if got content

    // BOTTOM RULER
    if ($this->ruler == "bottom") {
      $btable->tr();
      $rul = new hr("$this->ruler_width%", $this->ruler_size, $this->ruler_colour);
      $btable->td( $rul->render() );
      $btable->td_alignment($this->justify);
    }

    // END TABLE DIV
    $s .= $btable->render();

    debug_trace();
    // Return the HTML..
    return $s;
  } // html

} // blocklet class

// ----------------------------------------------------------------------
/** Utility function to extract content attributes from string.
* Incoming Format is:
*   <tagname attrname="attrvalue" attrname="attrvalue" ... >
* Returns an associative array:
* An element 'tagname' will contain the tag name:
*   returnedarray["tagname"] = tagname
* Other elements one for each attribute:
*   returnedarray["attrname"] = attrvalue
* @access private
*/
function extractAttributes($str){
  $attrs = array();
  if (preg_match("/<(.*?) (.*?)>/i", $str, $matches) ) {
    $attrs["tagname"] = $matches[1];
    $attributes = trim($matches[2]);
    if (preg_match_all("/(.*?)=\"(.*?)\"/i", $attributes, $matches)) {
      $total = count($matches[0]);
      for ($tagno = 0; $tagno < $total; $tagno++) {
        if (trim($matches[1][$tagno]) != "") {
          $attrname = trim(strtolower($matches[1][$tagno]));
          $attrval  = trim($matches[2][$tagno]);
          $attrs[$attrname] = $attrval;
        }
      }
    }
  }
  return $attrs;
} // extractAttributes

// ----------------------------------------------------------------------
?>