<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    control-panel.php                                       */
/* Author:      Paul Waite                                              */
/* Description: Axyl control panel                                      */
/*                                                                      */
/* ******************************************************************** */
/** @package core */

// Local library has Axyl images..
$LIBDIR = "/lib";

// Axyl installation settings..
$AXYL_HOME = "";
$AXYL_CONF = "/etc/axyl/axyl.conf";
if (file_exists($AXYL_CONF)) {
  $result = exec("grep \"AXYL_HOME=\" $AXYL_CONF");
  if ($result != "") {
    $bits = explode("=", $result);
    if (is_dir($bits[1])) {
      $AXYL_HOME = $bits[1];
    }
  }
}

// ----------------------------------------------------------------------
// Include required modules..
/** Sundry contants & defs */
require("constants.php");
/** Renderable module defs */
require("renderable.php");
/** Form handling */
require("form-defs.php");
/** Utilities */
require("utils.php");
/** Debugger defs */
require("debugger.php");
/** Record maintainer module */
require("recmaint-defs.php");
/** Application setup */
require("application-defs.php");

// ----------------------------------------------------------------------
// FUNCTIONS
/**
* Determine the index of Nth database entry..
* @access private
*/
function getdbindex($Nth) {
  global $app;
  $dbix = -1; $dbpos = 0;
  for ($ix = 0; $ix < count($app->settings); $ix++) {
    $setting = $app->settings[$ix];
    if ($setting->name == "database") {
      if ($dbpos == $Nth) {
        $dbix = $ix;
        break;
      }
      else {
        $dbpos += 1;
      }
    }
  }
  return $dbix;
}
/**
* Determine the index of last database entry..
* @access private
*/
function getlastdbindex() {
  global $app;
  $dbix = -1;
  for ($ix = 0; $ix < count($app->settings); $ix++) {
    $setting = $app->settings[$ix];
    if ($setting->name == "database") {
      $dbix = $ix;
    }
  }
  return $dbix;
}
/**
* Delete the Nth database entry. Database entries are numbered
* from zero (first database entry) upwards..
* @access private
*/
function deletedbentry($Nth) {
  global $app;
  $dbix = getdbindex($Nth);
  if ($dbix != -1) {
    $setting = $app->settings[$dbix];
    if ($setting->name == "database") {
      unset($app->settings[$dbix]);
    }
  }
  return $dbix;
}

// ----------------------------------------------------------------------
// CONVERSION OF OLD APPLICATION.PHP FILE TO NEW XML SCHEMA
$error = false;
$user_msg = "";
$appfile = new inputfile("application.php");
if ($appfile->opened) {
  $appfile->readall();
  $appfile->closefile();
  $appstuff = $appfile->content;
  if (strstr($appstuff, "\$TEMPLATESDIR =")) {
    if (file_exists("$AXYL_HOME/lib/default-application.xml")) {
      copy("application.php", "application.php.bak");
      if (file_exists("application.php.bak")) {

        if (is_writeable("application.php")) {
          copy("$AXYL_HOME/lib/application.php", "application.php");
          copy("$AXYL_HOME/lib/default-application.xml", "application.xml");

          if (file_exists("application.xml")) {

            $app = new application("application.xml");
            //echo "converting..<br>";

            // DEFINITIONS
            if (preg_match("/define\(\"APP_NAME\",[\s]*\"(.*?)\"/", $appstuff, $matches)) {
              $app->definitions["APP_NAME"] = $matches[1];
              //echo "setting APP_NAME to [" . $matches[1] . "]<br>";
            }
            if (preg_match("/define\(\"APP_PREFIX\",[\s]*\"(.*?)\"/", $appstuff, $matches)) {
              $app->definitions["APP_PREFIX"] = $matches[1];
              //echo "setting APP_PREFIX to [" . $matches[1] . "]<br>";
            }

            // GLOBALS
            if (preg_match("/^[$]TEMPLATESDIR[\s]*\=[\s]*\"(.*?)\"/m", $appstuff, $matches)) {
              $app->globals["TEMPLATESDIR"] = $matches[1];
              //echo "setting TEMPLATESDIR to [" . $matches[1] . "]<br>";
            }
            if (preg_match("/^[$]IMAGESDIR[\s]*\=[\s]*\"(.*?)\"/m", $appstuff, $matches)) {
              $app->globals["IMAGESDIR"] = $matches[1];
              //echo "setting IMAGESDIR to [" . $matches[1] . "]<br>";
            }
            if (preg_match("/^[$]WEBMASTER_PERSON[\s]*\=[\s]*\"(.*?)\"/m", $appstuff, $matches)) {
              $app->globals["WEBMASTER_PERSON"] = $matches[1];
              //echo "setting WEBMASTER_PERSON to [" . $matches[1] . "]<br>";
            }
            if (preg_match("/^[$]WEBMASTER_EMAIL[\s]*\=[\s]*\"(.*?)\"/m", $appstuff, $matches)) {
              $app->globals["WEBMASTER_EMAIL"] = $matches[1];
              //echo "setting WEBMASTER_EMAIL to [" . $matches[1] . "]<br>";
            }

            // SETTINGS
            if (preg_match("/^[$]RESPONSE->set_encoding\(\"(.*?)\"\)/m", $appstuff, $matches)) {
              $app->setparameter($matches[1], "encoding", "encoding");
              //echo "setting char encoding to [" . $matches[1] . "]<br>";
            }
            //else {
            //  echo "char encoding defaulted<br>";
            //}

            if (preg_match("/^[$]RESPONSE->set_blocked_ips\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter($matches[1], "badips", "badips");
              //echo "setting blocked ips to [" . $matches[1] . "]<br>";
            }
            //else {
            //  echo "blocked ips defaulted<br>";
            //}

            if (preg_match("/^[$]RESPONSE->set_sessiontype\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter(($matches[1] == "SESS_DATABASE_BACKED"), "database_backed", "database_backed");
              //echo "setting database-backed is " . ($matches[1] == "SESS_DATABASE_BACKED" ? "true" : "false") . "<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_lifetime\((.*?)\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "SESS_FOREVER":          $life = 315360000; break;
                case "SESS_1_YEAR":           $life = 31536000;  break;
                case "SESS_1_MONTH":          $life = 2592000;   break;
                case "SESS_1_WEEK":           $life = 604800;    break;
                case "SESS_1_DAY":            $life = 86400;     break;
                case "SESS_12_HOURS":         $life = 43200;     break;
                case "SESS_8_HOURS":          $life = 28800;     break;
                case "SESS_4_HOURS":          $life = 14400;     break;
                case "SESS_1_HOUR":           $life = 3600;      break;
                case "SESS_20_MINS":          $life = 1200;      break;
                case "SESS_BROWSER_LIFETIME": $life = -1;        break;
                case "SESS_ZERO_LIFETIME":    $life = 0;         break;
                default:  $life = -1;
              }
              $app->setparameter($life, "lifetime", "lifetime");
              //echo "setting cookie life to [" . $matches[1] . "($life)]<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_cookiename\((.*?)\)/m", $appstuff, $matches)) {
              if ($matches[1] != "APP_PREFIX . \"_session_id\"") {
                $app->setparameter($matches[1], "cookiename", "cookiename");
                //echo "setting cookiename to [" . $matches[1] . "]<br>";
              }
              //else {
              //  echo "setting cookiename to default<br>";
              //}
            }

            if (preg_match("/^[$]RESPONSE->set_keep\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter(($matches[1] == "KEEP_ENABLED"), "keep", "keep");
              //echo "setting keep status " . ($matches[1] == "KEEP_ENABLED" ? "ON" : "OFF") . "<br>";
            }

            if (preg_match("/^[$]RESPONSE->globalise_all\(\)/m", $appstuff, $matches)) {
              $app->setparameter(true, "globalise", "globalise");
              //echo "setting globalise all ON<br>";
            }
            else {
              $app->setparameter(false, "globalise", "globalise");
              //echo "setting globalise all OFF<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_compression_type\((.*?)\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "NO_COMPRESSION":       $comp = 0; break;
                case "BUILTIN_COMPRESSION":  $comp = 1; break;
                case "CUSTOM_COMPRESSION":   $comp = 2; break;
                default:  $comp = 0;
              }
              $app->setparameter($comp, "compression_type", "compression_type");
              //echo "setting compression type to [" . $matches[1] . "($comp)]<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_compression_minsize\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter($matches[1], "compression_threshold", "compression_threshold");
              //echo "setting compression threshold to [" . $matches[1] . "]<br>";
            }
            else {
              //echo "compression threshold is defaulted (0)<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_buffering_mode\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter(($matches[1] == "BUFFERED"), "buffered_output", "buffered_output");
              //echo "setting buffered output " . ($matches[1] == "BUFFERED" ? "ON" : "OFF") . "<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_page_expirysecs\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter($matches[1], "expiry", "expiry");
              //echo "setting page expiry to [" . $matches[1] . "]<br>";
            }
            //else {
            //  echo "compression page expiry is defaulted (-1)<br>";
            //}

            if (preg_match("/^[$]RESPONSE->set_authentication_type\((.*?)\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "NO_AUTHENTICATION":    $auth = 0; break;
                case "HTTP_AUTHENTICATION":  $auth = 1; break;
                case "FORM_AUTHENTICATION":  $auth = 2; break;
                default:  $auth = 2;
              }
              $app->setparameter($auth, "authtype", "authtype");
              //echo "setting authentication type to [" . $matches[1] . "($auth)]<br>";
            }

            if (preg_match("/^[$]RESPONSE->on_authentication_fail\((.*?)(,(.*?))*?\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "AUTHFAIL_DIE_MSG":     $authf = 0; break;
                case "AUTHFAIL_DIE_SILENT":  $authf = 1; break;
                case "AUTHFAIL_REDIRECT":    $authf = 2; break;
                case "AUTHFAIL_GUEST":       $authf = 3; break;
                default:  $authf = 0;
              }
              $app->setparameter($authf, "authfail", "authfailopt");
              //echo "setting auth fail option to [" . $matches[1] . "($authf)]<br>";
            }

            if (isset($matches[3])) {
              $authurl = preg_replace("/['\"]/", "", $matches[3]);
              $app->setparameter($authurl, "authfail", "authfailurl");
              //echo "setting auth fail URL to [$authurl]<br>";
            }
            //else {
            //  echo "no URL<br>";
            //}

            if (preg_match("/^[$]RESPONSE->on_logins_exceeded\((.*?)(,(.*?))*?\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "SESS_ALLOW":           $logexc = 0; break;
                case "SESS_ALLOW_CULL":      $logexc = 1; break;
                case "SESS_BLOCK_MSG":       $logexc = 2; break;
                case "SESS_BLOCK_SILENT":    $logexc = 3; break;
                case "SESS_BLOCK_REDIRECT":  $logexc = 4; break;
                case "SESS_BLOCK_GUEST":     $logexc = 5; break;
                default:  $logexc = 0;
              }
              $app->setparameter($logexc, "loglimit", "logexceedopt");
              //echo "setting logins exceeded option to [" . $matches[1] . "]<br>";
            }

            if (isset($matches[3])) {
              $logexcurl = preg_replace("/['\"]/", "", $matches[3]);
              $app->setparameter($logexcurl, "loglimit", "logexceedurl");
              //echo "setting logins exceeded URL to [$logexcurl]<br>";
            }
            //else {
            //  echo "no URL<br>";
            //}

            if (preg_match("/^[$]RESPONSE->set_persistent_hosts\((.*?)\)/m", $appstuff, $matches)) {
              if ($matches[1] != "\"\"") {
                $app->setparameter($matches[1], "permhosts", "permhosts");
                //echo "setting persistent hosts list to [" . $matches[1] . "]<br>";
              }
              //else {
              //  echo "null persistent hosts list.<br>";
              //}
            }
            //else {
            //  echo "no persistent hosts.<br>";
            //}

            $patt  = "RESPONSE->add_database\(\n";
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // DB type
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // name
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // user
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // password
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // host
            $patt .= "[\s]*?\"(.*?)\".*\n";        // port
            $patt .= "(([\s])*?(DEFAULT_DATASOURCE))*";  // default flag

            // Purge existing database settings..
            $newsettings = array();
            for ($ix=0; $ix < count($app->settings); $ix++) {
              $setting = $app->settings[$ix];
              if ($setting->name != "database") {
                $newsettings[] = $setting;
              }
            }
            $app->settings = $newsettings;

            preg_match_all("/$patt/", $appstuff, $matches);
            for ($i=0; $i < count($matches[0]); $i++) {
              /*
              echo "database defs:<br>";
              echo " type: " . $matches[1][$i] . "<br>";
              echo " name: " . $matches[2][$i] . "<br>";
              echo " user: " . $matches[3][$i] . "<br>";
              echo " pass: " . $matches[4][$i] . "<br>";
              echo " host: " . $matches[5][$i] . "<br>";
              echo " port: " . $matches[6][$i] . "<br>";
              */
              $dbsetting = new setting("database", "add_database");
              $parameter = new parameter("type", "string");
              $parameter->setvalue($matches[1][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("name", "string");
              $parameter->setvalue($matches[2][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("user", "string");
              $parameter->setvalue($matches[3][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("password", "string");
              $parameter->setvalue($matches[4][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("host", "string");
              $parameter->setvalue($matches[5][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("port", "integer");
              $parameter->setvalue($matches[6][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);
              if (isset($matches[9][$i]) && $matches[9][$i] == "DEFAULT_DATASOURCE") {
                $defaultdb = $dbsetting;
                //echo "default DB<br>";
              }
              else {
                $secdbs[] = $dbsetting;
                //echo "secondary DB<br>";
              }
            } // for

            if (isset($defaultdb)) {
              $app->settings[] = $defaultdb;
            }
            if (isset($secdbs)) {
              foreach ($secdbs as $db) {
                $app->settings[] = $db;
              }
            }

            // Save all conversion changes..
            $app->save();
            $user_msg  = "Your application configuration has been converted to XML format. ";
            $user_msg .= "The old file has been saved as 'application.php.bak'. A new file ";
            $user_msg .= "'application.xml' has been created, which should only ever be ";
            $user_msg .= "changed by using this Control Panel.";
          }
          else {
            $error = true;
            $user_msg = "Conversion aborted due to problems creating XML file."
                            . "<br>Please fix & retry.";
          }
        }
        else {
          $error = true;
          $user_msg = "Conversion aborted. File 'application.php must be "
                          . "writeable by webserver.<br>Please fix & retry.";
        }
      } // // application.php.bak exists
      else {
        $error = true;
        $user_msg = "Conversion aborted due to problems backing up data."
                        . "<br>Please fix & retry.";
      }
    } // lib/default-application.xml exists
  } // old format file
} // appfile opened

// ----------------------------------------------------------------------
// FAILSAFE TO DEFAULT
// Failsafe - if no application XML file, copy default into place..
if (!file_exists("application.xml")) {
  if (file_exists("$AXYL_HOME/lib/default-application.xml")) {
    copy("$AXYL_HOME/lib/default-application.xml", "application.xml");
  }
}

// ----------------------------------------------------------------------
// READ XML APPLICATION SETTINGS
// Read in current application..
if (file_exists("application.xml") && is_writeable("application.xml")) {
  $app = new application();
}
else {
  $error = true;
  $user_msg = "Error: Please make 'application.xml' writeable to the webserver.";
}

// ----------------------------------------------------------------------
// SYNCHRONIZE
// Make sure that the current application XML file has all the required
// defs, globals & settings. For this we have to refer to the Axyl HOME
// file default-application.xml, so find Axyl HOME first of all..
if (!$error) {
  if (is_dir($AXYL_HOME)) {
    $defaultapp = new application("$AXYL_HOME/lib/default-application.xml");
    $synced = $app->synchronize($defaultapp);
    if ($synced) {
      $app->save();
      $app = new application();
    }
  }
}

// ----------------------------------------------------------------------
// POST ACTION
// Check if they opted to set things to default..
if (!$error) {
  if (isset($_default_x)) {
    if (file_exists("$AXYL_HOME/lib/default-application.xml")) {
      copy("$AXYL_HOME/lib/default-application.xml", "application.xml");
      $app = new application();
    }
  }
  elseif (isset($_recmaintpost_form) && $_recmaintpost_form == "cpform") {
    /*
    // DEBUGGING: POSTED VARS DUMP
    $s .= "<table border=1 cellpadding=2 cellspacing=0>";
    if (isset($HTTP_POST_VARS)) {
      $s .= "<tr><td colspan=2><h4>POSTed Vars</h4></td></tr>";
      reset($HTTP_POST_VARS);
      while (list($key, $val) = each($HTTP_POST_VARS)) {
        $s .= "<tr><td>$key</td><td>" . displayvar($val) . "</td></tr>";
      }
    }
    $s .= "</table>";
    */
    // Database Definition Deletes
    if (isset($_recmaintpost_dels) && $_recmaintpost_dels != "") {
      $delids = explode(FIELD_DELIM, $_recmaintpost_dels);
      $delixs = array();
      foreach ($delids as $dbid) {
        $ix = getdbindex($dbid);
        if ($ix != -1) {
          $delixs[] = $ix;
        }
      }
      foreach ($delixs as $ix) {
        unset($app->settings[$ix]);
      }
    }

    // DEFINITIONS
    $app->definitions["APP_PREFIX"] = $cp_app_prefix;
    $app->definitions["APP_NAME"] = $cp_app_name;

    // GLOBALS
    $app->globals["TEMPLATESDIR"] = $cp_templatesdir;
    $app->globals["IMAGESDIR"] = $cp_imagesdir;
    $app->globals["CACHEDIR"] = $cp_cachedir;
    $app->globals["CATALOGDIR"] = $cp_catalogdir;
    $app->globals["CMDIR"] = $cp_cmdir;
    $app->globals["INCDIR"] = $cp_incdir;
    $app->globals["WEBMASTER_PERSON"] = $cp_webmaster_person;
    $app->globals["WEBMASTER_EMAIL"] = $cp_webmaster_email;
    $app->globals["SQL_EXEC_THRESHOLD"] = $cp_sql_exec_threshold;

    // DATABASES
    if (isset($_recmaintpost_data) && $_recmaintpost_data != "") {
      $dbrecs = explode(RECORD_DELIM, $_recmaintpost_data);
      $dbfields = explode(",", $_recmaintpost_flds);
      foreach ($dbrecs as $dbrec) {
        $dbvalues = explode(FIELD_DELIM, $dbrec);
        $dbid = array_shift($dbvalues);
        $dbsetting = new setting("database", "add_database");
        $pos = 0;
        foreach ($dbfields as $dbfield) {
          $value = $dbvalues[$pos++];
          switch ($dbfield) {
            case "dbname":
              $parameter = new parameter("name", "string");
              $dbname = $value;
              break;
            case "dbtype":
              $parameter = new parameter("type", "string");
              break;
            case "dbuser":
              $parameter = new parameter("user", "string");
              break;
            case "dbpassword":
              $parameter = new parameter("password", "string");
              break;
            case "dbhost":
              $parameter = new parameter("host", "string");
              break;
            case "dbport":
              $parameter = new parameter("port", "integer");
              break;
          }
          $parameter->setvalue($value);
          $dbsetting->addparameter($parameter->name, $parameter);
        }

        $ix = get_settingindex($app, $dbname);
        if ($ix > -1) {
          $app->settings[$ix] = $dbsetting;
        }
        else {
          // Insert new database at end of existing databases
          // so that they stay pleasingly grouped..
          $lastdbix = getlastdbindex();
          if ($lastdbix == -1) {
            $app->settings[] = $dbsetting;
          }
          else {
            $ix = 0;
            $settings = array();
            foreach ($app->settings as $setting) {
              $settings[] = $setting;
              if ($ix == $lastdbix) {
                $settings[] = $dbsetting;
              }
              $ix += 1;
            }
            $app->settings = $settings;
          }
        }
      } // foreach dbrecs
    } // database save
    // Database ordering - determines default database
    elseif (isset($_recmaintpost_order) && $_recmaintpost_order != "") {
      $dborderings = explode(FIELD_DELIM, $_recmaintpost_order);
      $dbsettings = array();
      foreach ($dborderings as $dborder) {
        $ix = getdbindex($dborder);
        $dbsettings[] = $app->settings[$ix];
      }
      $firstdbix = getdbindex(0);
      for ($ix=0; $ix < count($dbsettings); $ix++) {
        $app->settings[$ix + $firstdbix] = $dbsettings[$ix];
      }
    }

    // SETTINGS
    $app->setparameter($cp_dtd_html, "dtd", "dtd", "html");
    $app->setparameter($cp_dtd_wml,  "dtd", "dtd", "wml");
    if (isset($cp_multilang)) {
      $app->setparameter(true, "multilang", "multilang");
      $app->setparameter("UTF-8", "encoding", "encoding");
    }
    else {
      $app->setparameter(false, "multilang", "multilang");
      $app->setparameter($cp_encoding, "encoding", "encoding");
    }
    $app->setparameter(isset($cp_database_backed),        "database_backed", "database_backed");
    $app->setparameter($cp_permhosts,                     "permhosts", "permhosts");
    $app->setparameter($cp_cookiename,                    "cookiename", "cookiename");
    $app->setparameter($cp_lifetime,                      "lifetime", "lifetime");
    $app->setparameter(isset($cp_guest_browser_lifetime), "guest_browser_lifetime", "guest_browser_lifetime");
    $app->setparameter($cp_expiry,                        "expiry", "expiry");
    $app->setparameter(isset($cp_metadata_enabled),       "metadata_enabled", "metadata_enabled");
    $app->setparameter(isset($cp_buffered_output),        "buffered_output", "buffered_output");
    $app->setparameter($cp_compression_type,              "compression_type", "compression_type");
    $app->setparameter($cp_compression_threshold,         "compression_threshold", "compression_threshold");
    $app->setparameter(isset($cp_keep),                   "keep", "keep");
    $app->setparameter(isset($cp_globalise),              "globalise", "globalise");
    $app->setparameter($cp_authtype,                      "authtype", "authtype");
    $app->setparameter($cp_authfailopt,                   "authfail", "authfailopt");
    $app->setparameter($cp_authfailurl,                   "authfail", "authfailurl");
    $app->setparameter(isset($cp_encrypted_passwords),    "encrypted_passwords", "encrypted_passwords");
    $app->setparameter($cp_logexceedopt,                  "loginlimit", "logexceedopt");
    $app->setparameter($cp_logexceedurl,                  "loginlimit", "logexceedurl");
    $app->setparameter($cp_badips,                        "badips", "badips");
    $app->setparameter(isset($cp_debug_on),               "debug_on", "debug_on");
    // Unpack debug classes..
    $debug_classes = 0;
    foreach ($cp_debug_classes as $class) {
      $debug_classes |= $class;
    }
    $app->setparameter($debug_classes, "debug_classes", "debug_classes");
    // Unpack debug outputs..
    $debug_output = 0;
    foreach ($cp_debug_output as $output) {
      $debug_output |= $output;
    }
    $app->setparameter($debug_output, "debug_output", "debug_output");

    // Save it
    $app->save();
    $app = new application();
  }
}

// ----------------------------------------------------------------------
// BOILERPLATING
$s = <<< EOS
<html>
<head>
<title>Axyl Control Panel</title>
<meta http-equiv="content-type" content="text/html; charset=UTF-8">
<meta name="generator" content="Catalyst IT Axyl">
<style type="text/css">
 margin: 0px 0px 0px 0px;
 font-family: Verdana, Arial, Helvetica, sans-serif;
 color: #605728;
 font-size: 9pt;
 font-style: normal;
 font-weight: normal;
 scrollbar-face-color: #f7f7f7;
 scrollbar-highlight-color: #b2b1b1;
 scrollbar-shadow-color: #b2b1b1;
 scrollbar-3dlight-color: white;
 scrollbar-arrow-color: #c9c9c9;
 scrollbar-track-color: #f5f5f5;
 scrollbar-darkshadow-color: white;
}
p, td, th, ol, ul, li, input, textarea, select {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 9pt;
 font-style: normal;
 font-weight: normal;
 color: #605728;
}
input, textarea, select {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 9pt;
 font-style: normal;
 font-weight: normal;
}
p {
 line-height: 115%;
}
hr {
 height: 1px;
 color: black;
 margin-top: 0;
 margin-bottom: 0;
}
form {
 margin: 0px;
 padding: 0px;
}
a {
 color: #AC9D46;
 text-decoration: none;
}
a:hover {
 color: #AC9D46;
 text-decoration: underline;
}
a:active {
 color: #AC9D46;
}
a:visited {
 color: #AC9D46;
}
th {
 text-align: left;
}

h1, h2, h5, h3, h4, h6 {
 font-family: Verdana, Arial, Helvetica, sans-serif;
 font-weight: bold;
 margin-top: 2px;
 margin-bottom: 2px;
}
h1 { color:#605728; font-size:125%; text-transform:capitalize; }
h2 { color:#605728; font-size:120%; }
h3 { color:#605728; font-size:115%; font-weight:bold;}
h4 { color:#605728; font-size:105%; font-weight:bold;}
h5 { color:#605728; font-size:100%; font-weight:bold;}
h6 { color:#605728; font-size:96%;  font-weight:bold;}
.axform {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 padding: 0px;
}
.axcombo {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 height: 20px;
 padding-left: 2px;
}
.axlistbox {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 padding-left: 2px;
}
.axtxtbox {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 width: 250px;
 height: 22px;
 padding-left: 2px;
 vertical-align: middle;
}
.axmemo {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 width: 250px;
 height: 100px;
 padding-left: 2px;
}
.axdatetime {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 width: 150px;
 height: 22px;
 padding-left: 2px;
}
.axnumbox {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 width: 80px;
 height: 22px;
 padding-left: 2px;
 padding-right: 2px;
 vertical-align: middle;
 text-align: right;
}
.axchkbox {
 vertical-align: middle;
}
.axfmlbl {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 font-weight: normal;
 vertical-align: top;
 color: black;
}
.axtitle {
 font-family: Arial, Helvetica, sans-serif;
 font-size:110%;
 color: white;
 background-color: #66700F;
 font-weight: bold;
}
.axfoot {
 height: 12px;
 background-color: #66700F;
}
.axhdg {
 font-family: Arial, Helvetica, sans-serif;
 font-size:100%;
 color: white;
 background-color: #898437;
 font-weight: bold;
}
.axsubhdg {
 font-family: Arial, Helvetica, sans-serif;
 font-size:100%;
 color: white;
 background-color: #66700F;
 font-weight: bold;
}
.axfg {
 color: #605728;
 font-weight: normal;
}
.axhl {
 color: red;
 font-weight: bold;
}
.axerror {
 color: red;
}
.axbgwhite {
 color: black;
 background-color: white;
}
.axbglite {
 color: black;
 background-color: #EAEBDF;
}
.axbgdark {
 color: white;
 background-color: #DEDFD4;
}
.axbgdarker {
 color: white;
 background-color: #66700F;
}
</style>
<script language="javascript">
var keyfield = new Array();
var curid = new Array();
var newid = new Array();
function setUTF8mode(multilang) {
  if (multilang) {
    document.forms.cpform.cp_encoding.value='UTF-8';
    document.forms.cpform.cp_encoding.disabled=true;
  }
  else {
    document.forms.cpform.cp_encoding.readonly=false;
    document.forms.cpform.cp_encoding.disabled=false;
  }
  return true;
}
</script>
<script type="text/javascript" src="$LIBDIR/js/recmaint.js"></script>
<script type="text/javascript" src="$LIBDIR/js/fieldvalidation.js"></script>
</head>
<body>
EOS;

// ----------------------------------------------------------------------
// MAIN FORM GENERATION

// Width of large form elements..
$fullwidth  = 540;
$halfwidth  = ceil($fullwidth * 0.50);
$thirdwidth = ceil($fullwidth * 0.37);
$quartwidth = ceil($fullwidth * 0.25);
$ewidth   = $halfwidth  . "px"; // Normal text fields
$awidth   = $fullwidth  . "px"; // Full field width
$cbowidth = $quartwidth . "px"; // Normal combos
$cwidth   = $thirdwidth . "px"; // Wide combos

// DATABASE LISTBOX
// Defined early so that buttons can be registered..
$database_listbox = new form_combofield("dbid");
$database_listbox->setclass("axlistbox");
// Make a new record maintainer, and attach the buttons..
$maintainer = new recmaintainer("cpform", $database_listbox);

$bup    = new form_imagebutton("_up",    "", "", "$LIBDIR/img/_up.gif",     "Move up",              57, 15);
$bdown  = new form_imagebutton("_down",  "", "", "$LIBDIR/img/_down.gif",   "Move down",            57, 15);
$bdel   = new form_imagebutton("_del",   "", "", "$LIBDIR/img/_delete.gif", "Delete database",      57, 15);
$badd   = new form_imagebutton("_add",   "", "", "$LIBDIR/img/_add.gif",    "Add new database",     57, 15);
$bsave  = new form_imagebutton("_save",  "", "", "$LIBDIR/img/_save.gif",   "Save your settings",   57, 15);
$breset = new form_imagebutton("_reset", "", "", "$LIBDIR/img/_reset.gif",  "Reverse your changes", 57, 15);
$breset->set_onclick("document.forms.cpform.reset()");
$bdef   = new form_imagebutton("_default", "", "", "$LIBDIR/img/_default.gif", "Apply default configuration", 57, 15);
$bdef->set_confirm_text("This will over-write the current configuration. Continue?");

// Register all relevant buttons to the maintainer..
$maintainer->register_button("up" ,   $bup);
$maintainer->register_button("down",  $bdown);
$maintainer->register_button("del",   $bdel);
$maintainer->register_button("add",   $badd);
$maintainer->register_button("save",  $bsave);

$Tapp = new table();
$Tapp->setwidth($fullwidth);
$Tapp->setalign("center");

// ......................................................................
// Heading
$Tapp->tr("axtitle");
$Tapp->td("<b>AXYL CONTROL PANEL</b>", "axtitle");
$Tapp->td_css("vertical-align:center;height:30px;padding-left:5px;");
if ($user_msg != "") {
  $Tapp->tr("axsubhdg");
  $Tapp->td($user_msg, "color:#F5DD64;text-align:center;");
}
elseif ($synced) {
  $Tapp->tr("axsubhdg");
  $Tapp->td("The Axyl configuration structure was successfully updated.", "color:#F5DD64;text-align:center;");
}

if (!$error) {
  // ......................................................................
  // Toolbar..
  $toolbar = array();
  $toolbar[] = $breset;
  $toolbar[] = $bdef;
  $toolbar[] = $bsave;
  $Tbar = new table("toolbar");
  $Tbar->tr();
  $tools = "";
  foreach ($toolbar as $tool) {
    $tools .= $tool->render();
  }
  $Tbar->th($tools, "text-align:right");
  $Tapp->tr("axbglite");
  $Tapp->td( $Tbar->render() );

  $tbox = new form_textfield();
  $tbox->setstyle("width:$ewidth");
  $tbox->setclass("axtxtbox");

  $chkbox = new form_checkbox();
  $chkbox->setclass("axchkbox");
  $chkbox->setvalue("yes");
  $chkbox->checked = false;

  // ......................................................................
  // DEFINITIONS
  $Tapp->tr("axhdg");
  $Tapp->td("<b>DEFINITIONS</b>", "axhdg");

  $Tin = new table("definitions");
  $Tin->setpadding(2);

  // Installs text field in $Tin table..
  function entryField($label, $fieldname, &$valarray) {
    global $app, $Tin, $tbox, $bg;
    $mybox = $tbox;
    $bg = ($bg == "axbgdark" ? "axbglite" : "axbgdark");
    $Tin->tr($bg);
    $Tin->td( $label, "axfg" );
    $mybox->setvalue($valarray[$fieldname]);
    $Tin->td( $mybox->render("cp_" . strtolower($fieldname)) );
  }
  // Installs info row in $Tin table..
  function infoField($info) {
    global $app, $Tin, $tbox, $bg;
    $bg = ($bg == "axbgdark" ? "axbglite" : "axbgdark");
    $Tin->tr($bg);
    $Tin->td();
    $Tin->td($info, "axfg");
    $Tin->td_css("font-style:italic;font-size:80%");
  }
  // Installs text field in $Tin table..
  function integerField($label, $fieldname, &$valarray, $intlimit, $pxwidth=100) {
    global $app, $Tin, $tbox, $bg;
    $mybox = $tbox;
    $bg = ($bg == "axbgdark" ? "axbglite" : "axbgdark");
    $Tin->tr($bg);
    $Tin->td( $label, "axfg" );
    $mybox->setstyle("width:" . $pxwidth . "px");
    $mybox->set_onblur("limitInt(this, 0, $intlimit)");
    $mybox->setvalue($valarray[$fieldname]);
    $Tin->td( $mybox->render("cp_" . strtolower($fieldname)) );
  }

  $bg = "axbgdark";
  entryField("Application Name:",   "APP_NAME",   $app->definitions);
  entryField("Application Prefix:", "APP_PREFIX", $app->definitions);

  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // GLOBALS
  $Tapp->tr("axhdg");
  $Tapp->td("<b>GLOBALS</b>", "axhdg");

  $Tin = new table("globals");
  $Tin->setpadding(2);
  $Tin->tbody("fmlook");
  $bg = "axbgdark";
  entryField("Templates directory:",       "TEMPLATESDIR", $app->globals);
  entryField("Images directory:",          "IMAGESDIR",    $app->globals);
  entryField("Cached files directory:",    "CACHEDIR",     $app->globals);
  entryField("Media catalog directory:",   "CATALOGDIR",   $app->globals);
  entryField("Managed content directory:", "CMDIR",        $app->globals);
  entryField("Includes directory:",        "INCDIR",       $app->globals);
  infoField(
      "NB: all directories specified above should be relative to the "
    . "website root directory. Additionally, if they are to be writeable "
    . "then they should be under the 'var' subdirectory."
      );
  entryField("Webmaster name:",   "WEBMASTER_PERSON",      $app->globals);
  entryField("Webmaster e-mail:", "WEBMASTER_EMAIL",       $app->globals);

  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // SETTINGS

  $Tapp->tr("axhdg");
  $Tapp->td("<b>WEBSITE SETTINGS</b>", "axhdg");

  // ......................................................................
  // DTD & ENCODING
  $Tapp->tr("axsubhdg");
  $Tapp->td("<b>Default DTD and Encoding</b>", "axsubhdg");

  $Tin = new table("dtd_enc");
  $Tin->setpadding(2);

  $cboHTMLDTD = new form_combofield();
  $cboHTMLDTD->setclass("axcombo");
  $cboHTMLDTD->additem("", "None");
  $cboHTMLDTD->additem(
            rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">"),
            "HTML 3.2 Strict"
            );
  $cboHTMLDTD->additem(
            rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">"),
            "HTML 4.01 Transitional"
            );
  $cboHTMLDTD->additem(
            rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\">"),
            "HTML 4.01 Strict"
            );
  $cboHTMLDTD->additem(
            rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\">"),
            "HTML 4.01 with Frameset"
            );
  $cboHTMLDTD->additem(
            rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\">"),
            "XHTML 1.0 Transitional"
            );
  $cboHTMLDTD->additem(
            rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\">"),
            "XHTML 1.0 Strict"
            );
  $cboHTMLDTD->additem(
            rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD XHTML 1.0 Frameset//EN\">"),
            "XHTML 1.0 with Frameset"
            );
  $cboHTMLDTD->setvalue($app->getparameter("dtd", "dtd", "html"));

  $cboWMLDTD = new form_combofield();
  $cboWMLDTD->setclass("axcombo");
  //$cboWMLDTD->setstyle("width:$cbowidth");
  $cboWMLDTD->additem("", "None");
  $cboWMLDTD->additem(
            rawurlencode("<!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1_1.xml\">"),
            "WML 1.1"
            );
  $cboWMLDTD->additem(
            rawurlencode("<!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.2//EN\" \"http://www.wapforum.org/DTD/wml12.xml\">"),
            "WML 1.2"
            );
  $cboWMLDTD->setvalue($app->getparameter("dtd", "dtd", "wml"));

  $cboENC = new form_combofield();
  $cboENC->setclass("axcombo");
  $cboENC->additem("ISO-8859-1", "ISO-8859-1 Latin 1 (Western)");
  $cboENC->additem("US-ASCII",   "US-ASCII");
  $cboENC->additem("UTF-8",      "UTF-8 (Unicode)");
  $setting = $app->get_setting("encoding");
  $cboENC->setvalue($app->getparameter("encoding", "encoding"));
  if ($app->getparameter("multilang", "multilang")) {
    $cboENC->disabled = true;
  }

  $Tin->tr("axbgdark");
  $Tin->td( "DTD for HTML content:", "axfg" );
  $Tin->td( $cboHTMLDTD->render("cp_dtd_html") );

  $Tin->tr("axbglite");
  $Tin->td( "DTD for WAP content:", "axfg" );
  $Tin->td( $cboWMLDTD->render("cp_dtd_wml") );

  $Tin->tr("axbgdark");
  $Tin->td( "Character Encoding:", "axfg" );
  $Tin->td( $cboENC->render("cp_encoding") );

  $mychkbox = $chkbox;
  $mychkbox->checked = $app->getparameter("multilang", "multilang");
  $mychkbox->set_onclick("setUTF8mode(this.checked)");
  $Tin->tr("axbglite");
  $Tin->td( "Website uses multiple languages:", "axfg" );
  $Tin->td( $mychkbox->render("cp_multilang") . "&nbsp;(requires UTF-8 encoding above)" );
  $Tin->td_css("font-style:italic;font-size:80%");

  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // DATABASE SETTINGS
  $Tapp->tr("axsubhdg");
  $Tapp->td("<b>Database settings</b>", "axsubhdg");

  $Tin = new table("dbsettings");
  $Tin->setpadding(2);

  $chkbox->checked = $app->getparameter("database_backed", "database_backed");
  $Tin->tr("axbgdark");
  $Tin->td( "Website uses a Database:", "axfg" );
  $Tin->td( $chkbox->render("cp_database_backed") );

  $Tin->tr("axbglite");
  $Tin->td( "Hosts for persistent DB connection:", "axfg" );
  $tbox->setvalue(str_replace("\"", "", $app->getparameter("permhosts", "permhosts")));
  $Tin->td( $tbox->render("cp_permhosts") );
  $Tin->tr("axbglite");
  $Tin->td();
  $Tin->td(
      "A comma-delimited list of hostnames which will use persistent "
    . "database connections. Usually these would be your production "
    . "web-servers.",
      "axfg"
      );
  $Tin->td_css("font-style:italic;font-size:80%");
  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // DATABASES
  $Tapp->tr("axsubhdg");
  $Tapp->td("<b>Databases</b>", "axsubhdg");

  $Tin = new table("dbsettings");
  $Tin->setpadding(2);

  $database_listbox->setstyle("width:$ewidth;");
  $database_listbox->size = 6;

  // Get defined databases..
  $dbs = $app->get_setting("database");
  if ($dbs === false) $databases = array();
  elseif (is_array($dbs)) $databases = $dbs;
  else $databases[0] = $dbs;

  $dbid = 0;
  foreach ($databases as $database) {
    // Populate listbox..
    $dbname = $database->getparameter("name");
    $database_listbox->additem($dbid, $dbname);

    // Populate maintainer data. The maintainer add_record method
    // requires an associative array keyed on listbox key id..
    $rec = array(
            "dbtype"     => $database->getparameter("type"),
            "dbname"     => $dbname,
            "dbuser"     => $database->getparameter("user"),
            "dbpassword" => $database->getparameter("password"),
            "dbhost"     => $database->getparameter("host"),
            "dbport"     => $database->getparameter("port")
            );
    $maintainer->add_record($dbid, $rec);
    if (!isset($firstrec)) {
      $firstrec = $rec;
    }
    $dbid += 1;
  } // foreach

  // Now set the defaults for each of the fields. These are
  // necessary for when a new record is created..
  $defaults = array(
            "dbtype"     => "postgres",
            "dbname"     => "",
            "dbuser"     => "",
            "dbpassword" => "",
            "dbhost"     => "",
            "dbport"     => ""
            );
  $maintainer->add_defaults($defaults);
  if (!isset($firstrec)) {
    $firstrec = $defaults;
  }

  // The listbox field..
  $database_listbox->setvalue($firstrec["dbname"]);
  $Tin->tr("axbgdark");
  $Tin->td( $database_listbox->render() );
  $Tin->td_width("50%");

  $Tin2 = new table();
  $Tin2->td(
      "NB: The ordering of this list is important. The first "
    . "database will be the default connection.",
      "axfg"
      );
  $Tin2->td_css("font-style:italic;font-size:80%");
  $Tin2->td_alignment("", "top");
  $Tin2->td(
        $bup->render()   . "<br>"
      . $bdown->render() . "<br>"
      . $bdel->render()  . "<br>"
      . $badd->render()
      );
  $Tin2->td_alignment("right", "top");
  $Tin->td( $Tin2->render() );
  $Tin->td_width("50%");
  $Tin->td_alignment("", "top");
  // ..................................................................
  // Database type field..
  $Fdbtype = new form_combofield("dbtype", "", $firstrec["dbtype"]);
  $Fdbtype->setclass("axcombo");
  $maintainer->register_field($Fdbtype);
  $Fdbtype->additem("postgres", "Postgres");
  $Fdbtype->additem("odbc",     "ODBC");
  $Fdbtype->additem("mssql",    "MS SQL Server");
  $Fdbtype->additem("mysql",    "MySQL");
  $Fdbtype->additem("oracle",   "Oracle");
  $Fdbtype->setstyle("width:$cbowidth;");
  $Tin->tr("axbglite");
  $Tin->td( "Database type:", "axfg" );
  $Tin->td( $Fdbtype->render() );
  // ..................................................................
  // Database name field..
  $Fdbname = new form_textfield("dbname", "", $firstrec["dbname"]);
  $maintainer->register_field($Fdbname);
  $Fdbname->setstyle("width:$ewidth;");
  $Fdbname->setclass("axtxtbox");
  $Tin->tr("axbgdark");
  $Tin->td( "Database name:", "axfg" );
  $Tin->td( $Fdbname->render() );
  // ..................................................................
  // Database user field..
  $Fdbuser = new form_textfield("dbuser", "", $firstrec["dbuser"]);
  $maintainer->register_field($Fdbuser);
  $Fdbuser->setstyle("width:$ewidth;");
  $Fdbuser->setclass("axtxtbox");
  $Tin->tr("axbglite");
  $Tin->td( "Username:", "axfg" );
  $Tin->td( $Fdbuser->render() );
  // ..................................................................
  // Database password field..
  $Fdbpassword = new form_textfield("dbpassword", "", $firstrec["dbpassword"]);
  $maintainer->register_field($Fdbpassword);
  $Fdbpassword->setstyle("width:$ewidth;");
  $Fdbpassword->setclass("axtxtbox");
  $Tin->tr("axbgdark");
  $Tin->td( "User password:", "axfg" );
  $Tin->td( $Fdbpassword->render() );
  // ..................................................................
  // Database host field..
  $Fdbhost = new form_textfield("dbhost", "", $firstrec["dbhost"]);
  $maintainer->register_field($Fdbhost);
  $Fdbhost->setstyle("width:$ewidth;");
  $Fdbhost->setclass("axtxtbox");
  $Tin->tr("axbglite");
  $Tin->td( "Hostname:", "axfg" );
  $Tin->td( $Fdbhost->render() );
  // ..................................................................
  // Database port field..
  $Fdbport = new form_textfield("dbport", "", $firstrec["dbport"]);
  $maintainer->register_field($Fdbport);
  $Fdbport->setstyle("width:$ewidth;");
  $Fdbport->setclass("axtxtbox");
  $Tin->tr("axbgdark");
  $Tin->td( "Port number:", "axfg" );
  $Tin->td( $Fdbport->render() );
  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // SESSION
  $Tapp->tr("axsubhdg");
  $Tapp->td("<b>Session</b>", "axsubhdg");

  $Tin = new table("session");
  $Tin->setpadding(2);

  $Tin->tr("axbgdark");
  $Tin->td( "Cookie name:", "axfg" );
  $cookiename = $app->getparameter("cookiename", "cookiename");
  if ($cookiename == "") {
    $cookiename = $app->definitions["APP_PREFIX"] . "_session_id";
  }
  $tbox->setvalue($cookiename);
  $Tin->td( $tbox->render("cp_cookiename") );

  $Tin->tr("axbglite");
  $Tin->td( "Cookie/session lifetime:", "axfg" );
  $Flife = new form_combofield();
  $Flife->setclass("axcombo");
  $Flife->setstyle("width:$cwidth");
  $Flife->additem(-1, "Until browser closed");
  $Flife->additem(315360000, "Forever and a day");
  $Flife->additem(31536000, "A year");
  $Flife->additem(2592000, "A month");
  $Flife->additem(604800, "A week");
  $Flife->additem(86400, "24 hours");
  $Flife->additem(43200, "12 hours");
  $Flife->additem(28800, "8 hours");
  $Flife->additem(14400, "4 hours");
  $Flife->additem(3600, "An hour");
  $Flife->additem(1200, "20 minutes");
  $Flife->additem(0, "Immediate expiry");
  $Flife->setvalue($app->getparameter("lifetime", "lifetime"));
  $Tin->td( $Flife->render("cp_lifetime") );

  $Tin->tr("axbgdark");
  $Tin->td( "Page expiry (seconds):", "axfg" );

  $Fexpiry = new form_combofield();
  $Fexpiry->setclass("axcombo");
  $Fexpiry->setstyle("width:$cwidth");
  $Fexpiry->additem(-1, "Immediate (dynamic content)");
  $Fexpiry->additem(60, "1 minute");
  $Fexpiry->additem(120, "2 minutes");
  $Fexpiry->additem(180, "3 minutes");
  $Fexpiry->additem(240, "4 minutes");
  $Fexpiry->additem(300, "5 minutes");
  $Fexpiry->additem(600, "10 minutes");
  $Fexpiry->additem(1800, "30 minutes");
  $Fexpiry->additem(3600, "1 hour");
  $Fexpiry->additem(14400, "4 hours");
  $Fexpiry->additem(28800, "8 hours");
  $Fexpiry->additem(86400, "24 hours");
  $Fexpiry->additem(315360000, "Never (static content)");
  $Fexpiry->setvalue($app->getparameter("expiry", "expiry"));
  $Tin->td( $Fexpiry->render("cp_expiry") );

  $chkbox->checked = $app->getparameter("guest_browser_lifetime", "guest_browser_lifetime");
  $Tin->tr("axbglite");
  $Tin->td( "Guest cookies browser lifetime:", "axfg" );
  $Tin->td( $chkbox->render("cp_guest_browser_lifetime") );

  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // CONTENT
  $Tapp->tr("axsubhdg");
  $Tapp->td("<b>Content settings</b>", "axsubhdg");

  $Tin = new table("content");
  $Tin->setpadding(2);

  $chkbox->checked = $app->getparameter("metadata_enabled", "metadata_enabled");
  $Tin->tr("axbglite");
  $Tin->td( "Enable metadata edit/generation:", "axfg" );
  $Tin->td( $chkbox->render("cp_metadata_enabled") );

  $chkbox->checked = $app->getparameter("buffered_output", "buffered_output");
  $Tin->tr("axbgdark");
  $Tin->td( "Buffered output (recommended):", "axfg" );
  $Tin->td( $chkbox->render("cp_buffered_output") );

  $Tin->tr("axbglite");
  $Tin->td( "Compression type:", "axfg" );
  $Fcomp = new form_combofield();
  $Fcomp->setclass("axcombo");
  $Fcomp->setstyle("width:$cwidth");
  $Fcomp->additem(0, "No compression");
  $Fcomp->additem(1, "Built-in compression (Php >= 4.0.4)");
  $Fcomp->additem(2, "Axyl custom compression");
  $Fcomp->setvalue($app->getparameter("compression_type", "compression_type"));
  $Tin->td( $Fcomp->render("cp_compression_type") );

  $Tin->tr("axbgdark");
  $Tin->td( "Compression threshold:", "axfg" );
  $Fcomp = new form_combofield();
  $Fcomp->setclass("axcombo");
  $Fcomp->setstyle("width:$cwidth");
  $Fcomp->additem(0, "None (compress all content)");
  $Fcomp->additem(1024, "Over 1Kb");
  $Fcomp->additem(4096, "Over 4Kb");
  $Fcomp->additem(8192, "Over 8Kb");
  $Fcomp->additem(16384, "Over 16Kb");
  $Fcomp->additem(32768, "Over 32Kb");
  $Fcomp->additem(65536, "Over 64Kb");
  $Fcomp->additem(262144, "Over 256Kb");
  $Fcomp->setvalue($app->getparameter("compression_threshold", "compression_threshold"));
  $Tin->td( $Fcomp->render("cp_compression_threshold") );

  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // GET/POST settings
  $Tapp->tr("axsubhdg");
  $Tapp->td("<b>GET/POST settings</b>", "axsubhdg");

  $Tin = new table("getpost");
  $Tin->setpadding(2);

  $chkbox->checked = $app->getparameter("keep", "keep");
  $Tin->tr("axbgdark");
  $Tin->td( "Enable Axyl KEEP feature:", "axfg" );
  $Tin->td( $chkbox->render("cp_keep") );

  $chkbox->checked = $app->getparameter("globalise", "globalise");
  $Tin->tr("axbglite");
  $Tin->td( "Auto-globalise all GET/POST vars:", "axfg" );
  $Tin->td( $chkbox->render("cp_globalise") );

  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // AUTHENTICATION
  $Tapp->tr("axsubhdg");
  $Tapp->td("<b>Authentication</b>", "axsubhdg");

  $Tin = new table("authentication");
  $Tin->setpadding(2);

  $chkbox->checked = $app->getparameter("encrypted_passwords", "encrypted_passwords");
  $Tin->tr("axbgdark");
  $Tin->td( "Encrypted passwords:", "axfg" );
  $Tin->td( $chkbox->render("cp_encrypted_passwords") );

  $Tin->tr("axbglite");
  $Tin->td( "Authentication type:", "axfg" );
  $Fcomp = new form_combofield();
  $Fcomp->setclass("axcombo");
  $Fcomp->setstyle("width:$cwidth");
  $Fcomp->additem(0, "No authentication");
  $Fcomp->additem(1, "HTTP authentication");
  $Fcomp->additem(2, "Axyl login form");
  $Fcomp->setvalue($app->getparameter("authtype", "authtype"));
  $Tin->td( $Fcomp->render("cp_authtype") );

  $Tin->tr("axbgdark");
  $Tin->td( "On failed authentication:", "axfg" );
  $Fcomp = new form_combofield();
  $Fcomp->setclass("axcombo");
  $Fcomp->setstyle("width:$cwidth");
  $Fcomp->additem(0, "Display basic fail message");
  $Fcomp->additem(1, "Die silently");
  $Fcomp->additem(2, "Re-direct to URL (below)");
  $Fcomp->additem(3, "Login as guest instead");
  $Fcomp->setvalue($app->getparameter("authfail", "authfailopt"));
  $Tin->td( $Fcomp->render("cp_authfailopt") );

  $Tin->tr("axbgdark");
  $Tin->td( "Failed re-direct URL:", "axfg" );
  $tbox->setvalue($app->getparameter("authfail", "authfailurl"));
  $Tin->td( $tbox->render("cp_authfailurl") );

  $Tin->tr("axbglite");
  $Tin->td( "On login limit exceeded:", "axfg" );
  $Flogexc = new form_combofield();
  $Flogexc->setclass("axcombo");
  $Flogexc->setstyle("width:$cwidth");
  $Flogexc->additem(0, "Take no action");
  $Flogexc->additem(1, "Allow, cull oldest sessions");
  $Flogexc->additem(2, "Deny access, display message");
  $Flogexc->additem(3, "Deny access silently");
  $Flogexc->additem(4, "Redirect to a URL (below)");
  $Flogexc->additem(5, "Login as guest instead");
  $Flogexc->setvalue($app->getparameter("loginlimit", "logexceedopt"));
  $Tin->td( $Flogexc->render("cp_logexceedopt") );

  $Tin->tr("axbglite");
  $Tin->td( "Login excess re-direct URL:", "axfg" );
  $tbox->setvalue($app->getparameter("loginlimit", "logexceedurl"));
  $Tin->td( $tbox->render("cp_logexceedurl") );

  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // MISC SETTINGS
  $Tapp->tr("axsubhdg");
  $Tapp->td("<b>Miscellaneous settings</b>", "axsubhdg");

  $Tin = new table("misc");
  $Tin->setpadding(2);

  $Tin->tr("axbglite");
  $Tin->td( "IP addresses to block:", "axfg" );
  $tbox->setvalue(str_replace("\"", "", $app->getparameter("badips", "badips")));
  $Tin->td( $tbox->render("cp_badips") );
  $Tin->tr("axbglite");
  $Tin->td();
  $Tin->td(
      "A comma-delimited list of IP addresses which are to be denied access.",
      "axfg"
      );
  $Tin->td_css("font-style:italic;font-size:80%");
  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  // ......................................................................
  // DEBUGGING
  $Tapp->tr("axhdg");
  $Tapp->td("<b>DEBUGGING</b>", "axhdg");

  $Tin = new table("debugging");
  $Tin->setpadding(2);

  $debugging = $app->getparameter("debug_on", "debug_on");
  $chkbox->checked = $debugging;
  $Tin->tr("axbgdark");
  $Tin->td( "Enable debugging:", "axfg" );
  $Tin->td( $chkbox->render("cp_debug_on") );

  $Tin->tr("axbglite");
  $Tin->td( "Classes of output to show:", "axfg");
  $Tin->td_alignment("", "top");
  $Fdebugcl = new form_combofield();
  $Fdebugcl->multiselect = true;
  $Fdebugcl->set_size(6);
  $Fdebugcl->setstyle("width:$cwidth");
  $Fdebugcl->additem(2,   "User diagnostics (default)");
  $Fdebugcl->additem(4,   "SQL statements");
  $Fdebugcl->additem(8,   "All SQL data (verbose)");
  $Fdebugcl->additem(16,  "Dump of GET/POST vars etc.");
  $Fdebugcl->additem(32,  "Include traceback info");
  $Fdebugcl->additem(64,  "Show table outlines");
  $Fdebugcl->additem(128, "Execution profiler");
  $Fdebugcl->additem(1,   "System diagnostics");
  // Build value as array of set bits..
  $debugcl = $app->getparameter("debug_classes", "debug_classes");
  $debug_value = array();
  for ($i=1; $i < 256; $i*=2) {
    if ($debugcl & $i) {
      $debug_value[] = $i;
    }
  }
  $Fdebugcl->setvalue($debug_value);
  $Tin->td( $Fdebugcl->render("cp_debug_classes") );

  $Tin->tr("axbgdark");
  $Tin->td( "Output modes:", "axfg");
  $Tin->td_alignment("", "top");
  $Fdebugop = new form_combofield();
  $Fdebugop->multiselect = true;
  $Fdebugop->set_size(6);
  $Fdebugop->setstyle("width:$cwidth");
  $Fdebugop->additem(1, "Standard (default)");
  $Fdebugop->additem(2, "Unbuffered echo");
  $Fdebugop->additem(4, "CLI output (non-web mode)");
  $Fdebugop->additem(8, "To system logfile");
  // Build value as array of set bits..
  $debugop = $app->getparameter("debug_output", "debug_output");
  $debugop_value = array();
  for ($i=1; $i < 256; $i*=2) {
    if ($debugop & $i) {
      $debugop_value[] = $i;
    }
  }
  $Fdebugop->setvalue($debugop_value);
  $Tin->td( $Fdebugop->render("cp_debug_output") );

  $bg = "axbgdark";
  integerField("SQL Execution log threshold:", "SQL_EXEC_THRESHOLD", $app->globals, 60000, 80);
  infoField(
      "SQL queries exeeding the specified number of milliseconds will "
    . "be logged in the system log. To disable, set to zero."
      );

  $Tin->set_width_profile("50%,50%");
  $Tapp->tr();
  $Tapp->td( $Tin->render() );

  $Tapp->tr("axfoot");
  $Tapp->td("&nbsp;", "axfoot");

} // if no errors

// ----------------------------------------------------------------------
// Finish and return the page..
$s .= "<form name=\"cpform\" method=\"post\">\n";
$s .= $Tapp->render();
$s .= $maintainer->render();
$s .= "</form>\n";

//echo $app->htmldump();

$s .= "</body>\n";
$s .= "</html>\n";
echo $s;
// ----------------------------------------------------------------------
?>