<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    dbsearch-defs.php                                       */
/* Author:      Paul Waite                                              */
/* Description: Definitions for interfacing to the database to perform  */
/*              a search. This module sits on the same level as the     */
/*              swish-defs.php module, and like that module extends     */
/*              the functionality in search-defs.php.                   */
/*                                                                      */
/* ******************************************************************** */
/** @package search */

/** Generic searching */
include_once("search-defs.php");

// ----------------------------------------------------------------------
/**
* DB Search class
* This class inherits the functionality of the generic 'search'
* class. It extends it to implement a database search.
* @package search
*/
class db_search extends search {
  /** The query which runs the search */
  var $searchquery = false;
  // ....................................................................
  /**
  * Constructor
  *
  * The constructor. Usually the query will be built separately and added
  * via set_searchquery(), but we also allow a ready-made query object to
  * be passed in here. This object should be a valid dbquery() object as
  * returned by the likes of dbselect() (see query-defs.php).
  *
  * @param string $title       Title describing this search
  * @param mixed  $searchquery Either a dbselect() object, or false
  */
  function db_search($title="Search Results", $searchquery=false) {
    $this->search($title);
    $this->max_results = 25;
    $this->skip_results = 0;
    if (!$searchquery) {
      $this->searchquery = new dbselect();
    }
    $this->initialise();
  }
  // ....................................................................
  /**
  * Define the search query object
  *
  * The query object will usually be an executed dbselect() object
  * @see dbselect
  * @param mixed $query The dbselect() object to use for searching
  */
  function set_searchquery($query=false) {
    if ($query) {
      $this->searchquery = $query;
    }
  }
  // ....................................................................
  /**
  * Define search table FROM list
  * @see sqlquery::from()
  * @see sqlquery::tables()
  * @param string $table_spec The table list to add to the query
  */
  function from($table_spec) {
    $this->searchquery->from($table_spec);
  }
  // ....................................................................
  /**
  * Define search WHERE clause
  * @see sqlquery::where()
  * @param string $where_clause The WHERE clause, without the word "WHERE".
  */
  function where($where_clause) {
    $this->searchquery->where($where_clause);
  }
  // ....................................................................
  /**
  * Define search ORDER BY clause
  * @see sqlquery::orderby()
  * @param string $field_spec The ORDER BY field list, without the words "ORDER BY".
  */
  function orderby($field_spec="") {
    $this->searchquery->orderby($field_spec);
  }
  // ....................................................................
  /**
  * Define search LIMIT clause
  * @see sqlquery::limit()
  * @param string $limit A numeric value. Do not include the word "LIMIT".
  */
  function limit($limit) {
    $this->searchquery->limit($limit);
  }
  // ....................................................................
  /**
  * Define search fields list
  * @see sqlquery::fieldlist()
  * @param string $field_spec A list of fields separated by commas.
  */
  function fieldlist($field_spec="*") {
    $this->searchquery->fieldlist($field_spec);
  }
  // ....................................................................
  /**
  * Execute the search
  * Execute the database query search and store any hits.
  */
  function execute() {
    debug_trace($this);
    $this->hit = array();
    if (isset($this->searchquery)) {
      $this->searchquery->execute();
      if ($this->searchquery->hasdata) {
        $this->searchquery->get_first();
        do {
          // The db search query returns an associative array which
          // is a row. We store this as our hit information..
          $hitfields = $this->searchquery->current_row;
          $this->hit[] = $hitfields;
        } while ($this->searchquery->get_next());
      }
    }
    debug_trace();
  } // execute

} // db_search class

// ----------------------------------------------------------------------
?>