<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    utils.php                                               */
/* Author:      Paul Waite                                              */
/* Description: Various general utility routines.                       */
/*                                                                      */
/* ******************************************************************** */
/** @package utils */

// ----------------------------------------------------------------------
/**
* Insert vertical space in table
* Insert some vertical whitespace into a table.
* @param integer $height Whitespace height in pixels
* @param integer $cols   No. of columns in the target table
* @return string HTML for a row of the given height
*/
function vspace($height, $cols=1) {
  echo "<tr><td colspan=\"$cols\" height=\"$h\"></td></tr>";
} // vspace

// ----------------------------------------------------------------------
/**
* Send a message to the system logfile
* Send message to the syslog, prefixed with APP_NAME nicely. Saves
* having to prefix it each time.
* @param string $msg     Message to enter into the system log
* @param string $prefix  Prefix to the message
*/
function log_sys($msg, $prefix="") {
  if ($prefix == "") {
    if (defined("APP_NAME")) {
      $prefix = APP_NAME;
    }
    else {
      $prefix = "AXYL";
    }
  }
  error_log("$prefix: $msg", 0);
} // log_sys

// ----------------------------------------------------------------------
/**
* Exit the application with error message
* Echo the message and leave. Intended to handle 'emergencies'.
* @param string $heading Message heading, subject
* @param string $msg     Message in detail
*/
function error_exit($heading, $msg="") {
  global $RESPONSE;

  switch ($RESPONSE->browser_type) {
    case BROWSER_TYPE_XHTML:
    case BROWSER_TYPE_HTML:
      $RESPONSE->delete_cookie();
      echo "<h3>$heading</h3><strong>$msg</strong>";
      break;

    case BROWSER_TYPE_XHTMLMP:
    case BROWSER_TYPE_WML:
    case BROWSER_TYPE_WMLUP:
      include_once("wml-defs.php");
      $card = new WMLcard("error_exit", "System Error");
      $card->insert(
        "<p>" .
            "$heading<br/>" .
            "$msg" .
        "</p>"
        );
      // Create the card deck and output it..
      $deck = new WMLdeck($card);
      echo $deck->wml();
      break;

    default:
      $RESPONSE->delete_cookie();
      echo "<h3>$heading</h3><strong>$msg</strong>";
  } // switch
  exit;
} // error_exit

// ----------------------------------------------------------------------
/**
* Return an HTTP error message
* Returns a formatted HTTP Error
* @param integer $code HTTP error code
* @return string The formatted HTTP error code
*/
function HTTPError($code) {
  $s = "";
  switch ($code) {
    case 401:
      $s = "Unauthorised.";
      break;
    case 403:
      $s = "Forbidden.";
      break;
    case 404:
      $s = "Not Found.";
      break;
    case 405:
      $s = "Method Not Allowed.";
      break;
    case 406:
      $s = "Not Acceptable.";
      break;
    case 500:
      $s = "Infernal server error.";
      break;
    case 501:
      $s = "Not Implemented.";
      break;
    default:
      $s = "Unknown Server Error.";
      break;
  } // switch

  return "HTTP/1.1 $code: $s";
} // HTTPError

// ----------------------------------------------------------------------
/**
* Exit with HTTP error code
* Send a simple error code and then die.
* @param integer $code HTTP error code
*/
function errorcode_exit($code) {
  error_exit( HTTPError($code) );
} // errorcode_exit

// -----------------------------------------------------
/**
* Return a value which may be defaulted
* Returns the value of the variable, if it is valid, otherwise
* returns the specified default value.
* @param mixed $val     The value of interest
* @param mixed $default The default value to apply if $val is empty
* @return mixed Either the value, or the default
*/
function defaulted($val, $default) {
  if (empty($val) || $val == "") return $default;
  else return $val;
} // defaulted

// ----------------------------------------------------------------------
/**
* Format a string (dotted) form of an IP address. We make sure we have
* the full dotted quad, and if we have to pad it with zeroes, then we
* add the network spec (/24, /16 etc.) as well. If the given IP already
* has a netmask, then we don't change it, and assume the IP is already
* correctly formatted by someone who knows what they are doing!
*
* @param string $pi IP address in string dotted format
* @return string The padded IP address
*/
function ip_format($ip) {
  // Pad shorthand addresses with zeroes..
  $myipbits = explode("/", $ip);
  $myip = trim($myipbits[0]);
  $mynetmask = trim($myipbits[1]);

  $octets = explode(".", $myip);
  if (count($octets < 4)) {
    while ( count($octets) < 4) {
      $octets[] = "0";
    }
    $myip = implode(".", $octets);
  }

  // If not defined already, find the netmask bits,
  // by scanning for zeroes from the end..
  if ($mynetmask == "") {
    $mynetmask = 32;
    for ($ix=3; $ix >= 0; $ix--) {
      if ($octets[$ix] == 0) $mynetmask -= 8;
      else break;
    }
  }

  // Append netmask if a network..
  if ($mynetmask < 32) {
    $myip .= "/$mynetmask";
  }
  return $myip;
} // ip_format

// -----------------------------------------------------
/**
* Check string to see if it is in IPv4 format. This is
* pretty simplistic. It returns true if the string $ip
* is of the form 'n.n.n.n'.
* @return bool True if it is in IPv4 format
*/
function is_ipaddress($ip) {
  $isip = false;
  $myip = explode(".", $ip);
  if (count($myip) == 4) {
    // A dotted quad is close enough..
    $isip = true;
  }
  return $isip;
} // is_ipaddress

// -----------------------------------------------------
/** This function is a utility to allow composite fragments of javascript
* to be replaced or concatenated together to form a new string of
* javascript. This is designed to be used for building short javascript
* statement strings to go in things such as onclick events for form
* elements etc.
*/
/** Add script by replacing current content (default) */
define("SCRIPT_REPLACE", 0);
/** Add script by keeping current content and appending to the end */
define("SCRIPT_APPEND",  1);
/** Add script by keeping current content and prefixing to the beginning */
define("SCRIPT_PREFIX",  2);

function inline_script($newscript, $oldscript="", $mode=SCRIPT_REPLACE) {
  $res = "";
  switch ($mode) {
    case SCRIPT_APPEND:
      if ($oldscript != "") {
        if (substr($oldscript, -1) != ";") {
          $oldscript .= ";";
        }
      }
      $res = $oldscript . $newscript;
      break;
    case SCRIPT_PREFIX:
      if ($newscript != "") {
        if (substr($newscript, -1) != ";") {
          $newscript .= ";";
        }
      }
      $res = $newscript . $oldscript;
      break;
    default:
      $res = $newscript;
      break;
  } // switch

  // Return new inline script
  return $res;
} // inline_script

// -----------------------------------------------------
/**
* Resolve a foreign key value
* Resolve a simple single-level foreign key. The variable
* 'display_fields' can contain multiple fields separated
* by the "|" char. A literal is prefixed by "#".
* @param mixed  $fkey            The value of the foreign key field to look up
* @param string $tbl             The name of the table to look up in
* @param string $key_field       Name of the key field to look up
* @param string $display_fields  Name of the display fields to get data from
* @return mixed The returned value
*/
function resolveFK($fkey, $tbl, $key_field, $display_fields) {
  debug_trace("utils.php: resolveFK()");
  $res = "";
  $q = dbrecordset("SELECT * FROM $tbl WHERE $key_field = '$fkey'");
  if ($q->hasdata) {
    $dispflds = explode("|",$display_fields);
    for ($i=0; $i < count($dispflds); $i++) {
      if ($i > 0) $res .= " ";
      $fld = $dispflds[$i];
      if (substr($fld,0,1) == "#") $res .= substr($fld,1);
      else $res .= $q->field($fld);
    }
  }
  debug_trace();
  return $res;
} // resolveFK

// -----------------------------------------------------
/**
* DEPRECATED: This global utility function is now deprecated since the
* code is non-database-independent, and was written solely to support
* Postgresql sequences. Instead, please update your code to use the new
* 'get_next_sequencevalue()' global function, or 'next_sequencevalue()'
* methods on the 'dbupdate' and 'dbinsert' classes, or else use the new
* 'dbseq' class itself.
* Return next sequence value - get the next value of the given sequence
* @param string $seq The name of the sequence to get the next value of
* @return integer The next sequence value
*/
function next_sequencevalue($seq) {
  debug_trace("utils.php: next_sequencevalue()");
  $res = FALSE;
  $qseq = new dbrows("SELECT NEXTVAL('$seq')" );
  if ($qseq->valid) {
    $r = $qseq->get_current();
    $qseq->tidyup();
    $res = $r[0];
  }
  debug_trace();
  return $res;
} // next_sequencevalue

// -----------------------------------------------------
/**
* Strip off any http:// from a URL
*/
function strip_http_prefix($url){
  if ( strtolower(substr($url, 0, 7)) == "http://" ) {
    $url = substr($url, 7);
  }
  return $url;
} // strip_http_prefix

// -----------------------------------------------------
/**
* Make sure there is an http:// on a URL. If already
* present then string is returned untouched.
* @return string The URL prefixed with http://
*/
function add_http_prefix($url){
  if ( strtolower(substr($url, 0, 7)) != "http://" ) {
    $url = "http://$url";
  }
  return $url;
} // add_http_prefix

// -----------------------------------------------------
/**
* Add a parameter keyvalue pair to a URL. We check that
* it isn't already there, and use the right delimiter.
* @param string $href URL to append the new parm to
* @param string $pname Name of the parameter
* @param string $pvalue Value of the parameter
*/
function href_addparm($href, $pname, $pvalue) {
  if ($href != "") {
    $urlbits = explode("?", $href);
    $baseurl = $urlbits[0];
    $query = (isset($urlbits[1]) ? $urlbits[1] : "");
    parse_str($query, $qvars);
    $qvars[$pname] = $pvalue;
    $q = array();
    foreach ($qvars as $name => $val) {
      $q[] = "$name=$val";
    }
    $query = implode("&", $q);
    $href = $baseurl . (($query != "") ? "?$query" : "");
  }
  return $href;
} // href_addparm

// -----------------------------------------------------
/**
* Remove a parameter keyvalue pair from a URL.
* @param string $href URL to remove the parm from
* @param string $pname Name of the parameter
*/
function href_delparm($href, $pname) {
  if ($href != "") {
    $urlbits = explode("?", $href);
    $baseurl = $urlbits[0];
    $query = (isset($urlbits[1]) ? $urlbits[1] : "");
    parse_str($query, $qvars);
    if (isset($qvars[$pname])) {
      unset($qvars[$pname]);
    }
    $q = array();
    foreach ($qvars as $name => $val) {
      $q[] = "$name=$val";
    }
    $query = implode("&", $q);
    $href = $baseurl . (($query != "") ? "?$query" : "");
  }
  return $href;
} // href_delparm

// -----------------------------------------------------
/**
* Returns the name in the form SURNAME, Firstname.
* @param string $firstname First name
* @param string $lastname Last name(s)
* @return string The name in format SURNAME, Firstname
*/
function format_name($firstname, $lastname) {
  return strtoupper($lastname) . ", " . ucfirst($firstname);
} // format_name

// -----------------------------------------------------
/**
* Returns a value inside quotes. The type of quotes (single or
* double) are determined from the value content. If the content
* already has a double quote in it, then single quotes are used,
* else (default) double quotes are used.
*
* @return string The value string with appropriate quotes around it.
*/
function quoted_valuestring($val) {
  // Default the delimiter to double quote..
  $delim = chr(34);

  // Check content for quotes..
  $dq = strchr($val, chr(34)); // Double quote(s) in value
  $sq = strchr($val, chr(39)); // Single quote(s) in value
  // Both present in value, sanitise..
  if ($dq && $sq) {
    $val = str_replace(chr(34), chr(39), $val);
  }
  // Only double quotes in value, swap..
  elseif ($dq) {
    $delim = chr(39);
  }
  return $delim . $val . $delim;
} // quoted_valuestring

// -----------------------------------------------------
/**
* Returns a nicely formatted time string in '3d 12h 14m 33s'
* format, given a number of seconds. Leading elements are suppressed
* if they are zero.
*
* @param integer Number of seconds to convert to time string
* @return string A formatted time string eg: '12h 14m 33s'
*/
function nicetime($secs) {
  $days = (int) ($secs / 86400);
  $secs = $secs % 86400;
  $hrs  = (int) ($secs / 3600);
  $secs = $secs % 3600;
  $mins = (int) ($secs / 60);
  $secs = $secs % 60;
  $s = "";
  if ($days > 0) $s .= $days . "d ";
  if ($hrs > 0) $s .= $hrs . "h ";
  if ($mins > 0) $s .= $mins . "m ";
  $s .= $secs . "s";
  return $s;
} // nicetime

// -----------------------------------------------------
/**
* Returns a nicely formatted size string for displaying
* byte sizes. Eg. 1024 returns '1Kb', etc. Obviously this
* is a display-only string, not a number. You can specify
* the number of decimal places, and the thousands separator
* if you for example want nullstring instead of a comma.
*
* @param integer $bytes    Number of bytes.
* @param integer $decimals Number of decimal places to report
* @param string $thousep   Thousands separator
* @return string A formatted bytesize eg.
*/
function nicebytesize($bytes, $decimals=0, $thousep=",") {
  if ($bytes > GIGABYTE) {
    $s = number_format( ($bytes / GIGABYTE), $decimals, ".", $thousep) . "GB";
  }
  elseif ($bytes > MEGABYTE) {
    $s = number_format( ($bytes / MEGABYTE), $decimals, ".", $thousep) . "MB";
  }
  elseif ($bytes > KILOBYTE) {
    $s = number_format( ($bytes / KILOBYTE), $decimals, ".", $thousep) . "KB";
  }
  else {
    $s = number_format( $bytes, $decimals, ".", $thousep);
  }
  return $s;
} // nicebytesize

// -----------------------------------------------------
?>