<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    webpage-defs.php                                        */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing web-pages.                     */
/*                                                                      */
/* ******************************************************************** */
/** @package core */

/** Content plugin classes */
include_once("plugin-defs.php");
/** Date-time functions */
include_once("datetime-defs.php");

// ----------------------------------------------------------------------
// Flag determining whether webpage is cached or not..
/** The webpage is cached */
define("CACHED",     true);
/** The webpage is not cached */
define("NOT_CACHED", false);

// ----------------------------------------------------------------------
// Option to determine the buffering mode.
/** The webpage is buffered using Php buffering */
define("BUFFERED",   true);
/** The webpage is not buffered */
define("UNBUFFERED", false);

// ----------------------------------------------------------------------
/**
* The webstream class
* A class to manage buffering, cacheing, processing and output of the
* content to the user agent. This is the entity which manages the Php
* buffering mechanism, starting and stopping buffering and sending the
* buffer to the client's browser. It also manages any cacheing of the webpage.
* This class extends the session, since the whole point of the session is
* to output content back to the user agent.
* @package core
*/
class webstream extends session {
  /** Whether to use Php buffering */
  var $buffered = true;
  /** The content to send to browser */
  var $content = "";
  /** Replacements to make in template */
  var $replacement;
  /** Page is cached or dynamic */
  var $cached = NOT_CACHED;
  /** Seconds expiry for cached webpages */
  var $cache_expiry = 0;
  /** Path to use to save cached version of webpage */
  var $cache_path = "";
  /** If true, force regeneration of cached webpage */
  var $cache_regen = false;
  // .....................................................................
  /**
  * Constructor
  * Create a new webstream object. When this object is created it
  * always starts buffering with the ob_start() call. The Phplib
  * system always uses the Php buffering mechanism. This allows us
  * to process the output content and do 'clever things' right up
  * to the point of sending it all to the user.
  * @param string $initcontent Some intial content for the page
  */
  function webstream($initcontent="", $buffered=true) {
    // Create the session
    $this->session();

    // Set buffering mode..
    $this->buffered = $buffered;

    // Open for business..
    $this->open_webstream();

    // Kick off with any initial content..
    $this->add_content($initcontent);
  } // webstream constructor
  // .....................................................................
  /** Start webstream output channel.. */
  function open_webstream() {
    if ($this->buffered) {
      if ($this->multilang && $this->mbstring_avail) {
        ob_start("mb_output_handler");
        debugbr("webstream: mb_output_handler (multilang)", DBG_DEBUG);
      }
      else {
        ob_start();
        debugbr("webstream: standard buffering", DBG_DEBUG);
      }
    }
    else {
      debugbr("webstream: no buffering", DBG_DEBUG);
    }
  } // open_webstream
  // .....................................................................
  /**
  * Close the webstream. Return any current webpage content. Clear the
  * current content. This method clears content, but leaves any replacement
  * definitions untouched for further processing. It is designed to be
  * called as part of the final webpage rendering process.
  * @return string The current content, as it stands.
  */
  function close_webstream() {
    if ($this->buffered) {
      $content = ob_get_contents();
      ob_end_clean();
    }
    else {
      $content = $this->content;
    }

    // Clear content..
    $this->content = "";

    // Return what there was..
    return $content;
  } // close_webstream
  // .....................................................................
  /** Reset the webstream. This method clears any current content, and
  * also clears any stored replacement definitions. This resets the
  * stream to the point at which it was created - a virgin webstream.
  */
  function reset_webstream() {
    $this->content = "";
    unset($this->replacement);
  } // reset_webstream
  // .....................................................................
  /** Add new content to the webstream.. */
  function add_content($content) {
    if ($content != "") {
      if ($this->buffered) {
        echo $content;
      }
      else {
        $this->content .= $content;
      }
    }
  } // add_content
  // .....................................................................
  /**
  * Cache this webpage
  * Causes the current webpage to be regarded as a cached page.
  * This means we look for a file of the same name but with extension
  * 'cached' in the $CACHEDIR directory, and check the modification
  * time. If it isn't expired then we set the page content to that file,
  * and send it. Otherwise we behave as if it is a normal dynamic Php page.
  * @param $expirysecs integer   Seconds before page cacheing expires
  */
  function cache($expirysecs=0) {
    global $CACHEDIR;
    global $cachecontrol;
    $this->cache_path = "$this->site_docroot/$CACHEDIR/" . $this->theme . "_" . basename($this->requested) . ".cached";
    $this->cache_expiry = $expirysecs;
    $this->cached = CACHED;

    // Obey any cache control directives..
    if (isset($cachecontrol)) {
      // Possible forced refresh..
      if (strtolower($cachecontrol) == "refresh") {
        $this->cache_expiry = 0;
      }
      // Possible forced expirytime..
      else {
        $this->cache_expiry = (int) $cachecontrol;
      }
    }
    if (file_exists($this->cache_path)) {
      $tsnow = time();
      $tsfile = filemtime($this->cache_path);
      if ($tsnow > $tsfile + $this->cache_expiry) {
        // We are about to drop through and allow this cached
        // file to be regenerated, but we touch it here to decrease
        // the chance of multiple rebuilds in the meantime..
        $this->cache_regen = true;
        touch($this->cache_path);
      }
      else {
        // This cached file has a current version out on
        // disk, so we just return this file to the client..
        $this->discard();
        $cachefile = new inputfile($this->cache_path);
        if ($cachefile->opened) {
          $cachefile->readall();
          $this->content = $cachefile->content;
          $cachefile->closefile();
          $this->send_to_browser();
          exit;
        }
        else {
          log_sys("Failed to read cache file '$this->cache_path'");
        }
      }
    }
    else {
      // Non-existant cache file. Make sure it gets built..
      $this->cache_regen = true;
      touch($this->cache_path);
    }
  } // cache
  // .....................................................................
  /**
  * Length of output buffer
  * Returns the length of our output buffer. Be careful when this is
  * called, since the buffer might not be filled by make_content() yet!
  * @param $expirysecs integer   Seconds before page cacheing expires
  */
  function length() {
    if ($this->buffered) {
      return ob_get_length();
    }
    else {
      return strlen($this->content);
    }
  } // length
  // .....................................................................
  /**
  * Replace pattern in webpage content
  * Replaces multiple occurrences of the given tag (pattern) in the
  * body content with the specified new stuff. NB: when you call this
  * method the replacement isn't actually done there and then. It is
  * simply flagged as something to be done just before all of the
  * content is delivered to the user browser.
  * @param string $tag       Pattern to replace in content
  * @param string $newstuff  Stuff to replace the tag with
  */
  function replace($tag, $newstuff) {
    $this->replacement[$tag] = $newstuff;
    return $this;
  } // replace
  // .....................................................................
  /**
  * Replace all webpage content
  * For replacing the total contents of the buffer so far
  * with a new content. Throw buffer away and start anew with
  * immediate effect.
  * @param string $newcontent  Replacement webpage content
  */
  function replace_content($newcontent) {
    $this->close_webstream();
    $this->reset_webstream();
    $this->open_webstream($newcontent);
  } // replace_content
  // .....................................................................
  /**
  * Discard all webpage content
  * For discarding the content so far with immediate effect.
  */
  function discard() {
    $this->close_webstream();
    $this->reset_webstream();
  } // discard
  // .....................................................................
  /**
  * Make content
  * This function takes our buffered content so far, stops buffering,
  * makes any replacements, and puts the resulting content into
  * the staging ($this->content) variable. This is mainly an internal
  * method which is called prior to sending the output to the user's
  * browser.
  * @access private
  */
  function make_content() {
    global $BLOCK_DEFS;
    $content = $this->close_webstream();
    if (isset($this->replacement)) {
      foreach ($this->replacement as $tag => $newstuff) {
        $tmp = str_replace($tag, $newstuff, $content);
        $content = $tmp;
      }
    }
    // Now insert our debugging output, if any..
    if (isset($this->debugger) && $this->debugger->debug_hascontent()) {
      switch($this->browser_type) {
        case BROWSER_TYPE_XHTML:
        case BROWSER_TYPE_HTML:
          // Tuck debugging info in after <body> tag..
          $pos = strpos($content, "<body");
          if (!($pos === false)) {
            $pos = strpos($content, ">", $pos);
            if (!($pos === false)) {
              $pos++;
              $content1 = substr($content, 0, $pos);
              $content2 = substr($content, $pos);
              $content = $content1 . $this->debugger->render() . $content2;
            }
          }
          break;
        case BROWSER_TYPE_XHTMLMP:
        case BROWSER_TYPE_WML:
        case BROWSER_TYPE_WMLUP:
          // For WML debug output we replace the content with a
          // dedicated card containing the debugging info..
          $bits = explode("<wml>", $content);
          $head = $bits[0];
          $dbgcard = new WMLcard("debug", "Debug");
          $dbgcard->insert_para($this->debugger->render());
          $dbgdeck = new WMLdeck($dbgcard);
          $content = $head . "<wml>" . $dbgdeck->render() . "</wml>";
          break;
      } // switch
    }
    $this->content = $content;
    return $this;
  } // make_content
  // .....................................................................
  /**
  * Send content to user browser
  * Deliver the content to the browser. First check if the page is cached
  * and if so whether we are going to update the cache. Next we get the
  * current buffer and aply any compression required. Then we send the
  * output on its way using the simple echo() function.
  * NOTE: If the page is not cached then we always send headers which
  * will make the user browser avoid cacheing it locally. This makes
  * sure that our dynamic pages will always be requested by it.
  */
  function send_to_browser() {
    global $HTTP_ACCEPT_ENCODING;
    global $compression_minsize;
    global $compression_type;

    // Check if the webpage is cached or not..
    if ($this->cached) {
      if ($this->cache_regen) {
        $cachefile = new outputfile($this->cache_path);
        if ($cachefile->opened) {
          $cachefile->write($this->content);
          $cachefile->closefile();
        }
        else {
          log_sys("Failed to write cache file '$this->cache_path'");
        }
      }
    }
    // Deal with compression options..
    switch ($compression_type) {
      case BUILTIN_COMPRESSION:
        ob_start("ob_gzhandler");
        echo $this->content;
        ob_end_flush();
        break;
      case CUSTOM_COMPRESSION:
        $size = strlen($this->content);
        if ($size >= $compression_minsize) {
          if (isset($HTTP_ACCEPT_ENCODING) && stristr($HTTP_ACCEPT_ENCODING, "gzip")) {
            $crc = crc32($this->content);
            $this->content = gzcompress($this->content, 9);
            $gzlen = strlen($this->content);
            $this->content = substr($this->content, 0, $gzlen - 4);
            $this->content .= AsFourChars($crc);
            $this->content .= AsFourChars($size);
            header("Content-Encoding: gzip");
            echo "\x1f\x8b\x08\x00\x00\x00\x00\x00";
          }
        }
        echo $this->content;
        break;
      default:
        echo $this->content;
    } // switch
  } // send_to_browser
  // .....................................................................
  /**
  * Send content to file
  * Deliver the content to a given file.
  * Make our buffer content, and then deliver it to a file.
  * @see make_content()
  * @see output_to_file()
  * @param  string  $name The name of the file
  * @param  string  $dir  The directory the file is in
  * @return boolean True if file was opened, else false
  */
  function send_to_file($name, $dir="") {
    $this->make_content();
    return $this->output_to_file($name, $dir);
  } // send_to_file
  // .....................................................................
  /**
  * Output content to file
  * Raw function to output content to file..
  * @param  string  $name The name of the file
  * @param  string  $dir  The directory the file is in
  * @return boolean True if file was opened, else false
  * @access private
  */
  function output_to_file($name, $dir="") {
    $res = false;
    $f = new outputfile($name, $dir);
    if ($f->opened) {
      $f->write($this->content);
      $f->closefile();
    }
    return $res;
  } // output_to_file
  // .....................................................................
  /**
  * Return webpage content
  * Builds all of the webpage content and returns it to the caller.
  * @return string All of the webpage content as it would be sent to the user browser
  */
  function webpage_content() {
    $this->make_content();
    return $this->content;
  } // webpage_content
  // .....................................................................
  /**
  * Send error and die
  * Generic function to abort and send an error notification to the user
  * instead. This function is a one-way trip to oblivion.
  * @param string $heading The error heading or subject
  * @param string $msg     The detailed error message
  */
  function send_error_and_die($heading, $msg="") {
    // Discard current buffered output..
    $this->discard();

    // Return error message in error page..
    $errorpg = new error_page($heading, $msg);
    $this->content = $errorpg->render();
    $this->send_to_browser();
    exit;
  } // send_error_and_die
  // .....................................................................
  /**
  * Send HTTP error code and die
  * Generic function to abort and send an error code notification to the
  * user. These are lookalike errors for generic ones like 404: Page not found
  * etc. This function will not return.
  * @param integer $code The HTTP error code to generate
  */
  function send_errorcode_and_die($code) {
    $this->send_error_and_die( HTTPError($code) );
  } // send_errorcode_and_die
} // webstream class

// ----------------------------------------------------------------------
/**
* The webpage page class.
* This is the main focus of the Axyl framework for generating webpage
* content of any kind and for any device. We envisage the standard page
* structure with a head, body and foot section. Depending on the device
* this might not be exactly right, but it doesn't really matter, since
* you have complete control over the content.
* @package core
*/
class webpage extends webstream {
  // Public
  /** Webpage head object */
  var $head;
  /** Webpage body object */
  var $body;
  /** Webpage foot object */
  var $foot;
  /** The name of the template which was applied, if any */
  var $template = "";
  /** The file the template content is to be found in */
  var $templatefile = "";
  /**
  * Theme to apply to the page. This should be a single-word ID, and
  * there should be a sub-directory of the same name in the 'templates'
  * directory, with templates and stylesheet(s) in it which comprise
  * the theme. This is how we provide different template sets and
  * stylesheets for branding purposes.
  */
  var $theme = "";

  // Private
  /** True if page has been generated
      @access private */
  var $generated = false;
  /** Set of plugins with content
      @access private */
  var $pluginset;
  // .....................................................................
  /**
  * Constructor
  * Create a new webpage object. When this object is created it
  * implicitly starts buffering with the ob_start() call by creating
  * a webstream.
  * NOTES: If you name your stylesheets after the APP_PREFIX (see your
  * application.php setup file), eg: 'haka.css' and 'haka_ie.css', then
  * the system will find them for you. The same applies if you call them
  * 'sitestyle.css' and 'sitestyle_ie.css'.
  * @param string $title      Webpage title string
  * @param string $template   Template for this webpage
  * @param string $theme      Theme to apply. This is for branding purposes
  * @param string $stylesheet Name of stylesheet. Defaults to {APP_PREFIX}.css
  */
  function webpage($title="", $template="", $sitetheme="", $stylesheet="") {
    // Create webstream, start buffering..
    $this->webstream();

    // Define the theme, if any. We override the static
    // defined theme with one which might be coming in
    // on a URL or via a form submission..
    global $theme;
    if (isset($theme) && $theme != "") {
      $this->set_theme($theme);
    }
    else {
      $this->set_theme($sitetheme);
    }

    // HEAD - Set up the head page section..
    $this->head = new head($title);

    // BODY - Set up body OR deck page section..
    if ($this->browser == BROWSER_PHONE) {
      $this->body = new deck();
      if ($this->browser_type == BROWSER_TYPE_WMLUP) {
        $s  = "<meta forua=\"true\" http-equiv=\"Cache-Control\" content=\"max-age=0\"/>";
        $s .= "<meta forua=\"true\" http-equiv=\"Cache-Control\" content=\"must revalidate\"/>";
        $this->head->add($s);
      }
    }
    else {
      $this->body = new body();
    }

    // FOOT - Define foot page section
    $this->foot = new foot();

    // Set up the template..
    $this->set_template($template);

    // Set up the stylesheet(s)..
    $this->set_stylesheet($stylesheet);

    // Set up the DTDs..
    $this->assign_DTD();

    // Initialise a plugin set..
    $this->pluginset = new pluginset();

  } // webpage constructor
  // .....................................................................
  /**
  * Add to the body content
  * @param string $morestuff  Content to append to the body of the webpage
  */
  function add($morestuff) {
    $this->body->add($morestuff);
    return $this;
  } // add
  // .....................................................................
  /**
  * Add named script
  * This adds a specific lump of script to the body under a unique name.
  * It allows you to collect multiple additions of script content under
  * a single grouping, and so cause the final rendering to group it all
  * together.
  * @param string $script   Script content (omit <script> tags)
  * @param string $name     Script content grouping identity
  * @param string $language The language the script is in
  */
  function add_named_script($script, $name, $language="javascript") {
    $this->body->add_named_script($script, $name, $language);
  } // add_named_script
  // .....................................................................
  /**
  * Add named script to the body
  * @param string $script   Script to insert into the body of the webpage
  * @param string $language Language of this script (default javascript)
  */
  function add_script($script, $language="javascript") {
    $this->body->add_script($script, $language);
  } // add_script
  // .....................................................................
  /**
  * Add script URL to the body
  * @param string $src   URL reference to script to insert into the body
  * @param string $language Language of this script (default javascript)
  */
  function add_scriptsrc($src, $language="javascript") {
    $this->body->add_scriptsrc($src, $language);
  } // add_scriptsrc
  // .....................................................................
  /**
  * Add content to a plugin
  * Plugins are regions of a webpage which you want to plug content into.
  * This is the method to call to add content to a named region or plugin.
  * The plugin ID (name) is related to a COMMENT TAG which appears in the
  * template of the page. If the plugin doesn't exist then one is created.
  * The 'content' can be any one of the following types:
  *
  *  String   - A literal string of content to put in the plugin
  *  Path     - Path to a file containing content (can be relative)
  *  Function - A function definition, eg: "foo('xyx')" to return content
  *  Object   - An object. Must be a descendant of RenderableObject.
  *
  * NB: This method can be called multiple times for the given plugin ID,
  *     and content will be accumulated each time.
  *
  * NB: When you nominate an object or a function to provide content, the
  *     system assumes you want these generated "late" - ie. at the time
  *     the webpage is being sent back to the user agent. If you want
  *     content to be inserted early, then use literal strings.
  *
  * @param string $pluginid  ID of this plugin. Used to find plugin location
  * @param mixed  $content   Content to plug in. May be string, path, object or function def.
  */
  function plugin($pluginid, $content) {
    $this->pluginset->addto($pluginid, $content);
    return $this;
  } // plugin
  // .....................................................................
  /**
  * Start/stop inline plugin content definition
  * This method allows plugin content to be taken from inline content
  * in the script page itself. At the start of the content call this
  * method with the name of the plugin (pluginid) you want to put the
  * content into. At the end of the content, call this method with
  * no parameter.
  */
  function plugin_inline($pluginid="") {
    static $the_plugin_id = "";

    if ($the_plugin_id == "") {
      if ($pluginid != "") {
        $the_plugin_id = $pluginid;
        $this->inline_start();
      }
    }
    else {
      $content = $this->inline_end();
      $this->plugin($the_plugin_id, $content);
      $the_plugin_id = "";
    }
  } // plugin_inline
  // .....................................................................
  /**
  * Begin inline buffering
  * If you want to include some 'naked' HTML in-line with your webpage
  * content, then call this method, then close the Php script with the
  * usual "?>" tag. Follow this with your HTML, then restart Php script
  * and grab the resulting HTML with... <?php $s = $page->inline_end();
  * Then $s will contain all the HTML between inline_start/inline_end.
  * @see inline_end()
  */
  function inline_start() {
    ob_start();
  } // inline_start
  // .....................................................................
  /**
  * End inline buffering
  * Usually used as in: <?php $s = $page->inline_end();
  * Then $s will contain all the HTML between inline_start/inline_end.
  * @see inline_start()
  * @return string Contents of the buffer gathered since inline_start()
  */
  function inline_end() {
    $content = ob_get_contents();
    ob_end_clean();
    return $content;
  } // inline_end
  // .....................................................................
  /**
  * Define the theme
  * Set the theme to apply to the page. This should be a single-word ID,
  * and there should be a sub-directory of the same name in the 'templates'
  * directory, with templates and stylesheet(s) in it which comprise
  * the theme. This is how we provide different template sets and stylesheets
  * for branding purposes.
  * @param string $theme Theme name for this webpage
  */
  function set_theme($theme) {
    global $IMAGESDIR, $TEMPLATESDIR;
    // Store new theme name..
    $this->theme = $theme;
    if ($theme != "") {
      // Point at theme images directory..
      $bits = explode("/", $IMAGESDIR);
      $n = count($bits) - 1;
      if ($n >= 0) {
        $imgdirname = $bits[$n];
        $IMAGESDIR = "$TEMPLATESDIR/$this->theme/$imgdirname";
        debugbr("theme '$this->theme': setting images directory to '$IMAGESDIR'", DBG_DEBUG);
      }
    }
  } // set_theme
  // .....................................................................
  /**
  * Set the filenames for the main stylesheet and (optionally) the
  * stylesheet for Internet Explorer compatible browsers. Note: this
  * should only be a filename, not a path. If a theme has been defined
  * then the stylesheet will be expected to be in the subdirectory of
  * the same name as the theme, under the 'templates' directory. If not
  * then the stylesheet is expected to be in the website root dir.
  * Defaults are in effect as follows:
  * If the stylesheet is nullstring, then we look for a stylesheet in
  * the appropriate directory with a name of APP_PREFIX.css. If that
  * isn't found we try looking for 'sitestyle.css'. Same applies for
  * the IE stylesheet, but with "_ie" appended to the filename part.
  * @param string $ss     Name of normal stylesheet
  * @param string $ss_ie  Name of stylesheet applicable to Internet Explorer
  */
  function set_stylesheet($ss="") {
    global $TEMPLATESDIR;
    if ($this->theme != "") {
      $prefix = $this->site_docroot . "$TEMPLATESDIR/$this->theme/";
      $wwwprefix = "$TEMPLATESDIR/$this->theme/";
    }
    else {
      $prefix = $this->site_docroot . "/";
      $wwwprefix = "/";
    }
    // See if we can find defaults if null. We first try files
    // with a stem of APP_PREFIX, and if that fails then we
    // try the standard 'sitestyle.css' naming scheme..
    if ($ss == "") {
      $try = APP_PREFIX;
      if (file_exists($prefix.$try.".css")) {
        $ss = $try.".css";
      }
      else {
        $try = "sitestyle";
        if (file_exists($prefix.$try.".css")) {
          $ss = $try.".css";
        }
      }
    }
    // Check for IE & Netscape stylesheets. The rule is that if this
    // exists it will be the same name as the standard stylesheet but
    // with "_ie" & "_ns" suffixes respectively..
    $ss_ie = ""; $ss_ns = "";
    if ($ss != "") {
      $ssbits = explode(".", $ss);
      $try = $ssbits[0] . "_ie";
      if (file_exists($prefix.$try.".css")) {
        $ss_ie = $try.".css";
      }
      $try = $ssbits[0] . "_ns";
      if (file_exists($prefix.$try.".css")) {
        $ss_ns = $try.".css";
      }
    }

    // Define any stylesheets found..
    if ($ss    != "") $ss    = $wwwprefix.$ss;
    if ($ss_ie != "") $ss_ie = $wwwprefix.$ss_ie;
    if ($ss_ns != "") $ss_ns = $wwwprefix.$ss_ns;
    $this->head->set_stylesheet($ss, $ss_ie, $ss_ns);
  } // set_stylesheet
  // .....................................................................
  /**
  * Define the script to execute after page has loaded
  * @param string $onload Script to execute when page loads
  */
  function set_onload($onload) {
    $this->body->set_onload($onload);
    return $this;
  } // set_onload
  // .....................................................................
  /**
  * Define the webpage title
  * @param string $title Title to give the webpage
  */
  function set_title($title) {
    $this->head->set_title($title);
    return $this;
  } // set_title
  // .....................................................................
  /**
  * Define the style settings for the webpage. Usually the stylesheet
  * will be all you require, however occasionally it is necessary to
  * have specific styles for a page. Set these here, but make sure to
  * provide the FULL style complete with <style></style> tags.
  * @param string $style Style to insert into the webpage
  */
  function set_style($style) {
    $this->head->set_style($style);
    return $this;
  } // set_style
  // .....................................................................
  /**
  * Define the template for the webpage.
  * The $templatename can be the special "plain" which is a pseudo template
  * of vanilla content, a fll path to a file, or a name. If it is a simple
  * name, then the system knows where to find it and constructs a full
  * path to the file from it.
  * @param string $templatename The name of, or full path to the template file
  */
  function set_template($templatename) {
    global $TEMPLATESDIR;
    if ($templatename != "") {
      // Record it..
      $this->template = $templatename;
      $this->templatefile = "";

      if ($templatename != "plain") {
        if (strtolower(substr($templatename, 5)) == ".html" ||
            strtolower(substr($templatename, 4)) == ".wml") {
          // Absolute path to template as it stands..
          $templatefile = $templatename;
        }
        else {
          $extn = "html";
          if ($this->browser == BROWSER_PHONE) {
            $extn = "wml";
          }
          // Build assumed path to correct template file..
          $templatefile = "template_$templatename.$extn";
          if ($this->theme != "") {
            // Use a template from a theme sub-directory..
            $templatefile = "$TEMPLATESDIR/$this->theme/$templatefile";
          }
          else {
            // Use standard template..
            $templatefile = "$TEMPLATESDIR/$templatefile";
          }
        }
        $this->templatefile = $this->site_docroot . $templatefile;
      } // not plain

      // Now we have the template content, use it..
      $this->head->load_template($this->templatefile);
      $this->body->load_template($this->templatefile);
    }
    return $this;
  } // set_template
  // .....................................................................
  /**
  * Assign the DTD for the resident head page section. We have two ways
  * to do this: directly via passed parameter $DTD, or indirectly by
  * using the content (browser) type, and looking up the DTD from the
  * array of DTD specifiers created in the response object.
  * @param string $DTD Optional override DTD specifier string
  */
  function assign_DTD($DTD="") {
    if (isset($this->head)) {
      if ($DTD == "" && isset($this->DTD)) {
        switch ($this->browser_type) {
          case BROWSER_TYPE_XHTML:
          case BROWSER_TYPE_HTML:
            if (isset($this->DTD[BROWSER_TYPE_HTML])) {
              $DTD = $this->DTD[BROWSER_TYPE_HTML];
            }
            break;
          case BROWSER_TYPE_XHTMLMP:
          case BROWSER_TYPE_WML:
          case BROWSER_TYPE_WMLUP:
            if (isset($this->DTD[BROWSER_TYPE_WML])) {
              $DTD = $this->DTD[BROWSER_TYPE_WML];
            }
            break;
        }
      }
      $this->head->set_dtd($DTD);
    }
  } // assign_DTD
  // .....................................................................
  /**
  * Insert a ready-made meta tag object into the webpage head section.
  * @param string $id Meta tag unique ID
  * @param object $metatag Meta tag object to insert
  */
  function insert_metatag($id, $metatag) {
    $this->head->insert_metatag($id, $metatag);
  } // insert_metatag
  // .....................................................................
  /**
  * Add a meta tag to the webpage head section.
  * @param string $name Meta tag name
  * @param string $content Meta tag content
  * @param string $language Optional language specifier for content
  * @param string $scheme Optional scheme specifier for content
  */
  function set_metatag($name, $content, $language, $scheme) {
    $this->head->set_metatag($name, $content, $language, $scheme);
  } // set_metatag
  // .....................................................................
  /**
  * Generate the webpage content
  * This routine is usually called after all the set-up calls for the
  * head, body etc. have been made. This echoes each page section to the
  * buffer, in preparation for delivery to the browser.
  */
  function generate() {
    global $HTMLAREA;
    if (!$this->generated) {
      if ($this->pluginset->hascontent) {
        foreach ($this->pluginset->plugins as $id => $plugin) {
          if (is_object($plugin)) {
            $newstuff = $plugin->render();
            $tag = "<!--" . strtoupper($id) . "-->";
            $this->replace($tag, $newstuff);
          }
        }
      }
      // Set up the javascript environment for any
      // htmlarea widgest which have been used..
      if (isset($HTMLAREA) && $HTMLAREA->activated) {
        $HTMLAREA->render();
      }
      // Produce the content..
      if ($this->buffered) {
        // Acquire content via the Php buffer..
        echo $this->head->render();
        echo $this->body->render();
        echo $this->foot->render();
      }
      else {
        // Content goes directly in..
        $this->content .= $this->head->render();
        $this->content .= $this->body->render();
        $this->content .= $this->foot->render();
      }
      $this->generated = true;
      $this->make_content();
    }
  } // generate
  // .....................................................................
  /**
  * Generate the content and then send it to the user browser.
  */
  function send() {
    $this->generate();
    $this->send_to_browser();
  } // send
  // .....................................................................
  /**
  * Generate the content into the buffer, then return the content.
  * @return string The webpage content complete, as a string
  */
  function render() {
    $this->generate();
    return $this->webpage_content();
  } // render

} // webpage class

// ----------------------------------------------------------------------
/**
* The head class.
* The class is a special kind of page section. It contains all of the meta
* tags, style tags, title and other such things for the webpage. Note that
* this section is not just the head content, but really comprises everything
* which is prior to the /head tag. This might include the DTD specifier
* for instance.
* @package core
*/
class head extends page_section {
  /** Title of the webpage */
  var $title = "";
  /** Style settings for the webpage */
  var $style = "";
  /** The Document Type Definition for this head section */
  var $DTD = "";
  /** Meta tags array */
  var $meta;
  /** Name of the stylesheet associated with the webpage */
  var $stylesheet = "";
  /** Name of the IE stylesheet */
  var $stylesheet_ie = "";
  /** Name of the Netscape stylesheet */
  var $stylesheet_ns = "";
  /** Languages used in content of the page */
  var $languages = array();
  /** Charset for content of the page */
  var $charset = "ISO-8859-1";
  // .....................................................................
  /**
  * Constructor
  * Create a new head object.
  * @param string $title Title of the webpage
  * @param string $style Specific style settings for the webpage
  */
  function head($title="", $style="") {
    $this->page_section();
    $this->set_title($title);
    $this->set_style($style);
  } //head constructor
  // .....................................................................
  /**
  * Define the title
  * @param string $title Title of the webpage
  */
  function set_title($title) {
    $this->title = $title;
  } // set_title
  // .....................................................................
  /**
  * Define the style
  * NB: The way this is currently done, you are expected to supply
  * your style WITHOUT style tags here.
  * @param string $style Specific style settings for the webpage
  */
  function set_style($style) {
    $this->style = $style;
  } // set_style
  // .....................................................................
  /**
  * Add the given content to the current style. Appends style statements
  * to the style string, which is rendered when the page gets rendered.
  * @param string $style Style settings to add to existing ones.
  */
  function add_style($style) {
    $this->style .= $style;
  } // add_style
  // .....................................................................
  /**
  * Set the stylesheet to use. This should be a valid pathname to an
  * existing file. The second parm is for special styles for Internet
  * Explorer-compatible browsers.
  * @param string $ss     Path to normal stylesheet
  * @param string $ss_ie  Path to stylesheet for Internet Explorer
  */
  function set_stylesheet($ss, $ss_ie="", $ss_ns="") {
    $this->stylesheet    = $ss;
    $this->stylesheet_ie = $ss_ie;
    $this->stylesheet_ns = $ss_ns;
  } // set_stylesheet
  // .....................................................................
  /**
  * Insert a ready-made meta tag object into the metatags array.
  * @param string $id Meta tag unique ID
  * @param object $metatag Meta tag object to insert
  */
  function insert_metatag($id, $metatag) {
    $this->meta[$id] = $metatag;
  } // insert_metatag
  // .....................................................................
  /**
  * Add meta tag to the section
  * @param string $name Meta tag name
  * @param string $content Meta tag content
  * @param string $language Optional language specifier for content
  * @param string $scheme Optional scheme specifier for content
  */
  function set_metatag($name, $content, $language="", $scheme="") {
    if ($name != "" && $content != "") {
      $newtag = new HTMLtag("meta");
      $newtag->add_attribute("name", $name);
      $newtag->add_attribute("content", $content);
      if ($language != "") {
        $newtag->add_attribute("lang", $language);
      }
      if ($scheme != "") {
        $newtag->add_attribute("scheme", $scheme);
      }
      $this->insert_metatag($name, $newtag);
    }
  } //set_metatag
  // .....................................................................
  /**
  * Set the DTD specifier string for this head section.
  * @param string $DTD The Document Type Definition string for this head
  */
  function set_dtd($DTD) {
    $this->DTD = $DTD;
  } // set_dtd
  // ...................................................................
  /**
  * Adds another language for the current head. Webpages might
  * contain content in multiple languages, hence the need for a list.
  * @param integer $langid The new language ID to add for the webpage
  */
  function add_language($langid) {
    if (!in_array($langid, $this->languages)) {
      $this->languages[] = $langid;
    }
  } // add_language
  // .....................................................................
  /**
  * Set the charset for this head section
  * @param string $charset The charset code for content of this head section
  */
  function set_charset($charset) {
    $this->charset = $charset;
  } // set_charset
  // ....................................................................
  /**
  * Load given template content.
  * We scrape everything between the appropriate tags and use it as
  * the template for our content.
  * @param string $templatefile  The full path to the template file
  */
  function load_template($templatefile="") {
    $template = $this->get_template($templatefile);
    if ($template != "") {
      $bits = explode("<head>", $template);
      if (isset($bits[0]) && $bits[0] != "") {
        // Always use DTD from template if found..
        if (preg_match("/<!DOCTYPE (.+)>/i", $bits[0], $matches)) {
          $this->set_dtd($matches[0]);
        }
        if (isset($bits[1]) && $bits[1] != "") {
          $bits = explode("</head>", $bits[1]);
          if (isset($bits[0]) && $bits[0] != "") {
            $head = $bits[0];
            // Strip items which Axyl generates for itself..
            $patts = array(
                "<title>.*?<\/title>",                 // Title tag
                "<meta http-equiv=\"content-type.*?>", // Content type meta tag
                "<link rel=(.*?)stylesheet.*?>"        // All stylesheet links
                );
            foreach ($patts as $patt) {
              $head = preg_replace("/$patt\n|$patt/i", "", $head);
            }
            $this->content = $head;
          }
        }
      }
    }
    return $this;
  } // load_template
  // ....................................................................
  /**
  * This renders the head as HTML. After the title and the meta tags
  * are rendered, the stylesheets are next. For the stylesheets we first
  * render the standard links, and then overlay this with the Internet Explorer
  * stylesheet if it is defined. Finally any literal styles are rendered
  * so they will take precedence. The scripts are rendered after the styles,
  * and the head content comes last.
  * @see render()
  * @return string The head section as HTML.
  */
  function html() {
    global $RESPONSE;
    // Content type meta tag - always has to be first..
    $meta = new HTMLtag("meta");
    $meta->add_attribute("http-equiv", "content-type");
    $meta->add_attribute("content", "text/html; charset=$this->charset");
    $this->meta[] = $meta;

    // Language(s) meta tag..
    $languages = array();
    if (count($this->languages) > 0) {
      $q  = "SELECT * FROM ax_language";
      $q .= " WHERE lang_id IN (" . implode(",", $this->languages) . ")";
      $q .= "   AND lang_id <> 0";
      $q .= " ORDER BY display_order";
      $langs = dbrecordset($q);
      if ($langs->hasdata) {
        do {
          $languages[] = $langs->field("char_encoding");
        } while ($langs->get_next());
        // Create the meta tag..
        if (count($languages) > 0) {
          $meta = new HTMLtag("meta");
          $meta->add_attribute("http-equiv", "content-language");
          $meta->add_attribute("content", implode(",", $languages));
          $this->meta[] = $meta;
        }
      }
    }

    // Generator meta tag..
    $meta = new HTMLtag("meta");
    $meta->add_attribute("name", "generator");
    $meta->add_attribute("content", "Catalyst IT Axyl");
    $this->meta[] = $meta;

    // Assemble head section..
    $s = "";
    if ($this->DTD != "") $s .= "$this->DTD\n";
    $s .= "<html";
    if (count($languages) > 0) {
      $s .= " lang=\"" . implode(",", $languages) . "\"";
    }
    $s .= ">\n";
    $s .= "<head>\n";
    $s .= "<title>$this->title</title>\n";

    // Now insert all defined meta tags..
    foreach ($this->meta as $metatag) {
      $s .= $metatag->render();
    }

    // Literal head content..
    $s .= $this->get_trimcontent();

    // The General Stylesheet..
    if (isset($this->stylesheet) && $this->stylesheet != "") {
      $s .= "<link rel=\"stylesheet\" href=\"$this->stylesheet\" type=\"text/css\">\n";
    }

    // Specific IE or Netscape Stylesheet..
    switch ($RESPONSE->browser) {
      case BROWSER_IE:
        if (isset($this->stylesheet_ie) && $this->stylesheet_ie != "") {
          $s .= "<link rel=\"stylesheet\" href=\"$this->stylesheet_ie\" type=\"text/css\">\n";
        }
        break;
      case BROWSER_NETSCAPE:
        if (isset($this->stylesheet_ns) && $this->stylesheet_ns != "") {
          $s .= "<link rel=\"stylesheet\" href=\"$this->stylesheet_ns\" type=\"text/css\">\n";
        }
        break;
    } // switch

    // Static styles..
    if (isset($this->style) && $this->style != "") {
      $s .= "<style type=\"text/css\">\n";
      $s .= "$this->style\n";
      $s .= "</style>\n";
    }

    // Scripts..
    $s .= $this->script();

    $s .= "</head>\n";
    return $s;
  } // html
  // ....................................................................
  /**
  * Use render() to render this element in your page.
  * This renders the field as WML.
  * @see render()
  * @return string The field as WML.
  */
  function wml() {
    $s = "";
    $s .= "<?xml version=\"1.0\" encoding=\"$this->charset\"?>\n";
    if ($this->DTD != "") $s .= "$this->DTD\n";
    $s .= "<wml>";
    if ($this->content != "" || isset($this->meta)) {
      $s .= "<head>";
      if (isset($this->meta)) {
        while (list($name, $content) = each($this->meta)) {
          $s .= "<meta name=\"$name\" content=\"$content\">\n";
        }
      }
      $s .= $this->content;
      $s .= "</head>";
    }
    return $s;
  } // wml
} // head class

// ----------------------------------------------------------------------
/**
* The body class
* The class is a special kind of page section. It contains all of the
* main page content.
* @package core
*/
class body extends page_section {
  /** The script to execute when the page has loaded */
  var $onload = "";
  /** Miscellaneous content for the body tag */
  var $parms = "";
  // .....................................................................
  /**
  * Constructor
  * Create a new body object.
  * @param string $templatefile Template file to load content from
  * @param string $onload Script to execute when page loads
  * @param string $parms  Misc other content for the body tag
  */
  function body($templatefile="", $onload="", $parms="") {
    $this->page_section();
    $this->load_template($templatefile);
    $this->set_onload($onload);
    $this->set_parms($parms);
  } // body constructor
  // .....................................................................
  /**
  * Define the script to execute after page has loaded
  * @param string $onload Script to execute when page loads
  */
  function set_onload($onload) {
    $this->onload .= $onload;
  } // set_onload
  // .....................................................................
  /**
  * Define other miscellaneous content to be put into the body tag.
  * This method can be called multiple times. Each time the content
  * is added (appended) to.
  * @param string $parms  Misc other content for the body tag
  */
  function set_parms($parms) {
    $this->parms .= $parms;
  } // set_parms
  // .....................................................................
  /**
  * Load given template content.
  * We scrape everything between the appropriate tags and use it as
  * the template for our content.
  * @param string $templatefile  The full path to template file
  * @access private
  */
  function load_template($templatefile="") {
    if ($templatefile == "") {
      $this->content = "<!--MAIN_CONTENT-->";
    }
    else {
      $template = $this->get_template($templatefile);
      if ($template != "" && strstr($template, "<body")) {
        $bits = explode("<body", $template);
        if (isset($bits[1]) && $bits[1] != "") {
          $i = strpos($bits[1], ">");
          if ($i > 1) {
            // Scrape body tag parameters and keep these..
            $parms = trim( substr($bits[1], 0, $i) );
            if ($parms != "") {
              $this->set_parms($parms);
            }
          }
          // Scrape body content..
          $content = substr($bits[1], $i + 1);
          $bits = explode("</body>", $content);
          if (isset($bits[0])) {
            $this->content = $bits[0];
            $this->fix_imagerefs();
          }
        }
      }
      else {
        $this->content = "<!--MAIN_CONTENT-->";
      }
    }
  } // load_template
  // ....................................................................
  /**
  * Fixes up all the media src= references taking into account theme etc.
  * Operates on the current $content variable, and fixes it directly.
  * Fix up all src= references (etc) to point to the correct images
  * directory, appropriate to any active theme, but ignores absolute
  * http;// refs..
  * @access private
  */
  function fix_imagerefs() {
    global $IMAGESDIR;
    // Fix up main body content..
    $this->content = preg_replace(
            "/(src=[\'\"])((?!http)(.+?))([\'\"])/ie",
            "'src=\"' . '$IMAGESDIR/' . basename('\\2') . '\"'",
            $this->content
            );
    $this->content = preg_replace(
            "/(background=[\'\"])((?!http)(.+?))([\'\"])/ie",
            "'background=\"' . '$IMAGESDIR/' . basename('\\2') . '\"'",
            $this->content
            );
    $this->content = preg_replace(
            "/(param name=movie value=[\'\"])((?!http)(.+?))([\'\"])/ie",
            "'param name=movie value=\"' . '$IMAGESDIR/' . basename('\\2') . '\"'",
            $this->content
            );
    // Fix up background refs in <body> tag..
    $this->parms = preg_replace(
            "/(background=[\'\"])((?!http)(.+?))([\'\"])/ie",
            "'background=\"' . '$IMAGESDIR/' . basename('\\2') . '\"'",
            $this->parms
            );
  } // fix_imagerefs
  // ....................................................................
  /**
  * This renders the body as HTML.
  * @return string The body section as HTML.
  */
  function html() {
    $s  = "<body";
    if (!empty($this->onload)) $s .= " onload=\"" . $this->onload . "\"";
    if (!empty($this->parms))  $s .= " " . $this->parms;
    $s .= ">\n";
    $s .= $this->script();
    $s .= $this->get_trimcontent();
    $s .= "\n</body>\n";

    return $s;
  } //html
  // ....................................................................
  /**
  * This renders the body as WML.
  * @return string The body section as WML.
  */
  function wml() {
    return $this->content;
  } // wml
} // body class

// ----------------------------------------------------------------------
/**
* The deck class
* The class is a special kind of body section. It contains all of the
* main page content for a WAP phone (wml).
* @package core
*/
class deck extends body {
  /** Template object for WML deck */
  var $WMLtemplate;
  // .....................................................................
  /**
  * Constructor
  * Create a new deck object.
  * @param string $templatefile Template file to load content from
  * @param string $onload Script to execute when page loads
  * @param string $parms  Misc other parameters
  */
  function deck($templatefile="", $onload="", $parms="") {
    $this->body($templatefile, $onload, $parms);
  } // deck constructor
  // ....................................................................
  /**
  * Defines the template section for the deck.
  * @param object $wmltemplate The template object for the WML deck
  */
  function wml_template($wmltemplate) {
    $this->WMLtemplate = $wmltemplate;
  } // wml_template
  // .....................................................................
  /**
  * Load the template content.
  * We scrape everything between the appropriate tags and use it as the
  * template for our body content.
  * @param string $templatefile The full path to template file
  * @access private
  */
  function load_template($templatefile="") {
    if ($templatefile == "") {
      $this->content = "<!--MAIN_CONTENT-->";
    }
    else {
      $template = $this->get_template($templatefile);
      if ($template != "" && strstr($template, "<wml>")) {
        $bits = explode("<wml>", $template);
        if (isset($bits[1]) && $bits[1] != "") {
          $bits = explode("</wml>", $bits[1]);
          if (isset($bits[0]) && $bits[0] != "") {
            $this->content = $bits[0];
            $this->fix_imagerefs();
          }
        }
      }
      else {
        $this->content = "<!--MAIN_CONTENT-->";
      }
    }
  } // load_template
  // ....................................................................
  /**
  * This renders the body as HTML.
  * @return string The body section as HTML.
  */
  function html() {
    $s  = "<body>";
    $s .= $this->get_trimcontent();
    $s .= "\n</body>\n";
    return $s;
  } // html
  // ....................................................................
  /**
  * This renders the deck content.
  * @return string The body section as WML.
  */
  function wml() {
    $s = "";
    if (isset($this->WMLtemplate)) {
      $s .= $this->WMLtemplate->wml();
    }
    $s .= $this->get_trimcontent();
    return $s;
  } // wml

} // deck class

// ----------------------------------------------------------------------
/**
* The foot class
* The class is a special kind of page section. It contains all of the
* page content for the footer region. Actually the footer is the
* simplest of the page sections in a webpage. All it really has is
* the main content, some optional script content and that's about it.
* @package core
*/
class foot extends page_section {
  /**
  * Constructor
  * Create a new foot object.
  */
  function foot() {
    $this->page_section();
  } // foot constructor
  // ....................................................................
  /**
  * This renders the foot as HTML.
  * @return string The foot section as HTML.
  */
  function html() {
    $s = "";
    $s .= $this->content;
    $s .= $this->script();
    $s .= "</html>\n";
    return $s;
  } // html
  // ....................................................................
  /**
  * This renders the foot as WML.
  * @return string The foot section as WML.
  */
  function wml() {
    return $this->content . "</wml>";
  } // wml
}

// ----------------------------------------------------------------------
/**
* The error page class
* Allows us to render errors as either HTML or WML.
* @package core
*/
class error_page extends page_section {
  /** Heading or subject of the error */
  var $heading;
  /** Actual error message in detail */
  var $msg;
  // .....................................................................
  /**
  * Constructor
  * Create a new error page object.
  * @param string $heading The error heading or subject
  * @param string $msg     The detailed error message
  */
  function error_page($heading, $msg="") {
    $this->heading = $heading;
    $this->msg = $msg;
  } // error_page
  // ....................................................................
  /**
  * This renders the error page section as HTML. We do this simply
  * as a heading at level 3 (h3), and the message content.
  * @return string The error page section as HTML.
  */
  function html() {
    return "<h3>$this->heading</h3>" . $this->msg;
  } // html
  // ....................................................................
  /**
  * This renders the error page section as WML. We do this simply
  * as a single card with the appropriate title, and the content
  * on the card in paragraph tags.
  * @return string The error page section as WML.
  */
  function wml() {
    $card = new WMLcard("error", "Error");
    $card->insert(
      "<p><b>" . $this->heading . "</b><br/>" . $this->msg . "</p>"
      );
    // Create the card deck and output it..
    $deck = new WMLdeck($card);
    return $deck->wml();
  } // wml
}

// ----------------------------------------------------------------------
/**
* The templated webpage page class
* This is a special case of webpage, but is largely deprecated since
* the better way of using templates is via the kind of mechanism
* seen in the "site-webpage.php" example file which comes with
* the Phplib. This method is rather a DIY approach.
* @package core
*/
class templated_webpage extends webstream {
  /**
  * Constructor
  * Create a new templated webpage object.
  * @param string $path  Path to the template for the webpage
  */
  function templated_webpage($path="") {
    $this->webstream();
    if ($path != "") {
      $this->set_template($path);
    }
  } // templated_webpage
  // .....................................................................
  /**
  * Set the template file
  * Base webpage content on a template. We echo this to the buffer.
  * The expected usage is that further programming will then repeatedly
  * call the 'replace' function, to make the customised webpage. After
  * that calling 'send' will deliver it to the end browser.
  * @param string $path  Path to the template for the webpage
  */
  function set_template($path) {
    $fp = fopen($path, "r");
    if ($fp) {
      $content = fread($fp, filesize($path));
      fclose ($fp);
      echo $content;
    }
    return $this;
  } // set_template
  // .....................................................................
  /**
  * Send content to user browser
  */
  function send() {
    $this->make_content();
    $this->send_to_browser();
  } // send
  // .....................................................................
  /**
  * Return content to caller
  * @return string Webpage content
  */
  function render() {
    return $this->webpage_content();
  } // render
} // templated_webpage class

// ----------------------------------------------------------------------
/**
* gzip utility. Returns Value as a four-char string.
* @param integer $value Character value to return as four-char string
* @return string Four char string equivalent of the given value
* @access private
*/
function AsFourChars($value) {
  $s = "";
  for ($i = 0; $i < 4; $i++) {
    $s .= chr($value % 256);
    $value = floor($value / 256);
  }
  return $s;
} // AsFourChars

// ----------------------------------------------------------------------
?>