#!/usr/bin/php4 -q
<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    axyl-databases-upgrade.php                              */
/* Author:      Paul Waite                                              */
/* Description: Attempts to upgrade all Axyl databases on the local     */
/*              system.                                                 */
/*                                                                      */
/* ******************************************************************** */
// ROOT CHECK..
$uid = shell_exec("id -u");
if ($uid != 0) {
  abort("You must be root to run this script.");
}

// ----------------------------------------------------------------------
// STDIN File Handle..
$FH = fopen("php://stdin", "r");

// ----------------------------------------------------------------------
// LOGGING
$LOGFILE = "./axyl-databases-upgrade.log";
$LOG = fopen($LOGFILE, "a");
if ($LOG === false) {
  echo "FATAL: could not open logfile for append.";
  exit;
}

// ----------------------------------------------------------------------
// POSTGRES IS REQUIRED
if (!extension_loaded("pgsql")) {
  if (!dl("pgsql.so")) {
    exit;
  }
}
if (!extension_loaded("pgsql")) {
  abort("FATAL: failed to load Postgres PHP extension pgsql.so.");
}

// ----------------------------------------------------------------------
// ATTEMPT TO READ IN THE AXYL CONFIG
// ----------------------------------------------------------------------
$CONF_FILE = "/etc/axyl/axyl.conf";
$conf_content = readallfile($CONF_FILE);
if ($conf_content != "") {
  if (preg_match("/AXUSER=([\S]+)\s/", $conf_content, $matches)) {
    $AXUSER = $matches[1];
  }
  if (preg_match("/AXYL_HOME=([\S]+)\s/", $conf_content, $matches)) {
    $AXYL_HOME = $matches[1];
  }

  if ($AXYL_HOME != "") {
    if (!is_dir($AXYL_HOME)) {
      abort("AXYL_HOME is undefined. Check your /etc/axyl/axyl.conf.");
    }
    else {
      // More through checking..
      if (!is_dir("$AXYL_HOME/db/postgres")) {
        abort(
          "Your Axyl installation at $AXYL_HOME doesn't check out.\n"
        . "For example the 'db/postgres' sub-directory is not there."
        );
      }
      elseif (!is_dir("$AXYL_HOME/www")) {
        abort(
          "Your Axyl installation at $AXYL_HOME doesn't check out.\n"
        . "For example the 'www' sub-directory is not there."
        );
      }
    }
  }
  else {
    abort(
      "The Axyl configuration file at $CONF_FILE does not define your\n"
    . "AXYL_HOME. This script cannot proceed without that setting.\n"
    . "Please check your Axyl configuration."
    );
  }
}
else {
  abort(
    "File /etc/axyl/axyl.conf was invalid. Have you installed Axyl?\n"
  . "If you have, then have you remembered to set it up properly, by\n"
  . "runing the script install/setup-axyl.sh?\n"
  );
}

// ----------------------------------------------------------------------
// FUNCTIONS
// ----------------------------------------------------------------------
function abort($msg) {
  global $LOG;
  $msg = "ERROR: ABORTING DATABASE UPGRADE!\n\n$msg\n";
  logit($msg);
  fclose($LOG);
  exit;
}

function logit($msg) {
  global $LOG;
  if (substr($msg, -1) != "\n") {
    $msg .= "\n";
  }
  fwrite($LOG, $msg);
  echo $msg;
}

function timestamp_to_displaydate($displayformat, $timestamp="") {
  if ($timestamp == "") {
    $timestamp = time();
  }
  return date($displayformat, $timestamp);
}

function connect_pgdb($dbname, $dbuser="", $dbpass="", $dbhost="", $dbport=0) {
  $connstr = " dbname=" . $dbname;
  if ($dbuser != "") $connstr .= " user=" . $dbuser;
  if ($dbpass != "") $connstr .= " password=" . $dbpass;
  if ($dbhost != "") $connstr .= " host=" . $dbhost;
  if ($dbport != 0 ) $connstr .= " port=" . $dbport;
  $connstr = trim($connstr);
  return pg_connect($connstr);
}

function disconnect_pgdb($dbid) {
  if ($dbid !== false) {
    pg_close($dbid);
  }
}

function query_pgdb($dbid, $sql) {
  if (PHP_VERSION >= 4.2) {
    $rid = pg_query($dbid, $sql);
  }
  else {
    $rid = pg_exec($dbid, $sql);
  }
  if ($rid === false) {
    logit("QFAIL: $sql");
    $pgerr = trim(errormessage($dbid));
    if ($pgerr != "") {
      logit("POSTGRES ERROR: $pgerr");
    }
  }
  return $rid;
}

function errormessage($dbid) {
  if (PHP_VERSION >= 4.2) return pg_last_error($dbid);
  else return pg_errormessage($dbid);
}

function numrows_pgdb($rid) {
  if (PHP_VERSION >= 4.2) return pg_num_rows($rid);
  else return pg_numrows($rid);
}

function fetch_array_pgdb($rid, $rowno) {
  return pg_fetch_array($rid, $rowno);
}

function readallfile($path) {
  $content = "";
  if (file_exists($path)) {
    $fp = fopen($path, "r");
    if ($fp !== false) {
      $content = fread($fp, filesize($path));
      fclose($fp);
    }
  }
  return $content;
}

function readallsql($path) {
  $content = readallfile($path);
  if ($content != "") {
    $content = preg_replace("/;$/", "^!^", $content);
    $content = preg_replace("/[\s]+/", " ", $content);
    $coresql_lines = explode("^!^", $content);
    $content = "";
    foreach ($coresql_lines as $sql_statement) {
      $sql_statement = trim($sql_statement);
      if ($sql_statement != "") {
        $content .= "$sql_statement;\n";
      }
    }
  }
  return $content;
}

// ----------------------------------------------------------------------
// AXYL DATABASE UPGRADE
$stamp = timestamp_to_displaydate("M jS g:ia", time());
logit("\nAXYL DATABASES UPGRADE\n");
logit("$stamp\n\n");
logit("This upgrade process will attempt to upgrade all Axyl databases\n");
logit("on this machine to the latest Axyl core schema. The process will\n");
logit("be done in 'no-drops' mode, which only adds missing schema.\n\n");
$doit = true;
echo "Continue? Y or N [Y] :";
$ans = rtrim(fgets($FH, 256));
if ($ans != "" && $ans != "Y" && $ans != "y") {
  $doit = false;
}

if ($doit) {
  logit("Axyl database upgrade begins..");

  // Get database names list..
  logit(" -- examining database environment..");
  $dbid = connect_pgdb("template1");
  if ($dbid !== false) {
    $database_names = array();
    $rid = query_pgdb($dbid, "SELECT datname FROM pg_database WHERE datname NOT LIKE 'template%'");
    if ($rid !== false) {
      $totrows = numrows_pgdb($rid);
      $axylcore_present = false;
      for ($rowno = 0; $rowno < $totrows; $rowno++) {
        $rowdata = fetch_array_pgdb($rid, $rowno);
        if (isset($rowdata["datname"]) && $rowdata["datname"] != "") {
          $dbname = $rowdata["datname"];
          if ($dbname == "axyl_core") {
            $axylcore_present = true;
          }
          else {
            $database_names[] = $rowdata["datname"];
          }
        }
      } // for
    }

    // Deal to Axyl core.. Create the latest..
    if ($axylcore_present) {
      $rid = query_pgdb($dbid, "DROP DATABASE axyl_core");
      if ($rid !== false) {
        logit(" -- dropped existing axyl_core");
      }
    }
    else {
      logit(" -- axyl_core not present");
    }
    $rid = query_pgdb($dbid, "CREATE DATABASE axyl_core");
    if ($rid !== false) {
      logit(" -- created new axyl_core");
    }
    disconnect_pgdb($dbid);

    // Define latest Axyl Core schema..
    $dbid = connect_pgdb("axyl_core");
    if ($dbid !== false) {
      logit(" -- connected to axyl_core");
      $coresql = readallsql("$AXYL_HOME/db/postgres/axyl_core.sql");
      $trigsql = readallsql("$AXYL_HOME/db/postgres/axyl_trig.sql");
      $rid = query_pgdb($dbid, $coresql);
      if ($rid !== false) {
        logit(" -- axyl_core schema now loaded");
      }
      else {
        abort(" -- error loading axyl_core schema");
      }
      $rid = query_pgdb($dbid, $trigsql);
      if ($rid !== false) {
        logit(" -- axyl_core triggers and procedures loaded");
      }
      else {
        abort(" -- error loading axyl_core triggers & procs");
      }
      disconnect_pgdb($dbid);
    }
  }

  // ----------------------------------------------------------------------
  // PROCESS ALL DATABASES
  if (count($database_names) > 0) {
    foreach ($database_names as $dbname) {
      // Check it is an Axyl database..
      $dbid = connect_pgdb($dbname);
      if ($dbid !== false) {
        $rid = query_pgdb($dbid, "SELECT * FROM pg_class WHERE relname='ax_user'");
        if ($rid !== false) {
          $totrows = numrows_pgdb($rid);
          if ($totrows == 1) {
            // Note that we loop because sometimes the diff have problems with
            // ordering which requires them to be iterated a few times..
            $upgrading_schema = true;
            while ($upgrading_schema) {
              logit("");
              echo "Upgrade Axyl database $dbname? Y or N [Y] :";
              $ans = rtrim(fgets($FH, 256));
              if ($ans == "" || $ans == "Y" || $ans == "y") {
                logit(" -- upgrading $dbname");
                $outputlines = array();
                exec("$AXYL_HOME/scripts/dbdiff.php --no-drops --target=$dbname --ref=axyl_core", $outputlines, $ret);
                if ($ret == 0) {
                  $loglines = "";
                  $diffs = array();
                  foreach($outputlines as $line) {
                    if (trim($line) != "" && substr($line, 0, 2) != "--") {
                      if (trim($line) != "") {
                        $loglines .= trim($line) . "\n";
                      }
                      $line = preg_replace("/;$/", "^!^", $line);
                      $diffs[] = trim($line);
                    }
                  }
                  if ($loglines != "") {
                    logit(" -- dbdiff output begins +=+=+=+=+=+=+=+=+=++=+=+=+=+=+=+=+=+=+\n");
                    logit($loglines);
                    logit(" -- dbdiff output ends   +=+=+=+=+=+=+=+=+=++=+=+=+=+=+=+=+=+=+\n");
                  }
                  $diffsql = trim(implode(" ", $diffs));
                  if ($diffsql != "") {
                    echo "\nApply the above SQL to upgrade $dbname (nothing was dropped)? Y or N [Y] :";
                    $ans = rtrim(fgets($FH, 256));
                    if ($ans == "" || $ans == "Y" || $ans == "y") {
                      $ok = true;
                      $diffs = explode("^!^", $diffsql);
                      foreach ($diffs as $diff) {
                        if ($diff != "") {
                          $rid = query_pgdb($dbid, $diff);
                          if ($rid === false) {
                            $ok = false;
                          }
                        }
                      } // foreach
                      // Final check that diff is now reporting identical schemas..
                      if ($ok) {
                        exec("$AXYL_HOME/scripts/dbdiff.php --no-drops --target=$dbname --ref=axyl_core", $outputlines);
                        foreach($outputlines as $line) {
                          if (trim($line) != "" && substr($line, 0, 2) != "--") {
                            $ok = false;
                            break;
                          }
                        }
                      }

                      // Report results..
                      if ($ok) {
                        $upgrading_schema = false;
                        logit(" -- done");
                      }
                      else {
                        logit(" -- failed - see any messages above for clues");
                        logit(" -- NB: failure might just be a case of statement ordering problems so");
                        logit(" -- please go around this loop until the SQL diffs don't change.");
                      }
                    }
                    else {
                      logit(" -- upgrade SQL was aborted (user choice)");
                      logit(" -- warning: not upgrading the database compromises the upgrade.");
                      $upgrading_schema = false;
                    }
                  }
                  else {
                    logit(" -- $dbname is up to date");
                    $upgrading_schema = false;
                  }
                } // fixups to make
              }
              else {
                logit(" -- user declined upgrade");
                $upgrading_schema = false;
              }
            } // while
          } // is Axyl db
        }
      } // can connect to db..
    } // foreach dbname
  }
  else {
    logit(" -- No databases found to process.");
  }
} // doit

?>