#!/bin/sh
# Upgrade/patch a POSTGRESQL database for an Axyl website.

# NOTE: This is not normally run standalone. The main DB upgrade
# script 'install/upgrade-axyl-databases.sh normally runs this.
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# INCOMING PARAMETERS
#  $1  - DBNAME
#  $2  - DBUSER
#  $3  - DBPASSWD ('none' means a blank password)
#  $4  - DBHOST ('direct' means a local database server)
#  $5  - DBPORT
#  $6  - AXYL_VERSION (Axyl version, '' or 'n.n.n'
DBNAME=""
DBUSER=""
DBPASSWD=""
DBHOST=""
DBPORT=5432
AXYL_VERSION="unknown"

POS=0
while [ $# -gt 0 ] ; do
  POS=`expr $POS + 1`
  case $POS in
    1)  DBNAME=$1
        ;;
    2)  DBUSER=$1
        ;;
    3)  DBPASSWD=$1
        ;;
    4)  DBHOST=$1
        ;;
    5)  DBPORT=$1
        ;;
    6)  AXYL_VERSION=$1
        ;;
  esac
  shift
done

# Cater for standalone running of this script. Normally we are called
# with AXYL_HOME et al already defined..
if [ -z $AXYL_HOME ] ; then
	CONFDIR=/etc/axyl
	CONF=${CONFDIR}/axyl.conf
	if [ ! -f $CONF ] ; then
	  echo "Axyl configuration file $CONF not found!"
	  exit 2
	else
	  . $CONF
	  if [ ! -d $AXYL_HOME ] ; then
	    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
	    exit 6
	  fi
	  . ${AXYL_HOME}/install/axyl-common.sh
	fi
fi

# We require Postgres to be locally installed, at least as postgresql-client
# even if no servers are created locally. This is based on the standard
# Debian location, with a few likely fallbacks..
PG_BIN=
PG_PATHS="/usr/lib/postgresql/bin /usr/bin /usr/local/pgsql/bin"
for pg_path in ${PG_PATHS}
do
  if [ -x ${pg_path}/psql ]
  then
    PG_BIN=$pg_path
    break
  fi
done

# Echo and log messages..
LOGFILE=${AXYL_LOGS}/upgrade-db.log
function logit() {
  echo $1
  echo `date` $1 >> $LOGFILE
}

# Exit if no Postgres executables..
[ -z $PG_BIN ] && exit 0

# Optional host settings for remotely accessed databases..
HOSTOPTS=""
[ "$DBHOST" != "direct" ] && HOSTOPTS="--host $DBHOST --port $DBPORT"

# Upgrade the database contents (tables and data)
PSQL=${PG_BIN}/psql
if [ -x $PSQL ] ; then
  # Acquire Axyl version from ax_control table. If this table doesn't
  # exist then skip this and we assume v2.1.0 as a default.
  AXYL_DB_VERSION="2.1.0"
  AXYL_CONTROL=`$PSQL --tuples-only --username $DBUSER --dbname $DBNAME $HOSTOPTS --command "SELECT relname FROM pg_class WHERE relname='ax_control'" | tr ' ' -d`
  if [ "$AXYL_CONTROL" = "ax_control" ] ; then
    AXYL_DB_VERSION=`$PSQL --tuples-only --username $DBUSER --dbname $DBNAME $HOSTOPTS --command "SELECT site_axyl_version FROM ax_control" | tr ' ' -d`
    if [ "$AXYL_DB_VERSION" = "" ] ; then
  	  AXYL_DB_VERSION="2.1.0"
    fi
  fi
  if [ "$AXYL_DB_VERSION" = "$AXYL_VERSION" ] ; then
    logit "${DBNAME} is up to date."
  else
    # Ok, we know the database version, and we know the Axyl installation version,
    # so let's see if we have a patch file series for this upgrade.
    # No patch files, no upgrade.
    UPGRADEROOT=${AXYL_HOME}/db/postgres/upgrade
  	cd ${UPGRADEROOT}
    patchfiles=`ls upgrade_*`
    if [ "$patchfiles" != "" ] ; then
    	patchSQL=`tempfile --prefix dbupgrade` 
    	currentver=$AXYL_DB_VERSION
    	gotpatches=0
    	for patchfile in $patchfiles ; do
   	      if [ "$currentver" = "$AXYL_VERSION" ] ; then
   	      	break
   	      fi
    	  patchnos=`echo $patchfile | sed "s;.sql;;"`
    	  fromver=`echo $patchnos | cut -d'_' -f2`
    	  nextver=`echo $patchnos | cut -d'_' -f4`
    	  if [ "$currentver" = "$fromver" ] ; then
			cat $patchfile >> $patchSQL
			currentver=$nextver
			gotpatches=1
    	  fi
    	done
    	if [ $gotpatches -eq 1 ] ; then
	  	  logit "patching ${DBNAME} ${AXYL_DB_VERSION} --> ${currentver}"
	  	  $PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --file $patchSQL  >>$LOGFILE 2>&1
	  	  if [ $? -eq 0 ] ; then
	  	    $PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --command "UPDATE ax_control set site_axyl_version='${currentver}'" >/dev/null
	  	  else
	  	    logit "errors occurred during the patch process"
	      fi
 	      echo "see patch log in $LOGFILE"
		else
		  logit "no patches are available for ${DBNAME} ${AXYL_DB_VERSION}"
		fi
		# Remove temporary patch file..
		rm -f $patchSQL
	fi
  fi  
else
  # No psql, no action..
  exit 0
fi

# END