#! /bin/bash
#
# Configure apache for Axyl
# P Waite
set -e

# Name of this script
THIS=`basename $0`

# Get common funcs and config vars etc.
# Acquire the local configuration..
CONFDIR=/etc/axyl
CONF=${CONFDIR}/axyl.conf
if [ ! -f $CONF ] ; then
  echo "Axyl configuration file $CONF not found!"
  exit 2
else
  . $CONF
  if [ ! -d $AXYL_HOME ] ; then
    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
    exit 6
  fi
  . ${AXYL_HOME}/install/axyl-common.sh
fi

# Parameters..
if [ $# -lt 6 ] ; then
  echo "usage: $THIS vhost servername admin_email documentroot maxmem maxupload [silent]"
  exit 3
fi

VHOST=$1
VHOST_MATCH=`echo $VHOST | sed -e "s;\*;\\\\\*;g"`
VSERVERNAME=$2
ADMIN_EMAIL=$3
DOCROOT=$4
MAXMEM=$5
MAXUPLOAD=$6
if [ "$7" != "" ] ; then
  MODE=$7
else
  MODE="interactive"
fi

# Functions
function fix_phpini () {
  TITLE=$1
  AXYL_HOME=$2
  PHP_INI=$3
  PHP_INI_OLD=$4
  TSTAMP=`date +"%Y%m%d%H%M%S"`
  PATH_PATT="^include_path[\s]*=[\s]*.*${AXYL_HOME}/lib"
  if [ -f $PHP_INI ] ; then
    # php ini file
    perl -pi -e "s;^[\s]*expose_php[\s]*.*$;expose_php = Off;i" ${PHP_INI}
    perl -pi -e "s;^[\s]*max_execution_time[\s]*.*$;max_execution_time = 0;i" ${PHP_INI}
    perl -pi -e "s;^[\s]*register_globals[\s]*.*$;register_globals = On;i" ${PHP_INI}
    perl -pi -e "s;^[\s]*magic_quotes_gpc[\s]*.*$;magic_quotes_gpc = Off;i" ${PHP_INI}
    CHKAX=`perl -n -e "m;${PATH_PATT};i && print;" ${PHP_INI}`
    if [ "$CHKAX" = "" ] ; then
      if [ "$MODE" = "interactive" ] ; then
        echo "IMPORTANT:"
        echo "For Axyl Php scripts to work properly for $TITLE we need to make sure"
        echo "the 'include_path' is set in ${PHP_INI}. The path can be"
        echo "re-written for you now, or if you already have a path you want to preserve,"
        echo "you can do it yourself after this installation."
        echo -n "Re-write this include path now? (y/n) [n] :"
        read ANS
      else
        ANS=y
      fi
      if [ "$ANS" = "y" -o "$ANS" = "Y" ] ; then
        cp -a ${PHP_INI} ${PHP_INI_OLD}
        A=`perl -n -e "m;^include_path[\s]*=;i && print;" ${PHP_INI}`
        if [ "$A" = "" ] ; then
          # Append include path to file..
          echo "" >> ${PHP_INI}
          echo "; Include path appended by Axyl installation.." >> ${PHP_INI}
          echo "include_path = \".:${AXYL_HOME}/lib\"" >> ${PHP_INI}
          echo "include path was appended to ${PHP_INI}"
        else
          # Replace include path
          perl -pi -e "s;^[\s]*include_path[\s]*.*$;include_path = \".:${AXYL_HOME}/lib\";i" ${PHP_INI}
          tell "include path was re-written in ${PHP_INI}"
        fi
        tell "NOTE: the original file was saved as:"
        tell "  ${PHP_INI_OLD}"
      else
        tell ""
        tell "Ok, no changes were made. You will need to make sure the following path"
        tell "is inserted into the 'include_path' in ${PHP_INI}: "
        tell "  '.:${AXYL_HOME}/lib'"
      fi
    fi
  fi
}

# Report Php module detection status. This will have been done
# along with Apache detection, in axyl-common.sh.
if [ $PHPDET -eq 1 ] ; then
	echo "Php version was detected as $PHP"
	if [ $PHP_APACHE_MOD -eq 1 ] ; then
		echo "Apache Php module seems to be installed also"
	fi
else
	echo "Warning: did not detect Php installed. Please install it."
	exit 6
fi

# Setup CONF depending on the apache config system..
case $APACHE_CONFSYS in
  ap2)
    CONF=${APACHE_CONFDIR}/sites-available/${VSERVERNAME}.conf
    [ -f $CONF ] &&  tell "WARNING: re-defining existing Axyl $APACHE_NAME configuration"
    >$CONF
    ;;
  confd)
    CONF=${APACHE_CONFDIR}/conf.d/${VSERVERNAME}.conf
    [ -f $CONF ] &&  tell "WARNING: re-defining existing Axyl $APACHE_NAME configuration"
    >$CONF
    ;;
  httpd)
    CONF=${APACHE_CONFDIR}/httpd.conf
    cp $CONF ${CONF}.axyl-old
    ;;
esac  

# Work file..
CONF_NEW=`tempfile -s axyl-new`
apache_reload=0
if [ -e $CONF ]
then
  if [ "$APACHE_CONFSYS" = "ap2" -o "$APACHE_CONFSYS" = "confd" ] ; then
    echo "# Axyl virtual host: $VSERVERNAME" >${CONF_NEW}
    echo "# Created:" `date` >>${CONF_NEW}
  else
    A=`perl -n -e "m;ServerName $VSERVERNAME; && print;" $CONF`
    if [ "$A" != "" ] ; then
      ${AXYL_HOME}/install/remove-apache-vh.sh $VSERVERNAME $MODE
    fi
    cp -a $CONF $CONF_NEW
  fi

  # Insert NameVirtualHost directive..
  if [ "$APACHE_CONFSYS" != "ap2" ] ; then
    A=`perl -n -e "m;^[\s]*NameVirtualHost.*${VHOST_MATCH}; && print;" $CONF`
    if [ "$A" = "" ] ; then
      echo "NameVirtualHost $VHOST" >>${CONF_NEW}
    fi
  fi

  # Logging directory. We use an Axyl location, since various distros
  # have an assortment of locations in use and it is impossible to
  # be sure of finding the correct area otherwise..
  if [ -z $AXYL_LOGS ] ; then
    AXYL_LOGS="/var/log/axyl"
  fi
  APACHE_LOGS=${AXYL_LOGS}/${APACHE_NAME}
  mkthisdir $APACHE_LOGS

  # Insert the virtual host definition..
  tell "adding vhost $VSERVERNAME"
  echo "<VirtualHost ${VHOST}>" >>${CONF_NEW}
  echo "  ServerName ${VSERVERNAME}" >>${CONF_NEW}
  echo "  ServerAdmin ${ADMIN_EMAIL}" >>${CONF_NEW}
  echo "  DocumentRoot ${DOCROOT}" >>${CONF_NEW}
  echo "  DirectoryIndex index.php" >>${CONF_NEW}
  echo "  ErrorLog ${APACHE_LOGS}/${VSERVERNAME}-error.log" >>${CONF_NEW}
  echo "  CustomLog ${APACHE_LOGS}/${VSERVERNAME}-access.log common" >>${CONF_NEW}
  echo "  php_value include_path \".:${DOCROOT}:${AXYL_HOME}/lib\"" >>${CONF_NEW}
  echo "  php_flag register_globals on" >>${CONF_NEW}
  echo "  php_flag display_errors off" >>${CONF_NEW}
  echo "  php_flag log_errors on" >>${CONF_NEW}
  echo "  php_flag magic_quotes_gpc off" >>${CONF_NEW}
  echo "  php_value memory_limit ${MAXMEM}M" >>${CONF_NEW}
  echo "  php_value upload_max_filesize ${MAXUPLOAD}M" >>${CONF_NEW}
  echo "  php_value post_max_size ${MAXUPLOAD}M" >>${CONF_NEW}
  echo "  <Directory ${DOCROOT}/>" >>${CONF_NEW}
  echo "    Order Deny,Allow" >>${CONF_NEW}
  echo "    Allow from All" >>${CONF_NEW}
  echo "    Options FollowSymLinks" >>${CONF_NEW}
  echo "    AllowOverride" >>${CONF_NEW}
  echo "  </Directory>" >>${CONF_NEW}
  echo "  Options FollowSymLinks" >>${CONF_NEW}
  echo "</VirtualHost>" >>${CONF_NEW}

  # Move new contents to applicable destination..
  mv $CONF_NEW $CONF
  
  # Apache2 site needs enabling now..
  if [ "$APACHE_CONFSYS" = "ap2" ] ; then
    if [ -x /usr/sbin/a2ensite ] ; then
      /usr/sbin/a2ensite ${VSERVERNAME}.conf
    else
      if [ -d ${APACHE_CONFDIR}/sites-enabled ] ; then
        ln -s ${APACHE_CONFDIR}/sites-available/${VSERVERNAME}.conf ${APACHE_CONFDIR}/sites-enabled/${VSERVERNAME}.conf
      fi
    fi
  fi

  # Make sure Php module is enabled
  if [ "$APACHE_CONFSYS" = "ap2" ] ; then
  	a2enmod ${PHP} >/dev/null 2>&1
  else
    APACHE_MODCONF=/usr/sbin/modules-config
    if [ -x ${APACHE_MODCONF} ] ; then
      APACHEID=`basename ${APACHE_CONFDIR}`
      EXISTS=`${APACHE_MODCONF} ${APACHEID} query mod_${PHP}`
      [ "$EXISTS" = "" ] && ${APACHE_MODCONF} ${APACHEID} enable mod_${PHP}
    else
      perl -pi -e "s/# *LoadModule ${PHP}_module/LoadModule ${PHP}_module/gi" $CONF
    fi
  fi
  apache_reload=1
else
  tell "ERROR: apache config files were not found."
  exit 7
fi
rm -f $CONF_NEW

# Make sure PHP ini files are ok
TSTAMP=`date +"%Y%m%d%H%M%S"`

# Apache is done via virtual hosts, and the global php.ini is probably
# best left alone. Uncomment as appropriate for your system..
# NOTICE: Left commented out as virtualhosts usually do what's required
#PHP_INI_APACHE_OLD=${PHP_INI_APACHE}.axyl-saved.${TSTAMP}
#fix_phpini "Apache" $AXYL_HOME $PHP_INI_APACHE $PHP_INI_APACHE_OLD

tell "checking your CLI/CGI script php.ini settings.."
PHP_INI_CLI_OLD=${PHP_INI_CLI}.axyl-saved.${TSTAMP}
fix_phpini "CLI" $AXYL_HOME $PHP_INI_CLI $PHP_INI_CLI_OLD

# Restart apache..
if [ $apache_reload -eq 1 ] ; then
  tell "reloading apache config.."
  ${AXYL_HOME}/install/prod-apache.sh reload $APACHE_NAME
fi

# END
