<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    calendar-defs.php                                       */
/* Author:      Paul Waite                                              */
/* Description: Definitions for a calendar 'widget'                     */
/*                                                                      */
/* ******************************************************************** */
/** @package datetime */

// ......................................................................
/**
* Calendar
* We define a class called 'Calendar' which renders a nice-looking
* graphical calendar 'widget' in the page and allows the user to click
* on dates and submit them to choose a date and 'do' something which
* is application-specific. The class contains various properties and
* methods to allow the user to find the current date that the calendar
* is set to, and acquire this date as a timestamp, nicely-formatted
* string or a DB-compliant value.
* @package datetime
*/
class Calendar extends RenderableObject {
  var $initDD = 1;                // Initial day-date 1-31
  var $initMM = 1;                // Initial month 1-12
  var $initYY = 1970;             // Initial year in YYYY format
  var $DD;                        // Current day-date 1-31
  var $MM;                        // Current month 1-12
  var $YY;                        // Current year in YYYY format
  var $dayname;                   // Current day name
  var $daysinmonth;               // Days in current month
  var $posturl = "";              // URL to POST the form to
  var $render_form = true;        // Whether to render as form elements
  var $render_subform = false;    // Whether to render elements as sub-form
  var $days_clickable = true;     // Whether days are rendered clickable or not
  var $formname = "fm_calendar";  // Form name to use

  var $days = 7;
  var $weeks = 6;
  var $months = 12;
  var $startyear = 1970;
  var $endyear = 9999;

  // ....................................................................
  /** Constructor. Create a new Calendar object with optional centre
  * date (defaults to 'today'). This date is the one we return to when
  * the 'today' button is clicked on the calendar & is the starting point.
  * @param mixed $today Today's date, either int timestamp or date string
  */
  function Calendar($today="") {
    global $RESPONSE;
    $this->posturl = $RESPONSE->requested;

    // Set initial 'todays' date..
    $this->set_today($today);

    // Do our normal processing..
    $this->POSTprocess();

    // Set default year limits. We do this after the POST processing
    // so that our default limits don't mess up the settings..
    $this->set_year_limits($this->initYY - 5, $this->initYY + 5);
  } // Calendar
  // ....................................................................
  /** Set the 'today' date for the calendar. This is the date that is
  * reset to when the user clicks on the 'TODAY' button. Of course it
  * defaults to the current date when the calendar object is created,
  * but this method allows for customised 'today' settings.
  * NB: If you call this with no paramters, it sets it to 'today'
  * @param mixed $today Today's date, either int timestamp or date string
  */
  function set_today($today="") {
    // Set initial date..
    $init_ts = time();
    if ($today !== "") {
      if (is_integer($today)) {
        $init_ts = $today;
      }
      elseif (is_string($today)) {
        $init_ts = strtotime($today);
      }
    }
    $this->initDD = date("j", $init_ts);
    $this->initMM = date("m", $init_ts);
    $this->initYY = date("Y", $init_ts);
  } // set_today
  // ....................................................................
  /** Set the starting and finishing year for the calendar - the range
  * of operation. This defaults to +/-10 years around the current year.
  * @param integer $startyear Starting year for calendar (>= 1970)
  * @param integer $endyear Ending year for calendar
  */
  function set_year_limits($startyear, $endyear) {
    $this->startyear = $startyear;
    if ($startyear < 1970) {
      $this->startyear = 1970;
    }
    if ($endyear >= $startyear) {
      $this->endyear = $endyear;
    }
    else {
      $endyear = $startyear;
    }
  } // set_year_limits
  // ....................................................................
  /** Set the URL of script to POST the calendar form to.
  * @param string $posturl URL to post the calendar to.
  */
  function post_to($posturl) {
    $this->posturl = $posturl;
  } // post_to
  // ....................................................................
  /** Render calendar dates (the individual days) as clickable. This will
  * cause a form submit, with the details of the date clicked on and the
  * calendar will be set to that date.
  * @param boolean $mode If true, then render days as clickable links
  */
  function render_days_clickable($mode=true) {
    $this->days_clickable = $mode;
  } // render_days_clickable
  // ....................................................................
  /** Set the calendar to be rendered in its own form. If your calendar
  * sits in an existing form, then call this method with 'false' to
  * prevent it rendering its own form.
  * @param boolean $mode If true, then render form, else just the calendar.
  */
  function render_in_form($mode=true) {
    $this->render_form = $mode;
  } // render_in_form
  // ....................................................................
  /** Set the calendar to be rendered in a sub-form. This is just the same
  * as rendering in a form, except we miss off the form tags.
  * @param boolean $mode If true, then render as a sub-form.
  */
  function render_as_subform($mode=true) {
    if ($mode == true) {
      $this->render_in_form(true);
    }
    $this->render_subform = $mode;
  } // render_in_form
  // ....................................................................
  /** Process GET/POST from form.
  * The form works in a GET mode, therefore new date settings are as if
  * passed in on the URL. This method processes these, and sets the
  * internal calendar date values accordingly.
  */
  function POSTprocess() {
    global $calDD, $calMM, $calYY;
    global $calDD_current, $calMM_current, $calYY_current;
    global $calReset, $calReset_x;

    // Report present settings..
    if (isset($calYY_current)) {
      debugbr("calendar: Pre-POSTprocess current year: $calYY_current", DBG_DEBUG);
    }
    if (isset($calMM_current)) {
      debugbr("calendar: Pre-POSTprocess current month: $calMM_current", DBG_DEBUG);
    }
    if (isset($calDD_current)) {
      debugbr("calendar: Pre-POSTprocess current day: $calDD_current", DBG_DEBUG);
    }

    // RESET TO TODAY
    if ( isset($calReset) || isset($calReset_x) ) {
      $this->DD = $this->initDD;
      $this->MM = $this->initMM;
      $this->YY = $this->initYY;
      debugbr("calendar: Today click (reset): YY=$this->YY MM=$this->MM DD=$this->DD", DBG_DEBUG);
    }
    // GET/POST PROCESSING
    else {
      for ( $i = 1; $i <= $this->days * $this->weeks; $i++ ) {
        global ${"calDD" . trim($i) . "_x"};
        if ( isset(${"calDD" . trim($i) . "_x"}) ) {
          $this->DD = $i;
          debugbr("calendar: day click: $this->DD", DBG_DEBUG);
        }
      } // for

      for ( $i = 0; $i <= $this->months + 1; $i++ ) {
        global ${"calMM" . trim($i) . "_x"};
        if ( isset(${"calMM" . trim($i) . "_x"}) ) {
          $this->MM = $i;
          debugbr("calendar: Prev/Next month click: $this->MM", DBG_DEBUG);
        }
      } // for

      // Assign values..
      // YEAR
      if (!isset($this->YY)) {
        if (isset($calYY)) {
          $this->YY = $calYY;
          debugbr("calendar: Year set from dropdown menu: $this->YY", DBG_DEBUG);
        }
        elseif (isset($calYY_current)) {
          $this->YY = $calYY_current;
          debugbr("calendar: Year set from current value: $this->YY", DBG_DEBUG);
        }
        else {
          $this->YY = $this->initYY;
          debugbr("calendar: Year fall-back to default/today: $this->YY", DBG_DEBUG);
        }
      }
      // MONTH
      if (!isset($this->MM)) {
        if (isset($calMM)) {
          $this->MM = $calMM;
          debugbr("calendar: Month set from dropdown menu: $this->MM", DBG_DEBUG);
        }
        elseif (isset($calMM_current)) {
          $this->MM = $calMM_current;
          debugbr("calendar: Month set to current value: $this->MM", DBG_DEBUG);
        }
        else {
          $this->MM = $this->initMM;
          debugbr("calendar: Month fall-back to default/today: $this->MM", DBG_DEBUG);
        }
      }
      // DAY
      if (!isset($this->DD)) {
        if (isset($calDD)) {
          $this->DD = $calDD;
          debugbr("calendar: Day set from dropdown menu: $this->DD", DBG_DEBUG);
        }
        elseif (isset($calDD_current)) {
          $this->DD = $calDD_current;
          debugbr("calendar: Day set from current value: $this->DD", DBG_DEBUG);
        }
        else {
          $this->DD = $this->initDD;
          debugbr("calendar: Day fall-back to default/today: $this->DD", DBG_DEBUG);
        }
      }
    } // GET/POST processing

    // BOUNDARY & SANITY CHECKING
    if ( $this->MM <= 0 ) {
      if ($this->YY > $this->startyear) {
        $this->MM += 12;
        $this->YY -= 1;
        debugbr("calendar: Crossed year boundary moving backwards: Month now: $this->MM, Year now: $this->YY", DBG_DEBUG);
      }
      else {
        $this->MM = 1;
        debugbr("calendar: Attempt to go back earlier than start year: Month now: $this->MM, Year stays as: $this->YY", DBG_DEBUG);
      }
    }

    if ( $this->MM > 12 ) {
      if ($this->YY < $this->endyear) {
        $this->MM -= 12;
        $this->YY += 1;
        debugbr("calendar: Crossed year boundary moving forwards: Month now: $this->MM, Year now: $this->YY", DBG_DEBUG);
      }
      else {
        $this->MM = 12;
        debugbr("calendar: Attempt to go forward beyond end year: Month now: $this->MM, Year stays as: $this->YY", DBG_DEBUG);
      }
    }

    // DAY NAME
    $this->dayname = date("l", $this->get_timestamp());
    debugbr("calendar: day name set to $this->dayname", DBG_DEBUG);

    // Set the days in this month using Php to check for leaps..
    $this->daysinmonth = 0;
    for ($i=28; $i < 32; $i++) {
      if ( checkdate($this->MM, $i, $this->YY) ) {
        $this->daysinmonth = $i;
      }
      else break;
    }
    debugbr("calendar: days in month set to $this->daysinmonth", DBG_DEBUG);

    // END-OF-MONTH BOUNDARY CHECK
    if ($this->DD > $this->daysinmonth) {
      debugbr("calendar: attempt to set days > day in month ($this->DD, day set to $this->daysinmonth", DBG_DEBUG);
      $this->DD = $this->daysinmonth;
    }

    // Final summary of what it got set to..
    if (debugging()) {
      debugbr("calendar: POSTprocess final date setting (ISO): " . $this->get_displaydate("Y-m-d"), DBG_DEBUG);
    }

  } // POSTprocess
  // ....................................................................
  /** Return the current calendar date as a timestamp. This will return
  * the timestamp at 00:00:00 (hh:mm:ss) ie. at the beginning of that day.
  * @return integer Return current calendar date as a timestamp.
  */
  function get_timestamp() {
    return mktime(0, 0, 0, $this->MM, $this->DD, $this->YY);
  } // get_timsetamp
  // ....................................................................
  /** Return the currently stored calendar date as a string in a given
  * date format (see Axyl datetime-defs.php for pre-defined formats).
  * The format is comprised of format characters as per the standard
  * Php function 'date()' (see Php manual entry).
  * @return string The current calendar date as a formatted date string.
  */
  function get_displaydate($displayformat) {
    return timestamp_to_displaydate($displayformat, $this->get_timestamp());
  } // get_displaydate
  // ....................................................................
  /** Return the currently stored calendar date as a string which will
  * be formatted so as to go into a database field nicely. This normally
  * means ISO format (YYYY-MM-DD).
  * @return string The current calendar date in DB-compatible format.
  */
  function get_DB_datetime() {
    return timestamp_to_datetime($this->get_timestamp());
  } // get_DB_datetime
  // ....................................................................
  /** Return the currently stored day-date as a two-digit string
  * @return string Set date in '99' format
  */
  function get_DDstr() {
    return sprintf("%02d", $this->DD);
  }
  // ....................................................................
  /** Return the currently stored month as a two-digit string
  * @return string Set month in '99' format
  */
  function get_MMstr() {
    return sprintf("%02d", $this->MM);
  }
  // ....................................................................
  /** Return the currently stored year as a two-digit string
  * @return string Set year in '9999' format
  */
  function get_YYstr() {
    return sprintf("%04d", $this->YY);
  }
  // ....................................................................
  /** Check that currently stored Month, Day and Year make a correct
  * date. Returns true if so.
  * @return boolean True if current date is valid
  */
  function is_valid() {
    return ( checkdate($this->MM, $this->DD, $this->YY) );
  } // is_valid
  // ....................................................................
  /**
  * Return the calendar HTML
  * @return string The calendar HTML.
  */
  function html() {
    global $RESPONSE, $LIBDIR;

    // -- Build the array of the day of the month

    // Calculate the Day of the Week offset for the 1st of the month
    $offset = date("w", mktime(0,0,0, $this->MM, 1, $this->YY));

    // Build a Day of the Month array for later display
    $dom = Array();
    for ( $i = 1; $i <= $offset; $i++ ) {
      $dom[$i] = 0;
    }

    for ( $i = 1; $i <= $this->days * $this->weeks; $i++ ) {
      if ( checkdate($this->MM, $i, $this->YY) ) {
        $dom[$i + $offset] = $i;
      }
      else {
        $dom[$i + $offset] = 0;
      }
    }

    // -- Now actually draw the calendar
    $s = "";
    if ( $this->render_form ) {
      if (!$this->render_subform) {
        $s .=  "\n<form name=\"$this->formname\" action=\"$this->posturl\" method=\"get\">\n";
      }
      $s .=  "\n<input type=\"hidden\" name=\"calDD_current\" value=\"$this->DD\">\n";
      $s .=  "\n<input type=\"hidden\" name=\"calMM_current\" value=\"$this->MM\">\n";
      $s .=  "\n<input type=\"hidden\" name=\"calYY_current\" value=\"$this->YY\">\n";
    }

    // Adjust the post url so it ends on the right thing ...
    if ( ! ereg("\?", $this->posturl) ) {
      $this->posturl .= "?";
    }
    else {
      $this->posturl .= "&amp;";
    }

    $s .= "
    <table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">
      <tr valign=\"top\">
        <td>
          <table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">
            <tr valign=\"top\">
              <td><img name=\"cal_1_1\" src=\"$LIBDIR/img/calendar/cal_1_1.gif\" width=\"64\" height=\"20\" border=\"0\"></td>
              <td>
                <table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">
                  <tr valign=\"top\">
                    <td><img name=\"cal_1_2\" src=\"$LIBDIR/img/calendar/cal_1_2.gif\" width=\"54\" height=\"9\" border=\"0\"></td>
                  </tr>
                  <tr valign=\"top\">
                    <td>";

                        if ( $this->render_form ) {
                          $s .= "<input type=\"image\" src=\"$LIBDIR/img/calendar/cal_1_4.gif\" width=\"54\" height=\"11\" border=\"0\" ";
                          $s .= "  name=\"calReset\" title=\"Reset to 'today'\" alt=\"\">";
                        } else {
                          $s .= "<a href=\"" . $this->posturl . "calReset=\">";
                          $s .= "<img src=\"$LIBDIR/img/calendar/cal_1_4.gif\" width=\"54\" height=\"11\" border=\"0\" title=\"Reset date\">";
                          $s .= "</a>";
                        }

             $s .= "</td>
                  </tr>
                </table>
              </td>
              <td><img name=\"cal_1_3\" src=\"$LIBDIR/img/calendar/cal_1_3.gif\" width=\"126\" height=\"20\" border=\"0\"></td>
            </tr>
          </table>
        </td>
      </tr>
      <tr valign=\"top\">
       <td><table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">
        <tr valign=\"top\">
            <td><img name=\"calendar_2\" src=\"$LIBDIR/img/calendar/calendar_2.jpg\" width=\"16\" height=\"256\" border=\"0\"></td>
         <td><table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">
          <tr valign=\"top\">
                <td width=\"212\" height=\"46\" background=\"$LIBDIR/img/calendar/calendar_3.jpg\" valign=\"middle\">";

                    if ( $this->render_form ) {
                        $s .= " <table border=\"0\" cellspacing=\"0\" cellpadding=\"0\" width=\"100%\">
                                 <tr align=\"left\" valign=\"middle\">
                                   <td>&nbsp;&nbsp;<select name=\"calMM\" onchange='javascript:document.$this->formname.submit();'>";

                        for ( $m = 1; $m <= 12; $m++ ) {
                          $selected = ( $this->MM == $m ) ? " selected" : "";
                          $s .= "<option value=\"$m\"$selected>" . date("F", mktime(0,0,0,$m,1,2000)) . "</option>\n";
                        }

                        $s .= "</select>
                                   </td>
                                   <td align=right>
                                     <select name=\"calYY\" onchange='javascript:document.$this->formname.submit();'>";

                        for ($y = $this->startyear; $y <= $this->endyear; $y++) {
                          $selected = ($this->YY == $y) ? " selected" : "";
                          $s .= "<option value=\"$y\"$selected>$y</option>\n";
                        }
                        $s .= "</select>&nbsp;&nbsp;
                                   </td>
                                 </tr>
                               </table>";
                    }
                    else {
                        $s .= "<p style=\"font-size:x-small;letter-spacing:+0.5pt;font-family:Arial, Helvetica, sans-serif;\" align=\"center\">";
                        $s .=  date("l, jS of F Y", mktime(0,0,0,$this->MM,$this->DD,$this->YY)) . "</p>";
                    }
         $s .= "</td>
          </tr>
          <tr valign=\"top\">
                <td><img name=\"calendar_5\" src=\"$LIBDIR/img/calendar/calendar_5.gif\" width=\"212\" height=\"14\" border=\"0\"></td>
          </tr>
          <tr valign=\"top\">
                <td width=\"212\" height=\"158\" background=\"$LIBDIR/img/calendar/calendar_6.jpg\" align=\"center\" valign=\"middle\">
                      <table border=\"0\" cellspacing=\"0\" cellpadding=\"0\" height=\"150\">";

                     for ( $w = 0; $w < $this->weeks; $w++ ) {
                       $s .= "<tr align=\"center\" valign=\"middle\">";
                       for ( $dow = 1; $dow <= $this->days; $dow++ ) {
                         $image = $dom[$dow + $this->days*$w];
                         $s .= "<td height=\"25\" width=\"27\">";
                         if ( $image ) {
                           $title = date("l jS of F Y", mktime(0,0,0,$this->MM,$image,$this->YY));
                           if ( $this->DD == $image ) {
                             $inverted = "-inverted";
                           }
                           else {
                             $inverted = "";
                           }
                           if ($this->days_clickable) {
                              if ( $this->render_form ) {
                                $s .= "<input type=\"image\" src=\"$LIBDIR/img/calendar/$image$inverted.gif\" width=\"27\" height=\"25\" border=\"0\" ";
                                $s .= " name=\"calDD$image\" title=\"$title\">";
                              } else {
                                $s .= "<a href=\"" . $this->posturl . "calDD=$image&amp;calMM=$this->MM&amp;calYY=$this->YY\">";
                                $s .= "<img src=\"$LIBDIR/img/calendar/$image$inverted.gif\" width=\"27\" height=\"25\" border=\"0\" title=\"$title\" alt=\"\">";
                                $s .= "</a>";
                              }
                           }
                           else {
                              $s .= "<img src=\"$LIBDIR/img/calendar/$image$inverted.gif\" width=\"27\" height=\"25\" border=\"0\" title=\"$title\" alt=\"\">";
                           }
                         }
                         else {
                           $s .= "<img src=\"$LIBDIR/img/calendar/$image.gif\" width=\"27\" height=\"25\" border=\"0\">";
                         }
                         $s .= "</td>\n";
                       }
                     $s .= "</tr>\n";
                   }

               $s .= "</table>
                    </td>
              </tr>
                  <tr valign=\"top\">
                    <td>
                  <table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">
                    <tr valign=\"top\">
                      <td>
                        <table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">
                          <tr valign=\"top\">
                            <td><img name=\"cal_7_1\" src=\"$LIBDIR/img/calendar/cal_7_1.gif\" width=\"68\" height=\"10\" border=\"0\"></td>
                          </tr>
                          <tr valign=\"top\">
                            <td>
                              <table border=\"0\" cellpadding=\"0\" cellspacing=\"0\" width=\"68\">
                                <tr valign=\"top\">
                                  <td>";

                                    if ( $this->render_form ) {
                                      $s .= "<input type=\"image\" src=\"$LIBDIR/img/calendar/cal_7_3.gif\" width=\"8\" height=\"16\" border=\"0\" ";
                                      $s .= "  name=\"calMM" . ($this->MM - 1) . "\" title=\"Previous month\">";
                                    }
                                    else {
                                      $s .= "<a href=\"" . $this->posturl . "calDD=$this->DD&amp;calMM=" . ( $this->MM - 1 ) . "&amp;calYY=$this->YY\">";
                                      $s .= "<img src=\"$LIBDIR/img/calendar/cal_7_3.gif\" width=\"8\" height=\"16\" border=\"0\" title=\"Previous month\">";
                                      $s .= "</a>";
                                    }
                           $s .= "</td>
                                  <td><img name=\"cal_7_4\" src=\"$LIBDIR/img/calendar/cal_7_4.gif\" width=\"50\" height=\"16\" border=\"0\"></td>
                                  <td>";

                                    if ( $this->render_form ) {
                                      $s .= "<input type=\"image\" src=\"$LIBDIR/img/calendar/cal_7_5.gif\" width=\"8\" height=\"16\" border=\"0\" ";
                                      $s .= "  name=\"calMM" . ( $this->MM + 1 ) . "\" title=\"Next month\">";
                                    }
                                    else {
                                      $s .= "<a href=\"" . $this->posturl . "calDD=$this->DD&amp;calMM=" . ( $this->MM + 1 ) . "&amp;calYY=$this->YY\">";
                                      $s .= "<img src=\"$LIBDIR/img/calendar/cal_7_5.gif\" width=\"8\" height=\"16\" border=\"0\" title=\"Next month\">";
                                      $s .= "</a>";
                                    }

                           $s .= "</td>
                                </tr>
                              </table>
                            </td>
                          </tr>
                          <tr valign=\"top\">
                            <td><img name=\"cal_7_6\" src=\"$LIBDIR/img/calendar/cal_7_6.gif\" width=\"68\" height=\"12\" border=\"0\"></td>
                          </tr>
                        </table>
                      </td>
                      <td><img name=\"cal_7_2\" src=\"$LIBDIR/img/calendar/cal_7_2.gif\" width=\"144\" height=\"37\" border=\"0\"></td>
                    </tr>
                  </table>
                </td>
                  </tr>
        </table></td>
            <td><img name=\"calendar_4\" src=\"$LIBDIR/img/calendar/calendar_4.jpg\" width=\"16\" height=\"256\" border=\"0\"></td>
        </tr>
      </table></td>
      </tr>
    </table>";

    if ($this->render_form && !$this->render_subform) {
      $s .= "</form>\n";
    }
    return $s;
  } // html

}
?>